# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements non-interface-specific configurations common code for
# IPv4 and IPv6.
#-------------------------------------------------------------------------------

from itertools import chain
import Tac
from TypeFuture import TacLazyType
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin import VrfCli
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from IpLibConsts import ALL_VRF_NAME

OptimizeProfileType = TacLazyType( 'Routing::Hardware::OptimizeProfileType' )
restartL3AgentWarnTemplate = (
      "Please restart layer 3 forwarding agent to ensure %s routes are " )

IPV4 = 4
IPV6 = 6

optimizeProfileToCliToken = {
   OptimizeProfileType.internet : 'internet',
   OptimizeProfileType.urpfInternet : 'urpf-internet',
}
cliTokenToOptimizeProfile = { y : x for x, y in optimizeProfileToCliToken.items() }

def ipv4RestartL3AgentWarnStr():
   return restartL3AgentWarnTemplate % 'IPv4'

def ipv6RestartL3AgentWarnStr():
   return restartL3AgentWarnTemplate % 'IPv6'

def setRoutesInExactMatch( mode, ipVersion, rhc, rhs, prefixLenSet=None,
                           expandLenSet=None, compressLenSet=None, urpf=False,
                           vrfs=None ):
   configChanged = False
   if ipVersion == IPV4:
      prefixLenSupportedInExactMatch = rhs.prefixLenSupportedInExactMatch
      numberOfPrefixLengthsInExactMatch = rhs.numberOfPrefixLengthsInExactMatch
      numberOfNonNibblePrefixLengthsInExactMatch = \
            rhs.numberOfNonNibblePrefixLengthsInExactMatch
      numberOfCompressedPrefixLengthsInExactMatch = \
            rhs.numberOfCompressedPrefixLengthsInExactMatch
      numberOfPrefixLengthInExactMatchWithUrpf = \
            rhs.numberOfPrefixLengthInExactMatchWithUrpf
      maxCompressionPrefixLenRange = rhs.maxCompressionPrefixLenRange
      maxExpandPrefixLenRange = rhs.maxExpandPrefixLenRange
      numberOfNonNibblePrefixLengthsInExactMatchWithUrpf = \
            rhs.numberOfNonNibblePrefixLengthsInExactMatchWithUrpf
      # Keep track of configuration changes that will require an agent restart.
      # If configChanged, we dont really care about the expandConfigChanged.
      # In future, we can eliminate the restart requirement if only expansion
      # configuration changes.
      configChanged = ( urpf != rhc.urpfExactMatchEnabled )
      rhc.urpfExactMatchEnabled = bool( urpf )
      msg = ipv4RestartL3AgentWarnStr() + "optimized"
   else:
      prefixLenSupportedInExactMatch = rhs.prefixLenSupportedInExactMatch
      numberOfPrefixLengthsInExactMatch = rhs.numberOfPrefixLengthsInExactMatch
      numberOfNonNibblePrefixLengthsInExactMatch = 0
      numberOfCompressedPrefixLengthsInExactMatch = 0
      numberOfPrefixLengthInExactMatchWithUrpf = 0
      maxCompressionPrefixLenRange = 0
      maxExpandPrefixLenRange = 0
      numberOfNonNibblePrefixLengthsInExactMatchWithUrpf = 0
      msg = ipv6RestartL3AgentWarnStr() + "optimized"

   configuredLengths = list( map( int, prefixLenSet or [] ) )
   expandLengths = list( map( int, expandLenSet ) ) if expandLenSet else []
   compressLengths = list( map( int, compressLenSet ) ) if compressLenSet else []
   supportedLengths = list( prefixLenSupportedInExactMatch )
   startupConfig = mode.session_.startupConfig()

   if not startupConfig and ( len( configuredLengths ) >
          numberOfPrefixLengthsInExactMatch ):
      mode.addError( "This platform supports %d lookups in the exact match table" %
                     numberOfPrefixLengthsInExactMatch )
      return
   numberOfNonNibblePrefixLengths = 0
   for configuredLength in configuredLengths:
      if configuredLength % 4:
         numberOfNonNibblePrefixLengths += 1
   if not startupConfig and ( numberOfNonNibblePrefixLengths >
         numberOfNonNibblePrefixLengthsInExactMatch ):
      mode.addError( "This platform supports %d non-nibble aligned lookups in the "
                     "exact match table" % \
                           numberOfNonNibblePrefixLengthsInExactMatch )
      return
   if not startupConfig and ( len( compressLengths ) >
         numberOfCompressedPrefixLengthsInExactMatch ):
      mode.addError( "This platform supports %d compressed prefix lengths in the "
                     "exact match table" % \
                           numberOfCompressedPrefixLengthsInExactMatch )
      return

   if urpf:
      if not startupConfig and ( len( configuredLengths ) >
           numberOfPrefixLengthInExactMatchWithUrpf ):
         mode.addError( "This platform supports %d lookups in the exact match table"
                        " along with Unicast RPF" %
                     numberOfPrefixLengthInExactMatchWithUrpf )
         return
      if numberOfNonNibblePrefixLengths:
         mode.addError( "This platform supports %d non-nibble aligned lookups in the"
               " exact match table along with Unicast RPF" %
                        numberOfNonNibblePrefixLengthsInExactMatchWithUrpf )
         return

   # Check value of the max prefix-expansion range in routingHwStatus
   # allow prefix-lengths for expansion in the range of
   # < REM-prefix-length - val> and <REM-prefix-length - 1 >
   val = maxExpandPrefixLenRange + 1
   for length in expandLengths:
      prefixLenAllowed = False
      for i in range( 1, val ):
         if length + i in configuredLengths:
            prefixLenAllowed = True
            break
      if not prefixLenAllowed:
         mode.addError( "Cannot expand prefix-length %d. No prefix-length in "
                        "range %d to %d is optimized." %
                        ( length, length + 1, length + val - 1 ) )
         return
      if length in configuredLengths:
         # We do not allow a prefix-length in the expanded list if it is already
         # present in the prefix list.
         # For e.g., /23 cannot be expanded at the same time it is optimized:
         # ip hardware fib optimize prefix-length 23 24 expand 22 23
         mode.addError( "Cannot expand prefix-length %d. It is already optimized."
                        % length )
         return
   val = maxCompressionPrefixLenRange + 1
   for length in compressLengths:
      prefixLenAllowed = False
      for i in range( 1, val ):
         if length - i in configuredLengths:
            prefixLenAllowed = True
            break
      if not startupConfig and not prefixLenAllowed:
         mode.addError( "Cannot compress prefix-length %d. Prefix-length %d is "
                        "not optimized." %
                        ( length, length - val + 1 ) )
         return
      if length in configuredLengths or length in expandLengths:
         # We do not allow a prefix-length in the compressed list if it is already
         # present in the prefix list or expanded list.
         # For e.g., /23 cannot be compressed at the same time it is optimized:
         # ip hardware fib optimize prefix-length 20 23 compress 23
         # For e.g., /21 cannot be compressed at the same time it is expanded:
         # ip hardware fib optimize prefix-length 20 23 expand 21 22 compress 21
         mode.addError( "Cannot compress prefix-length %d. It is already"
                        " optimized." % length )
         return

   if not startupConfig and compressLengths:
      minCompressLengthConfigured = min( compressLengths )
      # We are guaranteed to find a matching REM prefix length here because we
      # have already validated that we have a matching REM prefix length in
      # the lines above.
      matchingRemPrefixLength = [
         x for x in configuredLengths
         if x in range(
            minCompressLengthConfigured - rhs.maxCompressionPrefixLenRange,
            minCompressLengthConfigured ) ][ 0 ]
      maxCompressLengthPossible = matchingRemPrefixLength + \
                                  rhs.maxCompressionPrefixLenRange

      # The CLI command should throw an error if -
      # - compression configuration is not consecutive.
      # - does not include the max compression length possible.
      missingPrefixLengths = (
         set( range( minCompressLengthConfigured, maxCompressLengthPossible + 1 ) ) -
         set( compressLengths ) )
      if missingPrefixLengths:
         mode.addError(
            "Cannot compress prefix-length(s) %s. "
            "Prefix-length(s) %s must also be compressed." %
            ( ",".join( str( x ) for x in sorted( compressLengths ) ),
              ",".join( str( x ) for x in sorted( missingPrefixLengths ) ) ) )
         return

   # Disable optimization with profiles if enabled
   if vrfs:
      for vrf in vrfs:
         currentProfile = rhc.vrfOptimizeProfileName.get( vrf )
         configChanged |= currentProfile is not None
         del rhc.vrfOptimizeProfileName[ vrf ]
         vrfPrefixLensColl = rhc.vrfPrefixLensInExactMatch
         prefixLensInExactMatch = vrfPrefixLensColl.newMember( vrf )
         configChanged |= updatePrefixLensInExactMatch(
            ipVersion, rhs, prefixLensInExactMatch, supportedLengths,
            configuredLengths, expandLengths, compressLengths, startupConfig )
   else:
      if rhc.globalOptimizeProfileName:
         rhc.globalOptimizeProfileName = ''
         configChanged = True
      prefixLensInExactMatch = rhc.globalPrefixLensInExactMatch
      configChanged |= updatePrefixLensInExactMatch(
         ipVersion, rhs, prefixLensInExactMatch, supportedLengths,
         configuredLengths, expandLengths, compressLengths, startupConfig )

   if configChanged:
      mode.addWarning( msg )

def updatePrefixLensInExactMatch( ipVersion, rhs, prefixLensInExactMatch,
                                  supportedLengths, configuredLengths,
                                  expandLengths, compressLengths, startupConfig ):
   # Add all the prefix-lengths in current REM configuration, if valid
   # If this was not enabled in previous config, remConfigChanged = True
   remConfigChanged = False
   remPrefixLengths = set(
         chain( configuredLengths, expandLengths, compressLengths ) )
   for length in configuredLengths:
      if not startupConfig and length not in supportedLengths:
         continue
      if length not in prefixLensInExactMatch.prefixLen:
         remConfigChanged = True
      else:
         prevConfig = prefixLensInExactMatch.prefixLen[ length ] == length
         if not prevConfig:
            remConfigChanged = True
      prefixLensInExactMatch.prefixLen[ length ] = length

   # Check all the previously configured REM prefix-lengths. If not
   # set in current configuration, delete from config and
   # remConfigChanged = true
   for length in prefixLensInExactMatch.prefixLen:
      if prefixLensInExactMatch.prefixLen[ length ] == length:
         if length not in configuredLengths:
            del prefixLensInExactMatch.prefixLen[ length ]
            remConfigChanged = True

   # If any REM configuration changed, clear all the existing expansion
   # configuration, otherwise check and delete the values not configured now.
   if remConfigChanged:
      for length in prefixLensInExactMatch.prefixLen:
         if prefixLensInExactMatch.prefixLen[ length ] != length:
            del prefixLensInExactMatch.prefixLen[ length ]
   else:
      for length, remLength in \
            prefixLensInExactMatch.prefixLen.items():
         if length not in remPrefixLengths:
            del prefixLensInExactMatch.prefixLen[ length ]
            remConfigChanged = True

   # Add all the prefix-lengths in current expand configuration. If this was not
   # enabled in previous config, expandConfigChanged = True
   for length in expandLengths:
      for remLength in sorted( configuredLengths ):
         if remLength < length:
            continue
         if length not in prefixLensInExactMatch.prefixLen:
            remConfigChanged = True
         prefixLensInExactMatch.prefixLen[ length ] = remLength
         break

   # Add all the prefix-lengths in current compress configuration. If this was not
   # enabled in previous config, remConfigChanged = True
   # NOTE: We cannot use rhs.maxCompressionPrefixLenRange here because it is
   # not guaranteed to be set when the switch is booting up (startup-config).
   for length in compressLengths:
      for remLength in sorted( configuredLengths, reverse=True ):
         if remLength >= length:
            continue
         if length not in prefixLensInExactMatch.prefixLen:
            remConfigChanged = True
         prefixLensInExactMatch.prefixLen[ length ] = remLength
         break

   return remConfigChanged

def unsetRoutesInExactMatch( mode, ipVersion, rhc, rhs, vrfs=None,
                             clearPrefixLengthConfig=True,
                             clearPrefixesProfileConfig=True ):
   configChanged = False
   if ipVersion == IPV4:
      configChanged = rhc.urpfExactMatchEnabled
      # Keep track of configuration changes that will require an agent restart.
      # In future, we can eliminate the restart requirement if only expansion
      # configuration changes.
      rhc.urpfExactMatchEnabled = False
      msg = ipv4RestartL3AgentWarnStr() + "not optimized"
   else:
      msg = ipv6RestartL3AgentWarnStr() + "not optimized"

   # Disable optimization with profiles if enabled
   if vrfs:
      for vrf in vrfs:
         if clearPrefixesProfileConfig:
            profileName = rhc.vrfOptimizeProfileName.get( vrf )
            configChanged |= profileName is not None
            del rhc.vrfOptimizeProfileName[ vrf ]

         if clearPrefixLengthConfig:
            vrfPrefixLensColl = rhc.vrfPrefixLensInExactMatch
            vrfPrefixLenConfig = vrfPrefixLensColl.get( vrf, None )
            configChanged |= vrfPrefixLenConfig is not None
            del vrfPrefixLensColl[ vrf ]
   else:
      if clearPrefixesProfileConfig:
         configChanged |= bool( rhc.globalOptimizeProfileName )
         rhc.globalOptimizeProfileName = ''

      if clearPrefixLengthConfig:
         prefixLensInExactMatch = rhc.globalPrefixLensInExactMatch
         configChanged |= bool( prefixLensInExactMatch.prefixLen )
         prefixLensInExactMatch.prefixLen.clear()

   if configChanged:
      mode.addWarning( msg )

def enableOptimizePrefixes( mode, ipVersion, rhc, profileType='', vrfs=None ):
   if ipVersion == 4:
      rhc.urpfExactMatchEnabled = ( profileType == OptimizeProfileType.urpfInternet )
      msg = ipv4RestartL3AgentWarnStr() + "optimized"
   else:
      msg = ipv6RestartL3AgentWarnStr() + "optimized"

   configChanged = False
   if vrfs:
      for vrf in vrfs:
         prefixLens = rhc.vrfPrefixLensInExactMatch.get( vrf )
         configChanged |= prefixLens is not None
         del rhc.vrfPrefixLensInExactMatch[ vrf ]
         currentProfileType = rhc.vrfOptimizeProfileName.get( vrf )
         configChanged |= ( currentProfileType != profileType )
         if configChanged:
            rhc.vrfOptimizeProfileName[ vrf ] = profileType
   else:
      prefixLensInExactMatch = rhc.globalPrefixLensInExactMatch
      prefixLensInExactMatch.prefixLen.clear()
      configChanged |= (
            rhc.globalOptimizeProfileName != profileType )
      rhc.globalOptimizeProfileName = profileType

   if configChanged:
      mode.addWarning( msg )

def mergedDict( dict1, dict2 ):
   # A function that works in both python2 and python3.
   dict3 = dict1.copy()
   dict3.update( dict2 )
   return dict3

def supportedProfileType( mode, routingHwStatus ):
   supportedProfiles = {}

   helpMsgForProfileType = {
      OptimizeProfileType.internet : 'Optimize prefixes for Internet',
      OptimizeProfileType.urpfInternet : 'Optimize prefixes for Internet with URPF',
   }

   startupConfig = mode.session_.startupConfig()
   optimizeProfileSettings = []
   if startupConfig:
      # It is not possible to get the list of supported profiles when coming
      # here as part of startup config because platform wouldn't have published
      # the supported profiles yet. Accept all profiles that we know when this
      # is coming from startup config.
      optimizeProfileSettings = optimizeProfileToCliToken.keys()
   elif routingHwStatus.optimizeProfileSettingList:
      optimizeProfileSettings = \
         routingHwStatus.optimizeProfileSettingList.optimizeProfileSetting.keys()

   for optimizeProfileSetting in optimizeProfileSettings:
      cliToken = optimizeProfileToCliToken[ optimizeProfileSetting ]
      helpMsg = helpMsgForProfileType[ optimizeProfileSetting ]
      supportedProfiles[ cliToken ] = helpMsg
   return supportedProfiles

def vrfFilterWarningCheck( mode, vrfName, prefix ):
   if ( getEffectiveProtocolModel( mode ) == ProtoAgentModel.multiAgent and
        vrfName is not None and
        prefix is None and
        VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName ) != ALL_VRF_NAME ):
      mode.addWarning( 'VRF filtering is not supported.' )
