# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements interface-specific IP configuration.
#-------------------------------------------------------------------------------
'''Commands supported in interface config mode'''

import CliParser
import Arnet
from CliPlugin import IntfCli
from CliPlugin import IpAddrMatcher
from CliPlugin import IraCommonCli
from CliPlugin import IraIp6IntfCli
from CliPlugin import IraRouteCommon
from CliPlugin import IraUrpfCli
from CliPlugin.IraIpModel import InterfaceAddress
from CliPlugin.IraIpModel import IpAddress
from CliPlugin.VrfCli import vrfExists, vrfMatcher, DEFAULT_VRF
import CliToken.Ip, CliToken.Verify
import LazyMount
import Tracing
import Tac
import CliExtensions
import CliCommand
import CliMatcher
import ConfigMount
import IpLibTypes
import Toggles.IraToggleLib
from IpLibConsts import MAX_SECONDARIES_PER_INTERFACE
from RoutingIntfUtils import isManagement, mgmtPrefix

zeroIpAddress = Arnet.AddrWithMask( "0.0.0.0/0" )

af = Tac.Type( 'Arnet::AddressFamily' )
ipv4AddrSource = IpLibTypes.addrSource( af.ipv4 )
ipv6AddrSource = IpLibTypes.addrSource( af.ipv6 )
tristateBoolEnum = Tac.Type( "Ip::TristateBool" )

ipStatusDir = None
ipConfigDir = None
dynIpConfigDir = None
l3ConfigDir = None
ip6ConfigDir = None
allIntfStatusDir = None
allVrfConfig = None
routingHardwareStatus = None

ip = IraRouteCommon.Ip4()
routing = IraRouteCommon.routing( ip )

# the CLI hook for VRFs is used to allow individual features to block
# VRF config on interfaces on which it is otherwise supported,
# presumably due to some incompatible feature enabled on the
# interface.  Ira calls all hooks and ANDs the results, if any are
# False, the config is blocked.  This function is called as:
#
# (accept, message) = hook( intfname, oldVrf, newVrf, vrfDelete )
#
# intfName is the interface being changed.  oldVrf and newVrf are the
# names of the relevant vrfs, and vrfDelete indicates whether or not
# this hook is being called because of an attempt to delete a VRF.
#
# accept must be a boolean, message is a text string giving the specific
# reason why the config is blocked.  The message is required.

canSetVrfHook = CliExtensions.CliHook()

#-------------------------------------------------------------------------------
# Adds IP-specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class _IpIntfModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self, mode )
      self.ip = IpIntf( mode.intf, mode.sysdbRoot, createIfMissing=True )
      # XXX Shouldn't really need this - need a Vrf object instead

class IpIntfConfigModelet( _IpIntfModelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.routingSupported()

   # In the industry-standard CLI, the "ip address" commands work as follows.
   #
   # -  DHCP and a statically-configured address are mutually exclusive.
   #    -  Enabling DHCP (with "ip address dhcp") disables a previously-configured
   #       static IP address, and also forgets about that address so that a
   #       subsequent "no ip address dhcp" command leaves the interface with IP
   #       processing disabled.
   #    -  Configuring a static IP address (with "ip address <addr> <mask>")
   #       disables DHCP.
   # -  The "no ip address" command disables IP processing on the interface, whether
   #    it is configured with a static address (which is forgotten) or for DHCP.
   # -  The "no ip address <addr> <mask>" and "no ip address dhcp" commands behave
   #    identically to the "no ip address" command, except that they are rejected as
   #    invalid unless the interface is currently configured in agreement with the
   #    configuration specified in the command.  That is, "no ip address dhcp" only
   #    works if the interface is currently configured for DHCP, and
   #    "no ip address 1.2.3.4 255.255.255.0" only works if the interface is
   #    currently configured with the static IP address 1.2.3.4/24.
   #
   # For simplicity, we treat "no ip address dhcp" and "no ip address <addr> <mask>"
   # identically to "no ip address", regardless of the current configuration of the
   # interface.

   def setIpAddr( self, args ):
      addressWithMask = args[ 'ADDR' ]
      Tracing.trace0( "Enabling IP address", addressWithMask, "on", self.ip.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                                self.ip.name, configType="IP" )
      self.noIpUnnumberedIntf( None )
      self.ip.setIpAddr( addressWithMask )

   def noIpAddrCli( self, args ):
      self.ip.noIpAddr( None )

   def noIpAddr( self, message=None ):
      self.ip.noIpAddr( message )

   def noVirtualIpAddr( self, message=None ):
      self.ip.noVirtualIpAddr( message )

   def noPrimaryIpAddr( self, args ):
      addressWithMask = args[ 'ADDR' ]
      self.ip.noPrimaryIpAddr( addressWithMask )

   def setSecondaryIpAddr( self, args ):
      addressWithMask = args[ 'ADDR' ]
      Tracing.trace0( "Enabling SecondaryIP address", addressWithMask,
                      "on", self.ip.name )
      self.ip.setSecondaryIpAddr( addressWithMask )

   def noSecondaryIpAddr( self, args ):
      addressWithMask = args[ 'ADDR' ]
      Tracing.trace0( "Disabling SecondaryIP address", addressWithMask,
                      "on", self.ip.name )
      self.ip.noSecondaryIpAddr( addressWithMask )

   def setIpUnnumberedIntf( self, args ):
      intf = args[ 'INTF' ]
      Tracing.trace0( "Enabling unnumbered address borrowing from ", intf.name,
                      "on", self.ip.name )

      if intf.name == self.ip.name:
         self.mode.addError( "Interface cannot refer to itself" )
         return False
      if not intf.name.startswith( "Loopback" ):
         self.mode.addWarning( "Lending interface should be a Loopback interface" )
      self.ip.noIpAddr()
      self.ip.setIpUnnumberedIntf( intf )
      return True

   def noIpUnnumberedIntf( self, args ):
      Tracing.trace0( "Disabling unnumbered address on", self.ip.name )
      self.ip.noIpUnnumberedIntf()

   def setProxyArp( self, args ):
      cfg = self.ip.config()
      if CliCommand.isDefaultCmd( args ):
         value = cfg.proxyArpEnabledDefault
      else:
         value = not CliCommand.isNoCmd( args )
      cfg.proxyArpEnabled = value
      Tracing.trace0( "Set", self.ip.name, "enable proxy ARP to", value )
      if value:
         allowDefaultRoute = 'allow-default' in args
         cfg.proxyArpAllowDefaultRoute = allowDefaultRoute
         Tracing.trace0( "Set", self.ip.name, "proxy ARP allow default to",
                         allowDefaultRoute )
      else:
         cfg.proxyArpAllowDefaultRoute = cfg.proxyArpAllowDefaultRouteDefault

   def setLocalProxyArp( self, args ):
      cfg = self.ip.config()
      if CliCommand.isDefaultCmd( args ):
         value = cfg.localProxyArpEnabledDefault
      else:
         value = not CliCommand.isNoCmd( args )
      cfg.localProxyArpEnabled = value
      Tracing.trace0( "Set", self.ip.name, "enable local proxy ARP to", value )

   def setGratuitousArpAccepted( self, args ):
      cfg = self.ip.config()
      if CliCommand.isDefaultCmd( args ):
         value = cfg.gratuitousArpAcceptedDefault
      else:
         value = not CliCommand.isNoCmd( args )
      cfg.gratuitousArpAccepted = value
      Tracing.trace0( "Set", self.ip.name, "gratuitous arp accepted to", value )

   def setArpProxyReplyVirtualRouterMacAddress( self, args ):
      cfg = self.ip.config()
      if CliCommand.isDefaultCmd( args ):
         value = cfg.proxyMacVirtualMacEnabledDefault
      else:
         value = not CliCommand.isNoCmd( args )
      cfg.proxyMacVirtualMacEnabled = value
      Tracing.trace0( "Set", self.ip.name,
                      "enable ARP proxy reply virtual router MAC address to", value )

   def setVrf( self, args ):
      vrfName = args[ 'VRF_NAME' ]
      # Update the vrf forwarding command format.
      if self.mode.session.skipConfigCheck():
         self.ip.l3Config().vrf = vrfName
         return

      if not self.ip.l3Config().vrf == vrfName:
         if not vrfExists( vrfName ):
            self.mode.addError( "VRF %s not configured." % vrfName )
            return

         # check for interface support for VRFs
         if not self.mode.intf.vrfSupported():
            self.mode.addError( "VRF configuration is not supported on "
                                "this interface." )
            return

         # now check for incompatible features on the interfaces
         for hook in canSetVrfHook.extensions():
            (accept, hookMsg) = hook( self.ip.config().intfId,
                                      oldVrf=self.ip.l3Config().vrf,
                                      newVrf=vrfName, vrfDelete=False )
            if hookMsg:
               if accept:
                  self.mode.addWarning( hookMsg )
               else:
                  self.mode.addError( hookMsg )
            if not accept:
               return

         msg = 'Interface %s IP address %s removed due to enabling VRF ' + vrfName
         # note that 'noIpAddr' can *fail* if other code is blocking
         # the removal of config through the canDeleteIntfIpHook hook.
         # currently this is only used by Mlag, which should have also
         # registered through the canSetVrfHook above, so we should be OK.
         self.noIpAddr( msg )
         self.noIp6Addr( msg )
         msg = 'Interface %s virtual IP address %s removed ' + \
            'due to enabling VRF ' + vrfName
         self.noVirtualIpAddr( msg )
         #!!! do i really need to create the corresponding ip6 intf here?
         #!!! just doin this to make VrfIntfStatusTest.py pass right now
         IraIp6IntfCli.Ip6Intf( self.mode.intf, self.mode.sysdbRoot,
                                createIfMissing=True ).config()

         IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                                   self.ip.name, configType='VRF' )
         self.ip.l3Config().vrf = vrfName

         Tracing.trace0( "Set", self.ip.name, "VRF to", vrfName )

   def noVrf( self, args ):
      vrfName = args.get( 'VRF_NAME' )
      configVrf = self.ip.l3Config().vrf
      if vrfName not in ( None, configVrf ):
         self.mode.addWarning( "Interface %s not in VRF %s" %
                               ( self.ip.name, vrfName ) )
      elif configVrf:
         # as above, check for incompatible features on the interfaces
         for hook in canSetVrfHook.extensions():
            (accept, hookMsg) = hook( self.ip.config().intfId,
                                      oldVrf=self.ip.l3Config().vrf,
                                      newVrf=DEFAULT_VRF, vrfDelete=False )
            if hookMsg:
               if accept:
                  self.mode.addWarning( hookMsg )
               else:
                  self.mode.addError( hookMsg )
            if not accept:
               return

         msg = 'Interface %s IP address %s removed due to disabling VRF'
         self.noIpAddr( msg )
         self.noIp6Addr( msg )
         msg = 'Interface %s virtual IP address %s removed due to disabling VRF'
         self.noVirtualIpAddr( msg )
         #!!!do i really need to create the corresponding ip6Intf config here?
         IraIp6IntfCli.Ip6Intf( self.mode.intf,
                                self.mode.sysdbRoot ).config()
         Tracing.trace0( "Set", self.ip.name, "VRF to none" )
         self.ip.l3Config().vrf = DEFAULT_VRF

   def noIp6Addr( self, msg ):
      # Clear out the IPv6 addresses and the address source.
      # Ick, all the infra code for doing this is in IraIp6IntfCli.py,
      # and we can't call it from here.
      # Perhaps we need to factor the Vrf CLI out into its own intf modelet?
      if self.ip.config().intfId in ip6ConfigDir.intf:
         ip6IntfConfig = ip6ConfigDir.intf[ self.ip.config().intfId ]
         ip6IntfConfig.addrSource = ipv6AddrSource.manual
         for a in ip6IntfConfig.addr:
            self.mode.addWarning( msg % ( self.ip.config().intfId, a ) )
            del ip6IntfConfig.addr[ a ]

   def setUrpf( self, args ):
      urpfMode = ( IraUrpfCli.urpfMode.loose if 'any' in args
                                             else IraUrpfCli.urpfMode.strict )
      allowDefault = 'allow-default' in args or \
                     ( urpfMode == IraUrpfCli.urpfMode.loose and
                       'non-default' not in args )
      IraUrpfCli.doUrpf( self.ip, False, urpfMode,
                         allowDefault, args.get( 'ACL_NAME' ),
                         args.get( 'VRF_NAME' ) )

   def noUrpf( self, args ):
      IraUrpfCli.doUrpf( self.ip, no=True )

   def attachedRoutes( self, args ):
      self.ip.config().attachedRoutes = True

   def defaultAttachedRoutes( self, args ):
      self.ip.config().attachedRoutes = True

   def noAttachedRoutes( self, args ):
      self.ip.config().attachedRoutes = False

   def setIpRoutingAddrRequired( self, args ):
      self.ip.config().dpRoutingAddressRequiredDisabled = True
      # ipIntfConfig::forwarding for 00cli input is True
      # for all interfaces except Management interface. Setting
      # of this knob will also enable forwarding on Management
      # interfaces as we required addressless forwarding enabled
      # even on Management interfaces when this knob is configured
      if isManagement( self.ip.intf_.name ):
         self.ip.config().forwarding = True

   def noIpRoutingAddrRequired( self, args ):
      self.ip.config().dpRoutingAddressRequiredDisabled = False
      # If this interface is Management interface, reset ipIntfConfig::forwarding
      # to False as that is the default for 00cli input
      if isManagement( self.ip.intf_.name ):
         self.ip.config().forwarding = False

#-------------------------------------------------------------------------------
# Associate the IpIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( IpIntfConfigModelet )

#--------------------------------------------------------------------------------
# [ no | default ] ip address ADDR
#--------------------------------------------------------------------------------
addrWithMaskMatcher = IpAddrMatcher.ipAddrWithMaskExpr( 'IP address',
                                                        'Subnet\'s mask value',
                                                      'IP address with mask length' )

class IpAddressAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'ip address ADDR'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'address': 'Set IP address of an interface',
      'ADDR': addrWithMaskMatcher,
   }

   handler = IpIntfConfigModelet.setIpAddr
   noOrDefaultHandler = IpIntfConfigModelet.noPrimaryIpAddr

IpIntfConfigModelet.addCommandClass( IpAddressAddrCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ip address
#--------------------------------------------------------------------------------
class IpAddressCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip address'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'address': 'Set IP address of an interface',
   }

   noOrDefaultHandler = IpIntfConfigModelet.noIpAddrCli

IpIntfConfigModelet.addCommandClass( IpAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip address ADDR secondary
#--------------------------------------------------------------------------------
class IpAddressSecondaryCmd( CliCommand.CliCommandClass ):
   syntax = 'ip address ADDR secondary'
   noOrDefaultSyntax = 'ip address ADDR secondary'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'address': 'Set IP address of an interface',
      'ADDR': addrWithMaskMatcher,
      'secondary': 'Specify that address is secondary',
   }

   handler = IpIntfConfigModelet.setSecondaryIpAddr
   noOrDefaultHandler = IpIntfConfigModelet.noSecondaryIpAddr

IpIntfConfigModelet.addCommandClass( IpAddressSecondaryCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip address unnumbered INTF
#--------------------------------------------------------------------------------
class IpAddressUnnumberedIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip address unnumbered INTF'
   noOrDefaultSyntax = 'ip address unnumbered ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'address': 'Set IP address of an interface',
      'unnumbered': 'Specify the interface with the address to borrow',
      'INTF': IntfCli.Intf.matcher,
   }

   handler = IpIntfConfigModelet.setIpUnnumberedIntf
   noOrDefaultHandler = IpIntfConfigModelet.noIpUnnumberedIntf

IpIntfConfigModelet.addCommandClass( IpAddressUnnumberedIntfCmd )

#-------------------------------------------------------------------------------
# The "[no] ip mtu" command.
#-------------------------------------------------------------------------------
# This command is now an alias for "[no] mtu" which sets the mtu in the
# intfConfig
class HiddenIpMtu( CliCommand.CliCommandClass ):
   syntax = "ip mtu MTU"
   noOrDefaultSyntax = "ip mtu ..."
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'mtu': IntfCli.mtuKw,
      'MTU': IntfCli.mtuValue,
   }
   hidden = True
   handler = IntfCli.RoutedIntfModelet.setMtu
   noOrDefaultHandler = IntfCli.RoutedIntfModelet.noMtu

IntfCli.RoutedIntfModelet.addCommandClass( HiddenIpMtu )

#--------------------------------------------------------------------------------
# [ no | default ] ip proxy-arp [allow-default]
#--------------------------------------------------------------------------------
class IpProxyArpCmd( CliCommand.CliCommandClass ):
   syntax = 'ip proxy-arp [allow-default]'
   noOrDefaultSyntax = 'ip proxy-arp...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'proxy-arp': 'Proxy ARP for off-subnet addresses',
      'allow-default': ( 'Always consider the default route if it is present in the'
                         ' routing table' ),
   }

   handler = IpIntfConfigModelet.setProxyArp
   noOrDefaultHandler = IpIntfConfigModelet.setProxyArp

IpIntfConfigModelet.addCommandClass( IpProxyArpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip local-proxy-arp
#--------------------------------------------------------------------------------
class IpLocalProxyArpCmd( CliCommand.CliCommandClass ):
   syntax = 'ip local-proxy-arp'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'local-proxy-arp': 'Proxy ARP for subnet local addresses',
   }

   handler = IpIntfConfigModelet.setLocalProxyArp
   noOrDefaultHandler = IpIntfConfigModelet.setLocalProxyArp

IpIntfConfigModelet.addCommandClass( IpLocalProxyArpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp gratuitous accept
#--------------------------------------------------------------------------------
class ArpGratuitousAcceptCmd( CliCommand.CliCommandClass ):
   syntax = 'arp gratuitous accept'
   noOrDefaultSyntax = syntax
   data = {
      'arp': 'ARP config commands',
      'gratuitous': 'Gratuitous ARP config commands',
      'accept': 'Accept gratuitous ARP',
   }

   handler = IpIntfConfigModelet.setGratuitousArpAccepted
   noOrDefaultHandler = IpIntfConfigModelet.setGratuitousArpAccepted

IpIntfConfigModelet.addCommandClass( ArpGratuitousAcceptCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip arp proxy reply virtual-router mac-address
# --------------------------------------------------------------------------------
class IpArpProxyReplyVirtualRouterMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'ip arp proxy reply virtual-router mac-address'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'arp': 'ARP configuration',
      'proxy': 'Proxy ARP and local proxy ARP commands',
      'reply': 'ARP reply commands',
      'virtual-router': 'Virtual router commands',
      'mac-address': 'Use virtual router MAC address',
   }

   handler = IpIntfConfigModelet.setArpProxyReplyVirtualRouterMacAddress
   noOrDefaultHandler = IpIntfConfigModelet.setArpProxyReplyVirtualRouterMacAddress

IpIntfConfigModelet.addCommandClass( IpArpProxyReplyVirtualRouterMacAddressCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ip verify unicast source reachable-via (
#     ( any [ non-default ] [ lookup-vrf VRF_NAME ] [ exception-list ACL_NAME ] ) |
#     ( rx [ allow-default ] [ exception-list ACL_NAME ] ) )
#--------------------------------------------------------------------------------
class UrpfCmd( CliCommand.CliCommandClass ):
   _nonDefaultKwStr = '[ non-default ] ' \
         if Toggles.IraToggleLib.toggleUrpfIgnoreDefaultRouteEnabled() else ''
   _splitVrfUrpfKwStr = '[ lookup-vrf VRF_NAME ] ' \
         if Toggles.IraToggleLib.toggleSplitVrfUrpfResolutionEnabled() else ''
   syntax = ( 'ip verify unicast source reachable-via ('
                        '( any %s%s[ exception-list ACL_NAME ] ) |'
                        '( rx [ allow-default ] [ exception-list ACL_NAME ] ) )' %
                        ( _nonDefaultKwStr, _splitVrfUrpfKwStr ) )
   noOrDefaultSyntax = 'ip verify unicast ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'verify': CliToken.Verify.verifyMatcherForConfigIf,
      'unicast': CliCommand.guardedKeyword( 'unicast',
         helpdesc='Enable RPF on unicast traffic',
         guard=IraUrpfCli.urpf4SupportedGuard ),
      'source': IraUrpfCli.urpfSourceMatcher,
      'reachable-via': IraUrpfCli.urpfReachMatcher,
      'any': IraUrpfCli.urpfLooseMatcher,
      'non-default': IraUrpfCli.urpfLooseNonDefaultMatcher,
      'lookup-vrf': IraUrpfCli.urpfLooseLookupVrfMatcher,
      'VRF_NAME': vrfMatcher,
      'rx': IraUrpfCli.urpfStrictMatcher,
      'allow-default': IraUrpfCli.urpfAllowDefaultMatcher,
      'exception-list': CliCommand.guardedKeyword( 'exception-list',
         helpdesc='Exception list',
         guard=IraUrpfCli.urpf4ExceptionSupportedGuard ),
      'ACL_NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: IraUrpfCli.getUserAclNames( [ 'ip' ] ),
         helpdesc='Exception Acl' )
   }

   handler = IpIntfConfigModelet.setUrpf
   noOrDefaultHandler = IpIntfConfigModelet.noUrpf

IpIntfConfigModelet.addCommandClass( UrpfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf [ forwarding ] VRF_NAME
#--------------------------------------------------------------------------------
class VrfVrfnameCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf [ forwarding ] VRF_NAME'
   noOrDefaultSyntax = 'vrf [ forwarding ] [ VRF_NAME ]'
   data = {
      'vrf': 'Specify VRF for interface',
      'forwarding': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'forwarding',
            helpdesc='Specify VRF for interface' ),
         deprecatedByCmd='vrf [VRF_ID]' ),
      'VRF_NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: allVrfConfig.vrf.members(),
         helpdesc='VRF name' ),
   }

   handler = IpIntfConfigModelet.setVrf
   noOrDefaultHandler = IpIntfConfigModelet.noVrf

IpIntfConfigModelet.addCommandClass( VrfVrfnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip attached-routes
#--------------------------------------------------------------------------------
class IpAttachedRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'ip attached-routes'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'attached-routes': 'Install attached routes in FIB',
   }

   handler = IpIntfConfigModelet.attachedRoutes
   noHandler = IpIntfConfigModelet.noAttachedRoutes
   defaultHandler = IpIntfConfigModelet.defaultAttachedRoutes

IpIntfConfigModelet.addCommandClass( IpAttachedRoutesCmd )

class DirectedBcastConfigModelet( _IpIntfModelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.routingSupported() and mode.intf.name.startswith(
                  ( "Ethernet", "Port-Channel", "Vlan", "Switch", "Test" ) ) )

   def enableDirectedBroadcast( self, args ):
      self.ip.config().directedBroadcastEnabled = True

   def disableDirectedBroadcast( self, args ):
      self.ip.config().directedBroadcastEnabled = False

IntfCli.IntfConfigMode.addModelet( DirectedBcastConfigModelet )

#--------------------------------------------------------------------------------
# [ no | default ] ip directed-broadcast
#--------------------------------------------------------------------------------
class IpDirectedBroadcastCmd( CliCommand.CliCommandClass ):
   syntax = 'ip directed-broadcast'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'directed-broadcast': ( 'Forward directed broadcast packets destined for '
                               'this interface' ),
   }

   handler = DirectedBcastConfigModelet.enableDirectedBroadcast
   noOrDefaultHandler = DirectedBcastConfigModelet.disableDirectedBroadcast

DirectedBcastConfigModelet.addCommandClass( IpDirectedBroadcastCmd )

#-------------------------------------------------------------------------------
# Network-layer interface extension for IPv4.
#-------------------------------------------------------------------------------

# canSetIntfIpHook allows other packages to check if a proposed ip
# address (with mask) is acceptable.  The hook takes an Arnet::IpAddrWithMask
# and an interface name as arguments and returns a pair [ b, str ] where b is
# True if the proposed address is acceptable, False otherwise, and str is an
# error or warning string suitable for display in the CLI.
canSetIntfIpHook = CliExtensions.CliHook()

# canDeleteIntfIpHook allows other packages to check if the ip addresses of a
# proposed interface can be deleted. The hook takes an interface name as argument
# and returns a pair [ b, str ] where b is True if the deletion of any ip address
# deletion is acceptable, False otherwise, and str is an error or warning string
# suitable for display in the CLI.
canDeleteIntfIpHook = CliExtensions.CliHook()

class IpIntf( IntfCli.IntfDependentBase ):
   emptyIntf = None
   emptyL3Intf = None
   emptyIpIntfStatus = None
   def __init__( self, intf, sysdbRoot, createIfMissing=False ):
      self.ipIntfStatuses = ipStatusDir.ipIntfStatus
      self.ipIntfStatus = None
      self.createIfMissing_ = createIfMissing
      self.mode = intf.mode_
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   name = property( lambda self: self.intf_.name )

   #----------------------------------------------------------------------------
   # Returns whether the Ip::IpIntfStatus object exists for this interface.
   #----------------------------------------------------------------------------
   def lookup( self ):
      # The IpIntf exists if an associated Ip::IpIntfStatus object exists. Since
      # Ira is the one responsible for creating the status object, there is a
      # window in which the ip intf has been configured in the cli, but
      # 'show interfaces' does not display the ip intf status.
      self.ipIntfStatus = self.ipIntfStatuses.get( self.intf_.name )
      return self.ipIntfStatus is not None

   #----------------------------------------------------------------------------
   # Returns the IpIntfConfig object for this interface, creating it if it does
   # not already exist.  Will also create L3Config as side effect
   #----------------------------------------------------------------------------
   def config( self ):
      ( ipConfig, l3Config ) = self.bothConfigs()
      assert ipConfig and l3Config
      return ipConfig

   #----------------------------------------------------------------------------
   # Returns the L3Config object for this interface, creating it if it does
   # not already exist.  Will also create IpIntfConfig as a side effect
   #----------------------------------------------------------------------------
   def l3Config( self ):
      ( ipConfig, l3Config ) = self.bothConfigs()
      assert ipConfig and l3Config
      return l3Config

   def bothConfigs( self ):
      l3c = l3ConfigDir.intfConfig.get( self.intf_.name )
      if not l3c:
         if self.createIfMissing_:
            l3c = l3ConfigDir.intfConfig.newMember( self.intf_.name )
         else:
            if not IpIntf.emptyL3Intf:
               l3ConfigDirTmp = Tac.newInstance( "L3::Intf::ConfigDir", "" )
               IpIntf.emptyL3Intf = l3ConfigDirTmp.intfConfig.newMember( "" )
            l3c = IpIntf.emptyL3Intf

      iic = ipConfigDir.ipIntfConfig.get( self.intf_.name )
      # If addrSource == dhcp, the IpIntfConfig at ip/input/dynIpConfig/00cli
      # will contain the dhcp-provided IP address. The rest of the attributes
      # of this object will be same as ip/config
      if not iic or iic.addrSource == 'dhcp':
         if self.createIfMissing_:
            iic = ipConfigDir.ipIntfConfig.newMember( self.intf_.name )
            iic.l3Config = l3c
            if isManagement( self.intf_.name ):
               iic.forwarding = False
         elif self.lookup():
            configSource = self.ipIntfStatus.configSource
            addresslessForwarding = self.ipIntfStatus.addresslessForwarding
            if configSource in dynIpConfigDir.entityPtr and \
                  configSource in dynIpConfigDir.entryState and \
                  dynIpConfigDir[ configSource ]:
               iic = dynIpConfigDir[ configSource ].ipIntfConfig.get(
                       self.intf_.name )
            elif configSource == '00cli' and list( ipConfigDir.ipIntfConfig ):
               iic = ipConfigDir.ipIntfConfig.get( self.intf_.name )
            elif addresslessForwarding != 'IsInvalid':
               if not IpIntf.emptyIntf:
                  configDir = Tac.newInstance( "Ip::Config", "" )
                  IpIntf.emptyIntf = configDir.ipIntfConfig.newMember( "" )
                  IpIntf.emptyIntf.l3Config = l3c
               iic = IpIntf.emptyIntf
         if not iic:
            if not IpIntf.emptyIntf:
               configDir = Tac.newInstance( "Ip::Config", "" )
               IpIntf.emptyIntf = configDir.ipIntfConfig.newMember( "" )
               IpIntf.emptyIntf.l3Config = l3c
            iic = IpIntf.emptyIntf

      return( iic, l3c )

   def overlapHelper( self, addressWithMask, secondary ):
      if secondary:
         primaryMatch = "allowNoMatch"
         secondaryMatch = "allowExactMatch"
      else:
         primaryMatch = "allowAddressMatch"
         secondaryMatch = "allowExactMatch"
      return Tac.newInstance( "Ira::AddressQueryParams", ipConfigDir.force(),
                              self.config().intfId, mgmtPrefix(),
                              primaryMatch, secondaryMatch, addressWithMask )

   def canSetIntfIp( self, addressWithMask, secondary=False ):
      # The IpPrefixRule sets the addressWithMask to None if mask is invalid.
      if addressWithMask is None:
         self.mode.addError( "Mask must be contiguous" )
         return False

      ( valid, errorString ) = self.intf_.isValidHostInetAddress( addressWithMask )
      if not valid:
         self.mode.addError( errorString )
         return False

      checker = Tac.Value( "Ira::AddressAssignmentChecker" )
      ans = checker.check( self.overlapHelper( addressWithMask, secondary ) )
      if ans.acceptable:
         if ans.msg:
            self.mode.addWarning( ans.msg )
      else:
         self.mode.addError( ans.msg )
         return False

      # Check address against extension hooks, print warning or error message,
      # exit if the address is not acceptable.
      for hook in canSetIntfIpHook.extensions():
         [ accept, message ] = hook( self.config().intfId, addressWithMask,
                                     secondary )
         if message:
            if accept:
               self.mode.addWarning( message )
            else:
               self.mode.addError( message )
         if not accept:
            return False

      return True

   def canDeleteIntfIp( self, addressWithMask=None ):
      # Check address against extension hooks, print warning or error message,
      # exit if the address is not acceptable.
      for hook in canDeleteIntfIpHook.extensions():
         [ accept, message ] = hook( self.config().intfId, addressWithMask )
         if message:
            if accept:
               self.mode.addWarning( message )
            else:
               self.mode.addError( message )
         if not accept:
            return False

      return True

   def setIpAddr( self, addressWithMask ):
      self.config().addrSource = 'manual'
      # If startup-config, skip the sanity checks to speed up LoadConfig
      if not self.mode.session.skipConfigCheck():
         if addressWithMask == self.config().addrWithMask:
            # nothing to do
            return

         if not self.canSetIntfIp( addressWithMask ):
            return

         # a secondary address might be promoted
         if ( addressWithMask and
              addressWithMask in self.config().secondaryWithMask ):
            # for a short time the same address is both primary and secondary!
            self.config().addrWithMask = addressWithMask
            del self.config().secondaryWithMask[ addressWithMask ]
            return

      self.config().addrWithMask = addressWithMask

   def noIpAddr( self, message=None ):
      if not self.canDeleteIntfIp():
         return

      self.config().addrSource = 'manual'
      secondaries = self.config().secondaryWithMask
      for a in secondaries:
         if message:
            self.mode.addWarning( message % ( self.name, a.address ) )
         del self.config().secondaryWithMask[ a ]

      if message:
         addr = self.config().addrWithMask.address
         if not addr == '0.0.0.0':
            self.mode.addWarning( message % ( self.name, addr ) )
      self.config().addrWithMask = zeroIpAddress

      # Also remove any "unnumbered" config with this command
      self.config().unnumberedIntfId = Tac.Value( 'Arnet::IntfId', '' )
      Tracing.trace0( "Disabled IP processing on", self.name )

   def noVirtualIpAddr( self, message=None ):
      if not self.canDeleteIntfIp():
         return

      secondaries = self.config().virtualSecondaryWithMask
      for a in secondaries:
         if message:
            self.mode.addWarning( message % ( self.name, a.address ) )
         del self.config().virtualSecondaryWithMask[ a ]

      if message:
         addr = self.config().virtualAddrWithMask.address
         if not addr == '0.0.0.0':
            self.mode.addWarning( message % ( self.name, addr ) )
      self.config().virtualAddrWithMask = zeroIpAddress
      Tracing.trace0( "Disabled Virtual IP processing on", self.name )

   # If there are no secondaries and if the address is the current primary,
   # delete it
   def noPrimaryIpAddr( self, addressWithMask ):
      zap = addressWithMask
      pri = self.config().addrWithMask

      if self.config().addrSource == 'dhcp':
         self.mode.addError( "Address determined by DHCP" )
         return

      if zap != pri:
         self.mode.addError( "Address %s/%s does not match primary address "
                             "%s/%s" %
                             ( zap.address, zap.len, pri.address, pri.len ) )
         return

      if self.config().secondaryWithMask:
         self.mode.addError( "Primary address cannot be deleted before secondary" )
      elif not self.canDeleteIntfIp( addressWithMask ):
         return
      else:
         self.config().addrWithMask = zeroIpAddress

   def setSecondaryIpAddr( self, addressWithMask ):
      # If startup-config, skip the sanity checks to speed up LoadConfig
      if not self.mode.session.skipConfigCheck():
         pri = self.config().addrWithMask
         if pri == Arnet.AddrWithMask( "0.0.0.0/0" ):
            self.mode.addError( "Primary address must be assigned first" )
            return
         if pri.address == addressWithMask.address:
            self.mode.addError( "Address %s is assigned as primary" %
                                ( pri.address, ) )
            return

         if addressWithMask in self.config().secondaryWithMask:
            # nothing to do
            return

         # Secondary IPs not supported by DHCP at this time
         if self.config().addrSource == 'dhcp':
            return


         if len( self.config().secondaryWithMask ) >= MAX_SECONDARIES_PER_INTERFACE:
            self.mode.addError( "This interface already has %d secondary addresses "
                                "assigned" % ( MAX_SECONDARIES_PER_INTERFACE, ) )
            return

         if not self.canSetIntfIp( addressWithMask=addressWithMask, secondary=True ):
            return

      self.config().secondaryWithMask[ addressWithMask ] = True

   def noSecondaryIpAddr( self, addressWithMask ):
      if addressWithMask not in self.config().secondaryWithMask:
         # warning message, not error
         self.mode.addWarning( "Address %s/%s was not found for deletion" %
                               ( addressWithMask.address, addressWithMask.len ) )
         return

      if not self.canDeleteIntfIp( addressWithMask ):
         return

      del self.config().secondaryWithMask[ addressWithMask ]

   def setIpUnnumberedIntf( self, intf ):
      self.config().unnumberedIntfId = intf.name

   def noIpUnnumberedIntf( self ):
      self.config().unnumberedIntfId = Tac.Value( 'Arnet::IntfId', '' )

   def setIpAddrDhcp( self ):
      self.config().secondaryWithMask.clear()
      self.config().addrWithMask = zeroIpAddress
      self.config().addrSource = 'dhcp'

   def noIpAddrDhcp( self ):
      self.config().addrSource = 'manual'

   def setDhcpDefaultRoute( self ):
      self.config().defaultRouteSource = ipv4AddrSource.dhcp

   def noDhcpDefaultRoute( self ):
      self.config().defaultRouteSource = ipv4AddrSource.manual

   #----------------------------------------------------------------------------
   # Destroys the IpIntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      intf = self.intf_.name
      del ipConfigDir.ipIntfConfig[ intf ]
      if not intf in ip6ConfigDir.intf and intf in l3ConfigDir.intfConfig:
         del l3ConfigDir.intfConfig[ intf ]

   #----------------------------------------------------------------------------
   # Returns the IpIntfStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      return self.ipIntfStatus

   def getStatus( self ):
      if not self.lookup():
         if not IpIntf.emptyIpIntfStatus:
            l3Status = Tac.newInstance( "L3::Intf::Status", "" )
            IpIntf.emptyIpIntfStatus = \
                  Tac.newInstance( "Ip::IpIntfStatus", "", l3Status )
         return IpIntf.emptyIpIntfStatus
      return self.ipIntfStatus

   def state( self ):
      if self.config().enabled:
         return 'up'
      else:
         return 'down'

   def showNetworkAddr( self ):
      return self.showAddr( lookupIpIntfStatus=True )

   def showAddr( self, lookupIpIntfStatus=False ):
      if lookupIpIntfStatus and self.lookup() is False:
         return None

      interfaceAddress = InterfaceAddress()
      if self.config().addrSource == 'dhcp':
         interfaceAddress.dhcp = True

      # primary/secondary IP
      addressWithMask = self.getStatus().activeAddrWithMask
      if addressWithMask == zeroIpAddress:
         addressWithMask = self.config().addrWithMask

      if self.status() and not self.status().l3IntfConnected:
         interfaceAddress.lossOfConnectivityReason = 'cfm'

      interfaceAddress.primaryIp = IpAddress( address=addressWithMask.address,
                                                 maskLen=addressWithMask.len )
      if self.config().unnumberedIntfId:
         interfaceAddress.unnumberedIntf = self.config().unnumberedIntfId
      for sa in self.config().secondaryWithMask:
         secAddr = IpAddress( address=sa.address, maskLen=sa.len )
         interfaceAddress.secondaryIps[ sa.address ] = secAddr
         interfaceAddress.secondaryIpsOrderedList.append( secAddr )

      # virtual IP
      addressWithMask = self.config().virtualAddrWithMask
      if addressWithMask.address is None or addressWithMask.address == '0.0.0.0':
         interfaceAddress.virtualIp = IpAddress( address='0.0.0.0', maskLen=0 )
      else:
         interfaceAddress.virtualIp = IpAddress( address=addressWithMask.address,
                                                 maskLen=addressWithMask.len )

      for sa in self.config().virtualSecondaryWithMask:
         secAddr = IpAddress( address=sa.address, maskLen=sa.len )
         interfaceAddress.virtualSecondaryIps[ sa.address ] = secAddr
         interfaceAddress.virtualSecondaryIpsOrderedList.append( secAddr )

      if self.getStatus().useVirtualAddr:
         interfaceAddress.primaryIp = IpAddress( address='0.0.0.0', maskLen=0 )

      return interfaceAddress

   def showStat( self ):
      stat = self.status().statistics

      print( "    %d packets input, %d bytes" % ( stat.rxPackets, stat.rxBytes ) )
      print( "    %d input errors, %d drops, %d overrun, %d frame, %d mcast" % (
         stat.rxErrs, stat.rxDrops, stat.rxFifo, stat.rxFrameErrors, stat.rxMcast ) )
      print( "    %d packets output, %d bytes" % ( stat.txPackets, stat.txBytes ) )
      print( "    %d output errors, %d drops, %d overrun, %d collisions, "
            "%d carrier" % (
         stat.txErrs, stat.txDrops, stat.txFifo, stat.txCollisions,
         stat.txCarrier ) )

   # The following methods are used by "show ip interface"
   @staticmethod
   def addrMethod():
      return 'non-volatile memory'

   def proxyArp( self ):
      if self.config().proxyArpEnabled:
         return 'enabled'
      else:
         return 'disabled'

   def localProxyArp( self ):
      if self.config().localProxyArpEnabled:
         return 'enabled'
      else:
         return 'disabled'

   def gratuitousArp( self ):
      if self.config().gratuitousArpAccepted:
         return 'accepted'
      else:
         return 'ignored'

   def proxyMacVirtualMac( self ):
      if self.config().proxyMacVirtualMacEnabled:
         return 'enabled'
      else:
         return 'disabled'

#-------------------------------------------------------------------------------
# Adds IP-specific CLI commands to the "config-if" mode for routed ports.
#-------------------------------------------------------------------------------
class RoutingProtocolIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # we want to exclude management interfaces and loopback interfaces
      return ( mode.intf.routingSupported() and
               not mode.intf.name.startswith( "Management" ) and
               not mode.intf.name.startswith( "Loopback" ) )

#-------------------------------------------------------------------------------
# Associate the RoutingProtocolIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( RoutingProtocolIntfConfigModelet )

#-------------------------------------------------------------
# Adds IP-specific CLI commands to the "config-if" mode for
# SVIs, mgmt intfs and routed ports.
#-------------------------------------------------------------
class DhcpIntfConfigModelet( _IpIntfModelet ):
   def setIpAddrDhcp( self, args ):
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                       self.ip.name, configType="IP" )
      self.ip.setIpAddrDhcp()

   def noIpAddrDhcp( self, args ):
      self.ip.noIpAddrDhcp()

   def setDhcpDefaultRoute( self, args ):
      self.ip.setDhcpDefaultRoute()

   def noDhcpDefaultRoute( self, args ):
      self.ip.noDhcpDefaultRoute()

   @staticmethod
   def shouldAddModeletRule( mode ):
      # We want to exclude loopback interfaces
      return ( mode.intf.routingSupported() and
               not mode.intf.name.startswith( "Loopback" ) )

#--------------------------------------------------------------------------------
# [ no | default ] ip address dhcp
#--------------------------------------------------------------------------------
class IpAddressDhcpCmd( CliCommand.CliCommandClass ):
   syntax = 'ip address dhcp'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'address': 'Set IP address of an interface',
      'dhcp': 'IP address negotiated via DHCP',
   }

   handler = DhcpIntfConfigModelet.setIpAddrDhcp
   noOrDefaultHandler = DhcpIntfConfigModelet.noIpAddrDhcp

DhcpIntfConfigModelet.addCommandClass( IpAddressDhcpCmd )

dhcpHelp = 'DHCP server, client and relay interface configuration'
#--------------------------------------------------------------------------------
# [ no | default ] dhcp client accept default-route
#--------------------------------------------------------------------------------
class DhcpClientAcceptDefaultRouteCmd( CliCommand.CliCommandClass ):
   syntax = 'dhcp client accept default-route'
   noOrDefaultSyntax = syntax
   data = {
      'dhcp': dhcpHelp,
      'client': 'DHCP client options obtained via DHCP',
      'accept': 'Accept DHCP client options obtained via DHCP',
      'default-route': 'Install default-route obtained via DHCP',
   }

   handler = DhcpIntfConfigModelet.setDhcpDefaultRoute
   noOrDefaultHandler = DhcpIntfConfigModelet.noDhcpDefaultRoute

DhcpIntfConfigModelet.addCommandClass( DhcpClientAcceptDefaultRouteCmd )

#-------------------------------------------------------------------------------
# Associate the DhcpIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( DhcpIntfConfigModelet )

routingMatcher = CliMatcher.KeywordMatcher(
      'routing',
      helpdesc='Routing for IP packets' )
routingNode = CliCommand.Node(
      matcher=routingMatcher,
      guard=routing.hwSupportedGuard )

#--------------------------------------------------------------------------------
# [ no | default ] ip address required disabled
#--------------------------------------------------------------------------------
class IpRoutingAddressRequiredCmd( CliCommand.CliCommandClass ):
   syntax = 'ip routing address required disabled'
   noOrDefaultSyntax = 'ip ROUTING_UNGUARDED address required ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'routing': routingNode,
      'ROUTING_UNGUARDED': routingMatcher,
      'address': 'IP address',
      'required': 'Require IP address for routing',
      'disabled': 'Disable IP address requirement for routing'
   }

   handler = IpIntfConfigModelet.setIpRoutingAddrRequired
   noOrDefaultHandler = IpIntfConfigModelet.noIpRoutingAddrRequired

if Toggles.IraToggleLib.toggleIpv4AddresslessForwardingEnabled():
   IpIntfConfigModelet.addCommandClass( IpRoutingAddressRequiredCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )
   global ipStatusDir, ipConfigDir, allIntfStatusDir, allVrfConfig, ip6ConfigDir
   global dynIpConfigDir
   global l3ConfigDir
   ipStatusDir = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ipConfigDir = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   dynIpConfigDir = LazyMount.mount( entityManager, "ip/input/dynIpConfig",
           "Tac::Dir", "ri" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   IntfCli.Intf.registerDependentClass( IpIntf, priority=10 )
   allVrfConfig = LazyMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "r" )
   ip6ConfigDir = ConfigMount.mount(
      entityManager, "ip6/config", "Ip6::Config", "w" )
   l3ConfigDir = ConfigMount.mount(
      entityManager, 'l3/intf/config', 'L3::Intf::ConfigDir', 'w' )
