# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" Implements Cli show commands for displaying kernel information
All commands call Tac.run()
Errors from the bash command are returned as is
"""

import sys

import BasicCli
import CliMatcher
import CliParser
import CliToken.Ip
import CliToken.Ipv6
import LazyMount
import ShowCommand
import Tac
import re
from CliPlugin.VrfCli import (
   DEFAULT_VRF,
   VrfExecCmdDec,
   VrfExprFactory,
   getVrfNames,
   vrfExists,
   generateVrfCliModel
)
from CliPlugin.IraKernelIpCliModel import (
   AclDetailsInvertedString,
   AclDetailsInvertedIpAddress,
   ShowKernelIpCountersVrfDetails,
   ShowKernelIpCountersIcmp,
   ShowKernelInterfaceCountersVrf,
   ShowKernelInterfaceCountersVrfDetails,
   ShowKernelInterfaceCountersRxTxDetails,
   ShowKernelInterfaceCountersInet6,
   ShowKernelIpRouteVrf,
   ShowKernelIpRouteDetails,
   ShowKernelInterfaceAddrVrf,
   ShowKernelInterfaceAddrDetails,
   ShowKernelInterfaceAddrInet,
   ShowKernelIpAclVrf,
   ShowKernelIpAclChain,
   ShowKernelIpAclDetails,
   ShowKernelIpArpVrf,
   ShowKernelIpArpDetails,
   ShowKernelInUsePorts,
   ShowKernelInUsePortsDetails
)
from PyWrappers.IptablesIpv6 import ip6tables
from PyWrappers.Iptables import iptables

routingHardwareStatus = None

kernelKwForShow = CliMatcher.KeywordMatcher(
   'kernel', helpdesc='Show kernel information' )

ipKwForShow = CliToken.Ip.ipMatcherForShow
ipv6KwForShow = CliToken.Ipv6.ipv6MatcherForShow

interfaceKw = CliMatcher.KeywordMatcher( 'interface',
   helpdesc='Interface-specific details' )

def ipV6Vrf( mode, token ):
   # pylint: disable-next=singleton-comparison
   if routingHardwareStatus.vrfCapability.ipv6EnabledDefault == False:
      return CliParser.guardNotThisPlatform
   return None

vrfExprFactory = VrfExprFactory( helpdesc='VRF filter',
      inclDefaultVrf=True )
vrfExprFactory6 = VrfExprFactory( helpdesc='VRF filter',
      guard=ipV6Vrf,
      inclDefaultVrf=True )

#error message if the vrf doesn't exist
noVrfErrorMsg = "%s for VRF %s does not exist."

#-----------------------------------------------------------------------------------
# Tac run wrapper function
# - cmd takes in a string and converts it to a list for convenience
# - uses "ip net ns exec ns-<vrfName>" if vrfName is specified
# - 'ns-" will be prepended to vrfName
#-----------------------------------------------------------------------------------
def tacRun( cmd, vrfName=None, asRoot=False, captureOutput=True ):
   if vrfName is None:
      vrfName = DEFAULT_VRF
   if vrfName != DEFAULT_VRF:
      nsVrfName = 'ns-%s' % vrfName # pylint: disable=consider-using-f-string
      # pylint: disable-next=consider-using-f-string
      cmd = "nsenter --net=/var/run/netns/%s -- %s" % ( nsVrfName, cmd )
      asRoot = True
   cmdList = cmd.split()
   if captureOutput:
      items = Tac.run( cmdList, stdout=Tac.CAPTURE, stderr=sys.stderr,
                       ignoreReturnCode=True, asRoot=asRoot )
      items = items.splitlines()
      # pylint: disable-next=consider-using-f-string
      items.insert( 0, "VRF: %s" % vrfName )
   else:
      print( "VRF: %s" % vrfName ) # pylint: disable=consider-using-f-string
      sys.stdout.flush() # Needed to print VRF name before command output
      Tac.run( cmdList, stdout=sys.stdout, stderr=sys.stderr, ignoreReturnCode=True,
               asRoot=asRoot )
      items = []
   return items

def populateIpRouteModel( items, ipv6=False ):
   model = ShowKernelIpRouteVrf()
   addrRe = re.compile( r"^.+?(?=(dev|proto|scope))" )
   detailsRe =  re.compile( r"\S+\s\S+" )
   # items represents the original CLI output with each line stored
   # in a list entry for parsing.
   # e.g. i='10.80.0.0/13 via 172.30.114.129 dev ma1 proto gated'
   for i in items:
      i = i.strip()
      if 'VRF' in i:
         continue
      addr = addrRe.search( i ).group()
      routeType = None
      via = None
      addrDetails = addr.split()
      if "via" in addr:
         route, via = addrDetails[ 0 ].strip(), addrDetails[ 2 ].strip()
      elif len( addrDetails ) > 1:
         routeType, route = addrDetails[ 0 ].strip(), addrDetails[ 1 ].strip()
      else:
         route = addr.strip()
      routeDetails = ShowKernelIpRouteDetails(
                    routeType=routeType, via=via, linkdown="linkdown" in i,
                    dead="dead" in i, onlink="onlink" in i,
                    pervasive="pervasive" in i, offload="offload" in i,
                    notify="notify" in i, unresolved="unresolved" in i )
      details = addrRe.sub( "", i ).strip()
      details = detailsRe.findall( details )
      for d in details:
         kw, val = d.split()
         if val.lstrip( '-' ).isdigit():
            val = int( val )
         if kw == "error":
            kw = "errorNum"
         setattr( routeDetails, kw, val )
      if route == "default":
         if not ipv6:
            route = "0.0.0.0/0"
         else:
            route = "::/0"
      # BUG725015: this model is keyed on only the address portion of
      # the route, and not the prefix.  If it's desirable to key
      # on the prefix, the model must be changed in a customer-
      # visible way and likely needs a revision and a degrade
      # method.
      if '/' in route:
         addr, _ = route.split( '/' )
      else:
         addr = route
      model.routes[ addr ] = routeDetails
   return model

#-----------------------------------------------------------------------------------
# The "show kernel ip route [vrf <vrfName>]" command.
#-----------------------------------------------------------------------------------
routeKwForIp = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Entries used for routing packets' )

ShowKernelIpRouteVrfModel = generateVrfCliModel( ShowKernelIpRouteVrf,
                                             "A mapping of VRFs to their IP routes" )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIpRouteVrfModel )
def showKernelIpRoute( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IP routing table', vrfName ) )
      return None
   cmd = "ip route show"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateIpRouteModel( items, ipv6=False )
      model.addVrf( items )
   else:
      model = ShowKernelIpRouteVrf()
   return model

class ShowKernelIpRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ip route [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ip" : ipKwForShow,
      "route" : routeKwForIp,
      "VRF": vrfExprFactory,
   }
   handler = showKernelIpRoute
   cliModel = ShowKernelIpRouteVrfModel

BasicCli.addShowCommandClass( ShowKernelIpRouteCmd )

#-----------------------------------------------------------------------------------
# The "show kernel ipv6 route [vrf VRF]" command.
#-----------------------------------------------------------------------------------
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIpRouteVrfModel )
def showKernelIpv6Route( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IPv6 routing table', vrfName ) )
      return None
   cmd = "ip -f inet6 route show"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateIpRouteModel( items, ipv6=True )
      model.addVrf( items )
   else:
      model = ShowKernelIpRouteVrf()
   return model

class ShowKernelIpv6RouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ipv6 route [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ipv6" : ipv6KwForShow,
      "route" : routeKwForIp,
      "VRF": vrfExprFactory6,
   }
   handler = showKernelIpv6Route
   cliModel = ShowKernelIpRouteVrfModel

BasicCli.addShowCommandClass( ShowKernelIpv6RouteCmd )

def populateIpAclModel( items ):
   model = ShowKernelIpAclVrf()
   modelIntf = ShowKernelIpAclChain()
   detailsRe = r"^\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+"
   detailsCompRe = re.compile( detailsRe )
   chainCompRe = re.compile( r"Chain\s\S+\s\((.*)\)" )
   referencesCompRe = re.compile( r"\d+\sreferences" )
   policyCompRe = re.compile( r"policy\s+\S+" )
   packetsCompRe = re.compile( r"\d+\spackets" )
   bytesCompRe = re.compile( r"\d+\sbytes" )
   headingsCompRe = re.compile( r"pkts\s+bytes" )
   for i in items:
      details = detailsCompRe.search( i )
      if "VRF" in i:
         continue
      chainDev = chainCompRe.search( i )
      if chainDev:
         dev = chainDev.group().split()[ 1 ]
         model.chainAcls[ dev ] = ShowKernelIpAclChain()
         modelIntf = model.chainAcls[ dev ]
         chainDetails = chainDev.group( 1 )
         references = referencesCompRe.search( chainDetails )
         if references:
            refs = references.group().split()[ 0 ]
            modelIntf.references = int( refs )
         else:
            policy = policyCompRe.search( chainDetails ).group().split()[ 1 ]
            chainPackets = packetsCompRe.search( chainDetails ).group().split()[ 0 ]
            chainBytes = bytesCompRe.search( chainDetails ).group().split()[ 0 ]
            modelIntf.policy = policy
            modelIntf.chainPackets= int( chainPackets )
            modelIntf.chainBytes = int( chainBytes )
      elif headingsCompRe.search( i ) or i == "":
         continue
      elif details:
         d = details.group().split()
         extra = None
         values = detailsCompRe.sub( "", i ).strip()
         if values != "":
            extra = values
         # BUG726832: this model contains only the address portion of
         # the source and destination, and not the prefix.  If it's desirable
         # to report the prefix, the model must be changed in a customer-
         # visible way and likely needs a revision and a degrade
         # method.

         def stripPrefix( prefix ):
            if '/' in prefix:
               addr, _ = prefix.split( '/' )
            else:
               addr = prefix
            return addr
         # In IPv6 the 'opt' field is not represented by any value if it is empty
         # as opposed to the IPv4 version populates it with "--"
         try:
            AclDetailsInvertedIpAddress( value=stripPrefix(
                                                d[ 8 ].replace( "!", "" ) ),
                                         inverted=( "!" in d[ 8 ] ) )
         except ValueError:
            for index in range( len( d )-1, 3, -1 ):
               d[ index ] = d[ index-1 ]
            d[ 4 ] = "--"
         proto = AclDetailsInvertedString( value=d[ 3 ].replace( "!", "" ),
                                           inverted=( "!" in d[ 3 ] ) )
         opt = AclDetailsInvertedString( value=d[ 4 ].replace( "!", "" ),
                                         inverted=( "!" in d[ 4 ] ) )
         inbound = AclDetailsInvertedString( value=d[ 5 ].replace( "!", "" ),
                                             inverted=( "!" in d[ 5 ] ) )
         outbound = AclDetailsInvertedString( value=d[ 6 ].replace( "!", "" ),
                                              inverted=( "!" in d[ 6 ] ) )
         source = AclDetailsInvertedIpAddress( value=stripPrefix(
                                                      d[ 7 ].replace( "!", "" ) ),
                                               inverted=( "!" in d[ 7 ] ) )
         destination = AclDetailsInvertedIpAddress( value=stripPrefix(
                                                      d[ 8 ].replace( "!", "" ) ),
                                                inverted=( "!" in d[ 8 ] ) )
         aclDetails = ShowKernelIpAclDetails( numPackets=int( d[ 0 ] ),
                                              numBytes=int( d[ 1 ] ), target=d[ 2 ],
                                              proto=proto, opt=opt,
                                              inbound=inbound, outbound=outbound,
                                              source=source, destination=destination,
                                              aclExtraDetails=extra )
         modelIntf.details.append( aclDetails )
   return model

#-----------------------------------------------------------------------------------
# The "show kernel ip acl [vrf VRF]" command.
#-----------------------------------------------------------------------------------
aclKwForIp = CliMatcher.KeywordMatcher( 'acl', helpdesc='IP tables' )

ShowKernelIpAclVrfModel = generateVrfCliModel( ShowKernelIpAclVrf,
                                               "A mapping of VRFs to its ACL" )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIpAclVrfModel )
def showKernelIpAcl( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IP ACL table', vrfName ) )
      return None
   cmd = iptables() + " -nvxL"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, asRoot=True, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateIpAclModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelIpAclVrf()
   return model

class ShowKernelIpAclCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ip acl [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ip" : ipKwForShow,
      "acl" : aclKwForIp,
      "VRF": vrfExprFactory,
   }
   handler = showKernelIpAcl
   cliModel = ShowKernelIpAclVrfModel

BasicCli.addShowCommandClass( ShowKernelIpAclCmd )

#-----------------------------------------------------------------------------------
# The "show kernel ipv6 acl [vrf VRF]" command.
#-----------------------------------------------------------------------------------
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIpAclVrfModel )
def showKernelIpv6Acl( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IPv6 ACL table', vrfName ) )
      return None
   cmd = ip6tables() + " -nvxL"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, asRoot=True, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateIpAclModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelIpAclVrf()
   return model

class ShowKernelIpv6AclCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ipv6 acl [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ipv6" : ipv6KwForShow,
      "acl" : aclKwForIp,
      "VRF": vrfExprFactory6,
   }
   handler = showKernelIpv6Acl
   cliModel = ShowKernelIpAclVrfModel

BasicCli.addShowCommandClass( ShowKernelIpv6AclCmd )

kernelStrings = { 'valid_lft': 'validLft', 'preferred_lft': 'preferredLft',
                  'UNKNOWN': 'unknown', 'NOTPRESENT': 'notPresent', 'DOWN': 'down',
                  'LOWERLAYERDOWN': 'lowerLayerDown', 'TESTING': 'testing',
                  'DORMANT': 'dormant', 'UP': 'up', 'link-netnsid': 'linkNetnsid' }

def populateInterfaceAddrModel( items ):
   devRe = re.compile( r"^\d+:\s.+:" )
   valueRe = re.compile( r"\S+\s\S+" )
   inetAddrRe = re.compile( r"(inet6|inet)\s\S+" )
   flagsRe = re.compile( r"<(.*)>" )
   devLineDetailsRe = re.compile( r".*(?=>)>" )
   scopeRe = re.compile( r"scope\s\S+" )
   broadcastRe = re.compile( r"brd\s\S+" )
   model = ShowKernelInterfaceAddrVrf()
   intf = ShowKernelInterfaceAddrDetails()
   inetType = None
   for i in items:
      if "VRF" in i:
         continue
      dev = devRe.search( i )
      if dev:
         devName = dev.group().split()[ 1 ][ :-1 ]
         model.interfaces[ devName ] = ShowKernelInterfaceAddrDetails()
         intf = model.interfaces[ devName ]
         intf.flags = flagsRe.search( i ).group( 1 )
         values = devLineDetailsRe.sub( "", i ).strip()
         values = valueRe.findall( values )
         for v in values:
            kw = v.split()[ 0 ]
            val = v.split()[ 1 ]
            if val.lstrip( '-' ).isdigit():
               val = int( val )
            elif kw == "state":
               val = kernelStrings.get( val, val )
            elif kw == "master":
               kw = "parent"
            if kw not in intf:
               intf.extraDetails[ kw ] = val
            else:
               setattr( intf, kw, val )
      elif "link/" in i:
         values = valueRe.findall( i )
         for v in values:
            kw, val = v.split()
            kw = kw.split( "/" )[ 0 ]
            kw = kernelStrings.get( kw, kw )
            if val.lstrip( '-' ).isdigit():
               val = int( val )
            if kw not in intf:
               intf.extraDetails[ kw ] = val
            else:
               setattr( intf, kw, val )
      elif inetAddrRe.search( i ):
         scope = scopeRe.search( i ).group().split()[ 1 ]
         if "inet6" in i and not scope in intf.inet6:
            intf.inet6[ scope ] = ShowKernelInterfaceAddrInet()
            inetType = intf.inet6[ scope ]
         elif "inet" in i and not scope in intf.inet:
            intf.inet[ scope ] = ShowKernelInterfaceAddrInet()
            inetType = intf.inet[ scope ]
         prefix = inetAddrRe.search( i ).group().split()[ 1 ]
         # BUG725017: the addr is only the address portion of the assigned
         # subnet, and not the prefix.  If it's desirable to publish
         # the prefix, the model must be changed in a customer-
         # visible way and likely needs a revision and a degrade
         # method.
         addr, _ = prefix.split( '/' )
         inetType.addr = addr
         if broadcastRe.search( i ):
            inetType.brd = broadcastRe.search( i ).group().split()[ 1 ]
      elif "valid_lft" in i:
         values = valueRe.findall( i )
         for v in values:
            kw, val = v.split()
            kw = kernelStrings.get( kw, kw )
            setattr( inetType, kw, val )
   return model

#-----------------------------------------------------------------------------------
# The "show kernel ip interface addr [vrf VRF]" command.
#-----------------------------------------------------------------------------------
addrKwForIpIntf = CliMatcher.KeywordMatcher( 'addr',
   helpdesc='Show only IP/IPv6 interfaces' )

ShowKernelInterfaceAddrVrfModel = generateVrfCliModel( ShowKernelInterfaceAddrVrf,
                                             "A mapping of VRF to its interfaces" )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelInterfaceAddrVrfModel )
def showKernelIpIntfAddr( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IP interface address', vrfName ) )
      return None
   cmd = "ip -f inet addr show"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateInterfaceAddrModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelInterfaceAddrVrf()
   return model

class ShowKernelIpIntfAddrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ip interface addr [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ip" : ipKwForShow,
      "interface" : interfaceKw,
      "addr" : addrKwForIpIntf,
      "VRF": vrfExprFactory,
   }
   handler = showKernelIpIntfAddr
   cliModel = ShowKernelInterfaceAddrVrfModel

BasicCli.addShowCommandClass( ShowKernelIpIntfAddrCmd )

#-----------------------------------------------------------------------------------
# The "show kernel ipv6 interface addr [vrf VRF]" command.
#-----------------------------------------------------------------------------------
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelInterfaceAddrVrfModel )
def showKernelIpv6IntfAddr( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IPv6 interface address', vrfName ) )
      return None
   cmd = "ip -f inet6 addr show"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateInterfaceAddrModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelInterfaceAddrVrf()
   return model

class ShowKernelIpv6IntfAddrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ipv6 interface addr [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ipv6" : ipv6KwForShow,
      "interface" : interfaceKw,
      "addr" : addrKwForIpIntf,
      "VRF": vrfExprFactory6,
   }
   handler = showKernelIpv6IntfAddr
   cliModel = ShowKernelInterfaceAddrVrfModel

BasicCli.addShowCommandClass( ShowKernelIpv6IntfAddrCmd )

def camelize( string ):
   components = string.split()
   return components[ 0 ] + ''.join( x.title() for x in components[ 1: ] )

def populateIpCountersModel( items ):
   model = ShowKernelIpCountersVrfDetails()
   kw = ""
   icmpKw = "details"
   keyWithColonRe = re.compile( r"[^:]*" )
   valWithColonRe = re.compile( r".*:" )
   removePeriodRe = re.compile( r"\." )
   keyValWOColonRe = re.compile( r"\d+" )
   for i in items:
      split = i.split()
      if 'VRF' in i:
         continue
      if len( split ) == 1:
         kw = split[ 0 ][ 0 ].lower() + split[ 0 ][ 1:-1 ]
         continue
      attr = getattr( model, kw )
      if kw in ( "icmp", "icmp6" ):
         if not attr:
            if kw == "icmp":
               model.icmp = ShowKernelIpCountersIcmp()
               attr = model.icmp
            else:
               model.icmp6 = ShowKernelIpCountersIcmp()
               attr = model.icmp6
         if "histogram" in i:
            #Set histogram type (input|output)
            icmpKw = split[ 1 ] + split[ 2 ][ :-1 ].title()
            continue
         if ":" in i:
            key = keyWithColonRe.search( i ).group().strip()
            key = camelize( key )
            value = valWithColonRe.sub( "", i ).strip()

            # XXX: Due to a misformatting in earlier versions of netstat, we
            # are left with the legacy of replacing "echoRequests" in the
            # output histogram with "echoRequest". We should specify the model
            # here more completely - See BUG/755933
            if icmpKw == 'outputHistogram' and key == 'echoRequests':
               key = 'echoRequest'
            getattr( attr, icmpKw )[ key ] = int( value )
         else:
            if icmpKw != "details":
               icmpKw = "details"
            key = keyValWOColonRe.sub( "", i ).strip()
            key = camelize( key )
            if removePeriodRe.search( key ):
               key = key[ :-1 ]
            value = keyValWOColonRe.search(i ).group().strip()
            getattr( attr, icmpKw )[ key ] = int( value )
      elif ":" in i:
         key = keyWithColonRe.search( i ).group().strip()
         key = camelize( key )
         value = valWithColonRe.sub( "", i ).strip()
         attr[ key ] = int( value )
      else:
         key = keyValWOColonRe.sub( "", i ).strip()
         key = camelize( key )
         if removePeriodRe.search( key ):
            key = key[ :-1 ]
         value = keyValWOColonRe.search( i ).group().strip()
         attr[ key ] = int( value )
   return model

#-----------------------------------------------------------------------------------
# The "show kernel ip counters [vrf VRF]" command.
#-----------------------------------------------------------------------------------
countersKwForIp = CliMatcher.KeywordMatcher( 'counters',
   helpdesc='Network statistics' )

ShowKernelIpCountersVrfModel = generateVrfCliModel( ShowKernelIpCountersVrfDetails,
                                             "A mapping of VRFs to detail sections" )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIpCountersVrfModel )
def showKernelIpCounters( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IP counters', vrfName ) )
      return None
   cmd = "netstat -s -A inet"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateIpCountersModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelIpCountersVrfDetails()
   return model

class ShowKernelIpCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ip counters [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ip" : ipKwForShow,
      "counters" : countersKwForIp,
      "VRF": vrfExprFactory,
   }
   handler = showKernelIpCounters
   cliModel = ShowKernelIpCountersVrfModel

BasicCli.addShowCommandClass( ShowKernelIpCountersCmd )

#-----------------------------------------------------------------------------------
# The "show kernel ipv6 counters [vrf VRF]" command.
#-----------------------------------------------------------------------------------
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIpCountersVrfModel )
def showKernelIpv6Counters( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IPv6 counters', vrfName ) )
      return None
   cmd = "netstat -s -A inet6"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateIpCountersModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelIpCountersVrfDetails()
   return model

class ShowKernelIpv6CountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ipv6 counters [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ipv6" : ipv6KwForShow,
      "counters" : countersKwForIp,
      "VRF": vrfExprFactory6,
   }
   handler = showKernelIpv6Counters
   cliModel = ShowKernelIpCountersVrfModel

BasicCli.addShowCommandClass( ShowKernelIpv6CountersCmd )

def populateIpArpModel( items ):
   model = ShowKernelIpArpVrf()
   arpAddrRe = re.compile( r"^\S+" )
   devNameRe = re.compile( r"dev\s\S+" )
   lladdrRe = re.compile( r"lladdr\s\S+" )
   stateRe = re.compile( r"\w+$" )
   for i in items:
      i = i.strip()
      if "VRF" in i: # pylint: disable=no-else-continue
         continue
      else:
         arpDetails = ShowKernelIpArpDetails( proxy="proxy" in i,
                                              router="router" in i,
                                              externallyLearned="extern_learn" in i )
         dev = devNameRe.search( i )
         if dev:
            arpDetails.dev = dev.group().split()[ 1 ]
         lladdr = lladdrRe.search( i )
         if lladdr:
            arpDetails.lladdr = lladdr.group().split()[ 1 ]
         state = stateRe.search( i ).group().lower()
         if state == "noarp":
            state = "noArp"
         arpDetails.state = state
         addr = arpAddrRe.search( i ).group()
         model.arpAddresses[ addr ] = arpDetails
   return model

#-----------------------------------------------------------------------------------
# The "show kernel ip arp [vrf VRF]" command.
#-----------------------------------------------------------------------------------
arpKwForIp = CliMatcher.KeywordMatcher( 'arp', helpdesc='IP neighbors' )

ShowKernelIpArpVrfModel = generateVrfCliModel( ShowKernelIpArpVrf,
                                             "A mapping of VRFs to its ARP tables" )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIpArpVrfModel )
def showKernelIpNeighbors( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IP ARP table', vrfName ) )
      return None
   cmd = "ip -4 neigh show"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateIpArpModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelIpArpVrf()
   return model

class ShowKernelIpArpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ip arp [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ip" : ipKwForShow,
      "arp" : arpKwForIp,
      "VRF": vrfExprFactory,
   }
   handler = showKernelIpNeighbors
   cliModel = ShowKernelIpArpVrfModel

BasicCli.addShowCommandClass( ShowKernelIpArpCmd )

#-----------------------------------------------------------------------------------
# The "show kernel ipv6 neighbors [vrf VRF]" command.
#-----------------------------------------------------------------------------------
neighborsKwForIpv6 = CliMatcher.KeywordMatcher( 'neighbors',
   helpdesc='IPv6 neighbors' )

@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIpArpVrfModel )
def showKernelIpv6Neighbors( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'IPv6 neighbors', vrfName ) )
      return None
   cmd = "ip -f inet6 neigh show"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateIpArpModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelIpArpVrf()
   return model

class ShowKernelIpv6NeighborsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ipv6 neighbors [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "ipv6" : ipv6KwForShow,
      "neighbors" : neighborsKwForIpv6,
      "VRF": vrfExprFactory6,
   }
   handler = showKernelIpv6Neighbors
   cliModel = ShowKernelIpArpVrfModel

BasicCli.addShowCommandClass( ShowKernelIpv6NeighborsCmd )

def populateInterfaceCountersModel( items ):
   model = ShowKernelInterfaceCountersVrf()
   kw = ""
   kwRe= re.compile( r"^\w+:" )
   flagsRe = re.compile( r"=.*>" )
   mtuRe = re.compile( r"mtu.*" )
   rxtxRe = re.compile( r"(RX|TX)" )
   rxtxValsRe = re.compile( r"\w+\s\S+" )
   macAddrRe = re.compile( r"\w\w:\w\w:\w\w:\w\w:\w\w:\w\w" )
   unspecRe = re.compile( r"unspec\s\S+" )
   txQLenRe = re.compile( r"txqueuelen\s\d+" )
   inetRe = re.compile( r"inet\s\S+" )
   netmaskRe = re.compile( r"netmask\s\S+" )
   broadcastRe = re.compile( r"broadcast\s\S+" )
   inet6Re = re.compile( r"inet6\s\S+" )
   inet6TypeRe = re.compile( r"<(.+)>" )
   prefixlenRe = re.compile( r"prefixlen\s\S+" )
   scopeidRe = re.compile( r"scopeid[^<]*" )
   devInterruptRe = re.compile( r"device\sinterrupt\s\d+" )
   loopRe = re.compile( r"loop.*\((.*)\)" )

   for i in items:
      if i == "": # pylint: disable=no-else-continue
         continue
      elif "VRF" in i:
         continue
      kwSearch = kwRe.search( i )
      if kwSearch:
         kw = kwSearch.group()
         model.interfaces[ kw ] = ShowKernelInterfaceCountersVrfDetails()
         model.interfaces[ kw ].flags = flagsRe.search( i ).group()[ 1: ]
         model.interfaces[ kw ].mtu = int(
                                             mtuRe.search( i ).group().split()[ 1 ] )
      elif "RX" in i or "TX" in i:
         rxtx = ""
         if "RX" in i:
            if not model.interfaces[ kw ].rx:
               model.interfaces[ kw ].rx = ShowKernelInterfaceCountersRxTxDetails()
            rxtx = model.interfaces[ kw ].rx
         else:
            if not model.interfaces[ kw ].tx:
               model.interfaces[ kw ].tx = ShowKernelInterfaceCountersRxTxDetails()
            rxtx = model.interfaces[ kw ].tx
         rxtxVals = rxtxValsRe.findall( rxtxRe.sub( "", i ) )
         for r in rxtxVals:
            if ')' in r:
               continue
            key = r.split()[ 0 ]
            value = r.split()[ 1 ]
            if key == "bytes":
               key = "numBytes"
            elif key == "errors":
               key = "numErrors"
            elif key == "packets":
               key = "numPackets"
            elif key == "frame":
               key = "misaligned"
            setattr( rxtx, key, int( value ) )
      else:
         vrfDetails = model.interfaces[ kw ]
         macAddr = macAddrRe.search( i )
         if macAddr:
            vrfDetails.ether = macAddr.group()
         unspec = unspecRe.search( i )
         if unspec:
            vrfDetails.unspec = unspec.group().split()[ 1 ]
         txQLen = txQLenRe.search( i )
         if txQLen:
            vrfDetails.txqueuelen = int( txQLen.group().split()[ 1 ] )
         inet = inetRe.search( i )
         if inet:
            vrfDetails.inet = inet.group().split()[ 1 ]
            netmask = netmaskRe.search( i )
            if netmask:
               vrfDetails.netmask = netmask.group().split()[ 1 ]
            broadcast = broadcastRe.search( i )
            if broadcast:
               vrfDetails.broadcast = broadcast.group().split()[ 1 ]
         inet6 = inet6Re.search( i )
         if inet6:
            inet6Type = inet6TypeRe.search( i ).group( 1 )
            vrfDetails.inet6[ inet6Type ] = ShowKernelInterfaceCountersInet6()
            vrfDetails.inet6[ inet6Type ].address = inet6.group().split()[ 1 ]
            prefixlen = prefixlenRe.search( i )
            if prefixlen:
               vrfDetails.inet6[ inet6Type ].prefixlen = int(
                                                   prefixlen.group().split()[ 1 ] )
            scopeid = scopeidRe.search( i )
            if scopeid:
               vrfDetails.inet6[ inet6Type ].scopeid = int(
                                                   scopeid.group().split()[ 1 ], 16 )
         devInterrupt = devInterruptRe.search( i )
         if devInterrupt:
            vrfDetails.deviceInterrupt = int( devInterrupt.group().split()[ 2 ] )
         loop = loopRe.search( i )
         if loop:
            vrfDetails.loop = loop.group( 1 )
   return model

#-----------------------------------------------------------------------------------
# The 'show kernel interface counters [vrf VRF]" command.
#-----------------------------------------------------------------------------------
countersKwForIntf = CliMatcher.KeywordMatcher( 'counters',
   helpdesc='Show interface configurations' )

ShowKernelIntfCountersVrfModel = generateVrfCliModel(
                                             ShowKernelInterfaceCountersVrf,
                                             "A mapping of VRFs to interfaces" )
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelIntfCountersVrfModel )
def showKernelIntfCounters( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'Interface counters', vrfName ) )
      return None
   cmd = "ifconfig -a"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateInterfaceCountersModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelInterfaceCountersVrf()
   return model

class ShowKernelIntfCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel interface counters [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "interface" : interfaceKw,
      "counters" : countersKwForIntf,
      "VRF": vrfExprFactory,
   }
   handler = showKernelIntfCounters
   cliModel = ShowKernelIntfCountersVrfModel

BasicCli.addShowCommandClass( ShowKernelIntfCountersCmd )

#-----------------------------------------------------------------------------------
# The 'show kernel interface addr [vrf VRF]" command.
#-----------------------------------------------------------------------------------
addrKwForIntf = CliMatcher.KeywordMatcher( 'addr', helpdesc='Show interfaces' )

@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelInterfaceAddrVrfModel )
def showKernelAddr( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'Interface address', vrfName ) )
      return None
   cmd = "ip addr show"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateInterfaceAddrModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelInterfaceAddrVrf()
   return model

class ShowKernelIntfAddrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel interface addr [ VRF ]"
   data = {
      "kernel" : kernelKwForShow,
      "interface" : interfaceKw,
      "addr" : addrKwForIntf,
      "VRF": vrfExprFactory,
   }
   handler = showKernelAddr
   cliModel = ShowKernelInterfaceAddrVrfModel

BasicCli.addShowCommandClass( ShowKernelIntfAddrCmd )

def populateInUsePortsModel( items ):
   model = ShowKernelInUsePorts()
   protocolRe = re.compile( r"^(tcp|udp)" )
   processRe = re.compile( r"\(\(.*\)\)" )
   nameRe = re.compile( r"\"([^\"]*)\"" )
   pidRe = re.compile( r"\,pid=\d+" )

   for i in items:
      i = i.strip()
      if i == "":
         continue
      if not protocolRe.search( i ): # pylint: disable=no-else-continue
         continue
      else:
         portInfo = ShowKernelInUsePortsDetails()

         portDetails = i.split()
         if len( portDetails ) < 6:
            continue

         if processRe.search( i ):
            proc = processRe.search( i ).group()
            for name, pid in zip( nameRe.finditer( proc ), pidRe.finditer( proc ) ):
               name = name.group()[ 1 : -1 ]
               pid = int( pid.group()[ 5 : ] )
               portInfo.processes[ pid ] = name

         portInfo.proto = portDetails[ 0 ].strip()
         portInfo.recvQ = int( portDetails[ 2 ].strip() )
         portInfo.sendQ = int( portDetails[ 3 ].strip() )

         stateWords = portDetails[ 1 ].split( "-" )
         state = "".join( s[ 0 ] + s[ 1 : ].lower() for s in stateWords )
         portInfo.state = state[ 0 ].lower() + state[ 1 : ]

         socketInfo = []
         for j in range( 2 ):
            info = [ s.strip() for s in portDetails[ j + 4 ].rsplit( ":", 1 ) ]
            socketInfo.append( re.sub( r"(\[|\])", "", info[ 0 ] ) )
            socketInfo.append( info[ 1 ] )

         portInfo.peerAddr = socketInfo[ 2 ] if socketInfo[ 2 ] != "*" else None
         if socketInfo[ 3 ] != "*":
            portInfo.peerPort = int( socketInfo[ 3 ] )

         model.ports[ portDetails[ 4 ] ] = portInfo

   return model

# -----------------------------------------------------------------------------------
# The "show kernel ports in-use [vrf VRF]" command.
# -----------------------------------------------------------------------------------
PortsKw = CliMatcher.KeywordMatcher( 'ports',
   helpdesc='TCP and UDP ports' )
InUsePortsKw = CliMatcher.KeywordMatcher( 'in-use',
   helpdesc='Currenlty used ports' )

ShowKernelInUsePortsModel = generateVrfCliModel( ShowKernelInUsePorts,
                                             "Show all ports currently used" )

@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ShowKernelInUsePortsModel )
def showKernelInUsePorts( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName and not vrfExists( vrfName ):
      mode.addError( noVrfErrorMsg % ( 'Ports in use', vrfName ) )
      return None
   cmd = "ss -ntuap"
   captureOutput = not mode.session.shouldPrint()
   items = tacRun( cmd, vrfName=vrfName, asRoot=True, captureOutput=captureOutput )
   # Model is only populated when JSON tag is specified
   if captureOutput:
      model = populateInUsePortsModel( items )
      model.addVrf( items )
   else:
      model = ShowKernelInUsePorts()
   return model

class ShowKernelInUsePortsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show kernel ports in-use [ VRF ]"
   data = {
      "kernel": kernelKwForShow,
      "ports": PortsKw,
      "in-use": InUsePortsKw,
      "VRF": vrfExprFactory,
   }
   handler = showKernelInUsePorts
   cliModel = ShowKernelInUsePortsModel

BasicCli.addShowCommandClass( ShowKernelInUsePortsCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHardwareStatus
   routingHardwareStatus = LazyMount.mount( entityManager,
                                     "routing/hardware/statuscommon",
                                     "Routing::Hardware::StatusCommon", "r" )

