#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliToken.Service
import CliCommand
import CliMatcher
import BasicCli
import ConfigMount
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel

#-------------------------------------------------------------------------------
# The 'service routing protocols model [ribd|multi-agent]' command
#-------------------------------------------------------------------------------

ROUTING_PROTO_MODEL_HELP = {
      ProtoAgentModel.ribd : \
            "Configure protocols model as ribd (single agent)",
      ProtoAgentModel.multiAgent : \
            "Configure protocols model as multi-agent"
}

def getModelSyntax( mode ):
   data = { ProtoAgentModel.multiAgent:
            ROUTING_PROTO_MODEL_HELP[ ProtoAgentModel.multiAgent ] }
   if not l3Config.isRibdRemoved:
      data[ ProtoAgentModel.ribd ] = ROUTING_PROTO_MODEL_HELP[ ProtoAgentModel.ribd ]
   return data

tokenService = CliToken.Service.serviceKw
routingMatcherForConfig = CliMatcher.KeywordMatcher( 'routing',
    helpdesc="Routing service configuration" )
tokenProtocols = CliMatcher.KeywordMatcher( 'protocols',
    helpdesc="Configure routing protocols" )
tokenModel = CliMatcher.KeywordMatcher( 'model',
    helpdesc="Configure the model for how protocols are to be run" )
tokenForceRule = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'force',
                 helpdesc="Force protocol model change without switch reboot" ),
                 hidden=True )
modelTypeMatcher = CliMatcher.DynamicKeywordMatcher( getModelSyntax )

l3Config = None

configMode = BasicCli.GlobalConfigMode
def getEffectiveProtocolModel( mode ):
   """
   Per BUG391575, the mounted instance of l3Config may not return the actual
   effective protocol model running on the DUT (ie. Config replace). 
   This helper function allows you to properly retrieve the actual effective
   protocol model.
   """
   return mode.sysdbRoot.entity[ "l3/config" ].protocolAgentModel

def Plugin( entityManager ):
   global l3Config
   l3Config = ConfigMount.mount( entityManager, "l3/config", "L3::Config", 'w' )

class SetProtoModelCliExpression( CliCommand.CliCommandClass ):
   syntax = "SERVICE ROUTING PROTOCOLS MODEL MODEL_TYPE [ FORCE ]"

   data = {
      "SERVICE": tokenService,
      "ROUTING": routingMatcherForConfig,
      "PROTOCOLS": tokenProtocols,
      "MODEL": tokenModel,
      "MODEL_TYPE": modelTypeMatcher,
      "FORCE": tokenForceRule,
   }

   @staticmethod
   def handler( mode, args ):
      '''set the model for how protocols are to be run'''

      l3Config.configuredProtocolAgentModel = args[ "MODEL_TYPE" ]
      force = "FORCE" in args
      if mode.session.startupConfig() or force:
         l3Config.protocolAgentModel = l3Config.configuredProtocolAgentModel

      # We want to check a mismatch between the effective protocol model and the
      # configured state. We can not do that using l3Config. In a config replace
      # scenario l3Config will represent the temporary state used by config replace.
      # The getEffectiveProtocolModel is here to get the effective protocol model
      # properly.
      # See BUG391575
      if getEffectiveProtocolModel( mode ) != l3Config.configuredProtocolAgentModel:
         mode.addWarning( "Change will take effect only after switch reboot" )

configMode.addCommandClass( SetProtoModelCliExpression )

class NoProtoModelCliExpression( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = "SERVICE ROUTING PROTOCOLS MODEL ..."
   data = {
      "SERVICE": tokenService,
      "ROUTING": routingMatcherForConfig,
      "PROTOCOLS": tokenProtocols,
      "MODEL": tokenModel,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      '''revert to default model (ribd) for how
         protocols are to be run'''
      l3Config.configuredProtocolAgentModel = l3Config.protocolAgentModelDefault
      if l3Config.protocolAgentModel != l3Config.configuredProtocolAgentModel:
         mode.addWarning( "Change will take effect only after switch reboot" )

configMode.addCommandClass( NoProtoModelCliExpression )
