# Copyright (c) 2011-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# VRF-related CLI

'''Configuration commands for VRF'''

from collections import defaultdict

import BasicCli
import Tac
import CliMatcher
import CliCommand
import CliToken.Cli
import ShowCommand
import CliParser
import LazyMount
import ConfigMount
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import IraRouteCommon
from CliPlugin import RouteDistinguisher
from CliPlugin.IraRouterKernel import RouterKernelConfigMode
from CliPlugin.IraVrfModel import (
   Vrf,
   VrfSummary,
   VrfEntry,
   IpProtocolDesc,
   ProtocolInfo
)
from MultiRangeRule import MultiRangeMatcher
import Arnet
import Tracing
from IpLibConsts import (
      DEFAULT_VRF,
      DEFAULT_VRF_OLD,
)
from CliMode.Ira import VrfDefinitionBaseMode, VrfDefaultsMode
import BasicCliSession
import CliExtensions
from TypeFuture import TacLazyType

allVrfConfig = None
rdAutoInputDir = None
rdConfigInputDir = None
multiAgentVrfProtoConfig = None
allVrfStatusGlobal = None
routingVrfConfigDir = None
routingVrfRouteConfigDir = None
routingVrfRouteConfigDynamicDir = None
routingVrfInfoDir = None
routingConfig = None
routing6VrfConfigDir = None
routing6VrfRouteConfigDir = None
routing6VrfRouteConfigDynamicDir = None
routing6VrfInfoDir = None
routing6Config = None
arpInputConfigCli = None
routingHwStatusCommon = None
ip = None
ip6 = None

getAllPlusReservedVrfNames = VrfCli.getAllPlusReservedVrfNames
ipRedirect = IraRouteCommon.Ip4()
ip6Redirect = IraRouteCommon.Ip6()
routing = IraRouteCommon.routing( ipRedirect )
routing6 = IraRouteCommon.routing( ip6Redirect )
t0 = Tracing.trace0

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )

vrfHelpDesc = 'Configure VRFs'
vrfInstHelpDesc = 'Enter VRF definition sub-mode'
rdHelpDesc = 'BGP route distinguisher'

InternalIntfId = TacLazyType( 'Arnet::InternalIntfId' )
TristateU32 = TacLazyType( 'Ark::TristateU32' )

reservedPortsSysctlPath = '/proc/sys/net/ipv4/ip_local_reserved_ports'

# The CLI hook for VRFs is used to allow individual features to notify/block
# deletion of global VRF config which is otherwise supported. This allows those
# those features to remove any VRF specific configuration. Ira calls all hooks
# and ANDs the results, if any are False, the config is blocked. This function
# is called as:
#
# (accept, message) = hook( vrfName )
#
# vrfName is the name of relevant vrf.
#
# accept must be a boolean, message is a text string giving the specific
# reason why the deletion is blocked. The message is required.

canDeleteVrfHook = CliExtensions.CliHook()

deletedVrfHook = CliExtensions.CliHook()

######################################################################
# vrf instance sub-mode
#
# legacy:
# vrf definition <vrfName>
#
# Note that this is the new protocol-independent VRF definition mode.
# We are not supporting the old per-protocol 'ip vrf' and 'ipv6 vrf' CLI.
######################################################################

class VrfDefinitionMode( VrfDefinitionBaseMode, BasicCli.ConfigModeBase ):
   name = 'VRF Definition'
   vrfName = None
   vrfConfig = None

   def __init__( self, parent, session, vrfName ):
      VrfDefinitionBaseMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.vrfName = vrfName
      self.maybeMakeVrf()
      self.redirectDefault()

   def maybeMakeVrf( self ):
      vrfConfig = allVrfConfig.vrf.get( self.vrfName )
      if not vrfConfig:
         # Initialize default address family configuration to
         # True. The VrfSm will consider the configuration in the
         # context of the vrfCapability to determine the actual
         # enablement status of each address family.
         vrfConfig = allVrfConfig.vrf.newMember( self.vrfName )
         vrfConfig.addressFamily[ 'ipv4' ] = True
         vrfConfig.addressFamily[ 'ipv6' ] = True

      if self.vrfName not in routingVrfConfigDir.vrfConfig:
         vrfConfig = routingVrfConfigDir.newVrfConfig( self.vrfName )
         vrfRouteConfig = routingVrfRouteConfigDir.newVrfConfig( self.vrfName )
         vrfRouteConfigDynamic = routingVrfRouteConfigDynamicDir.newVrfConfig(
               self.vrfName )

      if self.vrfName not in routing6VrfConfigDir.vrfConfig:
         vrfConfig = routing6VrfConfigDir.newVrfConfig( self.vrfName )
         vrfRouteConfig = routing6VrfRouteConfigDir.newVrfConfig( self.vrfName )
         vrfRouteConfigDynamic = routing6VrfRouteConfigDynamicDir.newVrfConfig(
               self.vrfName )
         assert vrfConfig is not None
         assert vrfRouteConfig is not None
         assert vrfRouteConfigDynamic is not None

      if self.vrfName not in arpInputConfigCli.vrf:
         c = arpInputConfigCli.newVrf( self.vrfName )
         assert c is not None

   def redirectDefault( self ):
      r4 = routing.config( DEFAULT_VRF )
      routingVrfConfigDir.vrfConfig[ self.vrfName ].sendRedirects = r4.sendRedirects
      r6 = routing6.config( DEFAULT_VRF )
      routing6VrfConfigDir.vrfConfig[ self.vrfName ].sendRedirects = r6.sendRedirects

   def noAddressFamily( self, addressFamily ):
      allVrfConfig.vrf[ self.vrfName ].addressFamily[ addressFamily ] = False

class VrfAddressFamilyMode( BasicCli.ConfigModeBase ):
   name = 'VRF AF Mode'
   vrfName = None
   vrfConfig = None
   addressFamily = None

   def __init__( self, parent, session, vrfName, addressFamily ):
      self.modeKey = 'vrf-af'
      # pylint: disable-next=consider-using-f-string
      self.longModeKey = 'vrf-%s-af-%s' % ( vrfName, addressFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.vrfName = vrfName
      self.addressFamily = addressFamily
      self.vrfConfig = allVrfConfig.vrf.get( self.vrfName )
      assert self.vrfConfig
      self.vrfConfig.addressFamily[ addressFamily ] = True

   # route-target config commands go here when we add them

   def exitAddressFamily( self ):
      self.session_.gotoParentMode()

def gotoVrfAfMode( mode, addressFamily ):
   childMode = mode.childMode( VrfAddressFamilyMode,
                               vrfName=mode.vrfName,
                               addressFamily=addressFamily )
   mode.session_.gotoChildMode( childMode )

def getVrfNameFromIntf( _ipConfig, _ipStatus, _intfName, af=AddressFamily.ipv4 ):
   intfStatusAttr = 'ipIntfStatus' if af == AddressFamily.ipv4 else 'intf'
   ipIntfStatus = getattr( _ipStatus, intfStatusAttr ).get( _intfName )
   if ipIntfStatus:
      return ipIntfStatus.vrf
   intfConfigAttr = 'ipIntfConfig' if af == AddressFamily.ipv4 else 'intf'
   ipIntfConfig = getattr( _ipConfig, intfConfigAttr ).get( _intfName )
   if ipIntfConfig:
      return ipIntfConfig.vrf

   return DEFAULT_VRF

########## enter VRF mode
# 'definition' is actually a matcher wrapped by a Node, because only the node
# supports deprecatedBy.
# 'definition' is deliberately not hidden. I do not know why, but it was also not
# hidden before conversion to new parser.
vrfDefDeprecatedMatcher = CliMatcher.KeywordMatcher(
   'definition', 'Enter VRF definition sub-mode' )
vrfDefDeprecatedNode = CliCommand.Node( vrfDefDeprecatedMatcher,
                                        deprecatedByCmd='vrf instance [VRF_ID]' )

# Note: CliVrfTestLib.acceptedVrfNameCharacters should be updated to cover all
#       accepted characters in the VRF name pattern.
vrfNamePattern = r'[A-Za-z0-9_.:{}\[\]-]+'
vrfNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: allVrfConfig.vrf,
                                                'VRF name',
                                                pattern=vrfNamePattern )
allVrfNameMatcher = CliMatcher.DynamicNameMatcher( VrfCli.getAllVrfNames,
                                                   helpdesc='VRF name',
                                                   pattern=vrfNamePattern )

allPlusReservedVrfNameMatcher = CliMatcher.DynamicNameMatcher(
      getAllPlusReservedVrfNames, 'VRF name', pattern=vrfNamePattern )

class EnterVrfDefMode( CliCommand.CliCommandClass ):
   syntax = '''vrf ( instance | definition ) VRF_NAME'''
   noOrDefaultSyntax = syntax

   data = { "vrf": vrfHelpDesc,
            "instance": vrfInstHelpDesc,
            "definition": vrfDefDeprecatedNode,
            "VRF_NAME": vrfNameMatcher }

   handler = "IraVrfCliHandler.handlerEnterVrfDefMode"
   noOrDefaultHandler = "IraVrfCliHandler.noOrDefaultHandlerEnterVrfDefMode"

BasicCli.GlobalConfigMode.addCommandClass( EnterVrfDefMode )

########## set the RD

routeDistinguisherMatcher = RouteDistinguisher.RdDistinguisherMatcher(
                            helpdesc='BGP route distinguisher' )

class VrfRdCmd( CliCommand.CliCommandClass ):
   syntax = '''rd <rd>'''
   noOrDefaultSyntax = '''rd ...'''

   data = { "rd": rdHelpDesc,
            "<rd>": routeDistinguisherMatcher }

   handler = "IraVrfCliHandler.handlerVrfRdCmd"
   noOrDefaultHandler = "IraVrfCliHandler.noOrDefaultHandlerVrfRdCmd"

VrfDefinitionMode.addCommandClass( VrfRdCmd )

######### set description command

class VrfDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = '''description <desc>'''
   noOrDefaultSyntax = '''description ...'''

   data = { "description": 'VRF description',
            "<desc>": CliMatcher.StringMatcher( helpname='LINE',
                              helpdesc='Up to 240 characters describing this VRF' )
          }

   handler = "IraVrfCliHandler.handlerVrfDescriptionCmd"
   noOrDefaultHandler = "IraVrfCliHandler.noOrDefaultHandlerVrfDescriptionCmd"

VrfDefinitionMode.addCommandClass( VrfDescriptionCmd )

# set expected route count command

class VrfRouteExpectedCountCmd( CliCommand.CliCommandClass ):
   syntax = 'route expected count NUMBER'
   noOrDefaultSyntax = 'route expected count ...'

   data = {
      "route": 'Configure route parameters for this VRF',
      "expected": 'Configure expected route parameters for this VRF',
      "count": 'Configure expected route count for this VRF',
      'NUMBER': CliMatcher.IntegerMatcher( 0, 0xffffffff,
         helpdesc='Number of expected routes in this VRF' ),
   }

   handler = "IraVrfCliHandler.handlerVrfRouteExpectedCountCmd"
   noOrDefaultHandler = \
      "IraVrfCliHandler.noOrDefaultHandlerVrfRouteExpectedCountCmd"

VrfDefinitionMode.addCommandClass( VrfRouteExpectedCountCmd )

# set default expected route count command

class VrfDefaultsRouteExpectedCountCmd( CliCommand.CliCommandClass ):
   syntax = 'route expected count NUMBER'
   noOrDefaultSyntax = 'route expected count ...'

   data = {
      "route": 'Configure default route parameters for all VRFs',
      "expected": 'Configure default expected route parameters for all VRFs',
      "count": 'Configure default expected route count for all VRFs',
      'NUMBER': CliMatcher.IntegerMatcher( 0, 0xffffffff,
         helpdesc='Default number of expected routes per VRF' ),
   }

   handler = "IraVrfCliHandler.handlerVrfDefaultsRouteExpectedCountCmd"
   noOrDefaultHandler = \
      "IraVrfCliHandler.noOrDefaultHandlerVrfDefaultsRouteExpectedCountCmd"

VrfDefaultsMode.addCommandClass( VrfDefaultsRouteExpectedCountCmd )

class VrfDefaultsCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf defaults'
   noOrDefaultSyntax = syntax

   data = {
      "vrf": 'Configure VRFs',
      "defaults": 'Configure default parameters for all VRFs',
   }

   handler = "IraVrfCliHandler.handlerVrfDefaultsCmd"
   noOrDefaultHandler = "IraVrfCliHandler.noOrDefaultHandlerVrfDefaultsCmd"

BasicCli.GlobalConfigMode.addCommandClass( VrfDefaultsCmd )

########## AF mode entry/exit

#
# This is currently disabled.  For the first release we are only
# supporting IPv4.  Rather than force the user to type a bunch of
# useless commands to enable IPv4, we're going with the following
# semantics:
#
# IPv4 is enabled by default (and cannot be disabled in 4.10.0)
# IPv6 is disabled by default (and cannot be enabled in 4.10.0)
#
# to disable IPv4 in future releases we will do:
# ! address-family ipv4
# !   shutdown
# ! exit-address-family
#

#tokenAfInVrf = CliParser.KeywordRule( 'address-family',
#                                      helpdesc='Configure address family for VRF' )
#
#afOrRule = CliParser.OrRule( name='addressFamily' )
#afOrRule |= CliParser.KeywordRule( 'ipv4', helpdesc='IPv4 related' )
#afOrRule |= CliParser.KeywordRule( 'ipv6', helpdesc='IPv6 related' )
#
#VrfDefinitionMode.addCommand( ( tokenAfInVrf, afOrRule, gotoVrfAfMode ) )
#VrfDefinitionMode.addCommand( ( BasicCli.noOrDefault, tokenAfInVrf, afOrRule,
#                                VrfDefinitionMode.noAddressFamily ) )
#
#tokenExitAf = CliParser.KeywordRule( 'exit-address-family',
#                                     helpdesc='Exit from address-family mode' )
#
#VrfAddressFamilyMode.addCommand( ( tokenExitAf,
#                                   VrfAddressFamilyMode.exitAddressFamily ) )


######################################################################
# show vrf.
######################################################################

showMCastHook = CliExtensions.CliHook()
# showMCastHook allows Mroute to provide extra
# information about enabled/disabled multicast routing.

def intfsByVrf( ipIntfs, ip6Intfs ):
   '''
   return a list of interface assignment by VRFs
   '''
   intfsByVrfDict = defaultdict( set )
   for intfs in [ ipIntfs, ip6Intfs ]:
      for intfName, intf in intfs.items():
         intfsByVrfDict[ intf.vrf ].add( intfName )
   return intfsByVrfDict

def intfsByIpVrfKey( ipIntfs, vrfKey ):
   '''
   Return list holding interfaces with the provided VRF key.
   '''
   intfsByVrfKey = []
   for intfName, intf in ipIntfs.items():
      if intf.vrf == vrfKey:
         intfsByVrfKey.append( intfName )

   return intfsByVrfKey

def populateProtocolInfoModel( protocolInfoModel, af, vrfState, vrfKey,
                               isDefaultVrf, vrfConfigOrStatus, mode ):
   proto = af.strip( 'ip' )
   if af == 'ipv4':
      afEnabled = routingHwStatusCommon.vrfCapability.ipv4EnabledDefault
   else:
      afEnabled = routingHwStatusCommon.vrfCapability.ipv6EnabledDefault
   if isDefaultVrf or ( vrfConfigOrStatus is not None and
                        af in vrfConfigOrStatus.addressFamily and
                        vrfConfigOrStatus.addressFamily[ af ] and
                        afEnabled ):
      protocolInfoModel.supported = True

   if vrfState != "up":
      return

   vrfConfigDir = routingVrfConfigDir if af == 'ipv4' else routing6VrfConfigDir
   configDir = routingConfig if af == 'ipv4' else routing6Config
   vrfInfoDir = routingVrfInfoDir if af == 'ipv4' else routing6VrfInfoDir

   rc = configDir if isDefaultVrf else vrfConfigDir.vrfConfig.get( vrfKey )
   rs = vrfInfoDir.get( vrfKey )
   if not ( rc and rs ):
      protocolInfoModel.protocolState = "down"
      protocolInfoModel.routingState = "down"
   else:
      if rc.routing:
         protocolInfoModel.routingState = "up"
         for hook in showMCastHook.extensions():
            if hook( mode, vrfKey, proto ):
               if rc.routing != rs.routing:
                  protocolInfoModel.multicastState = "initializing"
               else:
                  protocolInfoModel.multicastState = "up"

         if rc.routing != rs.routing:
            protocolInfoModel.routingState = "initializingUp"
      else:
         protocolInfoModel.routingState = "down"
         if rc.routing != rs.routing:
            protocolInfoModel.routingState = "initializingDown"

def getVrfState( vrfS, vrfC, isDefaultVrf ):
   vrfState = "up"
   if vrfS is None and not isDefaultVrf:
      vrfState = "creating"
   if vrfC is None and not isDefaultVrf:
      vrfState = "deleting"
   return vrfState

def getVrfTextAddon( vrfKey, ipIntf ):
   intfsIp = intfsByIpVrfKey( ipIntf, vrfKey )
   interfacesAddon = []
   if intfsIp:
      for intfName in Arnet.sortIntf( intfsIp ):
         intfId = Arnet.IntfId( intfName )
         if not InternalIntfId.isInternalIntfId( intfId ):
            interfacesAddon.append( intfId )
   return interfacesAddon

def getVrfEntryModel( mode, vrfKey, intfsByVrfColl ):
   isDefaultVrf = vrfKey in ( DEFAULT_VRF, DEFAULT_VRF_OLD )

   vrfS = allVrfStatusGlobal.vrf.get( vrfKey )
   vrfC = allVrfConfig.vrf.get( vrfKey )
   if not vrfS and not vrfC and not isDefaultVrf:
      return None
   vrfEntryModel = VrfEntry()
   vrfEntryModel.protocols = IpProtocolDesc()
   vrfEntryModel.interfaces = []
   vrfEntryModel.routeDistinguisher = ""
   vrfEntryModel.protocols.ipv4 = ProtocolInfo(
         supported=False, protocolState="up",
         routingState="up" )
   vrfEntryModel.protocols.ipv6 = ProtocolInfo(
         supported=False, protocolState="up",
         routingState="up" )
   protoDict = { 'v4': vrfEntryModel.protocols.ipv4,
                 'v6': vrfEntryModel.protocols.ipv6 }
   if isDefaultVrf:
      vrfEntryModel.vrfId = 0
   else:
      nameToIdMap = routing.vrfNameStatus().nameToIdMap
      if nameToIdMap:
         vrfId = nameToIdMap.vrfNameToId.get( vrfKey )
         # don't populate vrfId unless we have it
         if vrfId:
            vrfEntryModel.vrfId = vrfId

   vrfConfigOrStatus = vrfS
   if vrfS is None and not isDefaultVrf:
      vrfConfigOrStatus = vrfC

   vrfEntryModel.vrfState = getVrfState( vrfS, vrfC, isDefaultVrf )

   for af in ( 'ipv4', 'ipv6' ):
      proto = af.strip( 'ip' )
      populateProtocolInfoModel(
            protoDict[ proto ], af, vrfEntryModel.vrfState, vrfKey,
            isDefaultVrf, vrfConfigOrStatus, mode )

   intfs = intfsByVrfColl.get( vrfKey )
   if intfs:
      for intfName in Arnet.sortIntf( intfs ):
         intfId = Arnet.IntfId( intfName )
         if not InternalIntfId.isInternalIntfId( intfId ):
            vrfEntryModel.interfaces.append( intfId )

   # For default vrf, we can only retrieve value
   # from BGP routeDistinguisherInputDir
   if isDefaultVrf:
      assert 'bgp' in rdConfigInputDir
      bgpRdInput = rdConfigInputDir.get( "bgp" ).routeDistinguisher
      rd = bgpRdInput.get( vrfKey )
      if rd and rd != 'INVALID':
         vrfEntryModel.routeDistinguisher = rd
      else:
         autoRdInput = rdAutoInputDir.get( "bgpAuto" )
         if autoRdInput:
            rd = autoRdInput.routeDistinguisher.get( vrfKey )
            if rd and rd != 'INVALID':
               vrfEntryModel.routeDistinguisher = rd
   else:
      if vrfS and vrfS.routeDistinguisher != 'INVALID':
         vrfEntryModel.routeDistinguisher = vrfS.routeDistinguisher

   # Add V4 and V6 classification for interfaces
   vrfEntryModel.interfacesV4 = getVrfTextAddon( vrfKey, ip.ipIntfStatus )
   vrfEntryModel.interfacesV6 = getVrfTextAddon( vrfKey, ip6.intf )

   return vrfEntryModel

def showVrfSummary( mode ):
   vrfCount = 0
   vrfUpCount = 0
   vrfV4RoutingCount = 0
   vrfV6RoutingCount = 0

   allVrfs = set( allVrfStatusGlobal.vrf ).\
      union( set( allVrfConfig.vrf ) )
   allVrfs.add( DEFAULT_VRF )

   for vrfKey in allVrfs:
      isDefaultVrf = vrfKey in ( DEFAULT_VRF, DEFAULT_VRF_OLD )
      vrfS = allVrfStatusGlobal.vrf.get( vrfKey )
      vrfC = allVrfConfig.vrf.get( vrfKey )
      if not vrfS and not vrfC and not isDefaultVrf:
         continue
      vrfConfigOrStatus = vrfS
      if vrfS is None and not isDefaultVrf:
         vrfConfigOrStatus = vrfC
      vrfCount += 1
      v4ProtocolInfo = ProtocolInfo( supported=False, protocolState="up",
                                     routingState="up" )
      v6ProtocolInfo = ProtocolInfo( supported=False, protocolState="up",
                                     routingState="up" )
      vrfState = getVrfState( vrfS, vrfC, isDefaultVrf )
      if vrfState == "up":
         vrfUpCount += 1
      else:
         # if the vrfState is not up we don't care about its protocols
         continue

      populateProtocolInfoModel(
         v4ProtocolInfo, 'ipv4', vrfState, vrfKey, isDefaultVrf, vrfConfigOrStatus,
         mode )
      populateProtocolInfoModel(
         v6ProtocolInfo, 'ipv6', vrfState, vrfKey, isDefaultVrf, vrfConfigOrStatus,
         mode )

      if v4ProtocolInfo.routingState == "up":
         vrfV4RoutingCount += 1
      if v6ProtocolInfo.routingState == "up":
         vrfV6RoutingCount += 1

   return VrfSummary( vrfCount=vrfCount, vrfUpCount=vrfUpCount,
                      vrfV4RoutingCount=vrfV4RoutingCount,
                      vrfV6RoutingCount=vrfV6RoutingCount )

def showVrf( mode, vrfName=None ):
   vrfModel = Vrf()
   assert( allVrfStatusGlobal is not None and
           allVrfConfig is not None )
   
   # get all intf -> vrf mappings
   intfsByVrfColl = intfsByVrf( ip.ipIntfStatus, ip6.intf )
      
   if not vrfName:
      vrfMax = routingHwStatusCommon.vrfCapability.maxVrfs
      if vrfMax > 0:
         # pylint: disable-next=consider-using-f-string
         print( "Maximum number of VRFs allowed: %d" % vrfMax )
      allVrfs = set( allVrfStatusGlobal.vrf.keys() ).\
         union( set( allVrfConfig.vrf.keys() ) )
      # get vrfEntryModel for all vrfs in
      # allVrfStatusGlobal/allVrfConfig collection
      for vrfKey in allVrfs:
         vrfEntryModel = getVrfEntryModel( mode, vrfKey, intfsByVrfColl )
         if vrfEntryModel is None:
            continue
         vrfModel.vrfs[ vrfKey ] = vrfEntryModel

      # special case: default vrf
      # default vrf is not in allVrfStatusGlobal/allVrfConfig collections
      vrfEntryModel = getVrfEntryModel( mode, DEFAULT_VRF, intfsByVrfColl )
      if vrfEntryModel:
         vrfModel.vrfs[ DEFAULT_VRF ] = vrfEntryModel
   else:
      vrfEntryModel = getVrfEntryModel( mode, vrfName, intfsByVrfColl )
      if vrfEntryModel:
         vrfModel.vrfs[ vrfName ] = vrfEntryModel

   return vrfModel

vrfKwForShow = CliMatcher.KeywordMatcher( 'vrf', helpdesc='Display VRF state' )

class ShowVrfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vrf [ <vrfName> ]'''
   data = { 'vrf': vrfKwForShow,
            '<vrfName>': allVrfNameMatcher
          }
   cliModel = Vrf

   @staticmethod
   def handler( mode, args ):
      return showVrf( mode, vrfName=args.get( "<vrfName>" ) )

BasicCli.addShowCommandClass( ShowVrfCmd )

class ShowVrfSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vrf summary'
   data = {
      'vrf': vrfKwForShow,
      'summary': 'VRF summary'
   }
   cliModel = VrfSummary

   @staticmethod
   def handler( mode, args ):
      return showVrfSummary( mode )

BasicCli.addShowCommandClass( ShowVrfSummaryCmd )

class ShowVrfReservedPortsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vrf [ VRF_NAME ] reserved-ports'''
   data = { 'vrf': vrfKwForShow,
            'VRF_NAME': allVrfNameMatcher,
            'reserved-ports': 'VRF reserved range of ports',
   }
   cliModel = "IraVrfModel.VrfReservedPorts"

   handler = "IraVrfCliHandler.handlerShowVrfReservedPortsCmd"

BasicCli.addShowCommandClass( ShowVrfReservedPortsCmd )

######################################################################
# cli vrf <vrfName>
#
# legacy:
# routing-context vrf <vrfName>
######################################################################

routeCtxKwDeprecated = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                             'routing-context',
                                             helpdesc='Enter VRF context' ),
                                        deprecatedByCmd='cli vrf [VRF_ID]' )

class VrfForRtCtxCmd( CliCommand.CliCommandClass ):
   # pylint: disable-next=consider-using-f-string
   syntax = "( cli | routing-context ) vrf ( <vrfName> | %s | %s )" % (
            DEFAULT_VRF, DEFAULT_VRF_OLD )

   data = { "cli": CliToken.Cli.cliForExecMatcher,
            "routing-context": routeCtxKwDeprecated,
            "vrf": 'Enter VRF context',
            "<vrfName>": vrfNameMatcher,
            DEFAULT_VRF: CliCommand.Node(
                  CliMatcher.KeywordMatcher( DEFAULT_VRF,
                        helpdesc='Default virtual routing and forwarding instance',
                                             priority=CliParser.PRIO_HIGH ) ),
            DEFAULT_VRF_OLD: CliCommand.Node(
                  CliMatcher.KeywordMatcher( DEFAULT_VRF_OLD,
                   helpdesc='Default virtual routing and forwarding instance (old)',
                                             priority=CliParser.PRIO_HIGH ),
                  hidden=True )
          }

   handler = "IraVrfCliHandler.handlerVrfForRtCtxCmd"

BasicCli.EnableMode.addCommandClass( VrfForRtCtxCmd )

######################################################################
# show routing-context vrf
# show cli vrf
######################################################################

cliKwForShow = CliToken.Cli.cliForShowMatcher
routeCtxKw = CliMatcher.KeywordMatcher( 'routing-context', helpdesc='VRF context' )
vrfKwForShowRtCtx = CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF context' )

class ShowCliVrfCtxCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ( cli | routing-context ) vrf'''
   data = { 'cli': cliKwForShow,
            'routing-context': routeCtxKw,
            'vrf': vrfKwForShowRtCtx }
   handler = "IraVrfCliHandler.handlerShowCliVrfCtxCmd"
   cliModel = "IraVrfModel.ShowCliVrfCtxModel"

BasicCli.addShowCommandClass( ShowCliVrfCtxCmd )

#-------------------------------------------------------------
# Register a format specifier for a vrf Name in the Cli prompt
#-------------------------------------------------------------
def getVrfCliPrompt( session ):
   vrfName = VrfCli.vrfMap.getCliSessVrf( session )
   if VrfCli.vrfMap.isDefaultVrf( vrfName ):
      return ""
   else:
      return "(vrf:%s)" % vrfName # pylint: disable=consider-using-f-string

#-------------------------------------------------------------
# Add vrfname to 'routing/multiAgentVrfProtocolConfig'
# when router <ospf/isis> <instance> vrf <vrfname>
# is run for the first time.
#-------------------------------------------------------------
def findOrCreateMultiAgentVrfProtoEntry( vrfName ):
   multiAgentVrfProtocolEntry = \
         multiAgentVrfProtoConfig.vrfConfig.newMember( vrfName )
   return multiAgentVrfProtocolEntry

#-------------------------------------------------------------
# Remove vrfName from 'routing/multiAgentVrfProtocolConfig'
# when no router <ospf/isis> <instance> vrf <vrfname>
# is run, and all agents have been unconfigured
#-------------------------------------------------------------
def maybeDelMultiAgentVrfProtoEntry( vrfName ):
   if len( multiAgentVrfProtoConfig.vrfConfig.get( vrfName ).agent ) == 0:
      del multiAgentVrfProtoConfig.vrfConfig[ vrfName ]

def addAgentVrfEntry( vrfName, agentName ):
   if vrfName == '' or vrfName is None:
      vrfName = DEFAULT_VRF
   multiAgentVrfProtocolEntry = \
         findOrCreateMultiAgentVrfProtoEntry( vrfName )
   multiAgentVrfProtocolEntry.agent.newMember( agentName, "", "" )

def removeAgentVrfEntry( vrfName, agentName ):
   if vrfName == '' or vrfName is None:
      vrfName = DEFAULT_VRF
   if vrfName in multiAgentVrfProtoConfig.vrfConfig:
      del multiAgentVrfProtoConfig.vrfConfig.get( vrfName ).agent[ agentName ]
      maybeDelMultiAgentVrfProtoEntry( vrfName )

# -------------------------------------------------------------------------------
# ip1(config-router-kernel-vrf-VRFNAME)#
# [ no | default ] reserved-ports PORTS
#
# This command writes the value specified by PORTS to Sysdb, which then will be used
# by Ira to write to the sysctl 'ipv4.net.ip_local_reserved_ports'.
# -------------------------------------------------------------------------------
portMatcher = MultiRangeMatcher( lambda: ( 1, 65535 ), noSingletons=False,
      helpdesc='List of comma separated ranges of port values' )

class ReservedPortsCmd( CliCommand.CliCommandClass ):
   syntax = 'reserved-ports PORT'
   noOrDefaultSyntax = 'reserved-ports ...'

   data = {
      'reserved-ports': 'Reserve a range of ports',
      'PORT': portMatcher,
      }

   handler = "IraVrfCliHandler.handlerReservedPortsCmd"
   noOrDefaultHandler = handler

RouterKernelConfigMode.addCommandClass( ReservedPortsCmd )

######################################################################
# boilerplate
######################################################################

def Plugin( entityManager ):
   routing.plugin( entityManager )
   routing6.plugin( entityManager )

   global allVrfConfig
   global rdAutoInputDir
   global rdConfigInputDir
   global allVrfStatusGlobal
   global ip
   global ip6
   global routingVrfConfigDir
   global routingVrfRouteConfigDir
   global routingVrfRouteConfigDynamicDir
   global routingConfig
   global routingVrfInfoDir
   global routing6VrfConfigDir
   global routing6VrfRouteConfigDir
   global routing6VrfRouteConfigDynamicDir
   global routing6VrfInfoDir
   global routing6Config
   global arpInputConfigCli

   global routingHwStatusCommon
   global multiAgentVrfProtoConfig

   allVrfConfig = ConfigMount.mount( entityManager, "ip/vrf/config",
                                     "Ip::AllVrfConfig", "w" )

   rdAutoInputDir = \
      LazyMount.mount( entityManager, 'ip/vrf/routeDistinguisherInputDir/auto',
                         'Tac::Dir', 'ri' )
   rdConfigInputDir = \
      ConfigMount.mount( entityManager, 'ip/vrf/routeDistinguisherInputDir/config',
                         'Tac::Dir', 'w' )

   multiAgentVrfProtoConfig = ConfigMount.mount( entityManager,
         'routing/multiAgentVrfProtocolConfig',
         'VrfTypes::MultiAgentVrfProtoConfig', 'w' )
   allVrfStatusGlobal = LazyMount.mount( entityManager, "ip/vrf/status/global",
                                         "Ip::AllVrfStatusGlobal", "r" )
   routingHwStatusCommon = LazyMount.mount(
      entityManager, "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )
   ip = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ip6 = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   routingVrfConfigDir = ConfigMount.mount( entityManager, "routing/vrf/config",
                                            "Routing::VrfConfigDir", "wi" )
   routingVrfRouteConfigDir = ConfigMount.mount( entityManager,
                                            "routing/vrf/routelistconfig",
                                            "Routing::VrfRouteListConfigDir", "wi" )
   routingVrfRouteConfigDynamicDir = LazyMount.mount( entityManager,
                                            "routing/vrf/routelistconfigdynamic",
                                            "Routing::VrfRouteListConfigDir", "wi" )
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                          "routing/vrf/routingInfo/status",
                                          "Tac::Dir", "ri" )
   routingConfig = LazyMount.mount( entityManager,
                                           "routing/config",
                                           "Routing::Config", "ri" )
   routing6VrfConfigDir = ConfigMount.mount( entityManager, "routing6/vrf/config",
                                            "Routing6::VrfConfigDir", "wi" )
   routing6VrfRouteConfigDir = ConfigMount.mount( entityManager,
                                            "routing6/vrf/routelistconfig",
                                            "Routing6::VrfRouteListConfigDir", "wi" )
   routing6VrfRouteConfigDynamicDir = LazyMount.mount( entityManager,
                                            "routing6/vrf/routelistconfigdynamic",
                                            "Routing6::VrfRouteListConfigDir", "wi" )
   routing6VrfInfoDir = LazyMount.mount( entityManager,
                                         "routing6/vrf/routingInfo/status",
                                         "Tac::Dir", "ri" )
   routing6Config = LazyMount.mount( entityManager,
                                           "routing6/config",
                                           "Routing6::Config", "ri" )
   arpInputConfigCli = ConfigMount.mount( entityManager,
                                       "arp/input/config/cli",
                                       "Arp::InputConfig", "wS" )

   BasicCliSession.addFormatSpecifier( 'v', getVrfCliPrompt )
