# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
      Bool,
      Dict,
      List,
      Model,
      Str,
      Enum,
      Submodel,
      Int
)
from TableOutput import (
      createTable,
      Format
)
from IntfModels import Interface
import textwrap
from CliMode.Intf import IntfMode

class ProtocolInfo( Model ):
   supported = Bool( help="Indicates whether the protocol is supported" )
   protocolState = Enum( values=( "up", "down" ),
         help="Protocol state is down when the vrf is being created, state "
         "is up otherwise" )
   routingState = Enum( values=(
         "up", "down", "initializingUp", "initializingDown" ),
         help="Indicates the various states of routing for the Vrf."
         "'up' indicates routing is enabled for this vrf. "
         "'down' indicates routing is disabled for this vrf. "
         "'initializing up/down' indicates routing state is being configured" )
   multicastState = Enum( values=( "up", "initializing" ),
         help="Indicates the state of the multicast routing after it is"
         " configured", optional=True )

   def formatStateStr( self ):
      if self.protocolState == "down":
         return "incomplete"
      elif self.routingState == "up":
         return "routing"
      elif self.routingState == "initializingUp":
         return "pending_routing"
      elif self.routingState == "initializingDown":
         return "pending_no routing"
      elif self.routingState == "down":
         return "no routing"
      else:
         return ""

   def formatMulticastStr( self ):
      if self.multicastState == "up":
         return "multicast"
      elif self.multicastState == "initializing":
         return "pending_multicast"
      return ""

   def formatStateStrWithMcast( self ):
      stateStr = self.formatStateStr()
      mcast = self.formatMulticastStr()
      if mcast:
         stateStr += "; "
         stateStr += mcast
      return stateStr

class IpProtocolDesc( Model ):
   ipv4 = Submodel( valueType=ProtocolInfo,
         help="ipv4 protocol information for a vrf" )
   ipv6 = Submodel( valueType=ProtocolInfo,
         help="ipv6 protocol information for a vrf" )

class VrfEntry( Model ):
   routeDistinguisher = Str( help="Route Distinguisher associated with this vrf" )
   protocols = Submodel( valueType=IpProtocolDesc,
         help="Information on ipv4 and ipv6 support for this vrf" )
   vrfState = Enum( values=( "creating", "up", "deleting" ),
         help="Various states of vrf. 'creating' indicates vrf is being"
         " created, 'up' indicates vrf is created successfully and 'deleting'"
         " indicates vrf is being deleted" )
   interfacesV4 = List( valueType=Interface, optional=True,
         help="List of IPv4 interfaces which are assigned to this vrf" )
   interfacesV6 = List( valueType=Interface, optional=True,
         help="List of IPv6 interfaces which are assigned to this vrf" )
   # Union of v4 and v6 interfaces.
   # Present in CAPI model to avoid incrementing model revision.
   interfaces = List( valueType=Interface,
         help="List of interfaces which are assigned to this vrf" )
   # Each L3-VRF in EOS has a unique Id associated with it which can
   # be found in quicktraces and other places. Having it as part of
   # show vrf allows us in debugging
   vrfId = Int( optional=True,
                help='Id associated with this vrf' )

   def formatStateStr( self, protocolType ):
      if self.vrfState == "up":
         if protocolType is None:
            return "up"
         elif protocolType == 4:
            return self.protocols.ipv4.formatStateStrWithMcast()
         else:
            return self.protocols.ipv6.formatStateStrWithMcast()
      elif self.vrfState == "creating":
         return "pending create"
      elif self.vrfState == "deleting":
         return "pending delete"
      else:
         return ""

class VrfSummary( Model ):
   vrfCount = Int( help="Number of VRFs" )
   vrfUpCount = Int( help="Number of up VRFs" )
   vrfV4RoutingCount = Int( help="Number of VRFs supporting IPv4 routing" )
   vrfV6RoutingCount = Int( help="Number of VRFs supporting IPv6 routing" )

   def render( self ):
      # pylint: disable-next=consider-using-f-string
      print( "VRF count: %d" % self.vrfCount )
      # pylint: disable-next=consider-using-f-string
      print( "VRF up count: %d" % self.vrfUpCount )
      # pylint: disable-next=consider-using-f-string
      print( "VRF IPv4 routing count: %d" % self.vrfV4RoutingCount )
      # pylint: disable-next=consider-using-f-string
      print( "VRF IPv6 routing count: %d" % self.vrfV6RoutingCount )

#--------------------------------------------------------
# show vrf Model
# Contains a dictionary of vrf entries based on vrf name
# and the output is rendered in the form of a table
#---------------------------------------------------------
class Vrf( Model ):
   ''' render all the goodies in the Vrf '''
   vrfs = Dict( keyType=str, valueType=VrfEntry,
         help="Mapping of Vrf name to Vrf information" )

   def render( self ):
      t = createTable( ( 'VRF', 'Protocols', 'State', 'Interfaces' ) )
      f1 = Format( justify='left' )
      f2 = Format( justify='left' )
      ifWidth = 40  # If more then 1 interface in table, we don't like it to wrap
      f3 = Format( justify='left', maxWidth=40 )
      f4 = Format( justify='left', maxWidth=ifWidth )
      t.formatColumns( f1, f2, f3, f4 )

      # Naive way to print two new rows.
      for vrf, vrfEntry in sorted( self.vrfs.items() ):
         # pylint: disable-next=singleton-comparison
         if vrfEntry.protocols.ipv4.supported == True:
            intfListIpv4 = [ IntfMode.getShortname( intf )
                           for intf in vrfEntry.interfacesV4 ]
            t.newRow( vrf,
                     "IPv4",
                     vrfEntry.formatStateStr( protocolType=4 ),
                     textwrap.fill( ', '.join( intfListIpv4 ), ifWidth ) )
         # pylint: disable-next=singleton-comparison
         if vrfEntry.protocols.ipv6.supported == True:
            intfListIpv6 = [ IntfMode.getShortname( intf )
                           for intf in vrfEntry.interfacesV6 ]
            t.newRow( vrf,
                     "IPv6",
                     vrfEntry.formatStateStr( protocolType=6 ),
                     textwrap.fill( ', '.join( intfListIpv6 ), ifWidth ) )
         if ( not vrfEntry.protocols.ipv4.supported and
              not vrfEntry.protocols.ipv6.supported ):
            # Interface list should be empty but print it anyway,
            # easier to spot any bug.
            intfList = [ IntfMode.getShortname( intf )
                        for intf in vrfEntry.interfaces ]
            t.newRow( vrf,
                      "not supported", # protocol string
                      vrfEntry.formatStateStr( protocolType=None ),
                      textwrap.fill( ', '.join( intfList ), ifWidth ) )

      print( t.output() )

