#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Clear
import CliToken.Monitor
from CliPlugin import ItsCliModel
from CliPlugin import TechSupportCli
from CliPlugin.DpsShowCli import getPathIndex, \
   pathNameMatcher, dstIpv4AddrMatcher, \
   srcIpv4AddrMatcher, peerIpv4AddrMatcher, \
   trafficClassMatcher, peerNameMatcher, pathGroupMatcher
from CliPlugin.WanTEShowCli import dpsSupported
import LazyMount
import ShowCommand
import SmashLazyMount
import Tac

itsStatus = None
itsStateStatus = None
itsCountersSnapShot = None
dpsPathStatusDir = None
dpsPeerStatus = None
dpsVrfStatus = None

ItsSessCounterData = Tac.Type( 'Its::ItsSessCounterData' )
ItsSessKey = Tac.Type( 'Its::ItsSessKey' )

def monitorTelemetrySupportedGuard( mode, token ):
   return None

def monitorTelemetryPathSupportedGuard( mode, token ):
   if dpsSupported():
      return None
   return CliParser.guardNotThisPlatform

telemetryKwMatcher = CliMatcher.KeywordMatcher( 'telemetry',
                         helpdesc='Show monitor telemetry information' )
telemetryNode = CliCommand.Node( matcher=telemetryKwMatcher,
                         guard=monitorTelemetrySupportedGuard )
telemetryClearKwMatcher = CliMatcher.KeywordMatcher( 'telemetry',
                         helpdesc='Clear monitor telemetry information' )
telemetryClearNode = CliCommand.Node( matcher=telemetryClearKwMatcher,
                         guard=monitorTelemetrySupportedGuard )
pathKwMatcher = CliMatcher.KeywordMatcher( 'path',
                         helpdesc='Show path telemetry information' )
pathNode = CliCommand.Node( matcher=pathKwMatcher,
                         guard=monitorTelemetryPathSupportedGuard )
clearMatcher = CliToken.Clear.clearKwNode

def getPathName( pathId ):
   return 'path' + str( pathId )

def getDpsPathStatus( vrfId, pathIndex ):
   vrfName = dpsVrfStatus.vrfIdToName.get( vrfId )
   if vrfName is None:
      return None
   vrfStatus = dpsPathStatusDir.pathStatus.get( vrfName )
   if vrfStatus is None:
      return None
   return vrfStatus.pathStatusEntry.get( pathIndex )

def getDpsPeerInfo( pathStatus ):
   peerIp = pathStatus.peerIp
   peerName = ''
   if peerIp in dpsPeerStatus.peerStatusEntry:
      peerName = dpsPeerStatus.peerStatusEntry[ peerIp ].peerName
   return peerIp.stringValue, peerName

def getItsSessions( mode, args, charact ):
   model = ItsCliModel.ItsPaths()
   model._detail = 'detail' in args  # pylint: disable=protected-access
   model._charact = charact # pylint: disable=protected-access

   dst = args.get( 'DSTIP' )
   pathName = args.get( 'NAME' )
   peer = args.get( 'PEERIP' )
   src = args.get( 'SRCIP' )
   tc = args.get( 'TC' )
   peerName = args.get( 'PEERNAME' )
   groupName = args.get( 'GROUPNAME' )

   sessionState = itsStateStatus.itsSessionState
   if charact:
      sessData = itsStatus.itsSessCharacData
   else:
      sessData = itsStatus.itsSessCounterData

   for sessKey, session in sessData.items():
      pathStatus = None
      sessPathId = sessKey.pathId
      sessVrfId = sessKey.vrfId
      pathStatus = getDpsPathStatus( sessVrfId, sessPathId )
      if not pathStatus:
         continue

      sessPeer, sessPeerName = getDpsPeerInfo( pathStatus )
      pathKey = pathStatus.pathKey
      sessDst = pathKey.dstAddr
      sessSrc = pathKey.srcAddr
      sessTrafficClass = sessKey.trafficClass
      sessPathName = getPathName( sessPathId )
      sessGroupName = pathStatus.pgName

      if ( ( dst and dst != sessDst.stringValue ) or
           ( pathName and getPathIndex( pathName ) != sessPathId ) or
           ( peer and peer != sessPeer ) or
           ( src and src != sessSrc.stringValue ) or
           ( tc and tc != sessTrafficClass ) or
           ( peerName and peerName != sessPeerName ) or
           ( groupName and groupName != sessGroupName ) ):
         continue

      if sessPeer in model.peers:
         modelPeer = model.peers[ sessPeer ]
      else:
         modelPeer = ItsCliModel.ItsPeer()
         modelPeer.peerName = sessPeerName
         model.peers[ sessPeer ] = modelPeer

      if sessGroupName in modelPeer.groups:
         modelGroup = modelPeer.groups[ sessGroupName ]
      else:
         modelGroup = ItsCliModel.ItsGroup()
         modelPeer.groups[ sessGroupName ] = modelGroup

      if sessPathName in modelGroup.paths:
         modelPath = modelGroup.paths[ sessPathName ]
      else:
         modelPath = ItsCliModel.ItsPath()
         modelPath.srcIp = sessSrc
         modelPath.dstIp = sessDst
         modelGroup.paths[ sessPathName ] = modelPath


      itsSess = ItsCliModel.ItsSession()

      sessionActive = False
      currentSessionState = sessionState.get( sessKey )
      if currentSessionState and currentSessionState.active:
         sessionActive = True

      if charact:
         # fill in characteristics
         itsSess.characteristics = ItsCliModel.ItsSession.SessionCharacts()
         load = 0.0
         if sessionActive:
            itsSess.characteristics.active = True
            # ns to seconds
            itsSess.characteristics.seconds = \
                           int( round( session.activeTime / 1e9 ) )
            # us to ms
            itsSess.characteristics.latency = session.latency / 1e3
            itsSess.characteristics.jitter = session.jitter / 1e3
            # Convert throughput from bps to Mbps
            itsSess.characteristics.throughput = \
               session.throughput / 1e6
            # Loss rate was scaled up with 1000000, convert them into percentage
            itsSess.characteristics.lossRate = session.loss / 1e4
            itsSess.characteristics.totalBandwidth = session.totalBandwidth
            itsSess.characteristics.availBandwidth = session.availBandwidth
            if session.totalBandwidth:
               load = ( ( session.totalBandwidth - session.availBandwidth ) /
                        session.totalBandwidth ) * 100
            load = round( load, 2 )
            itsSess.characteristics.load = load
         else:
            itsSess.characteristics.active = False
            # path is inactive, not make sense to show characteristics
            itsSess.characteristics.seconds = 0
            itsSess.characteristics.latency = 0.0
            itsSess.characteristics.jitter = 0.0
            itsSess.characteristics.throughput = 0.0
            itsSess.characteristics.lossRate = 0.0
            itsSess.characteristics.totalBandwidth = 0
            itsSess.characteristics.availBandwidth = 0
            itsSess.characteristics.load = 0.0

         # set mtu value as well
         itsSess.characteristics.mtu = 0
         itsSessKey = ItsSessKey( sessVrfId, sessPathId, 0 )
         itsPmtuStatus = itsStatus.itsSessPmtuStatus
         pmtuStatus = itsPmtuStatus.get( itsSessKey )
         if pmtuStatus:
            itsSess.characteristics.mtu = pmtuStatus.mtu
         else:
            itsSess.characteristics.mtu = 1500 # set default
      else:
         # fill in counters
         byteRecvSnapShot = 0
         pktRecvSnapShot = 0
         byteSentSnapShot = 0
         pktSentSnapShot = 0
         txInactiveDropSnapShot = 0
         securityErrSnapShot = 0
         countersSnapShot = itsCountersSnapShot.itsSessCounterData.get( sessKey )
         if countersSnapShot and countersSnapShot.timestamp > session.timestamp:
            byteRecvSnapShot = countersSnapShot.byteRecv
            pktRecvSnapShot = countersSnapShot.pktRecv
            byteSentSnapShot = countersSnapShot.byteSent
            pktSentSnapShot = countersSnapShot.pktSent
            txInactiveDropSnapShot = countersSnapShot.txInactiveDrop
            securityErrSnapShot = countersSnapShot.securityErr
         itsSess.counters = ItsCliModel.ItsSession.SessionCounters()
         itsSess.counters.inBytes = session.byteRecv - byteRecvSnapShot
         itsSess.counters.inPkts = session.pktRecv - pktRecvSnapShot
         itsSess.counters.outBytes = session.byteSent - byteSentSnapShot
         itsSess.counters.outPkts = session.pktSent - pktSentSnapShot
         itsSess.counters.inPktsDrop = session.securityErr - \
                                       securityErrSnapShot
         itsSess.counters.outPktsDrop = session.txInactiveDrop - \
                                        txInactiveDropSnapShot
      modelPath.sessions[ sessTrafficClass ] = itsSess

   return model

def getSessionCounters( mode, args ):
   return getItsSessions( mode, args, False )

def getSessionCharacts( mode, args ):
   return getItsSessions( mode, args, True )

#-----------------------------------------------------------------------------------
# show monitor telemetry path counters [ detail ]
#    [ destination DSTIP ] [ path-name NAME ]
#    [ peer PEERIP ] [ source SRCIP ] [ traffic-class TC ]
#    [ peer-name PEERNAME ] [ path-group GROUPNAME ]
#-----------------------------------------------------------------------------------
class ShowMonitorTelePathCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show monitor telemetry path counters [ detail ]
               [ destination DSTIP ] [ path-name NAME ] [ peer PEERIP ]
               [ source SRCIP ] [ traffic-class TC ]
               [ peer-name PEERNAME ] [ path-group GROUPNAME ]'''
   data = {
             'monitor': CliToken.Monitor.monitorMatcherForShow,
             'telemetry': telemetryNode,
             'path': pathNode,
             'counters': 'Show path counters',
             'detail': 'Show path counters in detail',
             'destination': 'Show path counters filtered by destination IP',
             'DSTIP': dstIpv4AddrMatcher,
             'path-name': 'Show path counters filtered by path name',
             'NAME': pathNameMatcher,
             'peer': 'Show path counters filtered by peer IP',
             'PEERIP': peerIpv4AddrMatcher,
             'source': 'Show path counters filtered by source IP',
             'SRCIP': srcIpv4AddrMatcher,
             'traffic-class': 'Show path counters filtered by traffic class',
             'TC': trafficClassMatcher,
             'peer-name': 'Show path counters filtered by peer name',
             'PEERNAME': peerNameMatcher,
             'path-group': 'Show path counters filtered by path group name',
             'GROUPNAME': pathGroupMatcher
          }
   cliModel = ItsCliModel.ItsPaths
   handler = getSessionCounters

BasicCli.addShowCommandClass( ShowMonitorTelePathCounters )

#-----------------------------------------------------------------------------------
# show monitor telemetry path characteristics [ detail ]
#    [ destination DSTIP ] [ path-name NAME ]
#    [ peer PEERIP ] [ source SRCIP ] [ traffic-class TC ]
#    [ peer-name PEERNAME ] [ path-group GROUPNAME ]
#-----------------------------------------------------------------------------------
class ShowMonitorTelePathCharacts( ShowCommand.ShowCliCommandClass ):
   syntax = '''show monitor telemetry path characteristics [ detail ]
               [ destination DSTIP ] [ path-name NAME ] [ peer PEERIP ]
               [ source SRCIP ] [ traffic-class TC ]
               [ peer-name PEERNAME ] [ path-group GROUPNAME ]'''
   data = {
             'monitor': CliToken.Monitor.monitorMatcherForShow,
             'telemetry': telemetryNode,
             'path': pathNode,
             'characteristics': 'Show path characteristics',
             'detail': 'Show path characteristics in detail',
             'destination': 'Show path characteristics filtered by destination IP',
             'DSTIP': dstIpv4AddrMatcher,
             'path-name': 'Show path characteristics filtered by path name',
             'NAME': pathNameMatcher,
             'peer': 'Show path characteristics filtered by peer IP',
             'PEERIP': peerIpv4AddrMatcher,
             'source': 'Show path characteristics filtered by source IP',
             'SRCIP': srcIpv4AddrMatcher,
             'traffic-class': 'Show path characteristics filtered by traffic class',
             'TC': trafficClassMatcher,
             'peer-name': 'Show path characteristics filtered by peer name',
             'PEERNAME': peerNameMatcher,
             'path-group': 'Show path characteristics filtered by path group name',
             'GROUPNAME': pathGroupMatcher
          }
   cliModel = ItsCliModel.ItsPaths
   handler = getSessionCharacts

BasicCli.addShowCommandClass( ShowMonitorTelePathCharacts )

#-----------------------------------------------------------------------------------
# clear monitor telemetry path counters
#-----------------------------------------------------------------------------------
class ClearMonitorTelePathCounters( CliCommand.CliCommandClass ):
   syntax = 'clear monitor telemetry path counters'
   data = {
             'clear': clearMatcher,
             'monitor': CliToken.Monitor.monitorMatcherForClear,
             'telemetry': telemetryClearNode,
             'path': pathNode,
             'counters': 'Clear path counters',
          }

   @staticmethod
   def handler( mode, args ):
      for sessKey, counters in itsStatus.itsSessCounterData.items():
         sessCounterSnapShot = ItsSessCounterData( sessKey,
                                                   counters.pktSent,
                                                   counters.byteSent,
                                                   counters.pktRecv,
                                                   counters.byteRecv,
                                                   counters.txInactiveDrop,
                                                   counters.securityErr,
                                                   counters.kaSent,
                                                   counters.kaRecv,
                                                   counters.kaGenErr,
                                                   counters.fbSent,
                                                   counters.fbRecv,
                                                   counters.txActivate,
                                                   counters.txInactivate,
                                                   counters.rxPacketLoss )
         sessCounterSnapShot.timestamp = Tac.now()
         itsCountersSnapShot.addItsSessCounterData( sessCounterSnapShot )
      # clear stale entries
      for sessKey in itsCountersSnapShot.itsSessCounterData:
         if sessKey not in itsStatus.itsSessCounterData:
            del itsCountersSnapShot.itsSessCounterData[ sessKey ]

BasicCli.EnableMode.addCommandClass( ClearMonitorTelePathCounters )

SHOW_TECH_CMDS = [
   'show monitor telemetry path counters',
   'show monitor telemetry path characteristics',
   'show monitor telemetry path counters detail',
   'show monitor telemetry path characteristics detail',
]
timeStamp = '2019-07-09 13:58:09'

TechSupportCli.registerShowTechSupportCmd(
   timeStamp,
   cmds=SHOW_TECH_CMDS,
   cmdsGuard=dpsSupported )

TechSupportCli.registerShowTechSupportCmd(
   timeStamp,
   cmds=SHOW_TECH_CMDS,
   cmdsGuard=dpsSupported,
   extended='sfe' )

def Plugin( em ):
   global itsStatus
   global itsStateStatus
   global itsCountersSnapShot
   global dpsPathStatusDir
   global dpsPeerStatus
   global dpsVrfStatus
   itsStatus = SmashLazyMount.mount( em, 'its/sessStatus',
                                     'ItsSmash::ItsSessStatus',
                                     SmashLazyMount.mountInfo( 'reader' ) )
   itsStateStatus = SmashLazyMount.mount( em, 'its/sessStateStatus',
                                          'ItsSmash::ItsSessStateStatus',
                                          SmashLazyMount.mountInfo( 'reader' ) )
   itsCountersSnapShot = LazyMount.mount( em, 'its/cli/counters/snapshot',
                                          'Its::ItsCountersSnapShot', 'wr' )
   dpsPathStatusDir = LazyMount.mount( em, 'dps/path/status',
                                       'Dps::PathStatusDir', 'r' )
   dpsPeerStatus = LazyMount.mount( em, 'dps/peer/status',
                                    'Dps::PeerStatus', 'r' )
   dpsVrfStatus = LazyMount.mount( em, 'dps/vrf/status',
                                   'Dps::VrfStatus', 'r' )

