# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the CLI config mode for defining L1 card profiles along with
all its constituent commands.
'''

import BasicCli
import CliCommand
from CliGlobal import CliGlobal
import CliMatcher
from CliMode.L1CardProfileDefinition import CardProfileDefinitionMode
import CliParserCommon
import CliToken
import CliToken.L1Profile
import ConfigMount
from Intf.CardIntfSlotRange import CardIntfSlotRangeMatcher
import LazyMount
from Plugins import SkipPluginModule
from Toggles import L1ProfileToggleLib
from TypeFuture import TacLazyType

if not L1ProfileToggleLib.toggleL1ProfileCustomModuleSupportEnabled():
   raise SkipPluginModule( 'L1 Profiles Custom Module Toggle Not Enabled' )

CardProfile = TacLazyType( 'L1Profile::CardProfile' )
CardProfileDescriptor = TacLazyType( 'L1Profile::CardProfileDescriptor' )
CardProfileSource = TacLazyType( 'L1Profile::CardProfileSource::CardProfileSource' )
GlobalConfig = TacLazyType( 'L1Profile::GlobalConfig' )
InterfaceSlotDescriptor = TacLazyType( 'L1Profile::InterfaceSlotDescriptor' )
InterfaceSlotProfileDescriptor = TacLazyType(
   'L1Profile::InterfaceSlotProfileDescriptor' )
InterfaceSlotProfileSource = TacLazyType(
   'L1Profile::InterfaceSlotProfileSource::InterfaceSlotProfileSource' )
IntfSlotProfileReader = TacLazyType( 'L1Profile::IntfSlotProfileReader' )
MountConstants = TacLazyType( 'L1Profile::MountConstants' )

gv = CliGlobal( dict( cardProfileLibraryCliDir=None,
                      intfSlotProfileLibraryRootDir=None ) )

l1ProfileDescriptionToken = CliMatcher.StringMatcher(
   helpdesc='Description for this L1 module profile' )

cliDefinedCardProfileMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: { desc.name: profile.description for desc, profile in
                  gv.cardProfileLibraryCliDir.cardProfile.items() },
   'L1 module profile name',
   extraEmptyTokenCompletionFn=lambda mode, context: [
      CliParserCommon.Completion( desc.name, profile.description ) for
      desc, profile in
      gv.cardProfileLibraryCliDir.cardProfile.items() ] )

class IntfSlotProfileMatcher( CliMatcher.DynamicNameMatcher ):
   '''A name matcher which has custom completion / help string logic to always show
   the resolved set of define interface slot profiles.
   '''
   @staticmethod
   def enumerateProfiles( mode ):
      acc = IntfSlotProfileReader( LazyMount.force(
         gv.intfSlotProfileLibraryRootDir ) )
      return sorted( desc.name for desc in acc.resolvedDescriptors().desc )

   @staticmethod
   def enumerateProfileCompletions( mode, context ):
      completions = []
      acc = IntfSlotProfileReader( LazyMount.force(
         gv.intfSlotProfileLibraryRootDir ) )
      for desc in acc.resolvedDescriptors().desc:
         profile = acc.profile( desc )
         completions.append(
            CliParserCommon.Completion( profile.descriptor.name,
                                        profile.description ) )

      return sorted( completions )

   def __init__( self ):
      super().__init__(
         self.enumerateProfiles,
         'Port profile name',
         extraEmptyTokenCompletionFn=self.enumerateProfileCompletions )

class CardProfileDefinitionConfigMode( CardProfileDefinitionMode,
                                       BasicCli.ConfigModeBase ):
   name = 'L1 Module Profiles Definition Configuration'

   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      self.profileDesc = CardProfileDescriptor( CardProfileSource.cli, profileName )

      # Create an local L1 card profile definition and copy the existing definition
      # to it from Sysdb if applicable. This serves two purposes:
      #  1. It allows for easy transaction rollback as
      #     the Sysdb copy wont be modified until we commit the transaction.
      #  2. CLI is multi threaded, so this protects the CLI session from operating on
      #     data which might be changing as other session commit their transactions.
      self.profile = CardProfile( self.profileDesc )
      # We need to insure that when we create a local L1 card profile that
      # displayName is set, similar to what the Parser does for builtin CardProfiles.
      if not self.profile.displayName:
         self.profile.displayName = self.profile.descriptor.name
      libraryProfile = gv.cardProfileLibraryCliDir.cardProfile.get(
         self.profileDesc )
      if libraryProfile:
         self.profile.copyFrom( libraryProfile )

      CardProfileDefinitionMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      # If we exited the mode normally then the transaction should be committed. If
      # "abort" was called prior to exit then no commit would occur.
      if self.profile:
         self.commitProfile()

      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.profile = None
      self.session_.gotoParentMode()

   def commitProfile( self ):
      libraryProfile = gv.cardProfileLibraryCliDir.cardProfile.newMember(
         self.profileDesc )
      libraryProfile.copyFrom( self.profile )

class DefineCardProfileCommand( CliCommand.CliCommandClass ):
   syntax = '''l1 module profile PROFILE'''
   noOrDefaultSyntax = syntax
   data = { 'l1': CliToken.L1Profile.profileDefineKeywordL1,
            'module': 'Configure module L1 parameters',
            'profile': 'Define a module L1 profile',
            'PROFILE': cliDefinedCardProfileMatcher }
   handler = 'L1CardProfileDefinitionHandler.DefineCardProfile.handler'
   noOrDefaultHandler = (
      'L1CardProfileDefinitionHandler.DefineCardProfile.noOrDefaultHandler' )

BasicCli.GlobalConfigMode.addCommandClass( DefineCardProfileCommand )

class AbortCardProfileDefinition( CliCommand.CliCommandClass ):
   syntax = '''abort'''
   data = { 'abort': CliToken.Cli.abortMatcher }
   handler = 'L1CardProfileDefinitionHandler.AbortCardProfileDefinition.handler'

CardProfileDefinitionConfigMode.addCommandClass( AbortCardProfileDefinition )

class CardProfileDescriptionCommand( CliCommand.CliCommandClass ):
   syntax = '''description DESC'''
   noOrDefaultSyntax = '''description [ DESC ]'''
   data = { 'description': 'Description string associated with the L1 module '
                           'profile',
            'DESC': CliMatcher.StringMatcher(
                           helpdesc='Description for this L1 module profile' ) }
   handler = 'L1CardProfileDefinitionHandler.CardProfileDescription.handler'
   noOrDefaultHandler = (
      'L1CardProfileDefinitionHandler.CardProfileDescription.noOrDefaultHandler' )

CardProfileDefinitionConfigMode.addCommandClass( CardProfileDescriptionCommand )

class InterfaceSlotProfileMapCommand( CliCommand.CliCommandClass ):
   syntax = '''port INTF_SLOT profile PROFILE'''
   noOrDefaultSyntax = '''port INTF_SLOT [ profile PROFILE ]'''
   data = { 'port': 'Configure port L1 parameters',
            'INTF_SLOT': CardIntfSlotRangeMatcher,
            'profile': 'Apply a L1 port profile to the port',
            'PROFILE': IntfSlotProfileMatcher() }
   handler = 'L1CardProfileDefinitionHandler.InterfaceSlotProfileMap.handler'
   noOrDefaultHandler = (
      'L1CardProfileDefinitionHandler.InterfaceSlotProfileMap.noOrDefaultHandler' )

CardProfileDefinitionConfigMode.addCommandClass( InterfaceSlotProfileMapCommand )

def Plugin( entityManager ):
   gv.cardProfileLibraryCliDir = ConfigMount.mount(
      entityManager,
      MountConstants.cardProfileLibraryCliDirPath(),
      'L1Profile::CardProfileDir',
      'w' )
   gv.intfSlotProfileLibraryRootDir = LazyMount.mount(
      entityManager,
      MountConstants.intfSlotProfileLibraryRootDirPath(),
      'Tac::Dir',
      'ri' )
