# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict
import os
import threading

import Arnet
import BasicCliUtil
import Cell
from CliPlugin import EthIntfCli
from CliPlugin import IntfPolicyManager
from CliPlugin import L1PolicyCliLib
from CliPlugin import XcvrAllStatusDir
from CliPlugin import SpeedGroupCli
from EthIntfLib import speedLanestoSpeedLanesStr
from Intf import IntfRange
import LazyMount
import Tac
from Toggles.L1PolicyAgentToggleLib import toggleL1PolicySpeedChangeHookEnabled
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( "L1PolicyCancelHooks" )
t0 = __defaultTraceHandle__.trace0
t1 = __defaultTraceHandle__.trace1

ethIntfSliceDefaultConfigDir = None
ethIntfGlobalDefaultConfigDir = None
allIntfStatusDir = None
xcvrAllStatusProxy = None
resourceConsumerSliceDir = None
resourceManagerSliceStatus = None
subdomainIntfSlotStatusDir = None
inactivePromptIpmCache = threading.local()
autonegConfigSliceDir = None

IntfModeAdapter = TacLazyType( "AlePhy::IntfModeAdapter::AdapterSm" )
IntfSwState = TacLazyType( "Interface::IntfSwState::IntfSwState" )
AllAutonegConfigDir = TacLazyType( "Interface::AllAutonegConfigDir" )
AutonegMode = TacLazyType( "Interface::AutonegMode" )
EthIntfModeDir = TacLazyType( "Interface::EthIntfModeDir" )
L1PolicyVersion = TacLazyType( 'L1::PolicyVersion' )

def getSliceId( ethPhyIntf ):
   sliceId = "FixedSystem"
   if Cell.cellType() != "fixed":
      sliceId = ethPhyIntf.sliceName()
   return sliceId

def getSliceDefaultConfig( ethPhyIntf ):
   sliceId = getSliceId( ethPhyIntf )
   return ethIntfSliceDefaultConfigDir.defaultConfig.get( sliceId )

def getResourceConsumerDir( ethPhyIntf ):
   # Iterate through all resource consumer dirs and
   # return the one managing the wanted intf
   for resourceConsumerDirs in resourceConsumerSliceDir.values():
      for resourceConsumerDir in resourceConsumerDirs.values():
         intfSerdesConsumer = resourceConsumerDir.serdesResourceConsumer.get(
               ethPhyIntf.name )
         if intfSerdesConsumer:
            return resourceConsumerDir
   return None

def getResourceConsumerDirFromSubdomain( sliceId, subdomain ):
   # Iterate through all resource consumer dirs and
   # return the one managing the subdomain
   resourceConsumerDirs = resourceConsumerSliceDir.get( sliceId )
   if not resourceConsumerDirs:
      return None
   for rcdSubdomain, resourceConsumerDir in resourceConsumerDirs.items():
      # BUG969431 : need to standardize subdomain name for rms/rcd
      if ( rcdSubdomain == subdomain or
            ( "FixedSystem" in rcdSubdomain and subdomain == "StrataPhy-0" ) ):
         return resourceConsumerDir
   return None

def getResourceManagerStatusDir( sliceId, subdomain ):
   resourceManagerStatusDirs = resourceManagerSliceStatus.get( sliceId )
   if not resourceManagerStatusDirs:
      return None
   for rmsSubdomain, rmsDir in resourceManagerStatusDirs.items():
      # BUG969431 : need to standardize subdomain name for rms/rcd
      if ( rmsSubdomain == subdomain or
            ( "FixedSystem" in rmsSubdomain and subdomain == "StrataPhy-0" ) ):
         return rmsDir
   return None

def getAutonegConfig( ethPhyIntf ):
   for autonegConfigDirs in autonegConfigSliceDir.values():
      for autonegConfigDir in autonegConfigDirs.values():
         autonegConfig = autonegConfigDir.autonegConfig.get( ethPhyIntf.name )
         if autonegConfig:
            return autonegConfig
   return None

def getAllAutonegConfigDir():
   allAutoNegConfigDir = AllAutonegConfigDir( 'allAutonegConfigDir' )
   for autonegConfigDirs in autonegConfigSliceDir.values():
      for autonegConfigDir in autonegConfigDirs.values():
         for autonegConfig in autonegConfigDir.autonegConfig.values():
            localAutonegConfig = Tac.newInstance( "Interface::AutonegConfig",
                                                  autonegConfig.intfId )
            Tac.newInstance( "Interface::AutonegConfigReplicatorSm",
                             autonegConfig.intfId,
                             autonegConfig, localAutonegConfig )
            allAutoNegConfigDir.autonegConfig.addMember( localAutonegConfig )
   return allAutoNegConfigDir

class L1PolicyCliCancelHookEnv():
   def __init__( self ):
      self.activeIpms = set()
      self.inactiveIntfs = set()
      self.errdisabledIntfs = set()
      self.flappingIntfs = set()
      self.subdomainToConfigResolverMap = {}
      self.subdomainToIpmMap = {}
      self.intfSpeedChangeMap = {}
      self.checkedRms = set()
      self.warningsPresent = False

   def createConfigResolverAndIpm( self, policyV3, sliceId, subdomain,
                                   resourceConsumerDir, resourceMgrStatus ):
      localAllIntfStatusDir = Tac.newInstance(
         "Interface::AllEthPhyIntfStatusDir", "local" )

      # Create pipeline if it doesn't exist for this subdomain
      if ( ( sliceId, subdomain ) not in self.subdomainToConfigResolverMap and
           policyV3 ):
         configResolver = L1PolicyCliLib.CliConfigResolver(
            sliceId, subdomain, localAllIntfStatusDir )
         # If all dirs are ready, move forward with l1policy pipeline
         if configResolver.setUpDirs():
            t1( "L1Policy active and ready to be used for", sliceId, subdomain )
            self.subdomainToConfigResolverMap[ (
               sliceId, subdomain ) ] = configResolver
         else:
            # If a subdomain managed by L1Policy does not have populated paths
            # break out of the loop so that ipm will not exist for it
            t1( "L1Policy does not have all dirs ready,",
                "skipping resolution for", subdomain )
            return
      t1( resourceConsumerDir, resourceMgrStatus )
      ipm = IntfPolicyManager.getIpm( resourceConsumerDir,
                                      resourceMgrStatus,
                                      localAllIntfStatusDir,
                                      inactivePromptIpmCache )
      if ipm is None:
         return

      if ipm not in self.activeIpms:
         self.subdomainToIpmMap[ ( sliceId, subdomain ) ] = ipm
         self.activeIpms.add( ipm )

   def runIntfModeAdapter( self, ipm ):
      t1( 'running intfModeAdapter for ipm', ipm )
      inputEim = EthIntfModeDir( 'input' )
      outputEim = ipm.mockEthIntfModeDir
      inputEim.copyFrom( outputEim )
      IntfModeAdapter( inputEim, outputEim, ipm.intfStatusDir )

   def compareIntfStates( self, ipm, resourceMgrStatus,
                          policyV3, configIntfIds=None ):
      if ( not ipm or not resourceMgrStatus or
          resourceMgrStatus in self.checkedRms ):
         return
      t1( 'comparing states for ipm', ipm )
      intfPolicyManager = ipm.ipm
      self.checkedRms.add( resourceMgrStatus )
      for intfId, intfState in resourceMgrStatus.intfStateDir.intfState.items():
         if configIntfIds and intfId in configIntfIds:
            # Ignore intfs under configuration for the speedchangehook
            continue
         if intfId not in intfPolicyManager.localIntfStateDir.intfState:
            continue
         eim = resourceMgrStatus.ethIntfModeDir.ethIntfMode.get( intfId )
         mockEim = ipm.mockEthIntfModeDir.ethIntfMode.get( intfId )
         currState = intfState.state
         newIntfState = intfPolicyManager.intfStateDir.intfState[ intfId ]
         newState = newIntfState.state
         t1( intfId, "eim:", eim, "mock eim:", mockEim )
         if ( currState.swState == IntfSwState.active and
              newState.swState != currState.swState ):
            t1( intfId, "(swState, hwState) changed from (", currState.swState,
                currState.hwState, ") to (", newState.swState,
                newState.hwState, ")" )
            if ( newState.swState == IntfSwState.errdisabled and policyV3 ):
               self.errdisabledIntfs.add( intfId )
            else:
               self.inactiveIntfs.add( intfId )
            self.warningsPresent = True
         elif ( eim and mockEim and eim.speed != mockEim.speed and policyV3 and
                  currState.swState == IntfSwState.active and
                  currState.swState == newState.swState ):
            # If the intf is not in the configIntfs and has a changed eim
            # then it has been auto adjusted due to L1PolicyAgent
            self.intfSpeedChangeMap[ intfId ] = ( eim, mockEim )
            self.warningsPresent = True
            t1( intfId, "eim:", eim, "mock eim:", mockEim )
         # TODO BUG907175: Need to remove the PD specific code to warn for flaps.
         #                 Current implementation is for gardena4 where remap
         #                 causes already linked-up intfs to flap.
         # If nothing has changed, link is still active but the intf has been
         # marked as Dirty; the link will flap. The link flapping because of the
         # dirty bit is true for everything in IPM that can set an intf as dirty.
         elif ( currState.swState == IntfSwState.active and
                currState.swState == newState.swState and
                newIntfState.dirty and
                ( ipm.resourceConsumerDir.blackhawkGen3Group or policyV3 ) ):
            t1( intfId, "has no changes but has Dirty Bit set" )
            self.flappingIntfs.add( intfId )
            self.warningsPresent = True

   def generateWarnings( self, mode, policyV3 ):
      if self.inactiveIntfs:
         inactiveIntfStr = ""
         for intfId in Arnet.sortIntf( self.inactiveIntfs ):
            inactiveIntfStr += f"   {intfId}\n"
         warning = f"""The requested speed change will cause the following
interfaces to become inactive:\n\n{inactiveIntfStr}"""
         mode.addWarning( warning )

      if self.errdisabledIntfs:
         errdisabledIntfStr = ""
         for intfId in Arnet.sortIntf( self.errdisabledIntfs ):
            errdisabledIntfStr += f"   {intfId}\n"
         warning = f"""The requested speed change will cause the following
interfaces to become errdisabled:\n\n{errdisabledIntfStr}"""
         mode.addWarning( warning )

      if self.flappingIntfs:
         flappingIntfStr = ""
         for intfId in Arnet.sortIntf( self.flappingIntfs ):
            flappingIntfStr += f"   {intfId}\n"
         warning = f"""The requested configuration change may cause the following
interfaces to flap:\n\n{flappingIntfStr}"""
         mode.addWarning( warning )

      # Generate speedchange warnings
      if ( policyV3 and self.intfSpeedChangeMap ):
         speedChangeWarnings = ""
         for intfId, eims in self.intfSpeedChangeMap.items():
            newSpeedLaneOutput = speedLanestoSpeedLanesStr.get(
               ( eims[ 1 ].speed, eims[ 1 ].laneCount ) )
            oldSpeedLaneOutput = speedLanestoSpeedLanesStr.get(
               ( eims[ 0 ].speed, eims[ 0 ].laneCount ) )
            if newSpeedLaneOutput and oldSpeedLaneOutput:
               speedChangeWarnings += ( f"{ intfId }: " +
                                        f"{ oldSpeedLaneOutput } => " +
                                        f"{ newSpeedLaneOutput }\n" )
         msg = ( "The requested configuration will cause the following "
                 "interfaces to change their auto adjusted configurations:" )
         warning = f"{ msg }\n\n{ speedChangeWarnings }"
         mode.addWarning( warning )


############ Speed Change Cancel Hook ###############
def warnIntfSecondaryEffects( mode, intfList, linkMode, advertisedModes ):
   if os.environ.get( 'SIMULATION_STRATA' ):
      return False
   t1( 'cancel hook called with', linkMode, advertisedModes )
   # Instantiate L1PolicyCliCancelHook Env and necessary sets
   configIntfIds = set()
   intfToIpmMap = {}
   intfToPolicyMap = {}
   l1PolicyHookEnv = L1PolicyCliCancelHookEnv()
   subdomainToIntfsMap = defaultdict( set )

   # Create an IntfPolicyManager (IPM) for each resourceConsumerDir
   # and synchronize its local states/status with the system's
   for intf in intfList:
      intfId = intf.config().intfId
      if intfId.startswith( "Switch" ) or intfId.startswith( 'Fabric' ):
         continue

      # Ignore interfaces which are not actually available
      # e.g. powered-off linecards
      ethIntfStatus = EthIntfCli.ethIntfStatusDir.intfStatus.get( intfId )
      if not ethIntfStatus:
         continue

      intfXcvrStatus = intf.intfXcvrStatus()
      if intfXcvrStatus:
         if intfXcvrStatus.xcvrPresence == 'xcvrPresent':
            # If the transceiver is not present, we still want to check for
            # interactions. The reason is that a speed configuration will be
            # effective on an interface without a transceiver.
            if not ethIntfStatus.linkModeSupported[ linkMode ]:
               continue

      if intf.ethIntfModeDir() is None:
         continue

      configIntfIds.add( intfId )
      intfToIpmMap[ intf ] = None
      resourceConsumerDir = getResourceConsumerDir( intf )
      resourceMgrStatus = intf.resourceManagerStatus()
      sliceId = getSliceId( intf )
      subdomain = intf.subdomain()
      policyV3 = False
      # Create a CliConfigResolver object per slot and subdomain if we
      # are using any policy
      policy = intf.l1PolicyVersion()
      t1( "Interface", intfId, "is on", subdomain, "with policy", policy )
      if ( toggleL1PolicySpeedChangeHookEnabled() and
           policy >= L1PolicyVersion( 3, 0 ) ):
         policyV3 = True
         intfToPolicyMap[ intf ] = policy
         subdomainToIntfsMap[ ( sliceId, subdomain ) ].add( intf )

      l1PolicyHookEnv.createConfigResolverAndIpm(
         policyV3, sliceId, subdomain, resourceConsumerDir, resourceMgrStatus )
      intfToIpmMap[ intf ] = l1PolicyHookEnv.subdomainToIpmMap.get( ( sliceId,
                                                                      subdomain ) )

   # Sync Interface statuses, mock them here if we do not have an active L1Policy
   for subdomainTuple, ipm in l1PolicyHookEnv.subdomainToIpmMap.items():
      if not ipm:
         continue
      configResolver = l1PolicyHookEnv.subdomainToConfigResolverMap.get(
         subdomainTuple )
      if not configResolver:
         t1( "not using L1Policy, mocking intfStatusDir for subdomain",
             subdomainTuple )
         intfStatusDir = ipm.intfStatusDir
         # If active l1policy, then CliConfigResolver has ownership of
         # localAllIntfStatusDir
         for intfStatus in allIntfStatusDir.intfStatus.values():
            intfStatusDir.intfStatus.addMember(
                  Tac.newInstance( "Interface::EthPhyIntfStatus",
                                    intfStatus.intfId,
                                    intfStatus.defaultConfig,
                                    intfStatus.genId,
                                    intfStatus.burnedInAddr,
                                    intfStatus.relativeIfindex ) )
            localIntfStatus = intfStatusDir.intfStatus[ intfStatus.intfId ]
            localIntfStatus.intfConfig = intfStatus.intfConfig
            Tac.newInstance( "Interface::EthPhyIntfStatusReplicatorSm",
                              intfStatus.intfId, intfStatus, localIntfStatus )
      # Synchronize IPM's mocked ethIntfModeDir
      t1( "synchronizing ethIntfModeDir and intfStateDir" )
      ipm.synchronize()

   # Simulate the L1Policy pipeline for every subdomain that has an active policy
   for subdomainTuple, configResolver in \
         l1PolicyHookEnv.subdomainToConfigResolverMap.items():
      ipm = l1PolicyHookEnv.subdomainToIpmMap.get( subdomainTuple )
      intfsInSubdomain = subdomainToIntfsMap.get( subdomainTuple )
      if not ipm or not intfsInSubdomain or not configResolver.setupReady:
         continue
      localIntfConfigs = []
      for intf in intfsInSubdomain:
         localEthPhyIntfConfig = Tac.newInstance( "Interface::EthPhyIntfConfig",
                                                  intf.config().intfId )
         Tac.newInstance( "Interface::EthPhyIntfConfigReplicatorSm",
                          intfId, intf.config(), localEthPhyIntfConfig )
         t1( "copying config lm", intf.config().intfId, "from",
             localEthPhyIntfConfig.linkModeLocal, "to", linkMode )
         localEthPhyIntfConfig.linkModeLocal = linkMode
         localIntfConfigs.append( localEthPhyIntfConfig )

      configResolver.resolve( localIntfConfigs,
                              {},
                              ipm.mockSpeedGroupStatusDir,
                              getAllAutonegConfigDir(),
                              ipm.mockEthIntfModeDir )

      ipm.intfStatusDir = configResolver.allIntfStatusDir

   # Simulate the Legacy pipeline for every interface on a subdomain that has no
   # active policy
   for intf in intfList:
      ipm = intfToIpmMap.get( intf )
      # Skip autonegSm if it has an active L1Policy on its subdomain
      if ipm is None or intfToPolicyMap.get( intf ):
         continue
      intfId = intf.config().intfId
      localEthIntfModeDir = ipm.mockEthIntfModeDir
      localEthPhyIntfConfig = Tac.newInstance( "Interface::EthPhyIntfConfig",
                                               intfId )

      Tac.newInstance( "Interface::EthPhyIntfConfigReplicatorSm",
                       intfId, intf.config(),
                       localEthPhyIntfConfig )
      hwAutonegStatus = Tac.newInstance( "Interface::HwAutonegStatus", intfId )
      # Ignore SpeedGroupStatusDir since it isn't needed when
      # linkMode isnot unknown/autoneg/auto40GbpsFull
      localSpeedGroupStatusDir = None
      t1( "AutonegSm intfId:", intfId )
      localEthPhyIntfStatus = ipm.intfStatusDir.intfStatus[ intfId ]
      sliceDefaultConfig = getSliceDefaultConfig( intf )

      localAutonegConfig = None
      autonegConfig = getAutonegConfig( intf )
      if autonegConfig:
         localAutonegConfig = Tac.newInstance( "Interface::AutonegConfig", intfId )
         Tac.newInstance( "Interface::AutonegConfigReplicatorSm",
                          intfId,
                          autonegConfig, localAutonegConfig )
         if linkMode not in [ "linkModeAutoneg", "linkModeAuto40GbpsFull" ]:
            localAutonegConfig.mode = AutonegMode.anegModeUnknown
         t1( "AutonegSm anMode:", localAutonegConfig.mode,
             "advertisedSpeeds:", localAutonegConfig.advertisedSpeeds )

      sissDir = ipm.subdomainIntfSlotStatusDir
      siss = None
      sis = None
      if sissDir:
         siss = sissDir.fetchSlot( intfId )
      if siss:
         sis = siss.subdomainIntfStatus[ intfId ]

      # Autoneg SM to react to when localEthPhyIntfConfig gets updated with user's
      # config.
      # pylint: disable=unused-variable
      autonegSm = Tac.newInstance( "Interface::AutonegSm",
         localEthPhyIntfConfig,
         hwAutonegStatus,
         localEthPhyIntfStatus,
         intf.intfXcvrStatus(),
         True, # autonegConfigSupported
         True, # clause37PhyAutonegStatusSupported
         localEthIntfModeDir,
         intf.speedGroupConfigDir(),
         localSpeedGroupStatusDir,
         localAutonegConfig,
         sliceDefaultConfig,
         ethIntfGlobalDefaultConfigDir,
         siss,
         sis,
         False,
         True,
         True )

      t1( "linkModeLocal changed on", intf.config().intfId,
          "from:", localEthPhyIntfConfig.linkModeLocal, "to:", linkMode )
      localEthPhyIntfConfig.linkModeLocal = linkMode

      if linkMode in [ "linkModeAutoneg", "linkModeAuto40GbpsFull" ]:
         if advertisedModes:
            localEthPhyIntfConfig.advertisedModesLocal = advertisedModes
            localEthPhyIntfConfig.advertisedModesConfiguredLocal = True
         else:
            localEthPhyIntfConfig.advertisedModesLocal = \
                  Tac.Value( "Interface::EthLinkModeSet" )
            localEthPhyIntfConfig.advertisedModesConfiguredLocal = False
         autonegAdvSm = Tac.newInstance( "Interface::AutonegAdvSm",
               localEthPhyIntfConfig,
               localEthPhyIntfStatus,
               intf.intfXcvrStatus(),
               sis
               )
         autonegAdvSm.autonegConfig = localAutonegConfig

      t1( "local ethPhyIntfStatus speed:", localEthPhyIntfStatus.speed,
          "duplex:", localEthPhyIntfStatus.duplex,
          "linkModeStatus:", localEthPhyIntfStatus.linkModeStatus )

   # Running IntfModeAdapter, then IPM with the user's changes.
   for ipm in l1PolicyHookEnv.activeIpms:
      l1PolicyHookEnv.runIntfModeAdapter( ipm )
      ipm.processIntfs()

   t1( "config intfs:", configIntfIds )
   # For every intf in each resourceManagerStatus, compare the current swState
   # with the new swState generated by the IPM. If the swState changed from active
   # to non-active, add it to the prompt list
   for intf in intfList:
      ipm = intfToIpmMap.get( intf )
      resourceMgrStatus = intf.resourceManagerStatus()
      policyV3 = ( intf in intfToPolicyMap and
                   intfToPolicyMap[ intf ] >= L1PolicyVersion( 3, 0 ) )
      l1PolicyHookEnv.compareIntfStates( ipm, resourceMgrStatus,
                                         policyV3, configIntfIds )

   # Determine which warning (or if both) are needed to prompt user.
   # speed change warnings only should be printed if we have subsumed intfs

   if l1PolicyHookEnv.warningsPresent:
      l1PolicyHookEnv.generateWarnings(
         mode, toggleL1PolicySpeedChangeHookEnabled() )
      prompt = "Do you wish to proceed with this command? [y/N]"
      if not BasicCliUtil.confirm( mode, prompt, answerForReturn=False ):
         return True
   return False

####### Speed-Group Change Cancel Hooks ############
def policyV3CheckSpeedGroupChangeCancel( mode, group, compatibility, onlyPolicyV3 ):
   if ( not mode.session.commandConfirmation() or
        not onlyPolicyV3 ):
      return False

   t1( mode, group, compatibility )
   l1PolicyHookEnv = L1PolicyCliCancelHookEnv()

   for name in IntfRange.intfListFromCanonical( [ group ] ):
      nameWithNoType = IntfRange.intfTypeFromName( name )[ 1 ]
      sliceId, subdomain = SpeedGroupCli.getSpeedGroupSliceSubdomain(
         nameWithNoType )
      speedGroupConfigDir = SpeedGroupCli.getSpeedGroupConfigDir(
         nameWithNoType )
      if not all( [ sliceId, subdomain, speedGroupConfigDir ] ):
         continue

      resourceConsumerDir = getResourceConsumerDirFromSubdomain( sliceId,
                                                      subdomain )
      resourceMgrStatus = getResourceManagerStatusDir( sliceId, subdomain )
      l1PolicyHookEnv.createConfigResolverAndIpm(
         onlyPolicyV3, sliceId, subdomain, resourceConsumerDir, resourceMgrStatus )
      ipm = l1PolicyHookEnv.subdomainToIpmMap.get( ( sliceId, subdomain ) )
      if ipm:
         # Synchronize IPM's mocked ethIntfModeDir
         t1( "synchronizing ethIntfModeDir and intfStateDir" )
         ipm.synchronize()

   # Simulate the L1Policy pipeline for every subdomain that has an active policy
   for subdomainTuple, configResolver in \
         l1PolicyHookEnv.subdomainToConfigResolverMap.items():
      ipm = l1PolicyHookEnv.subdomainToIpmMap.get( subdomainTuple )
      if not ipm or not configResolver.setupReady:
         continue
      localSgConfigs = {}
      for name in IntfRange.intfListFromCanonical( [ group ] ):
         nameWithNoType = IntfRange.intfTypeFromName( name )[ 1 ]
         if compatibility:
            newCompatibility = set( compatibility )
         else:
            newCompatibility = set()

         localSgConfigs[ nameWithNoType ] = newCompatibility
      configResolver.resolve( [],
                              localSgConfigs,
                              ipm.mockSpeedGroupStatusDir,
                              getAllAutonegConfigDir(),
                              ipm.mockEthIntfModeDir )

      ipm.intfStatusDir = configResolver.allIntfStatusDir

   # Running the IPM with the user's changes.
   for ipm in l1PolicyHookEnv.activeIpms:
      l1PolicyHookEnv.runIntfModeAdapter( ipm )
      ipm.processIntfs()

   # compare old and new intf states
   for ( sliceId, subdomain ), ipm in l1PolicyHookEnv.subdomainToIpmMap.items():
      resourceMgrStatus = getResourceManagerStatusDir( sliceId, subdomain )
      l1PolicyHookEnv.compareIntfStates( ipm, resourceMgrStatus, onlyPolicyV3 )

   # generate warnings
   if l1PolicyHookEnv.warningsPresent:
      l1PolicyHookEnv.generateWarnings( mode, onlyPolicyV3 )
      prompt = "Do you wish to proceed with this command? [y/N]"
      if not BasicCliUtil.confirm( mode, prompt, answerForReturn=False ):
         return True
   return False

####### Abort Hooks ##########
SpeedGroupCli.canPromptForAbortHook.addExtension(
   policyV3CheckSpeedGroupChangeCancel )

EthIntfCli.canPromptForAbortHook.addExtension( warnIntfSecondaryEffects )

def Plugin( em ):
   global ethIntfSliceDefaultConfigDir
   global ethIntfGlobalDefaultConfigDir
   global allIntfStatusDir
   global xcvrAllStatusProxy
   global resourceConsumerSliceDir
   global resourceManagerSliceStatus
   global autonegConfigSliceDir
   global subdomainIntfSlotStatusDir

   ethIntfSliceDefaultConfigDir = LazyMount.mount(
         em, "interface/archer/config/eth/phy/sliceDefault",
         "Interface::EthIntfSliceDefaultConfigDir", "r" )

   ethIntfGlobalDefaultConfigDir = LazyMount.mount(
         em, "interface/config/eth/phy/globalDefault",
         "Interface::EthPhyIntfGlobalDefaultConfigDir", "r" )

   allIntfStatusDir = LazyMount.mount(
         em, "interface/status/eth/phy/all",
         "Interface::AllEthPhyIntfStatusDir", "r" )

   xcvrAllStatusProxy = XcvrAllStatusDir.xcvrAllStatusDir( em )

   resourceConsumerSliceDir = LazyMount.mount(
         em, "interface/resources/consumers/slice",
         "Tac::Dir", "ri" )
   resourceManagerSliceStatus = LazyMount.mount(
         em, "interface/resources/status/slice", "Tac::Dir", "ri" )
   autonegConfigSliceDir = LazyMount.mount(
         em, "interface/archer/config/eth/autoneg/slice",
         "Tac::Dir", "ri" )
   subdomainIntfSlotStatusDir = LazyMount.mount(
         em, "interface/intfSlot/status", "Tac::Dir", "ri" )
