#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# pkgdeps: rpm AlePhy

''' This library instantiates an instance of the L1PolicyAgent pipeline by creating
L1PolicyAgent::ConfigResolver and L1PolicyAgent::DatapathMountResolver in order to
resolve l1ConfigDir.

This has originally been created for Cli CancelHooks to use when navigating speed
changes or pll speed group changes.

Usage: Create an instance of CliConfigResolver by passing in
slot, subdomain, and policyVersion.
To set up directories, run setUpDirs().
To run the pipeline, run resolve().
'''

from CliGlobal import CliGlobal
import LazyMount
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'L1PolicyCliLib' )

TERROR = Tracing.trace0
TWARN = Tracing.trace1
TNOTE = Tracing.trace3

AllEthPhyIntfConfigFilter = TacLazyType( 'Interface::AllEthPhyIntfConfigFilter' )
ConfigResolver = TacLazyType( 'L1PolicyAgent::ConfigResolver' )
DatapathConfigAdapter = TacLazyType( 'AlePhy::DatapathConfigurationAdapter' )
DatapathModeResolver = TacLazyType( 'L1PolicyAgent::DatapathModeResolver' )
EthPhyIntfConfigConstPtrDir = TacLazyType(
   'Interface::EthPhyIntfConfigConstPtrDir' )
EthPhyIntfStatus = TacLazyType( 'Interface::EthPhyIntfStatus' )
EthPhyIntfStatusReplicatorSm = TacLazyType(
   'Interface::EthPhyIntfStatusReplicatorSm' )
Generation = TacLazyType( 'Ark::Generation' )
L1ConfigDir = TacLazyType( 'L1::ConfigurationDir' )
MountConstants = TacLazyType( 'L1::MountConstants' )
SpeedGroupConfigDir = TacLazyType( 'Interface::SpeedGroupConfigDir' )
SubdomainIntfSlotStatusPtrConstDir = TacLazyType(
   'Interface::SubdomainIntfSlotStatusPtrConstDir' )
SubdomainLocationIntfSlotStatusAggregator = TacLazyType(
   'Interface::SubdomainLocationIntfSlotStatusAggregator' )

gv = CliGlobal( dict( allIntfConfigDir=None,
                      allIntfStatusDir=None,
                      autonegStatusRootDir=None,
                      capsRootDir=None,
                      defaultCapsRootDir=None,
                      defaultRootDir=None,
                      l1CardPowerGenerationRootDir=None,
                      hwL1TopoMetadataRootDir=None,
                      productConfigRootDir=None,
                      sissBaseDir=None,
                      speedGroupConfigSliceDir=None, ) )

class CliConfigResolver:
   def __init__( self, slotId, subdomain, allIntfStatusDir ):
      '''Instantiates CliConfigResolver and DatapathModeResolver Pipeline.

      Parameters
      ----------
      slotId : Tac::String
      subdomain : Tac::String
      allIntfStatusDir : Interface::AllEthPhyIntfStatusDir

      Returns
      -------
      An wrapper around L1PolicyAgent::ConfigResolver that computes states
      for all the interfaces in the provided subdomain and slot.
      '''

      # params for ConfigResolver
      self.epicDir = None
      self.capsDir = None
      self.pllSgConfigDir = None
      self.capsDefaultDir = None
      self.defaultsDir = None
      self.sissDir = None
      self.l1ConfigDir = None
      self.managedIntfIdDir = None
      self.policy = None

      # params for DatapathModeResolver
      self.autonegStatusSubdomainDir = None
      self.dpmL1ConfigDir = None

      # common attributes
      self.subdomain = subdomain
      self.slotId = slotId
      self.slotSubdomain = f'{ slotId }/{ subdomain }'
      self.configResolver = None
      self.datapathModeResolver = None
      self.datapathConfigAdapter = None
      self.mockSgConfigDir = None
      self.allIntfStatusDir = allIntfStatusDir
      self.setupReady = False

   def resolve( self,
                modifiedEpics,
                modifiedSgConfigs,
                speedGroupStatusDir,
                autonegConfigDir,
                ipmEimDir ):
      if not self.setupReady:
         TERROR( 'dirs were not set up' )
         return False
      if not speedGroupStatusDir:
         TERROR( 'cannot have empty speedGroupStatusDir' )
         return False
      if not autonegConfigDir:
         TERROR( 'cannot have empty autonegConfigDir' )
         return False
      if not ipmEimDir:
         TERROR( 'cannot have empty eimDir' )
         return False

      # overwrite modified epics if needed
      for epic in modifiedEpics:
         self.epicDir.intfConfig.addMember( epic )

      for group, compatibility in modifiedSgConfigs.items():
         if compatibility:
            speedGroupConfig = self.mockSgConfigDir.group.get( group )
            if speedGroupConfig is None:
               speedGroupConfig = self.mockSgConfigDir.newGroup( group )
            curConfig = set( speedGroupConfig.setting.keys() )
            curGenId = speedGroupConfig.settingGeneration.id
            # Delete unconfigured compatibility settings
            for setting in speedGroupConfig.setting:
               if setting not in compatibility:
                  del speedGroupConfig.setting[ setting ]
            # Add configured compatibility settings
            for attr in compatibility:
               speedGroupConfig.setting[ attr ] = True
            newConfig = set( speedGroupConfig.setting.keys() )
            if newConfig != curConfig:
               speedGroupConfig.settingGeneration = Generation( curGenId + 1, True )
         else:
            # no/default command delete the entity
            del self.mockSgConfigDir.group[ group ]

      TNOTE( 'creating configResolver:', self.slotSubdomain )
      self.configResolver = ConfigResolver( self.capsDir,
                                            self.managedIntfIdDir,
                                            self.epicDir,
                                            self.mockSgConfigDir,
                                            self.capsDefaultDir,
                                            self.defaultsDir,
                                            self.sissDir,
                                            self.l1ConfigDir,
                                            self.policy,
                                            None,
                                            True )

      TNOTE( 'creating datapathModeResolver:', self.slotSubdomain )
      self.datapathModeResolver = \
         DatapathModeResolver( self.slotId,
                               self.subdomain,
                               self.l1ConfigDir,
                               self.autonegStatusSubdomainDir,
                               self.dpmL1ConfigDir )

      TNOTE( 'creating datapathConfigAdapter:', self.slotSubdomain )
      self.datapathConfigAdapter = DatapathConfigAdapter( self.dpmL1ConfigDir,
                                                          self.allIntfStatusDir,
                                                          ipmEimDir,
                                                          speedGroupStatusDir,
                                                          autonegConfigDir )
      return True

   def setUpDirs( self ):
      if not self.subdomain or not self.slotId:
         TERROR( 'subdomain and slotId cannot be empty' )
         return False

      self.epicDir = EthPhyIntfConfigConstPtrDir( self.subdomain )
      self.l1ConfigDir = L1ConfigDir( self.subdomain )
      self.dpmL1ConfigDir = L1ConfigDir( self.subdomain )
      if not gv.allIntfConfigDir or not gv.allIntfStatusDir:
         TERROR( 'allIntfConfigDir or allIntfStatusDir do not exist' )
         return False

      capsSliceDir = gv.capsRootDir.get( self.slotId )
      if capsSliceDir:
         self.capsDir = capsSliceDir.get( self.subdomain )

      self.pllSgConfigDir = gv.speedGroupConfigSliceDir.get( self.slotId )
      capsDefaultSliceDir = gv.defaultCapsRootDir.get( self.slotId )
      if capsDefaultSliceDir:
         self.capsDefaultDir = capsDefaultSliceDir.get( self.subdomain )

      defaultsSliceDir = gv.defaultRootDir.get( self.slotId )
      if defaultsSliceDir:
         self.defaultsDir = defaultsSliceDir.get( self.subdomain )

      hwL1TopoMetadataSubdomainRootDir = gv.hwL1TopoMetadataRootDir.get(
         self.slotId )
      if not hwL1TopoMetadataSubdomainRootDir:
         TERROR( 'Managed IntfDir slot key error' )
         return False

      hwL1TopoMetadataSubdomainDir = hwL1TopoMetadataSubdomainRootDir.get(
         'subdomain' )
      if not hwL1TopoMetadataSubdomainDir:
         TERROR( 'Managed IntfDir subdomain root dir key error' )
         return False

      hwL1TopoMetadataDir = hwL1TopoMetadataSubdomainDir.get( self.subdomain )
      if not hwL1TopoMetadataDir:
         TERROR( 'Managed IntfDir subdomain key error' )
         return False

      self.managedIntfIdDir = hwL1TopoMetadataDir.get( 'memberIntfs' )
      prodCardSlotConfig = gv.productConfigRootDir.cardSlotConfig.get( self.slotId )
      if not prodCardSlotConfig:
         TERROR( 'Could not find ProductConfig for', self.slotId )
         return False

      prodSubdomainConfig = prodCardSlotConfig.subdomainConfig.get( self.subdomain )
      if not prodSubdomainConfig:
         TERROR( 'Could not find ProductCardSlotConfig for', self.subdomain )
         return False

      self.policy = prodSubdomainConfig.policyVersion
      autonegStatusSliceDir = self.autonegStatusSubdomainDir = \
         gv.autonegStatusRootDir.get( self.slotId )
      if autonegStatusSliceDir:
         self.autonegStatusSubdomainDir = autonegStatusSliceDir.get(
            self.subdomain )

      self.filterEpicDir()
      self.aggregateSissDir()

      # verify that all dirs exist
      if not all( [ self.capsDir,
                    self.pllSgConfigDir,
                    self.capsDefaultDir,
                    self.defaultsDir,
                    self.l1ConfigDir,
                    self.policy,
                    self.managedIntfIdDir.managedIntf.keys(),
                    self.autonegStatusSubdomainDir ] ):
         TERROR( 'mounted dirs do not have', self.slotId, ',', self.subdomain )
         return False

      # replicate allIntfStatusDir
      for intfId, srcStatus in gv.allIntfStatusDir.intfStatus.items():
         dstStatus = EthPhyIntfStatus( intfId,
                                       srcStatus.defaultConfig,
                                       srcStatus.genId,
                                       srcStatus.burnedInAddr,
                                       srcStatus.relativeIfindex )
         EthPhyIntfStatusReplicatorSm( intfId, srcStatus, dstStatus )
         self.allIntfStatusDir.intfStatus.addMember( dstStatus )

      # replicate pllSgConfigDir
      self.mockSgConfigDir = SpeedGroupConfigDir( self.subdomain )
      for group, sgConfig in self.pllSgConfigDir.group.items():
         mockSgConfig = self.mockSgConfigDir.group.newMember( group )
         for setting, valid in sgConfig.setting.items():
            mockSgConfig.setting[ setting ] = valid
         mockSgConfig.settingGeneration = Generation(
            sgConfig.settingGeneration.id, sgConfig.settingGeneration.valid )

      # cycle through managedIntfIds and verify all present
      for intfId in self.managedIntfIdDir.managedIntf.keys():
         if not self.epicDir.intfConfig[ intfId ]:
            TERROR( 'intfId', intfId, 'not present in filtered epicDir' )
            return False
         if not self.sissDir.fetchIntf( intfId ):
            TERROR( 'intfSlotId', intfId, 'not present in aggregated sissDir' )
            return False
         if not self.allIntfStatusDir.intfStatus[ intfId ]:
            TERROR( 'intfId', intfId, 'not present in allIntfStatusDir' )
            return False

      # clear to setup ConfigResolver
      self.setupReady = True
      return True

   def filterEpicDir( self ):
      AllEthPhyIntfConfigFilter( gv.allIntfConfigDir,
                                 self.managedIntfIdDir,
                                 self.epicDir )

   def aggregateSissDir( self ):
      self.sissDir = SubdomainIntfSlotStatusPtrConstDir( self.subdomain )
      SubdomainLocationIntfSlotStatusAggregator( gv.sissBaseDir,
                                                 gv.l1CardPowerGenerationRootDir,
                                                 self.slotId,
                                                 self.subdomain,
                                                 self.sissDir )

def Plugin( em ):
   try:
      gv.allIntfConfigDir = em.getLocalEntity( 'interface/config/eth/phy/all' )
   except KeyError:
      TNOTE( 'Could not retrieve locally aggregated EthPhyIntfConfig Dir' )

   try:
      gv.allIntfStatusDir = em.getLocalEntity( 'interface/status/eth/phy/all' )
   except KeyError:
      TNOTE( 'Could not retrieve locally aggregated EthPhyIntfStatus Dir' )

   gv.autonegStatusRootDir = LazyMount.mount(
      em, MountConstants.autonegStatusRootDirPath(), 'Tac::Dir', 'ri' )
   gv.capsRootDir = LazyMount.mount(
      em, MountConstants.l1PolicyCapsRootDirPath(), 'Tac::Dir', 'ri' )
   gv.defaultCapsRootDir = LazyMount.mount(
      em, MountConstants.l1PolicyDefaultCapsRootDirPath(), 'Tac::Dir', 'ri' )
   gv.defaultRootDir = LazyMount.mount(
      em, MountConstants.defaultRootDirPath(), 'Tac::Dir', 'ri' )
   gv.l1CardPowerGenerationRootDir = LazyMount.mount(
      em, MountConstants.l1CardPowerGenerationRootDirPath(), 'Tac::Dir', 'ri' )
   gv.hwL1TopoMetadataRootDir = LazyMount.mount(
      em, MountConstants.hwL1TopoMetadataDirPath(), 'Tac::Dir', 'ri' )
   gv.productConfigRootDir = LazyMount.mount(
      em, MountConstants.productConfigRootDirPath(), 'L1::ProductConfig', 'ri' )
   gv.sissBaseDir = LazyMount.mount(
      em, MountConstants.rootSissDirPath(), 'Tac::Dir', 'ri' )
   gv.speedGroupConfigSliceDir = LazyMount.mount(
      em, MountConstants.speedGroupConfigRootDirPath(), 'Tac::Dir', 'ri' )
