# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the config CLI used to configure a L1 module profile on a
module.
'''

from CliPlugin import (
   L1ProfileCliGuard,
   SystemL1,
)
from CliPlugin.L1ProfileCliCommon import (
   L1ProfileSupportedModulesRangeMatcher,
)
import CliCommand
from CliDynamicSymbol import LazyCallback
from CliGlobal import CliGlobal
import CliMatcher
import CliParserCommon
import LazyMount
from TypeFuture import TacLazyType

CardProfileReader = TacLazyType( 'L1Profile::CardProfileReader' )
CardProfileSource = TacLazyType( 'L1Profile::CardProfileSource::CardProfileSource' )
MountConstants = TacLazyType( 'L1Profile::MountConstants' )

gv = CliGlobal( cardProfileLibraryRootDir=None )

class CardProfileMatcher( CliMatcher.DynamicNameMatcher ):
   '''A name matcher which has custom completion / help string logic to always show
   the resolved set of defined card profiles.
   '''
   @staticmethod
   def enumerateProfiles( mode ):
      acc = CardProfileReader( LazyMount.force( gv.cardProfileLibraryRootDir ) )
      return sorted( acc.profile( desc ).displayName for desc
                     in acc.resolvedDescriptors( CardProfileSource.unknown ).desc
                     if acc.profile( desc ).visible )

   @staticmethod
   def enumerateProfileCompletions( mode, context ):
      completions = []
      acc = CardProfileReader( LazyMount.force( gv.cardProfileLibraryRootDir ) )
      for desc in acc.resolvedDescriptors( CardProfileSource.unknown ).desc:
         profile = acc.profile( desc )
         if not profile.visible:
            continue

         completions.append(
            CliParserCommon.Completion( profile.displayName,
                                        profile.description ) )

      return sorted( completions )

   def __init__( self ):
      super().__init__(
         self.enumerateProfiles,
         'Builtin module L1 profile name',
         extraEmptyTokenCompletionFn=self.enumerateProfileCompletions )

class CardProfileApplyCommand( CliCommand.CliCommandClass ):
   syntax = '''module ( MOD | switch ) profile PROFILE'''
   noOrDefaultSyntax = '''module ( MOD | switch ) profile [ PROFILE ]'''
   data = {
      'module': 'Configure module L1 parameters',
      'MOD': CliCommand.Node(
                matcher=L1ProfileSupportedModulesRangeMatcher(),
                guard=L1ProfileCliGuard.modularSystemGuard ),
      'switch': CliCommand.guardedKeyword(
                   'switch',
                   helpdesc='Switch module',
                   guard=L1ProfileCliGuard.fixedSystemGuard ),
      'profile': 'Apply an L1 module profile',
      'PROFILE': CardProfileMatcher(),
   }

   handler = 'L1ProfileApplyHandler.handler'
   noOrDefaultHandler = 'L1ProfileApplyHandler.noOrDefaultHandler'

SystemL1.SystemL1ConfigMode.addCommandClass( CardProfileApplyCommand )

SystemL1.registerCleanupHook( 'L1 Profile',
   LazyCallback( 'L1ProfileApplyHandler.cardProfileMapCleanup' ) )

def Plugin( entityManager ):
   gv.cardProfileLibraryRootDir = LazyMount.mount(
      entityManager,
      MountConstants.cardProfileLibraryRootDirPath(),
      'Tac::Dir',
      'ri' )
