# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
from CliDynamicSymbol import LazyCallback
from CliGlobal import CliGlobal
import CliMatcher
import CliToken.Hardware
import LazyMount
import ShowCommand
from TypeFuture import TacLazyType

# pylint: disable=pointless-string-statement

'''
Implementation of the CLI command(s):

   show hardware l1 topology component [ module <sliceID> ] 
      chip [ <chipType> [ <chipId> ] ] [ core <phyType> [ <coreId> ] ]
'''

PhyChipId = TacLazyType( 'PhyEee::PhyChipId' )
PhyCoreId = TacLazyType( 'PhyEee::PhyCoreId' )
PhyScope = TacLazyType( 'PhyEee::PhyScope' )
PhyScopeDescriptor = TacLazyType( 'PhyEee::PhyScopeDescriptor' )
PhyHelper = TacLazyType( 'Hardware::L1Topology::PhyHelper' )

L1TopoSysdbPaths = TacLazyType( 'Hardware::L1Topology::SysdbPathConstants' )
gv = CliGlobal( l1TopoDir=None )

def getSliceId( mode ):
   return list( gv.l1TopoDir.entityPtr[ 'slice' ] )

def getChipType( mode, context ):
   phyScopeDescriptor = PhyScopeDescriptor(
      ( context.sharedResult or {} ).get( 'MODULE', '' ),
      '',
      PhyChipId.unknownPhyChipId,
      '',
      PhyCoreId.unknownPhyCoreId,
      PhyScope.phyScopeUnknown
   )

   phyHelper = PhyHelper( gv.l1TopoDir )
   serdesTopologySet = phyHelper.getSerdesContainedBy( phyScopeDescriptor )

   return { serdes.chipType for serdes in serdesTopologySet.polarity.keys() }

def getPhyType( mode, context ):
   phyScopeDescriptor = PhyScopeDescriptor(
      ( context.sharedResult or {} ).get( 'MODULE', '' ),
      ( context.sharedResult or {} ).get( 'CHIPTYPE', '' ),
      ( context.sharedResult or {} ).get( 'CHIPID', PhyChipId.unknownPhyChipId ),
      '',
      PhyCoreId.unknownPhyCoreId,
      PhyScope.phyScopeUnknown
   )

   phyHelper = PhyHelper( gv.l1TopoDir )
   serdesTopologySet = phyHelper.getSerdesContainedBy( phyScopeDescriptor )

   return { serdes.phyType for serdes in serdesTopologySet.polarity.keys() }

matcherSliceId = CliMatcher.DynamicNameMatcher( getSliceId,
                                                helpdesc='Module name' )
matcherChipType = CliMatcher.DynamicNameMatcher( getChipType, passContext=True,
                                                 helpdesc='Chip type' )
matcherPhyType = CliMatcher.DynamicNameMatcher( getPhyType, passContext=True,
                                                helpdesc='Phy type' )

class ShowL1TopologyComponentCli( ShowCommand.ShowCliCommandClass ):
   syntax = '''show hardware l1 topology component [ module MODULE ]
               chip [ CHIPTYPE [ CHIPID ] ] [ core PHYTYPE [ COREID ] ]'''

   data = {
      'hardware': CliToken.Hardware.hardwareMatcherForShow,
      'l1': 'L1',
      'topology': 'Topology',
      'component': 'Component',
      'module': 'Limit output to a module',
      'MODULE': CliCommand.Node( matcherSliceId, storeSharedResult=True ),
      'chip': 'Limit output to a chip',
      'CHIPTYPE': CliCommand.Node( matcherChipType, storeSharedResult=True ),
      'CHIPID': CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 0, PhyChipId.unknownPhyChipId,
                                            helpdesc='ID of chip' ),
         storeSharedResult=True
      ),
      'core': 'Limit output to a core',
      'PHYTYPE': CliCommand.Node( matcherPhyType ),
      'COREID': CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 0, PhyCoreId.unknownPhyCoreId,
                                            helpdesc='ID of core' )
      ),
   }

   cliModel = 'L1TopologyComponentModel.L1TopologyComponent'

   @staticmethod
   def handler( mode, args ):
      sliceId = args.get( 'MODULE', '' )
      chipType = args.get( 'CHIPTYPE', '' )
      chipId = int( args.get( 'CHIPID', PhyChipId.unknownPhyChipId ) )
      phyType = args.get( 'PHYTYPE', '' )
      coreId = int( args.get( 'COREID', PhyCoreId.unknownPhyCoreId ) )
      scope = PhyScope.phyScopeUnknown

      phyScopeDescriptor = PhyScopeDescriptor(
         sliceId,
         chipType,
         chipId,
         phyType,
         coreId,
         scope
      )

      phyHelper = PhyHelper( gv.l1TopoDir )
      serdesTopologySet = phyHelper.getSerdesContainedBy( phyScopeDescriptor )

      componentModel = \
         LazyCallback( 'L1TopologyComponentModel.L1TopologyComponent' )()
      populateSerdesModel = \
         LazyCallback( 'L1TopologyComponentModel.SerdesComponent.populate' )

      for serdesDesc, polarity in serdesTopologySet.polarity.items():
         logicalLane = serdesTopologySet.logicalLane[ serdesDesc ]
         componentModel.serdesGroups.append(
            populateSerdesModel( serdesDesc, logicalLane, polarity ) )

      return componentModel

BasicCli.addShowCommandClass( ShowL1TopologyComponentCli )

# ----------------------------------------------------------------------------
# Plugin
# ----------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.l1TopoDir = LazyMount.mount( entityManager,
                                   L1TopoSysdbPaths.topologyDirPath,
                                   'Tac::Dir',
                                   'ri' )
