# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import collections
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.FabricIntfCli as FabricIntfCli
from CliPlugin import L1TopologyTraversalModel as TopoModel
from CliPlugin.VirtualIntfRule import IntfMatcher
import CliToken.Hardware
from EthIntfLib import fecEncodingToToken
from EthIntfLib import speedLanestoSpeedLanesStr
from natsort import natsorted
import ShowCommand
from TypeFuture import TacLazyType
import LazyMount
import Tracing
import Tac

# pylint: disable=pointless-string-statement

"""
Implementation of the CLI command(s):

   Show hardware l1 topology traversal interface <intfId> 
      speed <eth speed> fec <fec>

"""

traceHandle = Tracing.Handle( "L1TopologyTraversalCli" )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

EthSpeeds = TacLazyType( 'Interface::EthSpeed' )
EthDuplex = TacLazyType( 'Interface::EthDuplex' )
EthTypesApi = TacLazyType( 'Interface::EthTypesApi' )
EthFecEncoding = TacLazyType( 'Interface::EthFecEncoding' )
ComponentType = TacLazyType( 'Hardware::L1Topology::ComponentType' )
IdOps = TacLazyType( 'Hardware::L1Topology::IdentifierOperators' )

tableFmt = "{:<30} {:<30}"

l1MappingDir = None
l1TopoDir = None

matchFec = CliMatcher.EnumMatcher( {
   'reed-solomon544': 'Show reed-solomon544 forward error correction',
   'reed-solomon': 'Show reed-solomon forward error correction',
   'fire-code': 'Show fire-code forward error correction',
   'disabled': 'Show disabled forward error correction'
} )

matchDirection = CliMatcher.EnumMatcher( {
   'ingress': 'Show traversal of the ingress path',
   'egress': 'Show traversal of the egress path',
} )

# Formatting mappings for EthIntf Enums.
speedDuplextoSpeedDuplexStr = collections.OrderedDict(
      [ ( ( 'speed100Mbps', 'duplexFull' ), '100full' ),
        ( ( 'speed100Mbps', 'duplexHalf' ), '100half' ),
        ( ( 'speed10Mbps', 'duplexFull' ), '10full' ),
        ( ( 'speed10Mbps', 'duplexHalf' ), '10half' ),
      ] )

def _speedStrToEthSpeed( speed ):
   linkMode = EthIntfCli.speedLinkMode[ speed ]
   ethSpeed = EthTypesApi.linkModeToEthSpeed( linkMode )
   ethLaneCount = EthTypesApi.linkModeToEthLaneCount( linkMode )
   ethDuplex = EthTypesApi.linkModeToEthDuplex( linkMode )
   return ethSpeed, ethLaneCount, ethDuplex

def _fecStrToEthFecEncoding( fec ):
   encoding = EthFecEncoding.fecEncodingUnknown
   if fec == "reed-solomon544":
      encoding = EthFecEncoding.fecEncodingReedSolomon544
   elif fec == "reed-solomon":
      encoding = EthFecEncoding.fecEncodingReedSolomon
   elif fec == "fire-code":
      encoding = EthFecEncoding.fecEncodingFireCode
   elif fec == "disabled":
      encoding = EthFecEncoding.fecEncodingDisabled
   return encoding

def populateMode( componentGroup ):
   cliIntfMode = TopoModel.L1TopologyMode()
   cliIntfMode.speed = componentGroup.mode.intfSpeed
   cliIntfMode.laneCount = componentGroup.mode.laneCount
   cliIntfMode.fec = componentGroup.mode.fec
   cliIntfMode.duplex = componentGroup.mode.duplex
   return cliIntfMode

def populateChannelMode( componentGroup ):
   cliChannelMode = TopoModel.L1TopologyChannelMode()
   cliChannelMode.channelId = componentGroup.channelMode.channelId
   cliChannelMode.channelSpeed = componentGroup.channelMode.channelSpeed
   cliChannelMode.channelDuplex = componentGroup.channelMode.channelDuplex
   return cliChannelMode

def populateSerdes( componentGroup, component ):
   if component.tx:
      direction = "tx"
      logicalSerdes = component.phyScopeTopology().txP2L[ component.physicalLane ]
   else:
      direction = "rx"
      logicalSerdes = component.phyScopeTopology().rxP2L[ component.physicalLane ]
   phyChip = component.chipType()
   phyChipId = component.chipId()

   cliSerdes = TopoModel.SerdesTopology()
   cliSerdes.slotId = componentGroup.sliceId
   cliSerdes.phyChip = phyChip
   cliSerdes.phyChipId = phyChipId
   cliSerdes.phyType = component.phyType()
   cliSerdes.phyCoreId = component.phyCoreId()
   cliSerdes.phyScope = component.phyScope()
   cliSerdes.physicalSerdes = component.physicalLane
   cliSerdes.direction = direction
   cliSerdes.logicalSerdes = logicalSerdes
   cliSerdes.traceId = component.traceId
   return cliSerdes

def populateConnector( componentGroup, component ):
   cliConnector = TopoModel.ConnectorTopology()
   cliConnector.slotId = componentGroup.sliceId
   cliConnector.family = component.midplaneFamily().family
   cliConnector.connectorId = component.connectorId()
   cliConnector.pinId = component.pinId
   cliConnector.traceId = component.traceId
   return cliConnector

def populateXcvr( componentGroup, component ):
   cliXcvr = TopoModel.XcvrTopology()
   cliXcvr.slotId = componentGroup.sliceId
   cliXcvr.xcvrSlot = component.xcvrTopology().id
   cliXcvr.xcvrLane = component.id
   cliXcvr.traceId = component.traceId
   cliXcvr.direction = "tx" if component.tx else "rx" 
   cliXcvr.polarity = component.polarity
   return cliXcvr

def populateIntfTopology( ethIntfMode ):
   cliIntfTopology = TopoModel.IntfTopology()
   cliIntfTopology.intfId = ethIntfMode.intfId
   cliIntfTopology.speed = ethIntfMode.speed
   cliIntfTopology.fec = ethIntfMode.fec
   cliIntfTopology.laneCount = ethIntfMode.laneCount
   cliIntfTopology.duplex = ethIntfMode.duplex
   return cliIntfTopology

def populateCompGroup( componentGroup ):
   cliCompGroup = TopoModel.L1TopologyComponentGroup()
   cliCompGroup.componentType = componentGroup.componentType
   cliCompGroup.intfMode = populateMode( componentGroup )
   cliCompGroup.intfChannelMode = populateChannelMode( componentGroup )
   cliCompGroup.slotId = componentGroup.sliceId
   for component in componentGroup.component.values():
      if componentGroup.componentType == ComponentType.componentSerdes:
         cliCompGroup.components.append(
               populateSerdes( componentGroup, component ) )
      if componentGroup.componentType == ComponentType.componentXcvr:
         cliCompGroup.components.append(
               populateXcvr( componentGroup, component ) )
      if componentGroup.componentType == ComponentType.componentMidplane:
         cliCompGroup.components.append(
               populateConnector( componentGroup, component ) )
   return cliCompGroup

class ShowL1TopologyTraversalCli( ShowCommand.ShowCliCommandClass ):
   syntax = '''show hardware l1 topology traversal interface INTFID
         speed SPEED fec FEC [ DIRECTION ]'''


   intfMatcher_ = IntfMatcher()
   intfMatcher_ |= EthIntfCli.EthPhyIntf.ethMatcher
   intfMatcher_ |= FabricIntfCli.FabricIntf.matcher

   data = {
      'hardware': CliToken.Hardware.hardwareMatcherForShow,
      'l1': 'L1',
      'topology': 'Topology',
      'traversal': 'Traversal',
      'interface': 'Interface to traverse on',
      'INTFID': intfMatcher_,
      'speed': 'Interface speed',
      'SPEED': EthIntfCli.speedExpression, #BUG681105 only allow supported speeds.
      'fec': 'Forward error correction',
      'FEC': matchFec,
      'DIRECTION': matchDirection,
   }

   cliModel = TopoModel.L1TopologyTraversal

   @staticmethod
   def handler( mode, args ):
      intfId = args[ 'INTFID' ]
      speed = args[ 'SPEED_TYPE' ]
      fec = args[ 'FEC' ]
      direction = args.get( 'DIRECTION', 'egress' )

      topoDir = LazyMount.force( l1TopoDir )
      mappingDir = LazyMount.force( l1MappingDir )
      topoHelper = Tac.newInstance( "Hardware::L1Topology::TraversalHelper",
                                    topoDir, mappingDir )

      intfId = Tac.Value( 'Arnet::IntfId', intfId.name )
      ethIntfMode = Tac.Value( "Interface::EthIntfMode", intfId )
      ethIntfMode.speed, ethIntfMode.laneCount, ethIntfMode.duplex = \
            _speedStrToEthSpeed( speed )
      ethIntfMode.fec = _fecStrToEthFecEncoding( fec )

      # Fetching all component groups
      isTx = direction == 'egress'
      compGroupList = []
      compGroup = topoHelper.getComponentGroupForIntfId( intfId, ethIntfMode, isTx )
      travDirection = topoHelper.getDirectionForIntfId( intfId, True )
      maxHops = 10
      while compGroup:
         compGroupList.append( compGroup )
         compGroup = topoHelper.nextComponentGroup( compGroup, travDirection, True )
         travDirection = IdOps.invertDirection( travDirection )

         # In case there is a bug that causes us to call nextComponentGroup
         # in a loop forever, lets terminate after a set number of hops
         maxHops -= 1
         if maxHops <= 0:
            print( "WARNING: Max hops reached" )
            break

      # Creating the model
      supportedComponents = { ComponentType.componentXcvr,
                              ComponentType.componentSerdes,
                              ComponentType.componentMidplane }
      traversalModel = TopoModel.L1TopologyTraversal()
      traversalModel.intfTopology = populateIntfTopology( ethIntfMode )
      # Fetch speed mode for intfId and validate speed and fec intput from user
      speedModes = topoHelper.getSpeedModesForIntfId( intfId )

      # fecEncodingReedSolomon544 is originally mapped to reed-solomon. May be risky
      # to modify EthIntfLib.tokenToFecEncodingAttrs so we modify it here instead
      fecEncodingToToken[ 'fecEncodingReedSolomon544' ] = 'reed-solomon544'
      # Converts all viable fec encodings into corresponding tokens and adds to set
      fecTokens = { fecEncodingToToken[ eim.fec ] for eim in speedModes.intfMode }

      if ethIntfMode not in speedModes.intfMode:
         availableSpeeds = set()

         for availableMode in speedModes.intfMode:
            speed = availableMode.speed
            lanes = availableMode.laneCount
            duplex = availableMode.duplex

            if 'Mbps' in speed:
               availableSpeeds.add(
               speedDuplextoSpeedDuplexStr[ speed, duplex ] )
            else:
               availableSpeeds.add(
                  speedLanestoSpeedLanesStr[ speed, lanes ] )

         # pylint: disable-next=consider-using-f-string
         mode.addWarning( "Try one of these speeds: {}".format(
            natsorted( availableSpeeds ) ) )
         # pylint: disable-next=consider-using-f-string
         mode.addWarning( "Try one of these fec: {}".format(
               sorted( list( fecTokens ) ) ) )
         mode.addErrorAndStop( "This interface does not support the specified speed"
                               + " and/or fec." )
         
      for compGroup in compGroupList:
         if compGroup.componentType in supportedComponents:
            traversalModel.componentGroups.append(
                  populateCompGroup( compGroup ) )
      return traversalModel

BasicCli.addShowCommandClass( ShowL1TopologyTraversalCli )

#----------------------------------------------------------------------------
# Plugin
#----------------------------------------------------------------------------
def Plugin( entityManager ):
   global l1MappingDir
   global l1TopoDir

   l1MappingDir = LazyMount.mount( entityManager,
                                   "hardware/l1/mapping", "Tac::Dir", "ri" )
   l1TopoDir = LazyMount.mount( entityManager,
                                "hardware/l1/topology", "Tac::Dir", "ri" )
