# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Enum, Int, List, Model, Str, Submodel
from IntfModels import Interface

def printIndented( indentLevel, msg ):
   print( "   " * indentLevel + msg )
tableFmt = "{:<30} {:<30}"
smTableFmt = "{:<27} {:<27}"

class ComponentTopology( Model ):
   # Mark this model as not public so customers never see it
   __public__ = False

class ConnectorTopology( ComponentTopology ):
   # Mark this model as not public so customers never see it
   __public__ = False

   slotId = Str( help="Slot ID" )
   family = Str( help="Connector component family" )
   connectorId = Int( help="ID of the connector" )
   pinId = Int( help="ID of the pin within the connector" )
   traceId = Int( help="ID of trace connecting to this SerDes" )

   def render( self ):
      printIndented( 1, '--- Connector Topology ---' )
      printIndented( 1, tableFmt.format( "Slot:", self.slotId ) )
      printIndented( 1, tableFmt.format( "Family:", self.family ) )
      printIndented( 1, tableFmt.format( "Connector ID:", self.connectorId ) )
      printIndented( 1, tableFmt.format( "Pin ID:", self.pinId ) )
      printIndented( 1, tableFmt.format( "Trace ID:", self.traceId ) )

class XcvrTopology( ComponentTopology ):
   # Mark this model as not public so customers never see it
   __public__ = False

   slotId = Str( help="Slot ID" )
   xcvrSlot = Int( help="Transceiver slot ID" )
   xcvrLane = Int( help="Transceiver slot lane" )
   traceId = Int( help="ID of trace connecting to this SerDes" )
   direction = Enum( values=[ "tx", "rx" ],
                     help="tx or rx for the lane" )
   polarity = Str( help="Polarity of the Transceiver slot module" )

   def render( self ):
      printIndented( 1, '--- Transceiver Topology ---' )
      printIndented( 1, tableFmt.format( "Slot:", self.slotId ) )
      printIndented( 1, tableFmt.format( "Transceiver Slot ID:", self.xcvrSlot ) )
      printIndented( 1, tableFmt.format( "Transceiver Lane ID:", self.xcvrLane ) )
      printIndented( 1, tableFmt.format( "Direction:", self.direction ) )
      printIndented( 1, tableFmt.format( "Polarity:", self.polarity) )
      printIndented( 1, tableFmt.format( "Trace ID:", self.traceId ) )

class SerdesTopology( ComponentTopology ):
   # Mark this model as not public so customers never see it
   __public__ = False

   slotId = Str( help="Slot ID" )
   phyChip = Str( help="Type of chip" )
   phyChipId = Int( help="ID of the chip" )
   phyType = Str( help="Type of phy core" )
   phyCoreId = Int( help="ID of phy core" )
   phyScope = Enum( values=[ "phyScopeLine", "phyScopeSystem" ],
                   help="Side of the chip" )
   physicalSerdes = Int( help="Physical SerDes ID" )
   direction = Enum( values=[ "tx", "rx" ],
                     help="tx or rx for the lane" )
   logicalSerdes = Int( help="Logical SerDes ID" )
   traceId = Int( help="ID of trace connecting to this SerDes" )

   def render( self ):
      printIndented( 1, '--- SerDes Topology ---' )
      printIndented( 1, tableFmt.format( "Slot:", self.slotId ) )
      printIndented( 1, tableFmt.format( "Phy Chip:", self.phyChip ) )
      printIndented( 1, tableFmt.format( "Phy Chip ID:", self.phyChipId ) )
      printIndented( 1, tableFmt.format( "Phy Type:", self.phyType ) )
      printIndented( 1, tableFmt.format( "Phy Core ID:", self.phyCoreId ) )
      printIndented( 1, tableFmt.format( "Phy Side:", self.phyScope ) )
      printIndented( 1, tableFmt.format( "Physical SerDes:", self.physicalSerdes ) )
      printIndented( 1, tableFmt.format( "Direction:", self.direction ) )
      printIndented( 1, tableFmt.format( "Logical SerDes:", self.logicalSerdes ) )
      printIndented( 1, tableFmt.format( "Trace ID:", self.traceId ) )

class L1TopologyMode( Model ):
   # Mark this model as not public so customers never see it
   __public__ = False

   speed = Str( help="Interface speed" )
   fec = Str( help="Interface FEC" )
   laneCount = Str( help="Number of lanes" )
   duplex = Str( help="Interface duplex" )

   def render( self ):
      printIndented( 1, smTableFmt.format( "Speed:", self.speed ) )
      printIndented( 1, smTableFmt.format( "FEC:", self.fec ) )
      printIndented( 1, smTableFmt.format( "Lane Count:", self.laneCount ) )
      printIndented( 1, smTableFmt.format( "Duplex:", self.duplex ) )

class L1TopologyChannelMode( Model ):
   # Mark this model as not public so customers never see it
   __public__ = False

   channelId = Int( help="ID of the channel" )
   channelSpeed = Str( help="Speed of the channel" )
   channelDuplex = Str( help="Duplex of the channel" )

   def render( self ):
      printIndented( 1, smTableFmt.format( "ID:", self.channelId ) )
      printIndented( 1, smTableFmt.format( "Channel Speed:", self.channelSpeed ) )
      printIndented( 1, smTableFmt.format( "Channel Duplex:", self.channelDuplex ) )

class L1TopologyComponentGroup( Model ):
   # Mark this model as not public so customers never see it
   __public__ = False

   componentType = Enum( values=[ "componentSerdes",
                                  "componentXcvr",
                                  "componentMidplane" ],
                         help="The type of the Components in the ComponentGroup" )
   components = List( valueType=ComponentTopology,
                      help=( "Member components of the ComponentGroup" ) )
   intfMode = Submodel( valueType=L1TopologyMode,
                        help=( "The interface configuration the group of "
                               "components should operate at" ) )
   intfChannelMode = Submodel( valueType=L1TopologyChannelMode,
                               help="The channel mode of the CompoonentGroup" )
   slotId = Str( help="The slot ID" )

   def render( self ):
      printIndented( 0, '--- Component Group ---' )
      printIndented( 0, tableFmt.format( "Slot:", self.slotId ) )
      printIndented( 0, tableFmt.format( "Component Type:", self.componentType ) )
      printIndented( 0, "Component Mode:" )
      self.intfMode.render()
      printIndented( 0, "Component Channel Mode:" )
      self.intfChannelMode.render()
      printIndented( 0, "Components:" )
      for component in self.components:
         component.render()

class IntfTopology( Model ):
   # Mark this model as not public so customers never see it
   __public__ = False

   intfId = Interface( help="Name of interface traversed" )
   speed = Str( help="Interface speed" )
   fec = Str( help="Interface FEC" )
   laneCount = Str( help="Number of lanes" )
   duplex = Str( help="Interface duplex" )

   def render( self ):
      printIndented( 0, '--- Interface Speed Topology ---' )
      printIndented( 0, tableFmt.format( "Interface:", self.intfId.stringValue ) )
      printIndented( 0, tableFmt.format( "Speed:", self.speed ) )
      printIndented( 0, tableFmt.format( "FEC:", self.fec ) )
      printIndented( 0, tableFmt.format( "Lane Count:", self.laneCount ) )
      printIndented( 0, tableFmt.format( "Duplex:", self.duplex ) )

class L1TopologyTraversal( Model ):
   # Mark this model as not public so customers never see it
   __public__ = False

   intfTopology = Submodel( valueType=IntfTopology,
                        help='L1 Topology Intf Topology' )
   componentGroups = List( valueType=L1TopologyComponentGroup,
                           help=( "An ordered list of hops in an "
                                  "L1 Topology Traversal" ) )
   def render( self ): 
      print( "WARNING! This is a debug command only for use under supervision " \
             "of Arista customer support or engineering\n" )
      print( "--- Starting Traversal Path ---\n" )
      self.intfTopology.render()
      for compGroup in self.componentGroups:
         compGroup.render()
      print( "\n--- Finishing Traversal Path ---" )

