# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import BasicCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from CliPlugin import MacAddr
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import ConfigMount, LazyMount
from CliMode.L2Protocol import L2ProtocolModeBase, L2ProtocolProfileModeBase
from EbraLib import ( tacL2PtAction, tacL2PtTagFormat, l2PtForwardTarget,
                      l2ProtocolMacAddress, cliTokenToL2PtProtocol )
from TypeFuture import TacLazyType
import Toggles.EbraToggleLib

__defaultTraceHandle__ = Tracing.Handle( 'L2PtCli' )
t0 = Tracing.trace0

PortChannelIntfId = TacLazyType( 'Arnet::PortChannelIntfId' )

# Module globals set up by the Plugin function below
l2PtProfileConfig = None
l2PtIntfConfig = None
bridgingHwCapabilities = None

#-------------------------------------------------------------------------------
# The "config-l2-protocol" mode.
#-------------------------------------------------------------------------------
class L2ProtocolMode( L2ProtocolModeBase, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'L2 Protocol Configuration'

   def __init__( self, parent, session ):
      L2ProtocolModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def clear( self, args ): # pylint: disable=arguments-renamed
      l2PtProfileConfig.profile.clear()

# Used in global mode to guard profile creation
def l2ProtocolModeGuard( mode, token ):
   if bridgingHwCapabilities.l2ProtocolTransparencySupported:
      return None
   return CliParser.guardNotThisPlatform

# Used in interface mode to guard profile attachment to the interface
def l2ProtocolFwdModeGuard( mode, token ):
   if l2ProtocolModeGuard( mode, token ) is None:
      if mode.intf.isSubIntf():
         if bridgingHwCapabilities.l2ProtocolForwardingSubIntfSupported:
            return None
      elif PortChannelIntfId.isPortChannelIntfId( mode.intf.name ):
         if bridgingHwCapabilities.l2ProtocolForwardingLagSupported and \
               Toggles.EbraToggleLib.toggleLagL2ProtocolFwdEnabled():
            return None
      else:
         return None
   return CliParser.guardNotThisPlatform

def gotoL2ProtocolMode( mode, args ):
   childMode = mode.childMode( L2ProtocolMode )
   mode.session_.gotoChildMode( childMode )

def targetsForProtocol( mode, context ):
   targets = {}
   protocol = context.sharedResult[ 'PROTOCOL' ]
   protocol = ' '.join( protocol )
   action = context.sharedResult[ 'ACTION' ]
   if ( action == tacL2PtAction.forward ) and ( mode.session_.startupConfig() or
         cliTokenToL2PtProtocol[ protocol ] in
         bridgingHwCapabilities.l2ProtocolForwardingTrapSupportedProtocol ):
      targets[ l2PtForwardTarget() ] = l2PtForwardTarget()
   return targets

def actionsForProtocol( mode, context ):
   actions = { tacL2PtAction.forward: tacL2PtAction.forward }
   protocol = context.sharedResult[ 'PROTOCOL' ]
   protocol = ' '.join( protocol )
   if mode.session_.startupConfig() or cliTokenToL2PtProtocol[ protocol ] in \
         bridgingHwCapabilities.l2ProtocolForwardingDropSupportedProtocol:
      actions[ tacL2PtAction.drop ] = tacL2PtAction.drop
   return actions

#--------------------------------------------------------------------------------
# l2-protocol
#--------------------------------------------------------------------------------
class L2ProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'l2-protocol'
   noOrDefaultSyntax = syntax
   data = {
      'l2-protocol': CliCommand.guardedKeyword( 'l2-protocol',
                        helpdesc='Enter L2 Protocol configuration mode',
                        guard=l2ProtocolModeGuard ),
   }
   handler = gotoL2ProtocolMode

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      l2PtProfileConfig.profile.clear()

BasicCliModes.GlobalConfigMode.addCommandClass( L2ProtocolCmd )

#-------------------------------------------------------------------------------
# The "forwarding profile <profileName>" mode.
#-------------------------------------------------------------------------------
class L2ProtocolProfileModeContext:
   def __init__( self, mode, profileName ):
      self.mode = mode
      self.profileName_ = profileName
      # Current profile
      profile = l2PtProfileConfig.profile.get( profileName )
      # Profile undergoing editing
      self.editProfile_ = Tac.newInstance( 'Ebra::L2Pt::L2PtProfile',
                                           self.profileName_ )
      if profile:
         self.copyProfile( self.editProfile_, profile )
         self.editProfile_.version = profile.version

   def copyProfile( self, dstProfile, srcProfile ):
      copier = Tac.newInstance( 'Cli::Session::EntityCopy' )
      copier.handler = Tac.newInstance( "Ebra::L2Pt::L2PtProfileCopyHandler" )
      copier.handler.handleEntity( copier, dstProfile, srcProfile, '' )

   def editProfile( self ):
      return self.editProfile_

   def profileName( self ):
      return self.profileName_

   def commit( self ):
      profile = l2PtProfileConfig.profile.newMember( self.profileName_ )
      if profile.version != self.editProfile_.version:
         self.mode.addWarning( 'Version mismatch possibly due to profile being '
                               'edited in another CLI session. Aborting change.')
         return
      self.copyProfile( profile, self.editProfile_ )
      # pylint: disable-next=consider-using-f-string
      t0( 'Profile Version = %d' % profile.version )

class L2ProtocolProfileMode( L2ProtocolProfileModeBase, BasicCli.ConfigModeBase ):
   name = "L2 Protocol Forwarding Profile Configuration"

   def __init__( self, parent, session, profileName ):
      self.l2ProtocolProfileModeContext = L2ProtocolProfileModeContext( self,
                                                                        profileName )
      self.profileName_ = profileName
      L2ProtocolProfileModeBase.__init__( self, param=self.profileName_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.l2ProtocolProfileModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      context = self.l2ProtocolProfileModeContext
      self.l2ProtocolProfileModeContext = None
      context.commit()

def gotoL2PtProfileMode( mode, args ):
   childMode = mode.childMode( L2ProtocolProfileMode,
                               profileName=args[ 'PROFILENAME' ] )
   mode.session_.gotoChildMode( childMode )

def deleteL2PtProfile( mode, args ):
   # Delete L2Pt profile from config
   del l2PtProfileConfig.profile[ args[ 'PROFILENAME' ] ]

matcherL2ProfileName = CliMatcher.DynamicNameMatcher(
                         lambda mode: l2PtProfileConfig.profile,
                         'L2 protocol forwarding profile name',
                         pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

#--------------------------------------------------------------------------------
# [ no | default ] forwarding profile PROFILENAME
#--------------------------------------------------------------------------------
class ForwardingProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding profile PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'forwarding': '',
      'profile': 'Enter forwarding profile configuration mode',
      'PROFILENAME': matcherL2ProfileName,
   }
   handler = gotoL2PtProfileMode
   noOrDefaultHandler = deleteL2PtProfile

L2ProtocolMode.addCommandClass( ForwardingProfileProfilenameCmd )

tagFormats = { tacL2PtTagFormat.tagged : tacL2PtTagFormat.tagged,
               tacL2PtTagFormat.untagged : tacL2PtTagFormat.untagged }

def getProtocols( mode ):
   # pylint: disable-next=unnecessary-comprehension
   cliTokens = [ token for token in cliTokenToL2PtProtocol ]
   if not mode.session_.startupConfig() and bridgingHwCapabilities:
      if "bfd per-link rfc-7130" in cliTokens and \
            not( Toggles.EbraToggleLib.toggleL2ProtocolFwdMbfdEnabled() and
            bridgingHwCapabilities.l2ProtocolForwardingMbfdSupported ):
         cliTokens.remove( "bfd per-link rfc-7130" )
      if bridgingHwCapabilities.l2ProtocolTransparencySubset:
         l2Protocol = [ [ 'lacp' ], [ 'lldp' ], [ 'stp' ] ]
         if bridgingHwCapabilities.l2ProtocolMacsecEapolTransparencySupported:
            l2Protocol.append( [ 'macsec' ] )
         return l2Protocol
   protocols = [ protocol.split( " " ) for protocol in cliTokens ]
   return protocols

def getTagFormats( mode ):
   if bridgingHwCapabilities:
      if bridgingHwCapabilities.l2ProtocolTransparencySubset:
         # we only support tacL2PtTagFormat.all
         return {}
   return tagFormats

dynTagMatcher = CliMatcher.DynamicKeywordMatcher( getTagFormats )

def deleteRule( profile, l2PtProtocolInfo, l2ProtocolMatch ):
   del profile.protocolToAction[ l2PtProtocolInfo ]
   seq = profile.protocolInfoToSeq.get( l2ProtocolMatch )
   if seq is not None:
      del profile.seqToProtocolInfo[ seq ]
      del profile.protocolInfoToSeq[ l2ProtocolMatch ]

def addRule( profile, l2PtProtocolInfo, l2ProtocolMatch, action, target=None ):
   seq = max( profile.seqToProtocolInfo, default=0 ) + 10
   if target == l2PtForwardTarget():
      action = tacL2PtAction.trap
   if l2ProtocolMatch not in profile.protocolInfoToSeq or \
            profile.protocolToAction[ l2PtProtocolInfo ] != action:
      if l2ProtocolMatch in profile.protocolInfoToSeq:
            # This is a conflicting rule (forward vs. trap or vice versa).
            # Delete the old rule before adding the new one.
         deleteRule( profile, l2PtProtocolInfo, l2ProtocolMatch )

      l2ProtocolFwdAction = Tac.newInstance( "Ebra::L2Pt::L2ProtocolFwdAction",
                                                action )
      l2ProtocolFwdInfo = Tac.newInstance( "Ebra::L2Pt::L2ProtocolFwdInfo",
                                                l2ProtocolMatch,
                                                l2ProtocolFwdAction )
      profile.protocolToAction[ l2PtProtocolInfo ] = action
      profile.protocolInfoToSeq[ l2ProtocolMatch ] = seq
      profile.seqToProtocolInfo[ seq ] = l2ProtocolFwdInfo

def configL2ProtocolFwdProtocol( mode, args ):
   protocol = " ".join( args[ 'PROTOCOL' ] )
   action = args[ 'ACTION' ]
   tagFormat = args.get( 'TAGFORMAT' )
   target = args.get( 'TARGET' )
   macAddrSet = l2ProtocolMacAddress( cliTokenToL2PtProtocol[ protocol ] )
   if not tagFormat:
      tagFormat = tacL2PtTagFormat.all
   profile = mode.l2ProtocolProfileModeContext.editProfile()

   for macAddr in macAddrSet:
      l2PtProtocolInfo = Tac.newInstance( "Ebra::L2Pt::L2ProtocolInfo", macAddr,
                                          tagFormat, 0 )
      l2ProtocolMatch = Tac.newInstance( "Ebra::L2Pt::L2ProtocolMatch", macAddr,
                                         tagFormat, 0 )

      if CliCommand.isNoOrDefaultCmd( args ):
         deleteRule( profile, l2PtProtocolInfo, l2ProtocolMatch )
      else:
         addRule( profile, l2PtProtocolInfo, l2ProtocolMatch, action, target )

# -----------------------------------------------------------------------------------
# [ no | default ] PROTOCOL [ TAGFORMAT ] ACTION [ TARGET ] in L2ProtocolProfileMode
# -----------------------------------------------------------------------------------
class L2ProtocolFwdProtocolConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'PROTOCOL [ TAGFORMAT ] ACTION [ TARGET ]'
   noOrDefaultSyntax = syntax
   data = {
      'PROTOCOL': CliCommand.Node(
         CliMatcher.KeywordListsMatcher( getProtocols ),
         storeSharedResult=True ),
      'TAGFORMAT': dynTagMatcher,
      'ACTION': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( actionsForProtocol,
         passContext=True ), storeSharedResult=True ),
      'TARGET': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( targetsForProtocol,
         passContext=True ) ),
   }
   handler = configL2ProtocolFwdProtocol
   noOrDefaultHandler = configL2ProtocolFwdProtocol
L2ProtocolProfileMode.addCommandClass( L2ProtocolFwdProtocolConfigCmd )

def l2ProtocolFwdAddressGuard( mode, token ):
   if bridgingHwCapabilities.l2ProtocolForwardingAddressSupported:
      return None
   return CliParser.guardNotThisPlatform

def configL2ProtocolFwdAddress( mode, args ):
   macAddr = args[ 'ADDRESS' ]
   action = args[ 'ACTION' ]
   tagFormat = args.get( 'TAGFORMAT' )
   if not tagFormat:
      tagFormat = tacL2PtTagFormat.all
   profile = mode.l2ProtocolProfileModeContext.editProfile()

   l2PtProtocolInfo = Tac.newInstance( "Ebra::L2Pt::L2ProtocolInfo", macAddr,
                                       tagFormat, 0 )
   l2PtProtocolInfo.addrConfig = True
   l2ProtocolMatch = Tac.newInstance( "Ebra::L2Pt::L2ProtocolMatch", macAddr,
                                       tagFormat, 0 )
   l2ProtocolMatch.addrConfig = True

   if CliCommand.isNoOrDefaultCmd( args ):
      deleteRule( profile, l2PtProtocolInfo, l2ProtocolMatch )
   else:
      addRule( profile, l2PtProtocolInfo, l2ProtocolMatch, action )

# -----------------------------------------------------------------------------------
# [ no | default ] mac-address [ TAGFORMAT ] ACTION  in L2ProtocolProfileMode
# -----------------------------------------------------------------------------------
class L2ProtocolFwdAddressConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-address ADDRESS [ TAGFORMAT ] ACTION'
   noOrDefaultSyntax = syntax
   data = {
      'mac-address': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'mac-address', helpdesc='MAC address to match' ),
         guard=l2ProtocolFwdAddressGuard
         ),
      'ADDRESS': MacAddr.macAddrMatcher,
      'TAGFORMAT': dynTagMatcher,
      'ACTION': CliMatcher.EnumMatcher( { 'forward': 'forward',
                                         'drop': 'drop' } )
   }
   handler = configL2ProtocolFwdAddress
   noOrDefaultHandler = configL2ProtocolFwdAddress

L2ProtocolProfileMode.addCommandClass( L2ProtocolFwdAddressConfigCmd )


class L2ProtocolIntfProfileModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      intfTypesSupported = ( 'Ethernet', 'Switch', 'Port-Channel' )
      return mode.intf.name.startswith( intfTypesSupported )

IntfCli.IntfConfigMode.addModelet( L2ProtocolIntfProfileModelet )

def setL2PtProfile( mode, args ):
   intf = mode.intf
   profileName = args.get( 'PROFILENAME' )
   if CliCommand.isNoOrDefaultCmd( args ):
      if profileName:
         if intf.name in l2PtIntfConfig.intfToProfile and \
               l2PtIntfConfig.intfToProfile[ intf.name ] == profileName:
            del l2PtIntfConfig.intfToProfile[ intf.name ]
      else:
         del l2PtIntfConfig.intfToProfile[ intf.name ]
   else:
      l2PtIntfConfig.intfToProfile[ intf.name ] = profileName

# The guard is configured with "forwarding" token so that we don't
# interfere with other Intf mode commands that start with "l2-protocol"
#------------------------------------------------------------------------
# [ no | default ] l2-protocol forwarding profile PROFILENAME
#------------------------------------------------------------------------
class L2ProtocolProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'l2-protocol forwarding profile PROFILENAME'
   noOrDefaultSyntax = 'l2-protocol forwarding profile [ PROFILENAME ]'
   data = {
      'l2-protocol': 'Set L2 protocol characteristics of the interface',
      'forwarding': CliCommand.guardedKeyword( 'forwarding',
                        helpdesc='',
                        guard=l2ProtocolFwdModeGuard ),
      'profile': 'Enter forwarding profile configuration mode',
      'PROFILENAME': matcherL2ProfileName,
   }
   handler = setL2PtProfile
   noOrDefaultHandler = setL2PtProfile

L2ProtocolIntfProfileModelet.addCommandClass( L2ProtocolProfilenameCmd )

#---------------------------------------------------------------------------------
# 1) To delete L2 Protocol forwarding config when the corresponding interface gets
#    deleted.
# 2) Support for default interface command.
#---------------------------------------------------------------------------------
class L2ProtocolFwdIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "L2ProtocolFwdIntfJanitor: interface", self.intf_.name, "going away" )
      del l2PtIntfConfig.intfToProfile[ self.intf_.name ]


def Plugin( entityManager ):
   global l2PtProfileConfig
   global l2PtIntfConfig
   global bridgingHwCapabilities

   l2PtProfileConfig = ConfigMount.mount( entityManager,
         "l2protocolforwarding/profileconfig",
         "Ebra::L2Pt::L2PtProfileConfig", "w" )
   l2PtIntfConfig = ConfigMount.mount( entityManager,
         "l2protocolforwarding/intfconfig",
         "Ebra::L2Pt::L2PtIntfConfig", "w" )

   IntfCli.Intf.registerDependentClass( L2ProtocolFwdIntfJanitor )
   mount = LazyMount.mount
   bridgingHwCapabilities = mount( entityManager, "bridging/hwcapabilities",
                                   "Bridging::HwCapabilities", "r" )
