# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Enum, Dict, List, Str, Bool
from IntfModels import Interface
from EbraLib import ( tacL2PtProtocol, tacL2PtTagFormat, tacL2PtAction,
                      l2PtProtocolToCliToken, l2PtForwardTarget,
                      l2PtActionToCapiAction, l2PtCliActionToCapiAction )
from Intf import IntfRange
from ArnetModel import MacAddress
import Arnet
import TableOutput
import Toggles.EbraToggleLib

class L2ProtocolForwardingInfo( Model ):
   protocol = Enum( values=tacL2PtProtocol.attributes,
                    optional=True,
                    help="L2 Protocol", default=None )
   address = MacAddress( help='Protocol MAC address', optional=True )
   tagFormat = Enum( values=tacL2PtTagFormat.attributes,
                     optional=True,
                     help="VLAN tag format" )
   action = Enum( values=list( l2PtCliActionToCapiAction.values() ),
                  help="Action to be performed on protocol packets" )

   def configuredUsingAddress( self ):
      return self.protocol is None

class L2ProtocolForwardingProfile( Model ):
   protocolInfo = List( valueType=L2ProtocolForwardingInfo,
                           optional=True,
                           help="L2 protocol forwarding information " )
   def render( self ):
      for info in self.protocolInfo:
         if info.configuredUsingAddress():
            protocolName = info.address.displayString
         else:
            protocolName = l2PtProtocolToCliToken[ info.protocol ].upper()
         # pylint: disable-next=consider-using-f-string
         print( "  %s:" % protocolName, end=" " )
         if info.tagFormat and info.tagFormat != tacL2PtTagFormat.all:
            # pylint: disable-next=consider-using-f-string
            print( "%s" % info.tagFormat.title(), end=" " )
         print( l2PtCliActionToCapiAction.reverse()[ info.action ].title() )

class L2ProtocolForwardingProfiles( Model ):
   profiles = Dict( valueType=L2ProtocolForwardingProfile,
                    help="A mapping of L2 protocol forwarding profile name to "
                         "profile information" )
   def render( self ):
      for profileName, profile in self.profiles.items():
         # pylint: disable-next=consider-using-f-string
         print( "Profile: %s" % profileName )
         profile.render()

class L2ProtocolForwardingProfileSummary( Model ):
   configuredIntfs = List( valueType=Interface,
                           optional=True,
                           help="Interfaces on which profile is configured" )
   activeIntfs = List( valueType=Interface,
                       optional=True,
                       help="Interfaces on which profile is active" )
   def render( self ):
      print( '  Configured:', end=' ' )
      print( ','.join( IntfRange.intfListToCanonical( self.configuredIntfs ) ) )
      print( '  Active:', end=' ' )
      print( ','.join( IntfRange.intfListToCanonical( self.activeIntfs ) ) )

class L2ProtocolForwardingProfileSummaries( Model ):
   profiles = Dict( valueType=L2ProtocolForwardingProfileSummary,
                    help="A mapping of L2 protocol forwarding profile name to "
                         " interfaces on which it is active/configured." )
   def render( self ):
      for profileName, profile in self.profiles.items():
         # pylint: disable-next=consider-using-f-string
         print( "Profile: %s" % profileName )
         profile.render()

class L2ProtocolForwardingInterface( Model ):
   profileName = Str( help="Active L2 protocol forwarding profile" )
   protocolInfo = List( valueType=L2ProtocolForwardingInfo,
                        help="L2 protocol forwarding information",
                        optional=True )

class L2ProtocolForwardingInterfaces( Model ):
   interfaces = Dict( valueType=L2ProtocolForwardingInterface,
                      keyType=Interface,
                      help="L2 protocol forwarding information for interfaces" )
   _detail = Bool( help="User requested detailed L2 protocol forwarding information",
                   optional=True )

   def detailIs( self, detail ):
      self._detail = detail

   def _mapIntfProtocolTagToAction( self ):
      intfProtocolTagToAction = {}
      for intfName, intfInfo in self.interfaces.items():
         protocolInfo = intfInfo.protocolInfo
         for info in protocolInfo:
            actionLegend = info.action.upper()[ 0 ]
            if info.configuredUsingAddress():
               continue
            if info.action == l2PtActionToCapiAction[ tacL2PtAction.trap ]:
               actionLegend = l2PtForwardTarget().upper()[ 0 ]
            if info.tagFormat in [ tacL2PtTagFormat.all,
                                   tacL2PtTagFormat.tagged ]:
               intfProtocolTagToAction[ ( intfName, info.protocol, 'T' ) ] = \
                                                         actionLegend
            if info.tagFormat in [ tacL2PtTagFormat.all,
                                   tacL2PtTagFormat.untagged ]:
               intfProtocolTagToAction[ ( intfName, info.protocol, 'U' ) ] = \
                                                         actionLegend
      return intfProtocolTagToAction

   def render( self ):
      fc = TableOutput.Format( justify='center' )
      fc.padLimitIs( True )
      fl = TableOutput.Format( justify='left' )
      fl.padLimitIs( True )

      # This means that user asked for detailed information
      if self._detail:
         intfProtocolTagToAction = self._mapIntfProtocolTagToAction()

         headers = [ "Interface", "Profile" ]
         tagFormats = ( 'T', 'U' )
         formats = [ fl, fl ]

         protocols = sorted( tacL2PtProtocol.attributes,
                     key=lambda protocol: l2PtProtocolToCliToken[ protocol ] )

         if not Toggles.EbraToggleLib.toggleL2ProtocolFwdMbfdEnabled():
            protocols.remove( 'mbfd' )
         for protocol in protocols:
            # shorten # BFD PER-LINK RFC-7130 to BFD RFC-7130
            if protocol == 'mbfd':
               headers += [ ( "BFD RFC-7130", tagFormats ) ]
            else:
               headers += [ ( l2PtProtocolToCliToken[ protocol ].upper(),
                  tagFormats ) ]
            formats += [ fc ] * len( tagFormats )

         table = TableOutput.createTable( headers )
         table.formatColumns( *formats )
         for intfName in Arnet.sortIntf( self.interfaces ):
            profileName = self.interfaces[ intfName ].profileName
            # limit width of profile name column to 16 characters
            # this was decided in cli-review@ discussion
            entry = [ intfName, profileName[ : 16 ] ]
            for protocol in protocols:
               for tagFormat in tagFormats:
                  entry.append( intfProtocolTagToAction.get( ( intfName, protocol,
                                                               tagFormat ), '-' ) )
            table.newRow( *entry )
         print( 'Tagging Types: T: tagged U: untagged' )
         # pylint: disable-next=consider-using-f-string
         print( 'Actions: F: forward %s: forward %s D: drop' %
               ( l2PtForwardTarget().upper()[ 0 ], l2PtForwardTarget() ) )
      else:
         headers = [ 'Interface', 'Profile' ]
         table = TableOutput.createTable( headers )
         table.formatColumns( fl, fl )
         for intfName in Arnet.sortIntf( self.interfaces ):
            table.newRow( intfName, self.interfaces[ intfName ].profileName )
      print( table.output() )
