#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import deque
from CliMatcher import SlashedIntegerMatcher
from CliPlugin import LagModel
from LacpConstants import * # pylint: disable=wildcard-import
import CliGlobal
import Tac, Arnet, CliParser
import LazyMount, ConfigMount
import CliExtensions
import MultiRangeRule
from TypeFuture import TacLazyType

# Globals written by the Plugin function at the end of this file
bridgingHwCapabilities = None
lacpCliConfig = None
lacpCliOverrideDir = None
lacpCounters = None
lacpCountersCheckpoint = None
lacpEventTable = None
lacpStatus = None
lagConfigCli = None
lagConfigDir = None
lagIntfConfigDir = None
lagIntfStatusDir = None
recircHwConfig = None
gv = CliGlobal.CliGlobal( swagHwCapabilities=None )

lagConfigAllowCallbacks = []
lagConfigValidCallbacks = []

MemberId = TacLazyType( 'Swag::MemberId' )
PortChannelNum = Tac.Type( "Lag::PortChannelNum" )
PortChannelIntfId = Tac.Type( "Arnet::PortChannelIntfId" )
FabricChannelIntfId = TacLazyType( "Arnet::FabricChannelIntfId" )
portChannel = PortChannelIntfId.portChannel
isPortChannelIntfId = PortChannelIntfId.isPortChannelIntfId
isRecirc = PortChannelIntfId.isRecirc
portChannelConfigLimit = Tac.Type( "Lag::PortChannelConfigLimit" ).limit
# pylint: disable-next=consider-using-f-string
configLimitExceededMsg = ( "The number of configured port channels exceeds "
                           "the config limit %d." % portChannelConfigLimit )

# Used for annotating Port-Channels with additional information
annotateMarkers = ( "*" )

# showLacpAnnotateHook allows another package to annotate Port-Channels with
# additional information. Use registerLacpAnnotateHook to register the hook.
showLacpAnnotateHook = CliExtensions.CliHook()

# Association marker-->detail message
markerMsgs = {}
# List of tuples (marker, annotateFunc) for registered showLacpAnnotateHook 
# extensions
afmAssoc = []

# showLacpSysIdHook allows another package to provide extra information
# about system ids used by LACP. The hook should take the lacpConfig
# and the brief parameters and print out the relevant output.
showLacpSysIdHook = CliExtensions.CliHook()

# showLacpInternalSysIdHook allows another package to provide extra information
# about system ids used by LACP. The hook takes the lacpConfig as a parameter
# and is responsible for printing out the relevant information.
showLacpInternalSysIdHook = CliExtensions.CliHook()

def memberIdRangeFn( mode, context=None ):
   if gv.swagHwCapabilities:
      return ( MemberId.min, gv.swagHwCapabilities.maxMembersSupported )
   else:
      return ( MemberId.min, MemberId.max )

# Fabric-channel interface number matcher.
FabricChannelIntfNumberMatcher = SlashedIntegerMatcher(
   rangeFn=memberIdRangeFn,
   helpname="MEMBER ID/GROUP ID",
   helpdesc="Fabric-channel",
   subLbound=PortChannelNum.min,
   subUbound=PortChannelNum.max )

def getCurrentSystemPriority():
   with ConfigMount.ConfigMountDisabler( disable=True ):
      return lacpCliConfig.priority

# guard for recirc channel group
def recircGuard( mode, token ):
   if not recircHwConfig.recircSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

# guard for fabric channel-group. Will be fleshed out once hardware
# capability merges. BUG977080 tracks this.
def fabricChannelGuard( mode, token ):
   if not gv.swagHwCapabilities.swagSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def hashOutputIntfGuard( mode, token ):
   # Supports cli to determine ECMP or LAG hash interface output
   if ecmpOutputIntfGuard( mode, token ) is None or \
      lagOutputIntfGuard( mode, token ) is None:
      return None
   return CliParser.guardNotThisPlatform

def lagOutputNonUnicastGuard( mode, token ):
   if not bridgingHwCapabilities.lagShowOutputIntfNonUnicastSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

# guard for 'show LAG output member' command
def lagOutputIntfGuard( mode, token ):
   if not bridgingHwCapabilities.lagShowOutputIntfSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def registerLagConfigCallback( callback ):
   # Any function that wants to get called back when a physical
   # port is configured as a lag member should register here.
   # a callback can allow or deny the configuration of the
   # physical member as a lag.
   #
   # callback parameters:
   # @modes: interface modes for the Cli
   # @newLagName: the new lag name the ports are put into, or None
   # @oldLagNames: the old lag names the ports were in, or None
   lagConfigAllowCallbacks.append( callback )

def registerLagConfigValidCallback( callback ):
   # Any function that wants to make sure that after the addition/removal
   # of a physical port to a lag, the configuration is valid registers here.
   # On denial addition/deletion of lag member will be reverted back.
   #
   # callback parameters:
   # @mode: mode for the Cli
   # @portName: name of the physical port
   # @newLagName: the new lag name the port is put into, or None
   # @oldLagName: the old lag name the port was in, or None
   lagConfigValidCallbacks.append( callback )

def registerLacpAnnotateHook( hook ):
   # Any package that wants to annotate a port-channel for 'show lacp'
   # commands (excluding show lacp aggregate) registers its hook here.
   # The hook should return a tuple ( message, annotateFunc )
   # with the detail message description and function that returns True if
   # the passed Port-Channel name should be annotated.
   markerIndex = len( showLacpAnnotateHook.extensions() )
   assert markerIndex < len( annotateMarkers )
   showLacpAnnotateHook.addExtension( hook )

   ( msg, af ) = hook()
   marker = annotateMarkers[ markerIndex ]

   markerMsgs[ marker ] = msg   
   afmAssoc.append( ( af, marker ) )

def ecmpOutputIntfGuard( mode, token ):
   # Supports cli to determine ECMP interface output
   if bridgingHwCapabilities.ecmpShowOutputIntfSupported:
      return None
   return CliParser.guardNotThisPlatform

def ethPhyIntfLagConfigDir( mode ):
   return lagConfigDir.phyIntf

def ethPhyIntfLagCliConfigDir( mode ):
   return lagConfigCli.phyIntf

def _ethPhyIntfLagConfig( mode, create, name, configDir ):
   if not name:
      name = mode.intf.name
   epilc = configDir.get( name )
   if not epilc:
      if create:
         epilc = configDir.newMember( name )
      else:
         return None
   return epilc

def ethPhyIntfLagConfig( mode, name=None ):
   # lag/config.phyIntf is a non-instantiating collection. create has to be
   # False
   return _ethPhyIntfLagConfig( mode, False , name,
                                ethPhyIntfLagConfigDir( mode ) )

def ethPhyIntfLagCliConfig( mode, create=False, name=None ):
   return _ethPhyIntfLagConfig( mode, create, name,
                                ethPhyIntfLagCliConfigDir( mode ))

def inactiveLag( intfId ):
   return ( lagIntfConfigDir.intfConfig.get( intfId, None ) and
            not lagIntfStatusDir.intfStatus.get( intfId, None ) and
            len( lagIntfStatusDir.intfStatus ) >= portChannelConfigLimit )

def portChannelFilter( intfId ):
   return isPortChannelIntfId( intfId ) and not isRecirc( intfId )

def fabricChannelFilter( intfId ):
   return FabricChannelIntfId.isFabricChannelIntfId( intfId )

# Filters fabric-channel interfaces from ethLagIntfConfigDir.
def fabricChannelIntfList( candidateIntfList ):
   if not candidateIntfList:
      candidateIntfList = lagIntfConfigDir.intfConfig.keys()
   return list( filter( fabricChannelFilter, candidateIntfList ) )

# pass a channelGroups and the mode in, and return a list of
# channel-groups that match the channelGroups *AND* are configured
# for LACP.  channelGroups of None means list all appropriate
# channel-groups
def portchannelList( mode, channelGroups, lacpOnly=True, inactiveWarn=True, 
                     filterFunc=portChannelFilter, silent=False ):
   if channelGroups is None:
      explicit = False
      candidateChannelList = lagIntfConfigDir.intfConfig
   else:
      explicit = True
      if not isinstance( channelGroups, list ):
         channelGroups = [ channelGroups ]
      # Filter out unconfigured channels
      candidateChannelSet = { c.idStr for c in channelGroups }
      configuredChannels = set( lagIntfConfigDir.intfConfig )
      unconfiguredChannels = candidateChannelSet.difference( configuredChannels )
      candidateChannelList = list( candidateChannelSet.difference(
         unconfiguredChannels ) )
   candidateChannelList = deque( Arnet.sortIntf( candidateChannelList ) )
   inactiveLagIds, filteredChannels = [], []
   while candidateChannelList:
      channel = candidateChannelList.popleft()
      if not filterFunc( channel ):
         if explicit and not silent:
            print( f"{channel} not configured as LAG" )
         continue
      lagIntfConfig = lagIntfConfigDir.intfConfig.get( channel )
      # Handle inactive port channels
      if inactiveLag( channel ):
         if inactiveWarn:
            # Save inactive lag ID for aggregated warning
            inactiveLagIds.append( portChannel( channel ) )
         else:
            # If inactiveWarn == False, we want the inactive channel
            # in our command's output
            filteredChannels.append( channel )
         continue
      # If lacpOnly, filter out non-LACP port channels
      if lacpOnly and lagIntfConfig.mode == 'lagModeOn':
         if explicit and not silent:
            print( f"{channel} not configured for LACP" )
         continue
      # Otherwise, give channel back to command to render
      filteredChannels.append( channel )
   if inactiveWarn and inactiveLagIds and not silent:
      pluralFirst = "s" if len( inactiveLagIds ) > 1 else ""
      ranges = MultiRangeRule.multiRangeToCanonicalString( inactiveLagIds )
      pluralSecond = "are" if len( inactiveLagIds ) > 1 else "is"
      print( f"Port-Channel{pluralFirst}{ranges} {pluralSecond} inactive.",
             configLimitExceededMsg )
   if explicit and unconfiguredChannels and not silent:
      unconfiguredChannels = Arnet.sortIntf( unconfiguredChannels )
      idStrToId = { c.idStr: c.id for c in channelGroups }
      unconfiguredLagIds = ( idStrToId[ c ] for c in unconfiguredChannels )

      plural = "s" if len( unconfiguredChannels ) > 1 else ""
      ranges = MultiRangeRule.multiRangeToCanonicalString( unconfiguredLagIds )
      print( f"Port-Channel{plural}{ranges} not configured as LAG" )
   return filteredChannels

def syncCounterCheckpoint( toDir, fromDir ):

   toIntfNames = set( toDir.portCounters.keys() )
   fromIntfNames = set( fromDir.portCounters.keys() )

   for intf in toIntfNames:
      if intf not in fromIntfNames:
         del toDir.portCounters[ intf ]

   toIntfNames = set( toDir.portCounters.keys() )

   for intf in fromIntfNames:
      if intf not in toIntfNames:
         toDir.portCounters.newMember( intf )

def updateLacpPortIdRangeForLag( poName ):
   if lacpCliConfig.portIdRange != Tac.Value( 'Lacp::PortIdRange' ):
      lacpCliOverrideDir.portIdRange[ poName ] = lacpCliConfig.portIdRange

def deleteLacpPortIdRangeForLag( poName ):
   del lacpCliOverrideDir.portIdRange[ poName ]

def updateLacpOverridePortIdRange():
   if lacpCliConfig.portIdRange == Tac.Value( 'Lacp::PortIdRange' ):
      for poName in lagIntfConfigDir.intfConfig:
         deleteLacpPortIdRangeForLag( poName )
   else:
      for poName in lagIntfConfigDir.intfConfig:
         updateLacpPortIdRangeForLag( poName )

def deleteLacpSystemIdForLag( poName ):
   del lacpCliOverrideDir.systemId[ poName ]

def lacpPortStateModel( state ):
   def stateFlag( flag ):
      return bool( state & flag )

   return LagModel.LacpPortState(
      activity = stateFlag( StateActivity ),
      timeout = stateFlag( StateTimeout ),
      aggregation = stateFlag( StateAggregation ),
      synchronization = stateFlag( StateSynchronization ),
      collecting = stateFlag( StateCollecting ),
      distributing = stateFlag( StateDistributing ),
      defaulted = stateFlag( StateDefaulted ),
      expired = stateFlag( StateExpired )
      )

# returns list of qualifying ports in the port-channel.
# Trampoline to channelPorts C++ implementation for faster execution.
def _channelPorts( tacLagCliSupport, channel,
                   lacpOnly=True, status=Tac.Type( "Lag::LagStatusFilter" ).\
                      filterOnActiveAndInactive, useLagConfig=False ):
   tacLagCliSupport.portList.clear()
   tacLagCliSupport.channelPorts( channel, lacpOnly, status, useLagConfig )
   return tacLagCliSupport.portList

# If channel is singleton, return list of ports in channel.  If channel is a list,
# returns a dict of channel:ports Status can be 'filterOnActiveAndInactive' (return
# both active and inactive), or 'filterOnActive' or 'filterOnInactive'.
# "active" is always defined consistently, whether we are talking about the
# lag as a whole (lacpOnly == False) or only about LACP (lacpOnly ==
# True). "inactive" is defined relative to whether you are dealing with the
# LAG as a whole, or LACP only.  A port is "active" in the LAG in the sense of being
# a member of the LAG, which (if the port-channel is in lacpMode) is the same as
# being a member of stat->agg->aggregate in LACP. When lacpOnly==False, then
# "inactive" means any port configured to be in the port-channel, but is not in
# lagStatus.member.  In the case lacpOnly==True, then "inactive" means those ports
# considered by LACP, (e.g. the LAG hasn't rejected them as incompatible) but not
# aggregated, e.g. it only considers those that have been added to
# lacpIntfLagConfig.member.  The inactive ports are those that are either in
# stat->agg->standby, stat->otherIntf, or stat->agg->selected but *not* in
# stat->agg->aggregate. 
# It is possible that a port is considered by LACP but is in none of those
# collections.
#  useLagConfig:
#     When this option is True, the mbrList is composed from lag/config/phyIntf/
def channelPorts( mode, channel, lacpOnly=True, 
                  status=Tac.Type( "Lag::LagStatusFilter" ).\
                     filterOnActiveAndInactive,
                  useLagConfig=False ):
   tacLagCliSupport = Tac.newInstance( "Lag::LagCliSupport",
                                       lagConfigDir.force(),
                                       lagIntfConfigDir.force(),
                                       lagIntfStatusDir.force(),
                                       lacpStatus.force() )
   if not isinstance( channel, list ):
      return _channelPorts( tacLagCliSupport, channel,
                            lacpOnly, status, useLagConfig )
   else:
      return { port:
                       list( _channelPorts( tacLagCliSupport,
                                      port, lacpOnly, status, useLagConfig ) )
                     for port in channel }

# This should return {} unless there is some bug, or we catch it in the middle
# of reactors deleting ports from a channel that has been unconfigured.
# Trampolines to lacpOrphanPorts C++ implementation for faster execution.
def lacpOrphanPorts( mode ):
   """Find ports in lls.portStatus whose lag is not in lagIntfConfigDir"""
   
   # Complete mounts if the required entities are not yet resolved.
   tacLagCliSupport = Tac.newInstance( "Lag::LagCliSupport",
                                       lagConfigDir.force(),
                                       lagIntfConfigDir.force(),
                                       lagIntfStatusDir.force(),
                                       lacpStatus.force() )
   tacLagCliSupport.lacpOrphanPorts()
   return list( tacLagCliSupport.portMap.items() )

LagProtocolMap = { "lagModeUnconfigured": "Unconfigured",
                   "lagModeOn":"Static",
                   "lagModeLacp":"LACP" } 
LacpModeMap = { None:"Unknown",
                "lacpModeOff":"off",
                "lacpModePassive":"passive",
                "lacpModeActive":"active" }

def lagModeToProtocol( mode ):
   assert mode in LagProtocolMap
   return LagProtocolMap[ mode ].lower()

# getLagMemberDict
#  This returns a Dictionary of Lists:
#     Key   - Lag Name
#     Value - List of Members in the Lag
def getLagMemberDict ( mode, channelGroupIdList=None, 
                       filterFunc=portChannelFilter ):
   channels = portchannelList( mode, channelGroupIdList, lacpOnly=False,
                                         filterFunc=filterFunc )
   # Add the Member Lists from the info in lag/config/phyIntf/
   return channelPorts( mode, channels, useLagConfig=True )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bridgingHwCapabilities
   global lacpCliConfig
   global lacpCliOverrideDir
   global lacpCounters, lacpCountersCheckpoint
   global lacpEventTable
   global lacpStatus
   global lagConfigCli, lagConfigDir
   global lagIntfConfigDir, lagIntfStatusDir
   global recircHwConfig

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   lacpCliConfig = ConfigMount.mount( entityManager, "lag/lacp/input/config/cli",
                                      "Lacp::CliConfig", "w" )
   lacpCliOverrideDir = ConfigMount.mount(entityManager,
                                 "lag/input/lacpoverride/cli",
                                 "Lag::Input::LacpOverrideDir", "w" )
   lacpCounters = LazyMount.mount( entityManager, "lag/lacp/counters",
                                   "Lacp::LacpCounters", "r" )
   lacpCountersCheckpoint = LazyMount.mount( entityManager, 
                                             "lag/lacp/countersCheckpoint",
                                             "Lacp::LacpCounters", "w" )
   lacpEventTable = LazyMount.mount( entityManager, "lag/lacp/eventtable",
                                     "Lacp::LacpEventTable", "r" )
   lacpStatus = LazyMount.mount( entityManager, "lag/lacp/status",
                                 "Lacp::LacpStatus", "r" )
   lagConfigCli = ConfigMount.mount( entityManager, "lag/input/config/cli",
                                       "Lag::Input::Config", "w" )
   lagConfigDir = LazyMount.mount( entityManager, "lag/config",
                                   "Lag::Config", "r" )
   lagIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/eth/lag",
                                       "Interface::EthLagIntfConfigDir", "w" )
   lagIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                       "Interface::EthLagIntfStatusDir", "r" )
   recircHwConfig = LazyMount.mount( entityManager, "lag/recirc/hwconfig", 
                                     "Lag::Recirc::HwConfig", "r" )
   gv.swagHwCapabilities = LazyMount.mount( entityManager, "swag/hwCapabilities",
                                            "Swag::HwCapabilities", "r" )
