# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Lag interface type.  In particular, it provides
# the EthLagIntf class.
#-------------------------------------------------------------------------------
import Tac, CliCommand, CliParser, CliMatcher
from CliPlugin import EbraEthIntfCli
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
from CliPlugin import IntfProfileCli
from CliPlugin import L2ProtocolShowCli
from CliPlugin import SubIntfCli
from CliPlugin import VirtualIntfRule
from CliPlugin.IntfModel import InterfaceCountersRate
from CliPlugin.LagIntfModel import LagEthInterfaceStatus
from CliPlugin.LagIntfModel import LagMemberInformation
from CliPlugin.LagCli import  maybeDeleteEthPhyIntfLagConfig
from CliPlugin.LagCliLib import (
   isPortChannelIntfId,
   portChannel,
   isRecirc,
   updateLacpPortIdRangeForLag,
   deleteLacpPortIdRangeForLag,
   deleteLacpSystemIdForLag,
)
# pylint: disable-next=consider-using-from-import
import CliPlugin.LagCommonCliLib as LagCommonCliLib
import Ethernet
import LazyMount, Arnet, ConfigMount
import Cell
from IntfRangePlugin.LagIntf import LagAutoIntfType, portChannelRangeFn
import SharedMem
import Smash
from Intf.IntfRange import intfTypeFromName, subintfSupportedGuard
from TypeFuture import TacLazyType
from LagLib import speedUnitEnumFromStr
from functools import reduce # pylint: disable=redefined-builtin
from Toggles.LagToggleLib import toggleMaxLinksPreemptivePriorityEnabled

entMan = None

lagIntfConfigDir = None
lagIntfStatusDir = None
lagGroupDir = None
lagIntfCounterDir = None
lagConfigCli = None
lagCheckpointCounterDir = None
bridgingHwCapabilities = None
lacpCliOverrideDir = None
ethPhyIntfStatusDir = None

PortChannelNum = TacLazyType( "Lag::PortChannelNum" )
LacpFallbackTimeoutValue = TacLazyType( "Interface::LacpFallbackTimeoutValue" )
LagSpeedValue = TacLazyType( "Interface::LagSpeedValue" )
LagSpeedUnit = TacLazyType( "Interface::LagSpeedUnit" )
IntfState = TacLazyType( "Interface::IntfEnabledState" )

def mixedSpeedSupportedGuard( mode, token ):
   if bridgingHwCapabilities.mixedSpeedLagSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# A subclass of the EthIntf class for Lag interfaces.
#-------------------------------------------------------------------------------
class EthLagIntf( EthIntfCli.EthIntf ):

   #----------------------------------------------------------------------------
   # Creates a new EthLagIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      EthIntfCli.EthIntf.__init__( self, name, mode )
      self.lagId = self.getLagId()
      self.ethIntfConfigDir = lagIntfConfigDir
      self.ethIntfStatusDir = lagIntfStatusDir

   def getLagId( self ):
      return portChannel( self.name )

   #----------------------------------------------------------------------------
   # Returns the shortname of the interface.
   #----------------------------------------------------------------------------
   def createShortname( self ):
      return 'Po%d' % self.lagId # pylint: disable=consider-using-f-string

   #----------------------------------------------------------------------------
   # Creates the Interface::EthLagIntfConfig object for this interface if it does
   # not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      ethLagIntfConfig = self.ethIntfConfigDir.intfConfig.newMember( self.name )
      if ethLagIntfConfig.adminEnabledState() == IntfState.unknownEnabledState:
         ethLagIntfConfig.adminEnabled = True

      updateLacpPortIdRangeForLag( self.name )

   #----------------------------------------------------------------------------
   # Deletes the Interface::EthLagIntfConfig object for this interface
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      lag = self.ethIntfConfigDir.intfConfig.get( self.name )
      lagId = None
      if lag:
         lagId = lag.intfId
         del self.ethIntfConfigDir.intfConfig[ self.name ]
         deleteLacpPortIdRangeForLag( self.name )
         deleteLacpSystemIdForLag( self.name )
      # Make sure no phy intfs point to this port-channel (or any other
      # obsolete channel)
      self.cleanEthPhyIntfLagConfigs( lagId )

   def cleanEthPhyIntfLagConfigs( self, lagId ):
      for intf in lagConfigCli.phyIntf.values():
         # snapshot lag, so another process clearing intf.lag doesn't cause
         # us to try to take an attribute of None
         lag = intf.lag
         # Cleanup lag ptr and lacpMode if epilc.lag.intfId matches the
         # requested lagId in the argument
         if lag and lag.intfId == lagId:
            intf.lag = None
            intf.mode = 'lacpModeOff'
         if not intf.lag:
            maybeDeleteEthPhyIntfLagConfig( intf, None )

   def setDefault( self ):
      elic = self.ethIntfConfigDir.intfConfig[ self.name ]
      elic.minLinks = 0
      elic.maxLinks = None
      elic.priorityEnforced = False
      elic.fallback = 'fallbackNone'
      elic.fallbackTimeout = elic.fallbackTimeoutDefault
      elic.lacpLoopEnable = False
      if lacpCliOverrideDir.systemId.get( self.name ):
         deleteLacpSystemIdForLag( self.name )
      EthIntfCli.EthIntf.setDefault( self )
      self.setMacAddr( None )

   #----------------------------------------------------------------------------
   # EthIntf is a subclass of PhysicalIntf, but we are actually a virtual
   # interface and we're allowed to be deleted.
   #----------------------------------------------------------------------------
   def noInterface( self ):
      self.destroy()

   #----------------------------------------------------------------------------
   # EthIntfCli.EthIntf.getIntfCounterDir assumes the counter dir leaf name
   # is the same as the status dir leaf name, which is not true for the status
   # dir at 'lag/input/interface/mlag'.
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      assert self.status()
      parent = self.status().parent
      if not parent:
         return None
      return lagCheckpointCounterDir

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def addrStr( self ):
      if self.status().forwardingModel == 'intfForwardingModelRouted':
         addr = self.routedAddr()
      else:
         addr = self.addr()
      return "address is %s" % addr # pylint: disable=consider-using-f-string

   def counter( self ):
      shmemEm = SharedMem.entityManager( sysdbEm=entMan )
      smi = Smash.mountInfo( 'reader' )
      smashCountersDir = shmemEm.doMount( 'interface/counter/lag/current',
                                          'Smash::Interface::AllIntfCounterDir',
                                          smi )
      return Tac.newInstance( 'Interface::SmashIntfCounter', 'current',
                              self.name,
                              smashCountersDir )

   def hardware( self ):
      return "portChannel"

   def bandwidth( self ):
      return self.status().speed * 1000

   def getIntfDuplexStr( self ):
      return "duplexFull"

   def xcvrTypeStr( self ):
      return "N/A"

   def autonegActive( self ):
      return False

   def flowcontrolRxPause( self ):
      return None

   def flowcontrolTxPause( self ):
      return None

   def getIntfStatusModel( self ):
      return LagEthInterfaceStatus( name=self.status().intfId )

   def showLoopbackMode( self, intfStatusModel ):
      pass

   def showLinkTypeSpecific( self, intfStatusModel ):
      for m in Arnet.sortIntf( self.status().member ):
         member = EthIntfCli.EthPhyIntf( m, self.mode_ )
         if member and member.lookupPhysical():
            config = member.config() is not None
            autoNeg = member.autonegActive() if config else False

            lagMemberInfo = LagMemberInformation( duplex=member.status().duplex,
                                                bandwidth=member.bandwidth() * 1000,
                                                _config=config,
                                                _autoNegotiateActive=autoNeg )
            intfStatusModel.memberInterfaces[ m ] =  lagMemberInfo

      # output fallback mode config and status
      elic = self.ethIntfConfigDir.intfConfig.get( self.name )
      elis = self.ethIntfStatusDir.intfStatus.get( self.name )
      fallbackEnabledStr = "na"
      if elis:
         fallbackEnabledStr = elis.debugFallbackSmState
         fallbackEnabledStr = fallbackEnabledStr[ len( "fallbackState" ): ].lower()
      if fallbackEnabledStr == "enabled" and elis.fallbackEnabled == 'fallbackNone':
         fallbackEnabledStr = "enabledLocally"
      if( elic ): # pylint: disable=superfluous-parens
         intfStatusModel.fallbackEnabled = elic.fallback != 'fallbackNone'
         intfStatusModel.fallbackEnabledType = elic.fallback
         if(( elic.fallback != 'fallbackNone' ) or
            ( elic.fallbackTimeout != elic.fallbackTimeoutDefault )):
            intfStatusModel.fallbackMode = fallbackEnabledStr
            intfStatusModel.fallbackTimeout = elic.fallbackTimeout
         if fallbackEnabledStr == "enabled" and \
               elic.minLinks != elic.minLinksDefault:
            msg = "min-links config will not be honored when fallback is active"
            self.mode_.addWarning( msg )
      else:
         intfStatusModel.fallbackEnabled = False

      intfStatusModel.bandwidth = self.bandwidth() * 1000

   #----------------------------------------------------------------------------
   # Is this port eligible for "switchport" commands?
   #----------------------------------------------------------------------------
   def switchportEligible( self ):
      return True

   #----------------------------------------------------------------------------
   # The rule for matching Lag interface names.  When this pattern matches, it
   # returns an instance of the EthLagIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = VirtualIntfRule.VirtualIntfMatcher( 'Port-Channel', PortChannelNum.min,
                                       PortChannelNum.max,
                                       helpdesc="Link Aggregation Group (LAG)",
                                       rangeFunc=portChannelRangeFn,
                                       value=lambda mode, intf:
                                       EthLagIntf( intf, mode ) )

   #----------------------------------------------------------------------------
   # Returns an unsorted list of LagEthIntf objects representing all the existing
   # Interface::LagEthIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in lagIntfStatusDir.intfStatus:
         if not isPortChannelIntfId( name ) or isRecirc( name ):
            continue
         intf = EthLagIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   def setL2Mtu( self, value ):
      cfg = self.config()
      cfg.l2Mtu = value

   def updateInterfaceCountersRateModel( self ):
      if self.counter() is None:
         return None

      elis = self.ethIntfStatusDir.intfStatus.get( self.name )
      if elis and len( elis.member ) == 1 and \
            list( elis.member )[ 0 ] in ethPhyIntfStatusDir.intfStatus:
         ethPhyIntfType, _ = intfTypeFromName( list( elis.member )[ 0 ] )
         intfObj = ethPhyIntfType.getCliIntf( self.mode_, list( elis.member )[ 0 ] )
         icrm = intfObj.updateInterfaceCountersRateModel()
         if icrm:
            intfCountersRateModel = InterfaceCountersRate( _name=self.name )
            intfCountersRateModel.description = self.description()
            intfCountersRateModel.interval = icrm.interval
            intfCountersRateModel.inBpsRate = icrm.inBpsRate
            intfCountersRateModel.inPpsRate = icrm.inPpsRate
            intfCountersRateModel.inPktsRate = icrm.inPktsRate
            intfCountersRateModel.outBpsRate = icrm.outBpsRate
            intfCountersRateModel.outPpsRate = icrm.outPpsRate
            intfCountersRateModel.outPktsRate = icrm.outPktsRate
            return intfCountersRateModel

      return super().updateInterfaceCountersRateModel()



#-------------------------------------------------------------------------------
# Register the EthLagIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( EthLagIntf, LagAutoIntfType,
                                  withIpSupport=True )
EbraEthIntfCli.switchPortMatcher |= EthLagIntf.matcher

#-------------------------------------------------------------------------------
# A subclass of the SubIntf class for Lag Sub-interfaces.
#-------------------------------------------------------------------------------
class LagSubIntf( SubIntfCli.SubIntf ):
   def vlanSubIntfSupported( self ):
      # Enable "vlan ..." CLI commands in the port-channel sub-interface
      # if-mode.
      return True

   @staticmethod
   def getAllPhysical( mode ):
      # parent class returns *all* sub-interfaces, nothing to do here.
      return []

def lagSubIntfRuleValueFunc( mode, intf ):
   if SubIntfCli.theSubintfSupportedGuard( mode, intf ) is not None:
      raise CliParser.InvalidInputError()
   return LagSubIntf( intf, mode )

SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
subLowerBound = SubIntfRangeDetails.subIntfExtendedIdMin
subUpperBound = SubIntfRangeDetails.subIntfExtendedIdMax

LagAutoIntfType.cliSubIntfClazzIs( LagSubIntf )
subMatcher = VirtualIntfRule.VirtualIntfMatcher( 'Port-Channel',
                                           PortChannelNum.min, PortChannelNum.max,
                                           helpdesc="Port-Channel Sub Interface",
                                           subIntf=True,
                                           subIntfGuard=subintfSupportedGuard,
                                           rangeFunc=portChannelRangeFn,
                                           value=lagSubIntfRuleValueFunc,
                                           subLbound=subLowerBound,
                                           subUbound=subUpperBound )
IntfCli.Intf.addSubIntf( subMatcher, LagSubIntf, withIpSupport=True )

def portChannelSubintfParentCfg( parentName ):
   if not parentName.startswith( 'Po' ):
      return None
   return lagIntfConfigDir.intfConfig.get( parentName )

SubIntfCli.subintfParentCfgHook.append( portChannelSubintfParentCfg )

#-------------------------------------------------------------------------------
# Register the EthLagIntf class as able to participate in interface ranges.
#-------------------------------------------------------------------------------
EbraEthIntfCli.registerSwitchPortIntfType( LagAutoIntfType )
L2ProtocolShowCli.l2ProtocolIntfTypes.append( LagAutoIntfType )

#-------------------------------------------------------------------------------
# Enable Lag-specific commands to be added to "config-if" mode.
#-------------------------------------------------------------------------------
class LagIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, EthLagIntf ) and not mode.intf.isSubIntf()

LagIntfConfigModelet.addCommandClass( IntfProfileCli.ApplyIntfProfile )
LagIntfConfigModelet.addCommandClass( IntfProfileCli.OverrideIntfProfile )

IntfCli.IntfConfigMode.addModelet( LagIntfConfigModelet )

#-------------------------------------------------------------------------------
# Register the EthLagIntf class as a type that supports L2 mtu.
#-------------------------------------------------------------------------------
EthIntfCli.registerL2MtuValidIntfType( EthLagIntf )

#-------------------------------------------------------------------------------
# The "[no|default] lacp max-bundle N" command, in "config-if" mode.
# This command is only valid on a Port-Channel interface.
# This is not yet supported.
#-------------------------------------------------------------------------------
matcherMaxBundle = CliMatcher.KeywordMatcher(
   'max-bundle',
   helpdesc='Set the maximum number of concurrent active links' )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel min-links N" command, in "config-if" mode.
# This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
description = 'Minimum no of ports required up before bringing up a port-channel'

def _rangeFn( mode, context ):
   # This code is called when the startup-config is being parsed,
   # which happens before the FruAgent has discovered all the hardware
   # and populated Sysdb.
   #
   # Therefore, we don't know the upper limit of this value yet, so we
   # need to accept *any* value, which we accomplish by returning a
   # ridiculously upper-end-of-the-range value in that case. I've
   # chosen 500, rather than some power-of-2, to show that this is a
   # meaningless, arbitrary value, rather than some actual limitation
   # in our SW or HW. Oh, and I guess this comment does that, too. :)
   if lagGroupDir is None:
      return ( 0, 500 )
   hwMax = reduce( max,
                   [ lg.maxPortsPerLagIntf for lg in
                     lagGroupDir.lagGroup.values() ], 0 )
   superMax = hwMax if hwMax else 500
   return ( 0, superMax )

def setMinLinks( mode, args ):
   minLinksVal= args[ 'MINLINKS' ]
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.minLinks = minLinksVal

def defaultMinLinks( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.minLinks = elic.minLinksDefault

matcherPortChannel = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'port-channel',
         helpdesc='Set port-channel parameters' ),
      guard=LagCommonCliLib.lagSupportedGuard )

class LagMinLinksCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel min-links MINLINKS'
   noOrDefaultSyntax = 'port-channel min-links ...'
   data = {
      'port-channel' : matcherPortChannel,
      'min-links' : description,
      'MINLINKS' : CliMatcher.DynamicIntegerMatcher(
         rangeFn=_rangeFn, helpname=None, helpdesc=description )
   }
   handler = setMinLinks
   noOrDefaultHandler = defaultMinLinks

LagIntfConfigModelet.addCommandClass( LagMinLinksCmd )

# -------------------------------------------------------------------------------
# The "[no|default] port-channel max-links N" command, in "config-if" mode.
# This command is only valid on a Port-Channel interface.
# -------------------------------------------------------------------------------
description = 'Maximum no of ports allowed to become active in the port-channel'

def setMaxLinks( mode, args ):
   maxLinksVal = args[ 'MAXLINKS' ]
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.maxLinks = maxLinksVal

def defaultMaxLinks( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.maxLinks = None

class LagMaxLinksCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel max-links MAXLINKS'
   noOrDefaultSyntax = 'port-channel max-links ...'
   data = {
      'port-channel': matcherPortChannel,
      'max-links': description,
      'MAXLINKS': CliMatcher.IntegerMatcher(
         # We chose an upper limit of 500, a number
         # big enough to cover any MPPL value, on any platform.
         # This is an arbitrary value, with no other meaning.
         1, 500,
         helpdesc='Maximum number of active ports in the port-channel' )
   }
   handler = setMaxLinks
   noOrDefaultHandler = defaultMaxLinks

if toggleMaxLinksPreemptivePriorityEnabled():
   LagIntfConfigModelet.addCommandClass( LagMaxLinksCmd )

# -------------------------------------------------------------------------------
# The "[no|default] port-channel member priority preemptive" command,
# in "config-if" mode.
# This command is only valid on a Port-Channel interface.
# -------------------------------------------------------------------------------
description = 'Enforce member port priority when selecting members to become active'

def setPreemptivePriority( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.priorityEnforced = True

def defaultPreemptivePriority( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.priorityEnforced = False

class PreemptivePriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel member priority preemptive'
   noOrDefaultSyntax = 'port-channel member priority'
   data = {
      'port-channel': matcherPortChannel,
      'member': 'Enforce member port priority for same speed port-channels',
      'priority': 'Enforce member port priority for same speed port-channels',
      'preemptive': 'Allow higher priority ports to take precedence'
   }
   handler = setPreemptivePriority
   noOrDefaultHandler = defaultPreemptivePriority

if toggleMaxLinksPreemptivePriorityEnabled():
   LagIntfConfigModelet.addCommandClass( PreemptivePriorityCmd )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel speed mixed" command, in "config-if" mode.
# This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
speedDescription = 'Configure port-channel speed related parameters'
description = 'Allow port-channel members of different speeds to become active'

matcherSpeed = CliMatcher.KeywordMatcher( 'speed', helpdesc=speedDescription )

def setMixedSpeed( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.mixedSpeed = True

def defaultMixedSpeed( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.mixedSpeed = False

class LagMixedSpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel speed mixed'
   noOrDefaultSyntax = syntax
   data = {
      'port-channel' : matcherPortChannel,
      'speed' : matcherSpeed,
      'mixed' : CliCommand.Node(
                   matcher=CliMatcher.KeywordMatcher(
                   'mixed',
                   helpdesc=description ),
                   guard=mixedSpeedSupportedGuard )
   }
   handler = setMixedSpeed
   noOrDefaultHandler = defaultMixedSpeed

LagIntfConfigModelet.addCommandClass( LagMixedSpeedCmd )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel speed minimum <value> {mbps|gpbs}"
# command, in "config-if" mode.
# This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def minSpeedUnitAdapter( mode, args, argsList ):
   args[ 'SPEED_UNIT' ] = speedUnitEnumFromStr( args.get( 'MINSPEED_UNIT', 'gbps' ) )

def setMinSpeed( mode, args ):
   speedValue = args[ 'MINSPEED_VAL' ]
   speedUnit = args[ 'SPEED_UNIT' ]

   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.minSpeed = Tac.Value( 'Interface::LagSpeed', speedValue, speedUnit )

def defaultMinSpeed( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.minSpeed = Tac.Value( 'Interface::LagSpeed' )

class LagMinSpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel speed minimum MINSPEED_VAL [MINSPEED_UNIT]'
   noOrDefaultSyntax = 'port-channel speed minimum ...'
   data = {
      'port-channel' : matcherPortChannel,
      'speed': matcherSpeed,
      'minimum': 'Minimum',
      'MINSPEED_VAL' : CliMatcher.IntegerMatcher(
         LagSpeedValue.min,
         LagSpeedValue.max,
         helpdesc='Minimum speed value' ),
      'MINSPEED_UNIT': CliMatcher.EnumMatcher( {
         'gbps': 'Speed in Gbps (default unit)',
         'mbps': 'Speed in Mbps' } ),
   }

   adapter = minSpeedUnitAdapter
   handler = setMinSpeed
   noOrDefaultHandler = defaultMinSpeed

LagIntfConfigModelet.addCommandClass( LagMinSpeedCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac-address router MAC_ADDR
#--------------------------------------------------------------------------------
LagIntfConfigModelet.addCommandClass( IntfCli.MacAddressRouterAddrCmd )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel lacp fallback [static|individual]" command,
# in "config-if" mode. This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def setFallback( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   if 'individual' in args:
      elic.fallback = 'fallbackIndividual'
   else:
      # Not specifying individual or static will default to static fallback
      elic.fallback = 'fallbackStatic'

def noFallback( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.fallback = 'fallbackNone'

matcherLacp = CliMatcher.KeywordMatcher( 'lacp', helpdesc='Set LACP parameter' )
matcherFallback = CliMatcher.KeywordMatcher(
   'fallback',
   helpdesc='Enable fallback to static LAG mode when LACP active mode times out' )

class LacpFallbackCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel lacp fallback [ individual | static ]'
   noOrDefaultSyntax = syntax
   data = {
      'port-channel' : matcherPortChannel,
      'lacp' : matcherLacp,
      'fallback' : matcherFallback,
      'individual' : 'Fallback to individual ports',
      'static' : 'Fallback to static LAG mode'
   }
   handler = setFallback
   noOrDefaultHandler = noFallback

LagIntfConfigModelet.addCommandClass( LacpFallbackCmd )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel lacp fallback timeout <N>" command,
# in "config-if" mode. This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def setFallbackTimeout( mode, args ):
   timeoutSeconds = args[ 'TIMEOUT' ]
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.fallbackTimeout = timeoutSeconds
   if not elic.fallback:
      mode.addWarning(
         'LACP fallback not configured yet. Configuring timeout does not ' +
         'enable LACP fallback automatically.' )

def defaultFallbackTimeout( mode, args ):
   elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
   elic.fallbackTimeout = elic.fallbackTimeoutDefault

class LacpFallbackTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel lacp fallback timeout TIMEOUT'
   noOrDefaultSyntax = 'port-channel lacp fallback timeout ...'
   data = {
      'port-channel' : matcherPortChannel,
      'lacp' : matcherLacp,
      'fallback' : matcherFallback,
      'timeout' : 'Set LACP fallback timeout (not enabling fallback automatically)',
      'TIMEOUT' : CliMatcher.IntegerMatcher(
         LacpFallbackTimeoutValue.min,
         LacpFallbackTimeoutValue.max,
         helpdesc='LACP fallback timeout in seconds (Default: 90)' )
   }
   handler = setFallbackTimeout
   noOrDefaultHandler = defaultFallbackTimeout

LagIntfConfigModelet.addCommandClass( LacpFallbackTimeoutCmd )

#-------------------------------------------------------------------------------
# The "[no|default] port-channel lacp loopback" command,
# in "config-if" mode. This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def setLacpLoopback( mode, args ):
   lagIntfConfigDir.intfConfig[ mode.intf.name ].lacpLoopEnable = True

def noLacpLoopback( mode, args ):
   lagIntfConfigDir.intfConfig[ mode.intf.name ].lacpLoopEnable = False

class LacpLoopbackCmd( CliCommand.CliCommandClass ):
   syntax = 'port-channel lacp loopback'
   noOrDefaultSyntax = 'port-channel lacp loopback'
   data = {
      'port-channel' : matcherPortChannel,
      'lacp' : matcherLacp,
      'loopback' : 'Enable LACP loopback negotiation'
   }
   handler = setLacpLoopback
   noOrDefaultHandler = noLacpLoopback

LagIntfConfigModelet.addCommandClass( LacpLoopbackCmd )

#-------------------------------------------------------------------------------
# The "[no|default] lacp system-id aaaa.bbbb.cccc" command,
# in "config-if" mode. This command is only valid on a Port-Channel interface.
#-------------------------------------------------------------------------------
def setLacpSystemId( mode, args ):
   systemId = args[ 'ADDR' ]
   intfId = mode.intf.name
   ethAddr = Tac.Value( 'Arnet::EthAddr', stringValue=systemId )
   lacpCliOverrideDir.systemId[ intfId ] = ethAddr

def noLacpSystemId( mode, args ):
   intfId = mode.intf.name
   if lacpCliOverrideDir.systemId.get( intfId, None ):
      del lacpCliOverrideDir.systemId[ intfId ]

class LacpSystemIdCmd( CliCommand.CliCommandClass ):
   syntax = 'lacp system-id ADDR'
   noOrDefaultSyntax = 'lacp system-id ...'
   data = {
      'lacp' : matcherLacp,
      'system-id' : 'Configure LACP system ID',
      'ADDR' : CliMatcher.PatternMatcher( Ethernet.macAddrPattern,
         helpname='H.H.H', helpdesc='LACP system ID' )
   }
   handler = setLacpSystemId
   noOrDefaultHandler = noLacpSystemId

LagIntfConfigModelet.addCommandClass( LacpSystemIdCmd )

#-------------------------------------------------------------------------------
# [ no | default ] l2 mtu MTU
# [ no | default ] l2 mru MRU
#-------------------------------------------------------------------------------
LagIntfConfigModelet.addCommandClass( EthIntfCli.L2MtuCmd )
LagIntfConfigModelet.addCommandClass( EthIntfCli.L2MruCmd )

# TODO Changes here needs to be reflected in the CliSavePlugin and ArosTest Plugin
# Add fallback priority to the show command

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global lagIntfConfigDir, lagIntfStatusDir, lagGroupDir, lagCheckpointCounterDir
   global entMan, lagConfigCli, bridgingHwCapabilities
   global lacpCliOverrideDir
   global ethPhyIntfStatusDir

   entMan = entityManager

   lagIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/eth/lag",
                                         "Interface::EthLagIntfConfigDir", "w" )
   lagConfigCli = ConfigMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "w" )
   lacpCliOverrideDir = ConfigMount.mount(entityManager,
                                 "lag/input/lacpoverride/cli",
                                 "Lag::Input::LacpOverrideDir", "w" )
   lagIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                       "Interface::EthLagIntfStatusDir", "r" )
   lagGroupDir = LazyMount.mount( entityManager, "hardware/lag/internalConfig",
                                  "Hardware::Lag::InternalConfig", "r" )
   lagCheckpointCounterDir = entityManager.mount( Cell.path( "lag/input/counters" ),
                                    "Interface::AllIntfCounterDir", "w" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   ethPhyIntfStatusDir = LazyMount.mount( entityManager,
                                          'interface/status/eth/phy/all',
                                          'Interface::AllEthPhyIntfStatusDir', 'r' )
