#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in
# pylint: disable=consider-using-from-import
# pylint: disable=consider-using-f-string

import BasicCli
import CliCommand
import CliMatcher
import LazyMount
import ShowCommand
import Tac
import Arnet
import CliParser
import CommonGuards
import CliPlugin.LagCli as LagCli
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.RecircCli as RecircCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.VlanCli as VlanCli 
import CliPlugin.LagModel as LagModel
import CliPlugin.LagCliLib as LagCliLib
import CliPlugin.LagCommonCliLib as LagCommonCliLib
from CliCommand import guardedKeyword
from CliMatcher import IntegerMatcher, KeywordMatcher
from CliPlugin.EthIntfCli import ethPhyIntfCounterDir
from CliPlugin.LagModel import ( 
      LacpSysId, 
      PortChannelLimits, 
      LacpPort, 
      LacpNeighbors, 
      LacpAggregates, 
      LacpMisconfig,
      LacpInternal, 
      LacpInternalEvents,
      PortChannels,
      PortChannelsTraffic, 
      RecircChannelLimits, 
      PortChannelsSummary, 
      LacpCounters, 
      FallbackConfigStatus,
)
from CliToken.PortChannel import portChannelMatcherForShow
from MultiRangeRule import MultiRangeMatcher
from LacpLib import dot43sysid
from functools import reduce # pylint: disable=redefined-builtin
from Toggles.LagToggleLib import toggleMaxLinksPreemptivePriorityEnabled
from Toggles.SwagCoreToggleLib import toggleSwagPhase1Enabled

lagMemberIntfMatcher = IntfCli.Intf.matcherWOSubIntf
bridgingHwCapabilities = None

PortChannelNum = Tac.Type( "Lag::PortChannelNum" )
def portChannelRange():
   if bridgingHwCapabilities and bridgingHwCapabilities.extendedLagIdSupported:
      return ( PortChannelNum.min, PortChannelNum.extendedMax )
   return ( PortChannelNum.min, PortChannelNum.max )

def recircChannelRange():
   return ( PortChannelNum.min, PortChannelNum.max )

srcMacMatcher = KeywordMatcher( 'src-mac-address',
      helpdesc='MAC address of the source' )
srcIpMatcher = KeywordMatcher( 'src-ip-address',
      helpdesc='Source IPv4 address' )
destinationMatcher = KeywordMatcher( 'destination',
      helpdesc='Destination interface' )
dstL4PortMatcher = KeywordMatcher( 'dst-l4-port',
      helpdesc='L4 destination port' )
dstMacMatcher = KeywordMatcher( 'dst-mac-address',
      helpdesc='MAC address of the destination' )
dstIpMatcher = KeywordMatcher( 'dst-ip-address',
      helpdesc='Destination IPv4 address' )
ipProtoMatcher = KeywordMatcher( 'ip-protocol', helpdesc='IP protocol' )
ipProtoNumberMatcher = IntegerMatcher( 1, 255, helpdesc='IP protocol No.' )
loadBalanceMatcherForShow = CliCommand.guardedKeyword( 'load-balance',
      helpdesc='Show load balancing information',
      guard=LagCliLib.hashOutputIntfGuard )
l4PortNumberMatcher = IntegerMatcher( 1, 65535, helpdesc='Port No.' )
srcL4PortMatcher = KeywordMatcher( 'src-l4-port',
      helpdesc='L4 source port' )
vlanMatcher = KeywordMatcher( 'vlan-id',
      helpdesc='Identifier for a virtual LAN' )
vlanIdMatcher = IntegerMatcher( 1, 4094,
      helpdesc='VLAN identifier in the frame' )
matcherActivePorts = CliMatcher.KeywordMatcher( 'active-ports', 
      helpdesc='Display info on ports that are active members of the LAG' )
matcherAggregates = CliMatcher.KeywordMatcher( 'aggregates', 
      helpdesc='Display info about LACP aggregates' )
matcherAllPortsUnselected = CliMatcher.KeywordMatcher( 'all-ports', 
      helpdesc='Display LACP info on all ports in LAG, even those not selected '
               ' by LACP to be part of aggregate' )
matcherAllPortsInactive = CliMatcher.KeywordMatcher( 'all-ports', 
      helpdesc='Display info on all ports configured for LAG, '
               'even those not currently active' )
matcherBriefRecirc = CliMatcher.KeywordMatcher( 'brief', 
      helpdesc='Display all recirc-channel status briefly' )
matcherBriefPortChannel = CliMatcher.KeywordMatcher( 'brief', 
      helpdesc='Display all port-channel status briefly' )
matcherBriefFabricChannel = CliMatcher.KeywordMatcher( 'brief',
      helpdesc='Display all fabric-channel status briefly' )
matcherBriefLacpStatus = CliMatcher.KeywordMatcher( 'brief', 
      helpdesc='Display relevant Link Aggregation Control Protocol (LACP) '
               'status briefly' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters', 
      helpdesc='Display LACP counters' )
matcherDetailedLacpStatus = CliMatcher.KeywordMatcher( 'detailed', 
      helpdesc='Display detailed Link Aggregation Control Protocol status' )
matcherDetailedPortChannel = CliMatcher.KeywordMatcher( 'detailed', 
      helpdesc='Display detailed port-channel status' )
matcherDetailedRecirc = CliMatcher.KeywordMatcher( 'detailed', 
      helpdesc='Display detailed recirc-channel status' )
matcherDetailedFabricChannel = CliMatcher.KeywordMatcher( 'detailed',
      helpdesc='Display detailed fabric-channel status' )
matcherEtherchannel = CliMatcher.KeywordMatcher( 'etherchannel', 
      helpdesc='Synonym for show port-channel parameters' )
matcherEthernet = CliMatcher.KeywordMatcher( 'Ethernet', 
      helpdesc='Ethernet interface' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface', 
      helpdesc='Display per-interface LACP info' )
matcherInternal = CliMatcher.KeywordMatcher( 'internal', 
      helpdesc='Display information internal to the '
               'Link Aggregation Control Protocol' )
matcherLacp = CliMatcher.KeywordMatcher( 'lacp', 
      helpdesc='Link Aggregation Control Protocol (LACP) status' )
matcherLoopback = CliMatcher.KeywordMatcher( 'Loopback', 
      helpdesc='Hardware interface used for looping packets' )
matcherManagement = CliMatcher.KeywordMatcher( 'Management', 
      helpdesc='Management interface' )
matcherMatchInterface = CliMatcher.KeywordMatcher( 'match-interface', 
      helpdesc='Filter results by intf' )
matcherPeer = CliMatcher.KeywordMatcher( 'peer', 
      helpdesc='Display identity and info about LACP neighbor(s)' )
matcherPortChannel = CliMatcher.KeywordMatcher( 'Port-Channel', 
      helpdesc='Link Aggregation Group (LAG)' )
matcherRecircChannelIntf = CliMatcher.KeywordMatcher( 'Recirc-Channel', 
      helpdesc='Recirc-Channel Interface' )
matcherRecircChannelParams = CliMatcher.KeywordMatcher( 'recirc-channel', 
      helpdesc='Show recirc-channel parameters' )
matcherFabricChannelParams = CliMatcher.KeywordMatcher( 'fabric-channel',
      helpdesc='Show fabric-channel parameters' )
matcherUnconnectedethernet = CliMatcher.KeywordMatcher( 'UnconnectedEthernet', 
      helpdesc='UnconnectedEthernet interface' )
matcherVlan = CliMatcher.KeywordMatcher( 'Vlan', 
      helpdesc='Logical interface into a VLAN' )
matcherChannelIds = MultiRangeMatcher(
      portChannelRange, False, 'Channel Group ID(s)' )
matcherRecircChannelIds = MultiRangeMatcher( 
      recircChannelRange, False, 'Recirculation Channel Group ID(s)' )
nodeEtherchannel = CliCommand.Node( matcher=matcherEtherchannel,
      guard=LagCommonCliLib.lagSupportedGuard,
      deprecatedByCmd='show port-channel' )
nodeLacp = CliCommand.Node( matcher=matcherLacp, 
      guard=LagCommonCliLib.lagSupportedGuard )
nodeNeighborDeprecated = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'neighbor', 
      helpdesc='Display identity and info about LACP neighbor(s)' ),
      deprecatedByCmd='show lacp peer' )
nodePortChannel = CliCommand.Node( matcher=portChannelMatcherForShow, 
      guard=LagCommonCliLib.lagSupportedGuard )
nodeRecircChannelParams = CliCommand.Node( matcher=matcherRecircChannelParams, 
      guard=LagCliLib.recircGuard )
nodeFabricChannelParams = CliCommand.Node( matcher=matcherFabricChannelParams,
      guard=LagCliLib.fabricChannelGuard )
nodeRecircChannelIntf = CliCommand.Node( matcher=matcherRecircChannelIntf, 
      guard=LagCliLib.recircGuard )
nodeUnconnectedethernet = CliCommand.Node( matcher=matcherUnconnectedethernet, 
      guard=EthIntfCli.ueGuard )
nodeVlan = CliCommand.Node( matcher=matcherVlan, 
      guard=VlanCli.bridgingSupportedGuard )
portChannelMatcher = guardedKeyword( 'port-channel',
      helpdesc='Show output member interface selected for a Port-Channel',
      guard=LagCliLib.lagOutputIntfGuard )

def getChannelGroupIdList( intfs ):
   if intfs is None:
      return None
   return list( map( LagCli.ChannelGroup, intfs.values() ) )

def getRecircGroupIdList( intfs ):
   if intfs is None:
      return None
   return list( map( LagCli.RecircChannelGroup, intfs.values() ) )

def doShowPortChannel( mode, args ):
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   brief = 'detailed' not in args
   allPorts = 'active-ports' not in args
   channels = LagCliLib.portchannelList( mode, channelGroupIdList, lacpOnly=False,
                                         inactiveWarn=False )
   return doShowChannelGroups( mode, channels, brief, allPorts )

class EtherchannelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show etherchannel [ INTFS ] '
              '[ ( ( brief | detailed ) [ active-ports | all-ports ] ) | ' 
              '  ( ( active-ports | all-ports ) [ brief | detailed ] ) ] ' )
   data = {
      'etherchannel': nodeEtherchannel,
      'INTFS': matcherChannelIds, 
      'brief': matcherBriefPortChannel,
      'detailed': matcherDetailedPortChannel,
      'active-ports': matcherActivePorts,
      'all-ports': matcherAllPortsInactive,
   }
   handler = doShowPortChannel
   cliModel = PortChannels

BasicCli.addShowCommandClass( EtherchannelCmd )

def genMarkers( portChannel ):
   # Function to generate annotation markers for a port channel
   markers = [ marker for af, marker in LagCliLib.afmAssoc if af( portChannel ) ]
   return markers

def usedDetailMarkerMessages( usedMarkers, markerMsgs ):
   mmList = []
   for marker in LagCliLib.annotateMarkers:
      if marker in usedMarkers:
         mmList.append( LagModel.LacpMarkerMessage(
                              marker=marker,
                              msg=markerMsgs[ marker ] ) )
   if mmList:
      return LagModel.LacpMarkerMessages( markerMessages=mmList )
   else:
      return None

def markersToModelSet( markerStrings ):
   if not markerStrings:
      return None
   else:
      return LagModel.LacpAnnotationMarkerSet( markers=markerStrings )

#
# Request the agent to update counters into sysdb
#
def updatePortCounters( mode ):
   # On standby supe, return counter values in Sysdb without waiting for
   # countersUpdated().
   if CommonGuards.standbyGuard( mode, "" ) == CliParser.guardNotThisSupervisor:
      LagCli.t0( "updatePortCounters called on standby supervisor" )
      return

   config = LagCliLib.lacpCliConfig
   status = LagCliLib.lacpStatus
   config.counterUpdateRequestTime = Tac.now() # monotonic time

   if mode.entityManager.isLocalEm():
      return

   def countersUpdated():
      return status.counterUpdateTime >= config.counterUpdateRequestTime
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.1,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      print ( "Warning: displaying stale counters" )

def lacpAggregateInfo( agg, lagStatus, allPorts, brief ):
   ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts,
     incompatPortsDetail ) = '', [], [], [], [], {}
   if agg:
      ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts ) = \
                              LagCli.describeAgg( agg, lagStatus, allPorts, brief )
   if ( lagStatus and lagStatus.otherIntf ):
      incompatPortsDetail = LagCli.incompatiblePortsDetail( LagCliLib.lacpStatus, 
                                                            lagStatus )
   return ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts,
            incompatPortsDetail )

def portChannelAggregate( portChannel, markers, lacpStatus, allPorts, brief ):
   lagStatus = lacpStatus.lagStatus.get( str( portChannel ) )
   agg = lagStatus and lagStatus.agg

   if lagStatus and ( agg or lagStatus.otherIntf ):
      ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts,
        incompatPortsDetail ) = lacpAggregateInfo( agg, lagStatus,
                                                   allPorts, brief )
   else:
      ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts,
        incompatPortsDetail ) = '', [], [], [], [], {}

   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelAggregate( aggregateId = aggregateId,
                                bundledPorts = bundledPorts,
                                otherPorts = otherPorts,
                                standbyPorts = standbyPorts,
                                negotiationPorts = negotiationPorts,
                                incompatiblePortsDetail = incompatPortsDetail,
                                markers = lacpAnnotationMarkers )

def otherLacpAggregates( lacpStatus, allPorts, brief ):
   otherAggsDict = {}
   otherAggs = dict( lacpStatus.aggStatus )
   for lls in lacpStatus.lagStatus.values():
      if lls.agg and lls.agg.lagId in otherAggs:
         del otherAggs[ lls.agg.lagId ]

   if otherAggs:
      for agg in otherAggs.values():
         ( aggregateId, bundledPorts, otherPorts, standbyPorts, negotiationPorts, \
           incompatPortsDetail ) = lacpAggregateInfo( agg, 
                                                      None, # no lls for otherAggs
                                                      allPorts, brief )

         otherAggsDict[ aggregateId ] = LagModel.PortChannelAggregate(
                                      aggregateId = aggregateId,
                                      bundledPorts = bundledPorts,
                                      otherPorts = otherPorts,
                                      standbyPorts = standbyPorts,
                                      negotiationPorts = negotiationPorts,
                                      incompatiblePortsDetail = incompatPortsDetail,
                                      markers = None )
   return otherAggsDict

def doShowLacpAggregates( mode, args ):
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   allPorts = 'all-ports' in args
   brief = 'detailed' not in args
   updatePortCounters( mode )
   brief = True if brief is None else brief
   channels = LagCliLib.portchannelList( mode, channelGroupIdList )

   usedMarkers = set()
   usedMarkerMessages = None
   portChannelDict = {}
   otherAggs = {}

   for channel in channels:
      markers = genMarkers( channel )
      usedMarkers = usedMarkers.union( set( markers ) )
      portChannelDict[ channel ] = portChannelAggregate( channel, markers,
                                                         LagCliLib.lacpStatus, 
                                                         allPorts, brief )

   if allPorts:
      otherAggs = otherLacpAggregates( LagCliLib.lacpStatus, allPorts, brief )

   if usedMarkers:
      usedMarkerMessages = usedDetailMarkerMessages( usedMarkers, 
                                                     LagCliLib.markerMsgs )

   return LacpAggregates( portChannels=portChannelDict,
                          otherAggregates=otherAggs,
                          _brief=brief,
                          _allPorts=allPorts,
                          markerMessages=usedMarkerMessages )

class LacpAggregatesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] aggregates '
              ' [ ( ( brief | detailed ) [ all-ports ] ) | '
              '   ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'aggregates': matcherAggregates,
      'brief': matcherBriefLacpStatus, 
      'detailed': matcherDetailedLacpStatus, 
      'all-ports': matcherAllPortsUnselected, 
   }
   handler = doShowLacpAggregates
   cliModel = LacpAggregates

BasicCli.addShowCommandClass( LacpAggregatesCmd )

# return ( brief, all, channelPortTree )
# pylint: disable-next=redefined-builtin
def lacpChannelsAndPorts( mode, channelGroups, intf, brief, all ):
   assert not ( intf and channelGroups ), \
       "Cannot specify both Interface and Channel" 
   channelPortTree = []
   if brief is None:
      brief = True
   ls = LagCliLib.lacpStatus
   if intf and intf.name:
      epilc = LagCliLib.ethPhyIntfLagConfig( mode, name=intf.name )
      # pylint: disable-next=unused-variable
      elic = LagCliLib.lagIntfConfigDir.intfConfig.get( intf.name )
      lpc = ls.portStatus.get( intf.name )
      elag = epilc and epilc.lag
      if not elag:
         print ( "Interface %s is not a member of a port channel." % intf.name )
         return ( brief, all, channelPortTree )
      if not lpc:
         print ( "Interface %s is not a member of a port channel configured for "
                 "LACP." % intf.name )
         return ( brief, all, channelPortTree )
      if LagCliLib.inactiveLag( elag.intfId ):
         print ( "Interface %s is a member of an inactive LACP port channel. "
                 "%s" % ( intf.name, LagCliLib.configLimitExceededMsg ) )
         return ( brief, all, channelPortTree )
      channels = [ elag.intfId ]
      all = True
   else:
      channels = LagCliLib.portchannelList( mode, channelGroups )
   if channels:
      ls = LagCliLib.lacpStatus
      activePorts = LagCliLib.channelPorts( mode, channels, True, 
                                         status=Tac.Type( "Lag::LagStatusFilter" ).\
                                         filterOnActive )   
      for channel in channels:
         ports = LagCli.cliLacpPorts( mode, channel,
                                      channels, all, activePorts[ channel ] )
         if intf and intf.name:
            if intf.name in ports:
               ports = [ intf.name ]
            else:
               ports = []
         channelPortTree.append( ( channel, ports, ) )
   if(( channelGroups is None ) and all and not brief ):
      # Find ports not associated with any active channels
      channelPortTree += LagCliLib.lacpOrphanPorts( mode )
   return( brief, all, channelPortTree )

def portsCountersDict( portChannel, ports, lacpStatus, lacpCounters,
                       lacpCountersCheckpoint, brief ):
   portDict = {}
   if ports:
      channelStatus = portChannel and lacpStatus.lagStatus.get( portChannel )
      for port in ports:
         pn = LagCli.portCounters( port, channelStatus, lacpCounters,
                                   lacpCountersCheckpoint, brief )
         if pn:
            portDict[ port ] = pn
   return portDict

def portChannelCounters( portChannel, ports, markers, lacpStatus, lacpCounters,
                         lacpCountersCheckpoint, brief ):
   portsDict = portsCountersDict( portChannel, ports, lacpStatus, lacpCounters,
                                  lacpCountersCheckpoint, brief )
   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelLacpCounters( interfaces=portsDict,
                                            markers=lacpAnnotationMarkers )

def doShowLacpCounters( mode, args ):
   brief = 'detailed' not in args
   allPorts = 'all-ports' in args
   intf = args.get( 'INTERFACE' )
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   updatePortCounters( mode )
   LagCliLib.syncCounterCheckpoint( LagCliLib.lacpCountersCheckpoint, 
                                    LagCliLib.lacpCounters )
   brief, allPorts, channelPortTree = \
       lacpChannelsAndPorts( mode, channelGroupIdList, intf, 
                             brief, allPorts )

   portChannelDict = {}
   orphanPorts = {}
   usedMarkerMessages = None

   if channelPortTree:
      haveOutput = False
      usedMarkers = set()

      for channel, ports in channelPortTree:
         haveOutput = True if ports else haveOutput
         if channel:
            markers = genMarkers( channel )
            usedMarkers = usedMarkers.union( set( markers ) )
            portChannelDict[ channel ] = \
      portChannelCounters( channel, ports, markers,
                                    LagCliLib.lacpStatus, LagCliLib.lacpCounters,
                                    LagCliLib.lacpCountersCheckpoint, brief )
         else:
            orphanPorts = portsCountersDict( None, ports, 
                                             LagCliLib.lacpStatus, 
                                             LagCliLib.lacpCounters, 
                                             LagCliLib.lacpCountersCheckpoint, 
                                             brief )
      if haveOutput:
         usedMarkerMessages = \
               usedDetailMarkerMessages( usedMarkers, LagCliLib.markerMsgs )

   return LacpCounters( portChannels=portChannelDict,
                        _brief=brief,
                        _allPorts=allPorts,
                        markerMessages=usedMarkerMessages,
                        orphanPorts=orphanPorts )

class LacpCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] counters '
              ' [ ( ( brief | detailed ) [ all-ports ] ) | '
              '   ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'counters': matcherCounters,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus, 
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpCounters
   cliModel = LacpCounters

BasicCli.addShowCommandClass( LacpCountersCmd )

# pylint: disable-next=inconsistent-return-statements
def portLacp( port, lacpStatus, lacpCounters, channelStatus, brief ):
   s = lacpStatus.portStatus.get( port )
   if not s:
      return
   lacpIntfConf = LagCli.lacpConfig.lacpIntfConfig.get( port )
   if not lacpIntfConf:
      return

   r = s.partnerPort
   l = s.actorPort

   details = None
   if not brief:
      c = lacpCounters.portCounters.get( port )
      details = LagModel.InterfaceLacpDetails(
        aggSelectionState = s.selected,
        partnerChurnState = s.partnerChurnState,
        partnerCollectorMaxDelay = s.partnerCollectorMaxDelay,
        actorAdminKey = "0x%04x" % l.key,
        actorChurnState = s.actorChurnState,
        actorLastRxTime = LagModel.toUtc( c.debugLastRxTime ) if c else 0.0,
        actorRxSmState = s.rxSmState,
        actorMuxSmState = s.muxSmState,
        actorMuxReason = c.debugMuxReason if c else '' )
      details.actorTimeoutMultiplier = lacpIntfConf.timeoutMult

   return LagModel.InterfaceLacp(
      actorPortStatus = LagCli.lacpPortStatus( port, channelStatus ),
      partnerSysId = dot43sysid( r.sysPriority, r.sysId ),
      partnerPortId = r.portId,
      partnerPortState = LagCliLib.lacpPortStateModel( s.partnerState ),
      partnerOperKey = "0x%04x" % r.key,
      partnerPortPriority = r.portPriority,
      actorPortId = l.portId,
      actorPortState = LagCliLib.lacpPortStateModel( s.actorState ),
      actorOperKey = "0x%04x" % l.key,
      actorPortPriority = l.portPriority,
      details = details )

def portsLacpDict( portChannel, ports, lacpStatus, lacpCounters, brief ):
   portDict = {}
   if ports:
      channelStatus = portChannel and lacpStatus.lagStatus.get( portChannel )
      for port in ports:
         pn = portLacp( port, lacpStatus, lacpCounters, channelStatus, brief )
         if pn:
            portDict[ port ] = pn
   return portDict

def portChannelLacp( portChannel, ports, markers, lacpStatus, lacpCounters, brief ):
   portsDict = portsLacpDict( portChannel, ports, lacpStatus, lacpCounters, brief )
   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelLacp( interfaces=portsDict,
                                    markers=lacpAnnotationMarkers )

def doShowLacpPort( mode, args ):
   intf = args.get( 'INTERFACE' )
   brief = 'detailed' not in args
   allPorts = 'all-ports' in args
   updatePortCounters( mode )
   brief, allPorts, channelPortTree = \
         lacpChannelsAndPorts( mode, None, intf, brief, allPorts )

   portChannelDict = {}
   orphanPorts = {}
   usedMarkerMessages = None
   interface = intf.name if intf else None

   if channelPortTree:
      haveOutput = False
      usedMarkers = set()

      for channel, ports in channelPortTree:
         haveOutput = True if ports else haveOutput
         if channel:
            markers = genMarkers( channel )
            usedMarkers = usedMarkers.union( set( markers ) )
            portChannelDict[ channel ] = \
                  portChannelLacp( channel, ports, markers, 
                                   LagCliLib.lacpStatus, LagCliLib.lacpCounters, 
                                   brief )
         else:
            orphanPorts = portsLacpDict( None, ports, LagCliLib.lacpStatus, 
                                         LagCliLib.lacpCounters, brief )

      if haveOutput:
         usedMarkerMessages = \
               usedDetailMarkerMessages( usedMarkers, LagCliLib.markerMsgs )

   return LacpPort( portChannels=portChannelDict,
                    interface=interface,
                    _brief=brief,
                    _allPorts=allPorts,
                    markerMessages=usedMarkerMessages,
                    orphanPorts=orphanPorts )

class LacpInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] ' 
              '[ ( ( brief | detailed ) [ all-ports ] ) | ' 
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpPort
   cliModel = LacpPort

BasicCli.addShowCommandClass( LacpInterfaceCmd )

class LacpInterfaceCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] counters ' 
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'counters': matcherCounters,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpCounters
   cliModel = LacpCounters

BasicCli.addShowCommandClass( LacpInterfaceCountersCmd )

def doShowLacpInternalEvents( mode, args ):
   model = LagModel.LacpInternalEvents()
   eventTable = LagCliLib.lacpEventTable

   records = list( eventTable.lacpEventRecord.values() )
   lastIndex = len( records ) - 1
   stopIndex = -1

   startTime = None
   N = args.get( 'N' )
   if N:
      N = args.get( 'N', 0 )
      if args[ 'UNIT' ] == 'seconds':
         startTime = Tac.utcNow() - N
      elif args[ 'UNIT' ] == 'minutes':
         startTime = Tac.utcNow() - N*60
      elif args[ 'UNIT' ] == 'hours':
         startTime = Tac.utcNow() - N*60*60
      elif args[ 'UNIT' ] == 'days':
         startTime = Tac.utcNow() - N*24*60*60
      elif args[ 'UNIT' ] == 'count':
         stopIndex = max( lastIndex - N, -1 )

   for i in range( lastIndex, stopIndex, -1 ):
      record = records[ i ]
      if startTime:
         if record.timestamp < startTime:
            break
      internalRecord = LagModel.LacpInternalEventRecord( 
         timestamp = record.timestamp, portChannel = record.portChannel, 
         intfId = record.intfId, smName = record.smName, state = record.state, 
         reason = record.reason, portEnabled = record.portEnabled, 
         lacpEnabled = record.lacpEnabled, currentWhileTimerExpired = 
         record.currentWhileTimerExpired, lacpduRx = record.lacpduRx,
         portMoved = record.portMoved, selected = record.selected,
         partnerSync = record.partnerSync )
      model.lacpInternalEventRecords.append( internalRecord )

   return model

class LacpInternalEventsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp internal events [ last N UNIT ]' )
   data = {
      'lacp': nodeLacp,
      'internal': matcherInternal,
      'events': 'Display Lacp state transition events',
      'last': 'Filter last few LACP state transition events',
      'N': CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1, 500, 
            helpdesc="Number of time units or events" ),
         maxMatches=1 ),
      'UNIT': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'seconds': 'Last N seconds',
            'minutes': 'Last N minutes',
            'hours': 'Last N hours',
            'days': 'Last N days',
            'count': 'Last N events'
            } ),
         maxMatches=1 )
   }
   handler = doShowLacpInternalEvents
   cliModel = LacpInternalEvents

BasicCli.addShowCommandClass( LacpInternalEventsCmd )

def portsInternalDict( portChannel, ports, lacpStatus, lacpCounters, brief ):
   portDict = {}
   if ports:
      channelStatus = portChannel and lacpStatus.lagStatus.get( portChannel )
      for port in ports:
         pn = LagCli.portInternal( port, lacpStatus, channelStatus, 
                                   lacpCounters, brief )
         if pn:
            portDict[ port ] = pn
   return portDict

def portChannelInternal( portChannel, ports, markers, lacpStatus, lacpCounters,
                         brief ):
   portsDict = portsInternalDict( portChannel, ports, lacpStatus, lacpCounters,
                                  brief )
   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelLacpInternal( interfaces=portsDict,
                                            markers=lacpAnnotationMarkers )

def doShowLacpInternal( mode, args ): 
   intf = args.get( 'INTERFACE' )
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   brief = 'detailed' not in args
   allPorts = 'all-ports' in args
   updatePortCounters( mode )
   brief, allPorts, channelPortTree = \
         lacpChannelsAndPorts( mode, channelGroupIdList, 
                               intf, brief, allPorts )
 
   portChannelDict = {}
   orphanPorts = {}
   systemIdentifier = None
   usedMarkerMessages = None
   sysIdList = None
 
   if channelPortTree:
      haveOutput = False
      usedMarkers = set()
 
      for channel, ports in channelPortTree:
         haveOutput = True if ports else haveOutput
         if channel:
            markers = genMarkers( channel )
            usedMarkers = usedMarkers.union( set( markers ) )
            portChannelDict[ channel ] = \
                  portChannelInternal( channel, ports, 
                                       markers, LagCliLib.lacpStatus, 
                                       LagCliLib.lacpCounters, brief )
         else:
            orphanPorts = portsInternalDict( None, ports, LagCliLib.lacpStatus, 
                                             LagCliLib.lacpCounters, brief )
 
      # LACP System priority:
      priority = LagCli.lacpConfig.priority
      addr = LagCli.bridgingConfig.bridgeMacAddr
      systemIdentifier = dot43sysid( priority, addr )
      sysIdList = []
      for hook in LagCliLib.showLacpInternalSysIdHook.extensions():
         internalSysId = hook( LagCli.lacpConfig )
         if internalSysId:
            sysIdList.append( internalSysId )
      if haveOutput:
         usedMarkerMessages = usedDetailMarkerMessages( usedMarkers, 
                                                        LagCliLib.markerMsgs )
 
   return LacpInternal( portChannels=portChannelDict,
                        systemId=systemIdentifier,
                        miscLacpSysIds=sysIdList,
                        _brief=brief,
                        _allPorts=allPorts,
                        markerMessages=usedMarkerMessages,
                        orphanPorts=orphanPorts )

class LacpInterfaceInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] internal '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'internal': matcherInternal,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpInternal
   cliModel = LacpInternal

BasicCli.addShowCommandClass( LacpInterfaceInternalCmd )

def portsNeighborDict( portChannel, ports, lacpStatus, brief ):
   portDict = {}
   if ports:
      channelStatus = portChannel and lacpStatus.lagStatus.get( portChannel )
      for port in ports:
         pn = LagCli.portNeighbor( port, lacpStatus, channelStatus, brief )
         if pn:
            portDict[ port ] = pn
   return portDict

def portChannelNeighbor( portChannel, ports, markers, lacpStatus, brief ):
   portsDict = portsNeighborDict( portChannel, ports, lacpStatus, brief )
   lacpAnnotationMarkers = markersToModelSet( markers ) if portChannel else None
   return LagModel.PortChannelLacpNeighbors( interfaces=portsDict,
                                    markers=lacpAnnotationMarkers )

def doShowLacpNeighbors( mode, args ):
   brief = 'detailed' not in args
   allPorts = 'all-ports' in args
   intf = args.get( 'INTERFACE' )
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   
   updatePortCounters( mode )
   brief, allPorts, channelPortTree = \
       lacpChannelsAndPorts( mode, channelGroupIdList, intf, 
                             brief, allPorts )

   portChannelDict = {}
   orphanPorts = {}
   usedMarkerMessages = None

   if channelPortTree:
      haveOutput = False
      usedMarkers = set()

      for channel, ports in channelPortTree:
         haveOutput = True if ports else haveOutput
         if channel:
            markers = genMarkers( channel )
            usedMarkers = usedMarkers.union( set( markers ) )
            portChannelDict[ channel ] = \
                  portChannelNeighbor( channel, ports,
                                       markers, LagCliLib.lacpStatus, brief )
         else:
            orphanPorts = portsNeighborDict( None, ports, 
                                             LagCliLib.lacpStatus, brief )

      if haveOutput:
         usedMarkerMessages = usedDetailMarkerMessages( usedMarkers, 
                                                        LagCliLib.markerMsgs )

   return LacpNeighbors( portChannels=portChannelDict,
                         _brief=brief,
                         _allPorts=allPorts,
                         markerMessages=usedMarkerMessages,
                         orphanPorts=orphanPorts )

class LacpInterfaceNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] neighbor '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'neighbor': nodeNeighborDeprecated,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpNeighbors
   cliModel = LacpNeighbors
   hidden = True
   privileged = True

BasicCli.addShowCommandClass( LacpInterfaceNeighborCmd )

class LacpInterfacePeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp interface [ INTERFACE ] peer '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'interface': matcherInterface,
      'INTERFACE': lagMemberIntfMatcher,
      'peer': matcherPeer,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpNeighbors
   cliModel = LacpNeighbors

BasicCli.addShowCommandClass( LacpInterfacePeerCmd )

class LacpInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] internal '  
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'internal': matcherInternal,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpInternal
   cliModel = LacpInternal

BasicCli.addShowCommandClass( LacpInternalCmd )

class LacpNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] neighbor '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'neighbor': nodeNeighborDeprecated,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpNeighbors
   cliModel = LacpNeighbors

BasicCli.addShowCommandClass( LacpNeighborCmd )

class LacpPeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show lacp [ INTFS ] peer '
              '[ ( ( brief | detailed ) [ all-ports ] ) | '
              '  ( all-ports [ brief | detailed ] ) ]' )
   data = {
      'lacp': nodeLacp,
      'INTFS': matcherChannelIds,
      'peer': matcherPeer,
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus, 
      'all-ports': matcherAllPortsUnselected,
   }
   handler = doShowLacpNeighbors
   cliModel = LacpNeighbors

BasicCli.addShowCommandClass( LacpPeerCmd )

def doShowLacpSysId( mode, args ):
   brief = 'detailed' not in args
   lc = LagCli.lacpConfig
   # LACP System priority:
   priority = lc.priority
   # Switch MAC Address
   bc = LagCli.bridgingConfig
   addr = bc.bridgeMacAddr
   sysIdRep = dot43sysid( priority, addr )
 
   sysIdList = []
   for hook in LagCliLib.showLacpSysIdHook.extensions():
      miscLacpSysId = hook( lc, brief )
      if miscLacpSysId:
         sysIdList.append( miscLacpSysId )
 
   details = None
   if not brief:
      details = LacpSysId.Details( priority=priority, address=addr )
   return LacpSysId( details=details,
                     miscLacpSysIds=sysIdList,
                     systemId=sysIdRep )

class LacpSysIdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show lacp sys-id [ ( brief | detailed ) ]'
   data = {
      'lacp': nodeLacp,
      'sys-id': 'Display System Identifier used by LACP',
      'brief': matcherBriefLacpStatus,
      'detailed': matcherDetailedLacpStatus,
   }
   handler = doShowLacpSysId
   cliModel = LacpSysId

BasicCli.addShowCommandClass( LacpSysIdCmd )

#
# LAG hidden command
# 
def doShowLagInitializationTime( mode, args ):
   print( "Lag initialization time: " )
   for name, value in LagCli.agentStatus.initTime.items():
      print( f"   {name} : {value:6.3f} seconds" )

class LagInitializationTimeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show lag initialization-time'
   data = {
      'lag': 'Link Aggregation Agent (LAG)',
      'initialization-time': 'Display time spent in LagAgent initialization',
   }
   handler = doShowLagInitializationTime
   hidden = True

BasicCli.addShowCommandClass( LagInitializationTimeCmd )

class PortChannelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show port-channel [ INTFS ] ' 
              '[ ( ( brief | detailed ) [ active-ports | all-ports ] ) | '
              '  ( ( active-ports | all-ports ) [ brief | detailed ] ) ] ' )
   data = {
      'port-channel': nodePortChannel,
      'INTFS': matcherChannelIds,
      'brief': matcherBriefPortChannel,
      'detailed': matcherDetailedPortChannel,
      'active-ports': matcherActivePorts, 
      'all-ports': matcherAllPortsInactive,
   }
   handler = doShowPortChannel
   cliModel = PortChannels

BasicCli.addShowCommandClass( PortChannelCmd )

def getInterfaceSummary( elis, port, elic ):
   linkDown = False
   suspended = False
   lacpMisconfig = None
   lacpState = None
   staticLag = False
   lagMember = False

   ethIntfstatus = LagCli.ethIntfStatusDir.intfStatus.get( port )
   # Check to see if the interface is on, if not print D and exit
   if (not ethIntfstatus or (ethIntfstatus.linkStatus != "linkUp" or 
           ethIntfstatus.operStatus != "intfOperUp") ):
      linkDown = True
   lacpIntfConf = LagCli.lacpConfig.lacpIntfLagConfig.get( port )
   # Check to see if part of the port channel
   if ( elis and elis.member ):
      if port in elis.member:
         lagMember = True
      else:
         # If the link is suspended, don't print any other flags 
         # Caller verifies that elic is not None
         # A channel could be suspended if it is the wrong speed for the channel
         # (among other reasons)
         if LagCliLib.LagProtocolMap[ elic.mode ] == "LACP":
            if( lacpIntfConf and
                lacpIntfConf.individualLagKey.individual is True and 
                lacpIntfConf.individualLagKey.lagIntfId == "" ):
               suspended = True
         elif LagCliLib.LagProtocolMap[ elic.mode ] == "Static": 
            suspended = True

   if LagCliLib.LagProtocolMap[ elic.mode ] == "Static":
      staticLag = True
   elif LagCliLib.LagProtocolMap[ elic.mode ] == "LACP":
      misConfig = None
      lss = LagCliLib.lacpStatus.lagStatus.get( elic.intfId )
      if not lss:
         misConfig = 'idle'        # LACP Idle- not flagged
      elif not lss.agg:
         misConfig = 'noAgg'       # LACP No Aggregate- not flagged yet
      elif port in lss.agg.aggregate:
         misConfig = 'bundled'     # LACP Bundled- already flagged above
      elif port in lss.agg.standby:
         misConfig = 'standby'     # LACP Standby- not flagged yet
      elif port in lss.otherIntf:
         misConfig = 'mismatchAgg' # LACP Mismatched Aggregate (incompatible)
      elif port in lss.agg.selected:
         misConfig = 'negotiation' # LACP In Negotation
      else:
         misConfig = 'unknown'     # LACP Unknown error- not flagged yet

      lacpMisconfig = LacpMisconfig( status=misConfig )

      s = LagCliLib.lacpStatus.portStatus.get( port )
      if s:
         lacpState = LagCliLib.lacpPortStateModel( s.actorState )

   return ( linkDown, suspended, lacpMisconfig, lacpState, staticLag, lagMember )

def getPortChannelLinkState( elis, elic ):
   portFlags = ""
   # Check to see if the port channel is active
   if( elis and elis.member ):
      portFlags += "up"
   else:
      portFlags += "down"
   return portFlags

def generateSummaryStats(lags, lagPortTable):
   # A channel is considered to be an aggregator if:
   # 1.) LACP channel in lss.agg
   # 2.) LACP channel in lss.otherIntf
   # 3.) Static configured channel
   numAggs = 0
   for elic in lags:
      lss = LagCliLib.lacpStatus.lagStatus.get( elic.intfId )
      if( lss and lss.agg ):
         numAggs += 1         
         numAggs += len( set( map( lambda p: p.lagId,
               lss.otherIntf.values() ) ) )
      # A static channel should always be considered an aggregator
      elif LagCliLib.LagProtocolMap[ elic.mode ] == "Static":
         numAggs += 1
   numChannels = 0
   for elic in lags:
      elis = LagCliLib.lagIntfStatusDir.intfStatus.get( elic.intfId, None )
      if( elis and elis.member ):
         numChannels += 1
   return ( numChannels, numAggs )

def doShowPortChannelSummary( mode, args ):
   elicd = LagCliLib.lagIntfConfigDir
   elisd = LagCliLib.lagIntfStatusDir
   channels = LagCliLib.portchannelList( mode, None, lacpOnly=False, 
                                         inactiveWarn=False )
   lags = list( map( elicd.intfConfig.get, channels ) )

   # Build up table of all ports in a LAG
   lagPortTable = {}
   for phyIntf in LagCliLib.ethPhyIntfLagConfigDir( mode ).values():
      if( not phyIntf.lag or ( phyIntf.lag not in lags )):
         continue
      lagPortTable.setdefault( phyIntf.lag.intfId, [] ).append( phyIntf.intfId )

   numberOfChannelsInUse, numberOfAggregators = \
         generateSummaryStats( lags, lagPortTable )
   portChannels = {}
   for elic in lags:
      # Don't print any RecircChannels
      if LagCliLib.isRecirc( elic.intfId ):
         continue

      elis = elisd.intfStatus.get( elic.intfId, None )

      # Determine Port-channel name and flags
      portChannelLinkState = getPortChannelLinkState( elis, elic )
      portName = elic.intfId

      intfs = lagPortTable.get( elic.intfId, [] )

      if portName in LagCli.lagInternalStatus.lagConfig:
         # We need to check whether the port-channel has
         # a lagStatus. CLI prevents configuring port-channels that exceed
         # the limit (2000) but if there are concurrent CLI sessions, this
         # check might be bypassed. In this case, for the port-channels that
         # exceed the configurable limit, Lag agent does not create a lagStatus
         collectDistribute = \
               LagCli.lagInternalStatus.lagConfig[ portName ].collectDistribute
         initialMembers = LagCli.hardwareLagConfig(
            portName ).initialMember[ portName ].member
         programmedMembers = LagCli.hardwareLagStatus(
            portName ).programmedMember.get( portName )

      memberDict = {}
      for intf in intfs:
         cfg = LagCliLib.ethPhyIntfLagConfig( mode, name=intf )
         linkDown, suspended, lacpMisconfig, lacpState, staticLag, lagMember = \
               getInterfaceSummary( elis, intf, elic )
         memberDict[ intf ] = \
               LagModel.PortChannelMemberSummary( intf=intf,
                                                  linkDown=linkDown,
                                                  lagMember=lagMember,
                                                  suspended=suspended,
                                                  lacpMisconfig=lacpMisconfig,
                                                  lacpState=lacpState,
                                                  staticLag=staticLag )
         # A member exceeds maximum weight if it is present in 
         # collectDistribute but not in initialMember collection
         exceedsMaxWeight = ( intf not in initialMembers and \
                                  intf in collectDistribute ) 
         memberDict[ intf ].exceedsMaxWeight = exceedsMaxWeight
         if programmedMembers and intf in programmedMembers.member:
            if not programmedMembers.member[ intf ].distributing:
               memberDict[ intf ].txDisabled = True

      if memberDict:
         lacpMode = None
         fallback = None
         cfg = LagCliLib.ethPhyIntfLagConfig( mode, name=min( intfs, 
                                                           key=Arnet.intfNameKey ) )
         if cfg and ( cfg.mode == 'lacpModeActive' or
                      cfg.mode == 'lacpModePassive' ):
            lacpMode = LagCliLib.LacpModeMap[ cfg.mode ] 
         if elis and elis.fallbackEnabled != 'fallbackNone':
            if elis.fallbackEnabled == 'fallbackStatic' or \
                   elis.fallbackEnabled == 'fallbackIndividual':
               fallback = FallbackConfigStatus( 
                     status=elis.fallbackEnabled )
         elif elic.fallback != 'fallbackNone':
            if elic.fallback == 'fallbackStatic' or \
                   elic.fallback == 'fallbackIndividual':
               fallback = FallbackConfigStatus( config=elic.fallback )
         protocol = LagCliLib.lagModeToProtocol( elic.mode )
         inactive = LagCliLib.inactiveLag( elic.intfId )
         portChannels[ portName ] = LagModel.PortChannelSummary(
                                          linkState=portChannelLinkState,
                                          protocol=protocol,
                                          lacpMode=lacpMode,
                                          fallback=fallback,
                                          ports=memberDict,
                                          inactive=inactive )
   return PortChannelsSummary( numberOfChannelsInUse=numberOfChannelsInUse,
                               numberOfAggregators=numberOfAggregators,
                               portChannels=portChannels )

class PortChannelDenseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel dense'
   data = {
      'port-channel': nodePortChannel,
      'dense': 'Display summary of port-channel status',
   }
   handler = doShowPortChannelSummary
   cliModel = PortChannelsSummary

BasicCli.addShowCommandClass( PortChannelDenseCmd )

def doShowChannelLimits( mode, channelType='port-channel', limits=None ):
   compatiblePorts = {}
   intfStatusColl = LagIntfCli.ethPhyIntfStatusDir.intfStatus
   for intf in LagCli.lagGroupDir.phyIntf.values():
      lg = intf.lagGroup
      if not lg or intf.name not in intfStatusColl or \
         intfStatusColl[ intf.name ].duplex != 'duplexFull':
         continue
      compatiblePorts.setdefault( lg.name, []).append( intf.name )
   groups = sorted( LagCli.lagGroupDir.lagGroup.values(),
                    key=lambda lg: lg.name )
   portChannelLimits = {}
   for lagGroup in groups:
      if compatiblePorts.get( lagGroup.name ) is None:
         continue
      portList = compatiblePorts.get( lagGroup.name )
      portChannelLimits[ lagGroup.name ] = LagModel.LagGroupLimits(
                       maxPortChannels=lagGroup.maxLagIntfs,
                       maxPortsPerPortChannel=lagGroup.maxPortsPerLagIntf,
                       portList=Arnet.sortIntf( portList ) )
   if channelType == 'port-channel':
      return PortChannelLimits( portChannelLimits=portChannelLimits )
   else:
      return RecircChannelLimits( recircChannelLimits=portChannelLimits )
   
def doShowPortChannelLimits( mode, args ):
   return doShowChannelLimits( mode, limits=None )

class PortChannelLimitsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel limits'
   data = {
      'port-channel': nodePortChannel,
      'limits': 'Display limits on port-channel membership',
   }
   handler = doShowPortChannelLimits
   cliModel = PortChannelLimits

BasicCli.addShowCommandClass( PortChannelLimitsCmd )

def doShowChannelGroupTraffic( mode, lagDict ):
   cntAttrList = [ "inUcastPkts", "outUcastPkts", "inMulticastPkts", \
                   "outMulticastPkts", "inBroadcastPkts", "outBroadcastPkts"]

   if not lagDict:
      return PortChannelsTraffic()
   
   firstLagToPrint = True # pylint: disable=unused-variable
   portChannelDict = {}
   # Each lag is a List of Members
   for ( lag, lagMembers ) in sorted( lagDict.items() ):
      portDict = {}
      if not lagMembers:
         continue
      # Build the counters:
      #   lagCntrDict --> Dictionary
      #        Key: attribute name
      #        Val: Sum of all interfaces per attribute
      #   lagMemberCntrDict - Dictionary of Dictionaries
      #        Key : Interface Name
      #        Val : Dictionary of Counters for the interface (intfCntrDict)
      #   intfCntrDict - Dictionary
      #        Key : Attribute name
      #        Val : Latest counter Value
      lagCntrDict = {}
      lagMemberCntrDict = {}
      for attr in cntAttrList:
         lagCntrDict[ attr ] = 0

      def stat(attr):
         sysdbValue = getattr( curIntfCounter.statistics, attr, None )
         if sysdbValue is None:
            return 'n/a'
         checkpointValue = 0
         if ckpt:
            checkpointValue = getattr( ckpt.statistics, attr, 0 )
         return sysdbValue - checkpointValue

      for intf in lagMembers:
         intfCounters = ethPhyIntfCounterDir().intfCounterDir.get( intf )
         if intfCounters is None:
            # Mlag peer interfaces don't have counters.
            continue
         curIntfCounter = intfCounters.intfCounter[ "current" ]
         intfObj = EthIntfCli.EthPhyIntf( intf, mode )
         ckpt = intfObj.getLatestCounterCheckpoint()
         phyIntf = LagCli.lagStatus.phyIntf.get( intf )
         if phyIntf is None:
            # It left the lag between the beginning of the function and now.
            continue
         lagMemberCkpt = phyIntf.lagMembershipCounterCheckpoint
         # access statsUpdateTime by 1st reading the nominal object rates
         # to get the latest shared memory values
         if not ckpt or ( ckpt.rates.statsUpdateTime < 
               lagMemberCkpt.rates.statsUpdateTime ) or ( 
                     ckpt.statistics.lastUpdate < 
                     lagMemberCkpt.statistics.lastUpdate ) :
         # Including check for statistics.lastUpdate because EtbaIntfCounter.tin
         # only updates statistics and not rates
            ckpt = lagMemberCkpt
         intfCntrDict = {}
         lagMemberCntrDict[ intf ] = intfCntrDict
         for attr in cntAttrList:
            counter = stat( attr )
            intfCntrDict[ attr ] = counter
            lagCntrDict[ attr ] = lagCntrDict[ attr ] + counter

      lagId = LagCli.portChannel( lag )  # pylint: disable=unused-variable
      for intf in lagMembers:
         intfCntrDict = lagMemberCntrDict.get( intf )
         if intfCntrDict is None:
            continue
         percentDict = {}
         for attr in cntAttrList:
            p = 0.0
            lagVal = lagCntrDict[ attr ]
            intfVal = intfCntrDict[ attr ]
            if lagVal > 0:
               p = ( float( intfVal ) / float( lagVal ) ) * 100
            percentDict[ attr ] = p
         tempPortDict = LagModel.InterfaceTraffic( **percentDict )
         portDict[ intf ] = tempPortDict
      portChannelDict[ lag ] = LagModel.PortChannelTraffic( 
                                    interfaceTraffic=portDict )
   return PortChannelsTraffic( portChannels=portChannelDict )

def doShowPortChannelTraffic( mode, args ):
   channelGroupIdList = getChannelGroupIdList( args.get( 'INTFS' ) )
   lagDict = LagCliLib.getLagMemberDict( mode, channelGroupIdList )
   return doShowChannelGroupTraffic( mode, lagDict )

class PortChannelLoadBalanceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel [ INTFS ] load-balance'
   data = {
      'port-channel': nodePortChannel,
      'INTFS': matcherChannelIds,
      'load-balance': 'Display traffic distribution of port-channel members',
   }
   handler = doShowPortChannelTraffic
   cliModel = PortChannelsTraffic

BasicCli.addShowCommandClass( PortChannelLoadBalanceCmd )

class PortChannelSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel summary'
   data = {
      'port-channel': nodePortChannel,
      'summary': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'summary',
                     helpdesc='Display summary of port-channel status' ),
         deprecatedByCmd='show port-channel dense'),
   }
   handler = doShowPortChannelSummary
   cliModel = PortChannelsSummary

BasicCli.addShowCommandClass( PortChannelSummaryCmd )

class PortChannelTrafficCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port-channel [ INTFS ] traffic'
   data = {
      'port-channel': nodePortChannel,
      'INTFS': matcherChannelIds,
      'traffic': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'traffic',
                  helpdesc='Display traffic distribution of port-channel members' ),
         deprecatedByCmd='show port-channel load-balance' ),
   }
   handler = doShowPortChannelTraffic
   cliModel = PortChannelsTraffic

BasicCli.addShowCommandClass( PortChannelTrafficCmd )

def doShowChannelGroups( mode, channels,
                         brief=True, allPorts=True ):
   if brief is None:
      brief = True
   if allPorts is None:
      allPorts = True
   elicd = LagCliLib.lagIntfConfigDir
   elisd = LagCliLib.lagIntfStatusDir
   lags = set( map( elicd.intfConfig.get, channels ) )
   if allPorts or not brief:
      epilsd = LagCli.lagStatus.phyIntf

   # Build up table of unconfigured ports in a LAG
   if allPorts:
      lagPortTable = {}
      for phyIntf in LagCliLib.ethPhyIntfLagConfigDir( mode ).values():
         pLag = phyIntf.lag
         if( not pLag or ( pLag not in lags )):
            continue
         lagPortTable.setdefault( pLag.intfId, [] ).append( phyIntf.intfId )

   # return True only if at least one port channel is configured to use
   # LACP *and* it has active members (We want to know whether we'll need
   # the "Lacp Mode" column in the output.
   def channelLacpActive( elic ):
      if elic.mode == "lagModeLacp":
         elis = elisd.intfStatus.get( elic.intfId, None )
         return elis and elis.member
      else:
         return False
   hasLacpChannels = reduce( lambda x, elic: x or channelLacpActive( elic ),
                             [ x for x in
         elicd.intfConfig.values() if x in lags ], False )
   hasFallback = reduce( lambda x, elic: x or ( elic.fallback or
                             elic.fallbackTimeout != elic.fallbackTimeoutDefault ),
            [ p for p in elicd.intfConfig.values() if p in lags ], False )

   portChannelDict = {}
   for elic in lags:
      activePortsDict = {}
      rxPortsDict = {}
      inactivePortsDict = {}
      elis = elisd.intfStatus.get( elic.intfId, None )
      programmedMembers = LagCli.hardwareLagStatus(
         elic.intfId ).programmedMember.get( elic.intfId )

      activeMembers = {}
      formerlyActiveMembers = []
      fallbackState = None
      checkMinLinks = False
      if hasFallback and elis and not brief:
         fallbackState = elis.debugFallbackSmState
         fallbackState = fallbackState[ len("fallbackState"): ]
         if fallbackState == "Enabled":
            checkMinLinks = True
         if( fallbackState == "Enabled" and not elis.fallbackEnabled ):
            fallbackState = "locallyOnly"
         else:
            fallbackState = fallbackState.lower()

      if checkMinLinks and elic.minLinks != elic.minLinksDefault:
         msg = "min-links config will not be honored when fallback is active"
         mode.addWarning( msg )

      # port channel current weight, the sum of all active
      # members' weights
      currWeight = 0
      if( elis and elis.member ):
         activeMembers = elis.member
         for port in Arnet.sortIntf( activeMembers ):
            if brief:
               activePortsDict[ port ] = LagModel.ActivePort()
            else:
               epils = epilsd.get( port )
               if epils:
                  time = LagModel.toUtc( epils.memberTime )
               else:
                  formerlyActiveMembers.append( port )
                  activePortsDict[ port ] = LagModel.ActivePort()
                  continue

               tempProtocol = None
               tempMode = None
               cfg = LagCliLib.ethPhyIntfLagConfig( mode, name=port )
               tempProtocol = LagCliLib.lagModeToProtocol( elic.mode )
               if hasLacpChannels and cfg:
                  tempMode = LagCliLib.LacpModeMap[ cfg.mode ]
               activePortsDict[ port ] = LagModel.ActivePort( timeBecameActive=time,
                                                              protocol=tempProtocol,
                                                              lacpMode=tempMode )
               portWeight = 0
               if programmedMembers and port in programmedMembers.member:
                  m = programmedMembers.member.get( port )
                  portWeight = m.weight
                  activePortsDict[ port ].collecting = m.collecting
                  activePortsDict[ port ].distributing = m.distributing
               currWeight += portWeight
               activePortsDict[ port ].weight = portWeight

      if allPorts:
         notActiveMembers = [ p for p in lagPortTable.get( elic.intfId, [] )
                              if not p in activeMembers or
                              ( p in formerlyActiveMembers ) ]
         
         for port in Arnet.sortIntf( notActiveMembers ):
            epils = epilsd.get( port )
            if epils:
               reason = epils.reason
               time = LagCli.LagModel.toUtc( epils.memberTime )
            elif not LagCli.ethIntfStatusDir.intfStatus.get( port ):
               reason = 'Interface unavailable'
               time = -1.0
            else:
               reason = 'unknown'
               time = 0.0
            if brief:
               time = None
            
            pmm = None
            if programmedMembers:
               pmm = programmedMembers.member.get( port )
            # for partially active ports that receive but don't
            # transmit, check if port in programmedMember and C/D=1/0
            if pmm and pmm.collecting and not pmm.distributing:
               rxPortsDict[ port ] = LagModel.RxPort(
                  time=time,
                  reason=reason )
            else:
               inactivePortsDict[ port ] = LagModel.InactivePort(
                  timeBecameInactive=time,
                  reasonUnconfigured=reason )

      portChannelDict[ elic.intfId ] = LagModel.PortChannel(
            fallbackState=fallbackState,
            activePorts=activePortsDict,
            rxPorts=rxPortsDict,
            inactivePorts=inactivePortsDict,
            recircFeature=list( elic.recircFeature ),
            inactiveLag=LagCliLib.inactiveLag( elic.intfId ),
            _formerlyActiveMembers=formerlyActiveMembers )

      minSpeed = str( elic.minSpeed.speedValue ) + " " + \
                    LagModel.speedUnitStr( elic.minSpeed.speedUnit )
      portChannelDict[ elic.intfId ].minLinks = elic.minLinks
      if toggleMaxLinksPreemptivePriorityEnabled():
         portChannelDict[ elic.intfId ].maxLinks = elic.maxLinks
      portChannelDict[ elic.intfId ].minSpeed = minSpeed
      mppl = 0
      for port in list( activePortsDict ) + list( inactivePortsDict ):
         if port in LagCli.lagGroupDir.phyIntf:
            lg = LagCli.lagGroupDir.phyIntf[ port ].lagGroup
            if lg:
               mppl = lg.maxPortsPerLagIntf
               break
      portChannelDict[ elic.intfId ].currWeight = currWeight
      portChannelDict[ elic.intfId ].maxWeight = mppl

   return PortChannels( portChannels=portChannelDict,
                        _brief=brief,
                        _allPorts=allPorts )
def doShowRecircChannel( mode, args ):
   recircChannelGroupIdList = getRecircGroupIdList( args.get( 'INTFS' ) )
   brief = 'detailed' not in args
   allPorts = 'active-ports' not in args
   channels = RecircCli.recircChannelList( mode, recircChannelGroupIdList )
   return doShowChannelGroups( mode, channels, brief, allPorts )

class RecircChannelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show recirc-channel [ INTFS ] '
              '[ ( ( brief | detailed ) [ active-ports | all-ports ] ) | '
              '  ( ( active-ports | all-ports ) [ brief | detailed ] ) ] ' )
   data = {
      'recirc-channel': nodeRecircChannelParams,
      'INTFS': matcherRecircChannelIds, 
      'brief': matcherBriefRecirc, 
      'detailed': matcherDetailedRecirc, 
      'active-ports': matcherActivePorts, 
      'all-ports': matcherAllPortsInactive, 
   }
   handler = doShowRecircChannel
   cliModel = PortChannels

BasicCli.addShowCommandClass( RecircChannelCmd )

def doShowFabricChannel( mode, args ):
   channelId = args.get( 'CHANNELID' )
   if channelId is not None:
      channelId = LagCli.FabricChannelGroup( channelId )
      if channelId.idStr not in LagCliLib.lagIntfConfigDir.intfConfig:
         mode.addErrorAndStop( f"{channelId.idStr} not configured as LAG" )
   brief = 'detailed' not in args
   allPorts = 'active-ports' not in args
   # Multiple channelIds are not supported yet. This is because we
   # need a MultiRangeMatcher on slashed integers or take in memberIds
   # slash groupIds as ranges. Adding that support can come as an
   # enhancement later.
   candidateFcIntfs = [] if not channelId else [ channelId.idStr ]
   fcIntfs = LagCliLib.fabricChannelIntfList( candidateFcIntfs )
   return doShowChannelGroups( mode, fcIntfs, brief, allPorts )

class FabricChannelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show fabric-channel [ CHANNELID ] '
              '[ ( ( brief | detailed ) [ active-ports | all-ports ] ) | '
              '  ( ( active-ports | all-ports ) [ brief | detailed ] ) ] ' )
   data = {
      'fabric-channel': nodeFabricChannelParams,
      'CHANNELID': LagCliLib.FabricChannelIntfNumberMatcher,
      'brief': matcherBriefFabricChannel,
      'detailed': matcherDetailedFabricChannel,
      'active-ports': matcherActivePorts,
      'all-ports': matcherAllPortsInactive,
   }
   handler = doShowFabricChannel
   cliModel = PortChannels

if toggleSwagPhase1Enabled():
   BasicCli.addShowCommandClass( FabricChannelCmd )

def doShowRecircChannelLimits( mode, args ):
   limits = 'limits' in args
   return doShowChannelLimits( mode, channelType='recirc-channel', 
                               limits=limits )

class RecircChannelLimitsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show recirc-channel limits'
   data = {
      'recirc-channel': nodeRecircChannelParams,
      'limits': 'Display limits on recirc-channel membership',
   }
   handler = doShowRecircChannelLimits
   cliModel = RecircChannelLimits

BasicCli.addShowCommandClass( RecircChannelLimitsCmd )

def doShowRecircChannelTraffic( mode, args ):
   recircChannelGroupIdList = getRecircGroupIdList( args.get( 'INTFS' ) )
   lagDict = LagCliLib.getLagMemberDict( mode, recircChannelGroupIdList, 
                                      filterFunc=LagCliLib.isRecirc )
   return doShowChannelGroupTraffic( mode, lagDict )

class RecircChannelTrafficCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show recirc-channel [ INTFS ] traffic'
   data = {
      'recirc-channel': nodeRecircChannelParams,
      'INTFS': matcherRecircChannelIds, 
      'traffic': 'Display traffic distribution of recirc-channel members',
   }
   handler = doShowRecircChannelTraffic
   cliModel = PortChannelsTraffic

BasicCli.addShowCommandClass( RecircChannelTrafficCmd )

def registerShowLoadBalancePortChannelCmd( showCommandClass ):
   # Strata and Mako implement their own syntax's for the port-channel hashing show
   # command ("show load-balance destination port-channel...") in that they accept
   # different combinations of headers to be hashed on.
   BasicCli.addShowCommandClass( showCommandClass )

def Plugin( entityManager ):
   global bridgingHwCapabilities

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
