#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=simplifiable-if-statement
# pylint: disable=no-self-argument

import io
import os
import re
from socket import IPPROTO_TCP

import Tac, BasicCli, CliParser, LazyMount
import ConfigMount
import Arnet
import CliPlugin.TechSupportCli
from CliPlugin import AclCli
from CliPlugin.LanzCliModel import * # pylint: disable=wildcard-import
from CliPlugin.VrfCli import vrfExists, DEFAULT_VRF
import AgentCommandRequest
from CliMode.Intf import IntfMode
import CliMode.Lanz
from LanzLib import lanzEnabled, lanzEnabledIs, lanzSupported, \
      lanzTxLatencyEnabled, LanzLocked
from MirroringLib import createV4ZeroGreTunnelKey, createGreTunnelKey
import AgentDirectory
import AclLib
import AclCliLib
from EthIntfUtil import isEthernetIntf
from MirroringCliLib import isMaxSessionsReached
import LanzWarningsLib
from TypeFuture import TacLazyType
from Toggles.LanzToggleLib import toggleMultiVrfStreamingEnabled

MirroringConstants = TacLazyType( "Mirroring::Constants" )
serviceName = "lanz"
lanzConfig = None
mirroringConfig = None
lagConfig = None
em = None
lanzHwStatus = None
mirroringHwCapability = None
ethPhyIntfStatusDir = None
streamingStatus = None
updateIntervalStatus = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
hwEpochStatus = None
lanzLatencyStatus = None

QueueLenUnit = Tac.Type( "Lanz::QueueLenUnit" )
QueueLength = Tac.Type( "Lanz::QueueLength" )
LanzHelper = Tac.Type( "Lanz::LanzHelper" )
IntfId = Tac.Type( "Arnet::IntfId" )
LanzMode = Tac.Type( "Lanz::LanzMode" )
defaultGreHdrProto = Tac.Type( "Mirroring::GreTunnelKey" ).defaultGreHdrProto
defaultGreTtl = 128
defaultGreDscp = 0

lanzModeDesc = { LanzMode.polling : "polling",
                 LanzMode.pollingDueToCountersExhausted :
                    "polling due to counter engines exhausted",
                 LanzMode.notifying : "notifying" }

def isSandRunning( sysname ):
   if AgentDirectory.agent( sysname, 'SandLanz' ) or \
          os.environ.get( 'SIMULATION_SANDLANZ' ):
      return True
   else:
      return False

def isStrataLanzRunning( sysname ):
   if AgentDirectory.agent( sysname, 'StrataLanz' ) or \
         os.environ.get( 'SIMULATION_STRATALANZ' ): 
      return True
   return False

def isStrataRunning( sysname ):
   if AgentDirectory.agent( sysname, 'StrataCentral' ) or \
          os.environ.get( 'SIMULATION_STRATALANZ' ):
      return True
   else:
      return False

def isXpRunning( sysname ):
   if AgentDirectory.agent( sysname, 'XpLanz' ) or \
          os.environ.get( 'SIMULATION_XPLANZ' ):
      return True
   else:
      return False

def isBfnRunning( sysname ):
   if AgentDirectory.agent( sysname, 'BfnLanz' ) or \
          os.environ.get( 'SIMULATION_BFNLANZ' ):
      return True
   else:
      return False

def isMakoRunning( sysname ):
   if AgentDirectory.agent( sysname, 'MakoSlice-FixedSystem' ) or \
      os.environ.get( 'SIMULATION_MAKOLANZ' ):
      return True
   else:
      return False


def isLanzAgentRunning( sysname ):
   if getForwardingPlaneName( sysname ) is None:
      return False

   if getForwardingPlaneName( sysname ) == 'Strata':
      if not isStrataLanzRunning( sysname ):
         return False
   return True

def getForwardingPlaneName( sysname ):
   if isSandRunning( sysname ):
      return 'Sand'
   if isStrataRunning( sysname ):
      return 'Strata'
   if isXpRunning( sysname ):
      return 'Xp'
   if isBfnRunning( sysname ):
      return 'Barefoot'
   if isMakoRunning( sysname ):
      return 'Mako'
   return None

# pylint: disable-next=inconsistent-return-statements
def getCongestionRecord( platformName, cmdType ):
   if platformName == 'Sand':
      if '-l' in cmdType:
         newRecord = SandLatencyRecord()
      else:
         newRecord = SandCongestionRecord()
      newRecord.globalProtectionModeEnabled = \
            hwEpochStatus.globalProtectionModeEnabled
      return newRecord
   elif platformName == 'Barefoot':
      if '-c' in cmdType:
         return BarefootStatisticsRecord()
      else:
         return BarefootCongestionRecord()
   elif platformName == 'Xp':
      if '-c' in cmdType:
         return XpStatisticsRecord()
      else:
         return XpCongestionRecord()
   elif platformName == 'Strata':
      if '-b' not in cmdType and \
             '-d' not in cmdType and \
             '-l' not in cmdType and \
             '-c' not in cmdType:
         return StrataCongestionRecord()
      elif '-d' in cmdType:
         return StrataDropsRecord()
      elif '-l' in cmdType:
         return StrataLatencyRecord()
      elif '-c' in cmdType:
         return StrataStatisticsRecord()
      else:
         return
   elif platformName == 'Mako':
      if '-c' in cmdType:
         return MakoStatisticsRecord()
      else:
         return MakoCongestionRecord()
   else:
      return

def lanzGuard( mode, token ):
   if lanzHwStatus.supported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzCpuPortMonitoringSupportedGuard( mode, token ):
   if lanzHwStatus.cpuPortMonitoringSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzFabricMonitoringSupportedGuard( mode, token ):
   if lanzHwStatus.fabricMonitoringSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def ethPortMirroringGuard( mode, token ):
   if lanzHwStatus.ethPortMirroringSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def cpuPortGuard( mode, token ):
   if lanzHwStatus.cpuPortMirroringSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzStatisticsGuard( mode, token ):
   if lanzHwStatus.statisticsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzLowThresholdGuard( mode, token ):
   if lanzHwStatus.lowThresholdSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzForceNotifyingSupportedGuard( mode, token ):
   if lanzHwStatus.forceNotifyingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform
   
def lanzStreamingGuard( mode, token ):
   if lanzHwStatus.streamingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzUpdateIntervalGuard( mode, token ):
   if lanzHwStatus.updateIntervalSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzDropGuard( mode, token ):
   if lanzHwStatus.dropSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzTxLatencyGuard( mode, token ):
   if lanzHwStatus.txLatencySupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzGlobalBufferGuard( mode, token ):
   if lanzHwStatus.globalBufferSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzFrameMirrorGuard( mode, token ):
   if lanzHwStatus.mirroringSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzDefaultThresholdsGuard( mode, token ):
   if lanzHwStatus.defaultThresholdsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzConfigDefaultThresholdsGuard( mode, token ):
   if lanzHwStatus.configDefThresholdsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def lanzTxLatencyEnablingGuard( mode, token ):
   if lanzHwStatus.txLatencyEnablingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mirroringDestinationGreSupported( mode, token ):
   if lanzHwStatus.mirroringSupported and \
         mirroringHwCapability.destinationGreSupported:
      return None
   return CliParser.guardNotThisPlatform

def isValidDestination( sessionName, destIntf ):
   # Destination is invalid, if it is present as a source or a destination in any
   # other session, or is present as a source interface in the current session
   cfg = mirroringConfig
   # cpu port can serve as destinations of multiple mirror sessions.
   if destIntf == "Cpu":
      return ( True, '' )

   # For ports other than cpu port, it cannot serve as the mirror destination
   # if it is already a source interface in a session or a destination in another
   # session
   for( name, session ) in cfg.session.items():
      if name != sessionName and ( destIntf in session.srcIntf or
                                   destIntf in session.targetIntf ):
         return( False, 'Duplicate' )
      if name == sessionName and destIntf in session.srcIntf:
         return( False, 'Duplicate' )

   # Destination is invalid if its part of any Lag 
   if destIntf in lagConfig.phyIntf and \
          lagConfig.phyIntf[ destIntf ].lag:
      return( False, 'Interface mode conflict, %s is part of lag %s' % \
                  ( destIntf, lagConfig.phyIntf[ destIntf ].lag.name ) )
   return ( True, '' )

def emitNonExistentSessionMsg( name, mode ):
   errMsg = "Monitor session " + name + " does not exist."
   mode.addError( errMsg )

def setMirrorDestination( name, intf, mode, delete=False ):

   cfg = mirroringConfig
   sessionCfg = cfg.session.get( name )
   mirroringEnabled = lanzConfig.mirroringEnabled

   if not lanzEnabled( lanzConfig, lanzHwStatus ):
      mode.addWarning( "Lanz agent not running." )

   if delete:
      del lanzConfig.mirrorDestIntf[ intf ]
      if sessionCfg:
         sessionCfg.destIntf = ""
         targetIntf = sessionCfg.targetIntf.get( intf )
         if targetIntf:
            if lanzEnabled( lanzConfig, lanzHwStatus ) and mirroringEnabled:
               del sessionCfg.targetIntf[ intf ]
               if not sessionCfg.targetIntf:
                  del cfg.session[ name ]
   else:
      # display error if creating more than max sessions
      if isMaxSessionsReached( mode, name, mirroringHwCapability, mirroringConfig ):
         emitMaxSessionsReachedMsg( name, mode )
         return

      ( status, reason ) = isValidDestination( name, intf )
      if not status:
         mode.addWarning( 'Mirror destination %s was rejected ' 
                                'with reason %s' % ( intf, reason ) )
         return
      if not sessionCfg:
         if lanzEnabled( lanzConfig, lanzHwStatus ) and mirroringEnabled:
            sessionCfg = cfg.session.newMember( name )
            sessionCfg.isInternal = True
            sessionCfg.owner = "Lanz"

      # No co-existence with non-gre destinations allowed.
      # Note that mirroringEnabled could be False, then the session won't exist.
      if sessionCfg and sessionCfg.targetIntf:
         if next( iter( sessionCfg.targetIntf ), '' ).startswith( "MirrorTunnel" ):
            emitGreWithOtherIntfNotSupportedMsg( intf, mode )
            return

      if ( mirroringHwCapability.multiDestSupported or
           ( not mirroringHwCapability.initialized and
             not mode.session_.guardsEnabled() ) ):
         lanzConfig.mirrorDestIntf[ intf ] = True 
         if ( lanzEnabled( lanzConfig, lanzHwStatus ) and mirroringEnabled and not
              sessionCfg.targetIntf.get( intf ) ):
            sessionCfg.targetIntf.newMember( intf ) 
      else:
         lanzConfig.mirrorDestIntf.clear()
         lanzConfig.mirrorDestIntf[ intf ] = True 
         if lanzEnabled( lanzConfig, lanzHwStatus ) and mirroringEnabled:
            sessionCfg.targetIntf.clear()
            sessionCfg.targetIntf.newMember( intf )
            sessionCfg.destIntf = intf

def setMirrorDestinations( name, intfs, mode, delete=False ):
   cfg = mirroringConfig
   sessionCfg = cfg.session.get( name )
   if delete:
      if not sessionCfg and lanzConfig.mirroringEnabled:
         emitNonExistentSessionMsg( name, mode )
   else:
      # display error if creating more than max sessions
      if isMaxSessionsReached( mode, name, mirroringHwCapability, mirroringConfig ):
         emitMaxSessionsReachedMsg( name, mode )
         return

   for intf in intfs:
      setMirrorDestination( name, intf, mode, delete )

def deleteLanzMirrorDestinationGre():

   name = MirroringConstants.lanzMirrorSession
   cfg = mirroringConfig
   sessionCfg = cfg.session.get( name )

   if lanzConfig and lanzConfig.mirrorDestIntf:
      lanzConfig.greTunnelConfig = Tac.Value( "Lanz::GreTunnelConfig",
         '0.0.0.0', '0.0.0.0', defaultGreTtl, defaultGreDscp, DEFAULT_VRF,
         defaultGreHdrProto )

      for dstIntf in lanzConfig.mirrorDestIntf:
         if dstIntf.startswith( 'MirrorTunnel' ):
            del lanzConfig.mirrorDestIntf[ dstIntf ]

   if sessionCfg and sessionCfg.targetIntf:
      for tgtIntf in sessionCfg.targetIntf:
         if tgtIntf.startswith( 'MirrorTunnel' ):
            del sessionCfg.targetIntf[ tgtIntf ]

   if not sessionCfg:
      return

   sessionCfg.greTunnelKey = createV4ZeroGreTunnelKey()
   sessionCfg.destIntf = ""

   # if there are no target intfs, we remove the session
   if not sessionCfg.targetIntf:
      del cfg.session[ name ]

class DestinationGreCfg:
   def __init__( self, srcIp, dstIp, ttl, dscp, vrf, protocol ):
      self.srcIp = srcIp
      self.dstIp = dstIp
      self.ttl = ttl
      self.dscp = dscp
      self.vrf = vrf
      self.protocol = protocol

# Adds a new gre tunnel destination to a Lanz mirroring session.
def setLanzMirrorDestinationGre( mode, destinationGreCfg, delete=False ):

   name = MirroringConstants.lanzMirrorSession
   cfg = mirroringConfig
   sessionCfg = cfg.session.get( name )
   mirroringEnabled = lanzConfig.mirroringEnabled

   if delete:
      deleteLanzMirrorDestinationGre()
      return

   if not lanzEnabled( lanzConfig, lanzHwStatus ):
      mode.addWarning( "Lanz agent not running." )

   # Max sessions check (can create a new session)
   if isMaxSessionsReached( mode, name, mirroringHwCapability, mirroringConfig ):
      emitMaxSessionsReachedMsg( name, mode )
      return

   # No co-existence with non-gre destinations allowed
   if sessionCfg and sessionCfg.targetIntf:
      firstTargetIntf = next( iter( sessionCfg.targetIntf ), '' )
      if firstTargetIntf.startswith( "MirrorTunnel" ):
         emitGreWithOtherIntfNotSupportedMsg( firstTargetIntf, mode )
         return

   # This is hardcoded dummy name for the gre interface for _InternalLanz session.
   greIntfFixedName = "MirrorTunnel9996"

   # Saving/updating the configuration
   if destinationGreCfg:
      dcfg = destinationGreCfg
      lanzConfig.greTunnelConfig = Tac.Value( "Lanz::GreTunnelConfig",
         dcfg.srcIp, dcfg.dstIp, dcfg.ttl, dcfg.dscp, dcfg.vrf, dcfg.protocol )

      lanzConfig.mirrorDestIntf[ greIntfFixedName ] = True

   # Create Lanz session if does no exist but was enabled
   if lanzEnabled( lanzConfig, lanzHwStatus ) and mirroringEnabled:
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.isInternal = True
         sessionCfg.owner = "Lanz"

      dcfg = lanzConfig.greTunnelConfig
      sessionCfg.greTunnelKey = createGreTunnelKey(
                                   dcfg.srcIp, dcfg.dstIp, dcfg.ttl, dcfg.dscp,
                                   greHdrProto=dcfg.protocol, vrf=dcfg.vrf )

      sessionCfg.targetIntf.newMember( greIntfFixedName )

def emitGreWithOtherIntfNotSupportedMsg( intfName, mode ):
   errMsg = "Destination cannot be GRE tunnel and " + intfName
   mode.addError( errMsg )

def emitMaxSessionsReachedMsg( name, mode ):
   errMsg = "Could not create session '" + name + \
       "'. Maximum mirroring sessions limit reached"
   mode.addError( errMsg )

def noSessionCommand( name, mode ):
   cfg = mirroringConfig
   if cfg.session.get( name ):
      del cfg.session[ name ]

#-------------------------------------------------------------------------------
# [ no|default ] queue-monitor length
#-------------------------------------------------------------------------------
def queueMonitorLength( mode, forceEnabledValue=True ):
   try:
      lanzEnabledIs( lanzConfig, forceEnabledValue )
   except LanzLocked as e:
      mode.addError( str( e ) )
      return
   sessionCfg = mirroringConfig.session.get( MirroringConstants.lanzMirrorSession )
   if sessionCfg:
      sessionCfg.targetIntf.clear()
   intfs = lanzConfig.mirrorDestIntf
   for intf in intfs:
      if intf.startswith( 'MirrorTunnel' ):
         setLanzMirrorDestinationGre( mode, None, False )
      else:
         setMirrorDestination( MirroringConstants.lanzMirrorSession, intf, mode )

def noQueueMonitorLength( mode, forceEnabledValue=False ):
   try:
      lanzEnabledIs( lanzConfig, forceEnabledValue )
   except LanzLocked as e:
      mode.addError( str( e ) )
      return
   if mirroringConfig.session.get( MirroringConstants.lanzMirrorSession ):
      del mirroringConfig.session[ MirroringConstants.lanzMirrorSession ]

def defaultQueueMonitorLength( mode, args ):
   if lanzHwStatus.lanzEnabledByDefault:
      queueMonitorLength( mode, forceEnabledValue=None )
   else:
      noQueueMonitorLength( mode, forceEnabledValue=None )

#-------------------------------------------------------------------------------
# [no|default] queue-monitor length tx-latency
#-------------------------------------------------------------------------------
def qMonitorLengthTxLatency( mode ):
   lanzConfig.txLatencyEnabled = True

def noQMonitorLengthTxLatency( mode ):
   lanzConfig.txLatencyEnabled = False

#-------------------------------------------------------------------------------
# [no|default] queue-monitor length update-interval <interval>
#-------------------------------------------------------------------------------
def qMonitorLengthUpdateInterval( mode, qLenInterval ):
   if qLenInterval < int( lanzHwStatus.defaultQlenUpdateInterval * 1000000 ):
      mode.addWarning( 'Setting the update interval to less than 1 second may '
                       'cause loss of congestion messages' )
   lanzConfig.qLenInterval = qLenInterval

def noQMonitorLengthUpdateInterval( mode ):
   lanzConfig.qLenInterval = int( lanzHwStatus.defaultQlenUpdateInterval * 1000000 )

#-------------------------------------------------------------------------------
# [no|default] queue-monitor length mode notifying
#-------------------------------------------------------------------------------
def qMonitorLengthNotifying( mode ):
   lanzConfig.forceNotifying = True

def noQMonitorLengthNotifying( mode ):
   lanzConfig.forceNotifying = False

#-------------------------------------------------------------------------------
# [no|default] queue-monitor length log <interval>
#-------------------------------------------------------------------------------
def qMonitorLengthLogInterval( mode, logInterval ):
   lanzConfig.logInterval = logInterval

def noQMonitorLengthLogInterval( mode ):
   lanzConfig.logInterval = 0

#-------------------------------------------------------------------------------
# The "[ no | default ] queue-monitor length global-buffer" command
#-------------------------------------------------------------------------------
def qMonitorLengthGlobalBuffer( mode, args ):
   lanzConfig.globalBufferEnabled = True

def noQMonitorLengthGlobalBuffer( mode, args ):
   lanzConfig.globalBufferEnabled = False

#----------------------------------------------------------------------------------
# The "queue-monitor length global-buffer thresholds <2-max_threshold>
# <1-max_threshold>" command
#----------------------------------------------------------------------------------
def _globalBufferHighThresholdRangeFn( mode, context ):
   return ( 2, lanzHwStatus.totalSegments )

def _globalBufferLowThresholdRangeFn( mode, context ):
   return ( 1, lanzHwStatus.totalSegments ) 

def setThresholds( mode, highThreshold, lowThreshold ):
   if lowThreshold > highThreshold:
      mode.addError( "Invalid input. High threshold is less than low threshold" )
      return

   if highThreshold >  lanzHwStatus.totalSegments:
      mode.addError( "High threshold is larger than maximum threshold %d"
                     %( lanzHwStatus.totalSegments ) )
      return

   lanzConfig.globalThresholdsConfig = Tac.Value( 'Lanz::GlobalThresholdsConfig',
                  highThreshold=highThreshold, lowThreshold=lowThreshold,
                  valid=True )

def setDefaultThresholds( mode ):
   highThreshold = lanzHwStatus.defGlobalBufHighThreshold
   lowThreshold = lanzHwStatus.defGlobalBufLowThreshold
   lanzConfig.globalThresholdsConfig = Tac.Value( 'Lanz::GlobalThresholdsConfig',
                  highThreshold=highThreshold, lowThreshold=lowThreshold,
                  valid=False )

def _highThresholdRangeFn( mode, context ):
   if lanzHwStatus.initialized or mode.session_.guardsEnabled():
      return ( lanzHwStatus.minHighThreshold, lanzHwStatus.maxQueueLen )
   else:
      return ( QueueLength.min, QueueLength.max )

def _lowThresholdRangeFn( mode, context ):
   if lanzHwStatus.initialized or mode.session_.guardsEnabled():
      return ( lanzHwStatus.minLowThreshold, lanzHwStatus.maxQueueLen )
   else:
      return ( QueueLength.min, QueueLength.max )

def validThresholds( mode, highThreshold, lowThreshold ):
   if lowThreshold > highThreshold:
      mode.addError( "Invalid input. High threshold is less than low threshold" )
      return False

   if highThreshold - lowThreshold < lanzHwStatus.minThresholdDifference:
      unit = "bytes" if lanzHwStatus.queueLenUnit == QueueLenUnit.queueLenUnitBytes \
                     else "segments"
      mode.addError( "High and low thresholds should be atleast %d %s apart." \
                     % ( lanzHwStatus.minThresholdDifference, unit ) )
      return False
   return True

#--------------------------------------------------------------------------------
# The "[ no | default ] queue-monitor length default thresholds <2-max_threshold>
# <1-max_threshold>" command
#--------------------------------------------------------------------------------
def setEthernetThresholds( mode, highThreshold, lowThreshold ):
   if not validThresholds( mode, highThreshold, lowThreshold ):
      return

   lanzConfig.defaultThresholdsConfig = Tac.Value( 'Lanz::DefaultThresholdsConfig',
                                                   defHighThreshold=highThreshold,
                                                   defLowThreshold=lowThreshold,
                                                   isActive=True )

   # Determine if any interface has been set to an invalid threshold and if so, print
   # warning message. This is currently only applicable on Sand platforms because
   # interfaces with different speeds have different buffer sizes.
   platformName = getForwardingPlaneName( mode.sysname )
   if platformName == "Sand":
      inactiveIntfs = []
      intfList = Arnet.sortIntf( ethPhyIntfStatusDir.intfStatus )
      for intf in intfList:
         if not isEthernetIntf( intf ):
            continue
         speed = ethPhyIntfStatusDir.intfStatus[ intf ].speed

         maxQueueLen = lanzHwStatus.maxQueueLenPerSpeed[ speed ]
         if highThreshold > maxQueueLen or lowThreshold > maxQueueLen:
            inactiveIntfs.append( intf )

      if inactiveIntfs:
         inactiveThreshold = LanzWarningsLib.InactiveThreshold()
         warningMessage = inactiveThreshold.message

         # Combine the list of invalid interfaces with the warning message
         intfs = ", ".join( inactiveIntfs )
         print( warningMessage % intfs )

def setDefaultEthernetThresholds( mode, args ):
   lanzConfig.defaultThresholdsConfig = Tac.Value( 'Lanz::DefaultThresholdsConfig',
                  defHighThreshold=lanzHwStatus.defHighThreshold,
                  defLowThreshold=lanzHwStatus.defLowThreshold,
                  isActive=False )

#--------------------------------------------------------------------------------
# The "[ no | default ] queue-monitor length default threshold <2-max_threshold>"
# command
#--------------------------------------------------------------------------------
def setEthernetThreshold( mode, highThreshold ):
   lowThreshold = highThreshold // 2
   setEthernetThresholds( mode, highThreshold, lowThreshold )

#-------------------------------------------------------------------------------
# The "[no/default] queue-monitor length global-buffer log interval" command
#-------------------------------------------------------------------------------
def qMonitorLengthGlobalLogInterval( mode, logInterval ):
   lanzConfig.globalLogInterval = logInterval
def setDefaultGlobalLogInterval( mode ):
   lanzConfig.globalLogInterval = 0 

#-------------------------------------------------------------------------------
# queue-monitor length default thresholds <0-100> percent <0-100> percent
# queue-monitor length default threshold <0-100> percent
# no/default queue-monitor length default thresholds
# no/default queue-monitor length default threshold
#-------------------------------------------------------------------------------
def qMonitorLengthDefaultThresholdsPercent( mode, percentHighThreshold,
                                           percentLowThreshold ):
   lanzConfig.defaultThresholdsConfig = Tac.newInstance(
         "Lanz::DefaultThresholdsConfig" )
   percentageThresholdsConfig = Tac.newInstance(
         "Lanz::PercentageThresholdsConfig" )
   percentageThresholdsConfig.percentHighThreshold = percentHighThreshold
   percentageThresholdsConfig.percentLowThreshold = percentLowThreshold
   percentageThresholdsConfig.isActive = True
   lanzConfig.percentageThresholdsConfig = percentageThresholdsConfig

def qMonitorLengthDefaultThresholdPercent( mode, percentHighThreshold ):
   qMonitorLengthDefaultThresholdsPercent( mode, percentHighThreshold,
                                           percentHighThreshold // 2 )

def noQMonitorLengthDefaultThreshold( mode, args ):
   lanzConfig.defaultThresholdsConfig = Tac.newInstance(
         "Lanz::DefaultThresholdsConfig" )
   lanzConfig.percentageThresholdsConfig = Tac.newInstance(
         "Lanz::PercentageThresholdsConfig" )

# Temporary disabling for Berlin release
# pylint: disable-next=pointless-string-statement
'''
BasicCli.GlobalConfigMode.addCommand( ( tokenQMonitor, tokenLength,
                                        tokenDefault, tokenDefaultThresholds,
                                        tokenPercentHighThreshold, tokenPercent,
                                        tokenPercentLowThreshold, tokenPercent,
                                        qMonitorLengthDefaultThresholdsPercent ) )

BasicCli.GlobalConfigMode.addCommand( ( tokenQMonitor, tokenLength,
                                        tokenDefault, tokenDefaultThreshold,
                                        tokenPercentHighThreshold, tokenPercent,
                                        qMonitorLengthDefaultThresholdPercent ) )

BasicCli.GlobalConfigMode.addCommand( ( BasicCli.noOrDefault,
                                        tokenQMonitor, tokenLength,
                                        tokenDefault, tokenDefaultThreshold,
                                        BasicCli.trailingGarbage,
                                        noQMonitorLengthDefaultThreshold ) )

BasicCli.GlobalConfigMode.addCommand( ( BasicCli.noOrDefault,
                                        tokenQMonitor, tokenLength,
                                        tokenDefault, tokenDefaultThresholds,
                                        BasicCli.trailingGarbage,
                                        noQMonitorLengthDefaultThreshold ) )
'''

#-------------------------------------------------------------------------------
# The "[ no | default ] queue-monitor length mirror destination <interface>" command
#-------------------------------------------------------------------------------
def qMonitorLengthFrameMirroring( mode, args ):
   intfs = [ 'Cpu' ] if 'Cpu' in args else args.get( 'INTFS', () )
   if intfs:
      # queue-monitor length mirror destination <interface>
      if not lanzConfig.mirroringEnabled:
         mode.addWarning( 'LanzConfig Mirroring is disabled' )
      setMirrorDestinations( MirroringConstants.lanzMirrorSession, intfs,
                             mode=mode )
   else:
      # queue-monitor length mirror
      lanzConfig.mirroringEnabled = True
      for intf in lanzConfig.mirrorDestIntf:
         if intf.startswith( 'MirrorTunnel' ):
            setLanzMirrorDestinationGre( mode, None, False )
         else:
            setMirrorDestination( MirroringConstants.lanzMirrorSession, intf,
                                  mode, delete=False )

def noQMonitorLengthFrameMirroring( mode, intfs=(), deleteDestAll=False ):
   if not lanzConfig.mirroringEnabled:
      mode.addWarning( 'LanzConfig Mirroring is disabled' )

   if deleteDestAll:
      # no queue-monitor length destination
      for intf in lanzConfig.mirrorDestIntf:
         if intf.startswith( 'MirrorTunnel' ):
            setLanzMirrorDestinationGre( mode, None, True )
         else:
            setMirrorDestination( MirroringConstants.lanzMirrorSession, intf,
                                  mode, delete=True )
   else:
      # no queue-monitor length mirror destination <interface>
      if intfs:
         setMirrorDestinations( MirroringConstants.lanzMirrorSession, intfs, mode,
                                delete=True )
      else:
         # no queue-monitor length mirror
         noSessionCommand( MirroringConstants.lanzMirrorSession, mode )
         lanzConfig.mirroringEnabled = False

def qMonitorLengthDeleteMirrorDest( mode, intfs ):
   noQMonitorLengthFrameMirroring( mode, intfs=intfs, deleteDestAll=not intfs )

# ----------------------------------------------------------------------
# [no | default] queue-monitor length mirror destination tunnel mode gre
# source <A.B.C.D> destination <W.X.Y.Z> [ttl <1-255>]
#                                        [dscp <0-63>]
#                                        [vrf <vrf>]
#                                        [protocol <0x0-0xFFFF>]
# ----------------------------------------------------------------------
def qMonitorLengthMirrorGreDestination( mode, greSrcAddr, greDstAddr,
                                        greTunnelOpts, no=False ):

   ttl = defaultGreTtl
   dscp = defaultGreDscp
   vrf = DEFAULT_VRF
   protocol = defaultGreHdrProto

   for opt in greTunnelOpts:
      if opt[ 0 ] == 'ttl':
         ttl = opt[ 1 ]
      elif opt[ 0 ] == 'dscp':
         dscp = opt[ 1 ]
      elif opt[ 0 ] == 'protocol':
         protocol = opt[ 1 ]
      elif opt[ 0 ] == 'vrf':
         vrf = opt[ 1 ]

   if greSrcAddr == greDstAddr:
      errMsg = 'source and destination cannot be same'
      mode.addError( errMsg )
      return

   sIp = Arnet.IpAddress( greSrcAddr )
   if sIp.isMulticast or sIp.isLinkLocal or sIp.isBroadcast:
      errMsg = 'Invalid source IP address'
      mode.addError( errMsg )
      return

   dIp = Arnet.IpAddress( greDstAddr )
   if dIp.isMulticast or dIp.isLinkLocal or dIp.isBroadcast:
      errMsg = 'Invalid destination IP address'
      mode.addError( errMsg )
      return

   setLanzMirrorDestinationGre(
      mode, DestinationGreCfg( greSrcAddr, greDstAddr, ttl, dscp,
                               vrf, protocol ), False )

def noQMonitorLengthMirrorGreDestination( mode ):
   if not lanzConfig.mirroringEnabled:
      mode.addWarning( 'LanzConfig Mirroring is disabled' )
      return
   setLanzMirrorDestinationGre( mode, None, True )

#-------------------------------------------------------------------------------
# show queue-monitor length [interfaces <range>]
#                           [limit <1-1000000> samples|seconds]
#                           [drops|tx-latency]
#
#-------------------------------------------------------------------------------
def showQMonLength( mode, intfs, info, csv=None, dropsOrLatency=None,
                    isls=None, globalBuffer=None, showAll=None, 
                    showStatistics=None ):
   showQMonLen = QueueMonitorLength()
   showQMonLen.lanzEnabled = lanzEnabled( lanzConfig, lanzHwStatus )
   showQMonLen.platformName = getForwardingPlaneName( mode.sysname )
   showQMonLen.bytesPerTxmpSegment = lanzHwStatus.bytesPerTxmpSegment
   showQMonLen.reportTime = Tac.utcNow()
   # pylint: disable-next=protected-access
   showQMonLen._lanzRunning = isLanzAgentRunning( mode.sysname )

   if not showQMonLen._lanzRunning: # pylint: disable=protected-access
      mode.addWarning( "Lanz agent not running." )
      return showQMonLen

   if not showQMonLen.lanzEnabled:
      return showQMonLen

   cmd = ""
   
   if showStatistics:
      cmd += ' -c'
   
   if intfs:
      if intfs == 'cpu':
         intfShortNames = [ lanzHwStatus.cpuInterfaceRegex ]
      elif intfs == 'fabric':
         intfShortNames = [ lanzHwStatus.fabricInterfaceRegex ]
      else:
         intfShortNames = [ IntfMode.getShortname( intf ) for intf in intfs ]

      cmd += '-i'
      cmd += ','.join( intfShortNames )

   if isls:
      cmd += '-i'
      cmd += ','.join( isls )

   if info:
      if info.get( 'samplesOrSeconds' ) == 'samples':
         cmd += ' -s'
      else:
         cmd += ' -t'
      cmd += '%s' % (info.get( 'limit' ) )
   if dropsOrLatency == 'drops':
      cmd += ' -d'
   elif dropsOrLatency == 'tx-latency':
      cmd += ' -l'
   if globalBuffer:
      cmd += ' -b'
   if csv:
      cmd += ' csv '
   if showAll:
      cmd += ' -a'
   if not info and not csv:
      # default is 1000 samples
      cmd += ' -s1000'

   output = io.StringIO()
   AgentCommandRequest.runSocketCommand( mode.entityManager,
                                         "lanz", "ShowLanzCallback",
                                         command=cmd, timeout=90,
                                         stringBuff=output )
   # Get the report time
   output.seek( 0, 0 )
   # Discard first empty line
   line = output.readline()
   if line and 'Cli connection' not in line:
      line = output.readline()
      if line and 'Cli connection' not in line:
         m = re.match( "Report generated at (?P<timeStr>.*)", line )
         if m:
            showQMonLen.reportTime = float( m.group( 'timeStr' ) )

   def getCongestionRecords():
      while True:
         line = output.readline()
         if not line or 'Cli connection' in line:
            break
         if line == '\n':
            continue
         if 'Global Hit' in line:
            regex = r'\s*Global Hit Count = (?P<count>\d+)'
            m = re.match( regex, line )
            showQMonLen.globalHitCount = int( m.group( 'count' ) )
         else:
            newRecord = getCongestionRecord( showQMonLen.platformName,
                                             # pylint: disable-next=protected-access
                                             showQMonLen._commandType )
            if not newRecord:
               break
            newRecord.parseShowQMonLengthRecord( line[:-1], cmd )
            yield newRecord

      output.close()

   showQMonLen._commandType = cmd # pylint: disable=protected-access
   showQMonLen.entryList = getCongestionRecords()
   return showQMonLen

#-------------------------------------------------------------------------------
# The "clear queue-monitor length statistics" command
#-------------------------------------------------------------------------------
def clearQMonitorLengthStatistics( mode, args ):
   output = io.StringIO()
   lanzRunning = isLanzAgentRunning( mode.sysname )
   if not lanzRunning:
      mode.addWarning( "Lanz agent not running." )
      return
   AgentCommandRequest.runSocketCommand( em, "lanz",
                                         "ShowLanzCallback",
                                         command="clearStatistics",
                                         timeout=90, stringBuff=output )

#-------------------------------------------------------------------------------
# show queue-monitor length status
#-------------------------------------------------------------------------------
def showQMonLenStatus( mode, args ):
   platformName = getForwardingPlaneName( mode.sysname )
   # pylint: disable-next=redefined-outer-name
   showQMonLenStatus = QueueMonitorLengthStatus()
   showQMonLenStatus.platformName = platformName

   showQMonLenStatus.lanzEnabled = lanzEnabled( lanzConfig, lanzHwStatus )

   if platformName == 'Mako':
      # The context in Mako completely different from other platforms. The
      # common showQMonLenStatus CLI model doesn't make sense.

      # This is a hack to map Mako specific buffer thresholds to existing
      # variables in lanzHwStatus.
      # def[High/Low]Threshold - ingress
      # defFabric[High/Low]Threshold - egress
      # defInternal[High/Low]Threshold - core memory
      makoShowQMonLenStatusModel = MakoQueueMonitorLengthStatusModel()
      makoShowQMonLenStatusModel.lanzEnabled = lanzEnabled(
         lanzConfig, lanzHwStatus )
      makoShowQMonLenStatusModel.ingressHighThreshold = lanzHwStatus.defHighThreshold
      makoShowQMonLenStatusModel.ingressLowThreshold = lanzHwStatus.defLowThreshold
      # This is another hack to not introduce more platform specific shabang
      # in the lanzHwStatus. We use "speed" to map to each different buffer type
      # 10M - ingress
      # 100M - egress
      # 1G - core memory
      EthSpeed = Tac.Type( 'Interface::EthSpeed' )
      makoShowQMonLenStatusModel.ingressMemorySize = \
         lanzHwStatus.maxQueueLenPerSpeed[ EthSpeed.speed10Mbps ]

      makoShowQMonLenStatusModel.egressHighThreshold = \
         lanzHwStatus.defFabricHighThreshold
      makoShowQMonLenStatusModel.egressLowThreshold = \
         lanzHwStatus.defFabricLowThreshold
      makoShowQMonLenStatusModel.egressMemorySize = \
         lanzHwStatus.maxQueueLenPerSpeed[ EthSpeed.speed100Mbps ]

      # Not every profile supports coremem, we need to check whether it's
      # valid or not
      if lanzHwStatus.defInternalHighThreshold:
         makoShowQMonLenStatusModel.corememHighThreshold = \
            lanzHwStatus.defInternalHighThreshold
         makoShowQMonLenStatusModel.corememLowThreshold = \
            lanzHwStatus.defInternalLowThreshold
         makoShowQMonLenStatusModel.corememMemorySize = \
            lanzHwStatus.maxQueueLenPerSpeed[ EthSpeed.speed1Gbps ]
      else:
         makoShowQMonLenStatusModel.corememMemorySize = 0

      showQMonLenStatus.makoShowQMonLenStatusModel = makoShowQMonLenStatusModel

   showQMonLenStatus.txLatencyEnablingSupported = \
         lanzHwStatus.txLatencyEnablingSupported
   showQMonLenStatus.txLatencyEnabled = lanzConfig.txLatencyEnabled
   showQMonLenStatus.ptpSupported = lanzLatencyStatus.ptpTimeSyncSupported

   if not lanzHwStatus.lanzSliceModes:
      showQMonLenStatus.lanzSliceModes = None
   else:
      # lanzMode and lanzSliceModes are used only on Sand platform
      showQMonLenStatus.lanzMode = \
            "notifying" if lanzConfig.forceNotifying else "polling"
      for sliceName, lanzMode in lanzHwStatus.lanzSliceModes.items():
         # pylint: disable-next=unsupported-assignment-operation
         showQMonLenStatus.lanzSliceModes[ sliceName ] = lanzModeDesc[ lanzMode ]

   if not lanzLatencyStatus.capableLinecards:
      showQMonLenStatus.latencyCapableLinecards = None
   else:
      for sliceName in lanzLatencyStatus.capableLinecards:
         # pylint: disable-next=unsupported-assignment-operation
         showQMonLenStatus.latencyCapableLinecards[ sliceName ] = True

   showQMonLenStatus.packetSamplingEnabled = lanzConfig.mirroringEnabled
   cfg = mirroringConfig
   sessionCfg = cfg.session.get( MirroringConstants.lanzMirrorSession )
   if sessionCfg:
      for key in sessionCfg.targetIntf:
         showQMonLenStatus.mirrorDestIntf.append("%s" %key )

   if lanzHwStatus.queueLenUnit == QueueLenUnit.queueLenUnitSegments:
      showQMonLenStatus.queueLenUnit = "segments"
   else:
      showQMonLenStatus.queueLenUnit = "bytes"
   showQMonLenStatus.maxQueueLength = lanzHwStatus.maxQueueLen

   showQMonLenStatus.bytesPerRxmpSegment = lanzHwStatus.bytesPerRxmpSegment
   showQMonLenStatus.bytesPerTxmpSegment = lanzHwStatus.bytesPerTxmpSegment
   showQMonLenStatus.logInterval = lanzConfig.logInterval
   showQMonLenStatus.globalBufferSupported = lanzHwStatus.globalBufferSupported
   showQMonLenStatus.globalBufferEnabled = lanzConfig.globalBufferEnabled

   if lanzConfig.globalThresholdsConfig.valid:
      showQMonLenStatus.globalHighThreshold = \
          lanzConfig.globalThresholdsConfig.highThreshold
      showQMonLenStatus.globalLowThreshold = \
          lanzConfig.globalThresholdsConfig.lowThreshold
   else:
      showQMonLenStatus.globalHighThreshold = \
          lanzHwStatus.defGlobalBufHighThreshold
      showQMonLenStatus.globalLowThreshold = \
          lanzHwStatus.defGlobalBufLowThreshold

   showQMonLenStatus.numTotalSegments = lanzHwStatus.totalSegments
   showQMonLenStatus.globalLogInterval = lanzConfig.globalLogInterval

   showQMonLenStatus.lowThresholdSupported = lanzHwStatus.lowThresholdSupported

   # if on Sand, use the updateIntervalStatus for display
   if platformName == "Sand":
      showQMonLenStatus.updateIntervalSupported = \
         updateIntervalStatus.updateIntervalSupported
      showQMonLenStatus.qLenInterval = updateIntervalStatus.qLenInterval
   else:
      showQMonLenStatus.updateIntervalSupported = \
         lanzHwStatus.updateIntervalSupported
      showQMonLenStatus.qLenInterval = lanzConfig.qLenInterval

   if not updateIntervalStatus.updateIntervalSupportedCards:
      showQMonLenStatus.updateIntervalSupportedCards = None
   else:
      for fapType in updateIntervalStatus.updateIntervalSupportedCards:
         # pylint: disable-next=unsupported-assignment-operation
         showQMonLenStatus.updateIntervalSupportedCards[ fapType ] = True

   showQMonLenStatus.mirroringSupported = lanzHwStatus.mirroringSupported

   # populate intfStatus models
   intfList = []
   if lanzHwStatus.cpuPortMonitoringSupported:
      intfList += [ 'Cpu' ]
   if lanzHwStatus.fabricMonitoringSupported:
      intfList += [ 'Fabric' ]
   intfList += Arnet.sortIntf( ethPhyIntfStatusDir.intfStatus )

   if platformName == "Sand" and \
         any( elt.startswith( 'Switch' ) for elt in intfList ):
      # On Sand systems with Switch interfaces, Ethernet interfaces are not
      # controlled by the ASIC so it doesn't make sense to report their status.
      intfList = list( elt for elt in intfList if not elt.startswith( 'Ethernet' ) )

   lanzHelper = LanzHelper( lanzConfig.force(), lanzHwStatus,
                            ethPhyIntfStatusDir.force() )

   for key in intfList:
      if not ( isEthernetIntf( key ) or key == 'Cpu' or key == 'Fabric' ):
         continue
      intfStatusModel = QueueMonitorLengthStatus.InterfaceStatusModel()

      thresholdInfo = lanzHelper.interfaceThresholds( key )
      intfStatusModel.highThreshold = thresholdInfo.high
      intfStatusModel.lowThreshold = thresholdInfo.low
      intfStatusModel.lanzDisabled = thresholdInfo.disabled
      if key == 'Cpu':
         intfStatusModel.mirroringEnabled = lanzHwStatus.cpuPortMirroringSupported
      elif key == 'Fabric':
         intfStatusModel.mirroringEnabled = False
      else:
         intfStatusModel.mirroringEnabled = lanzHwStatus.ethPortMirroringSupported
      intfStatusModel.mirroringEnabled &= thresholdInfo.mirroringEnabled

      # Add interface warnings
      if isEthernetIntf( key ):
         platformName = getForwardingPlaneName( mode.sysname )
         # Inactive warnings are only applicable on Sand platforms as the max queue
         # length for each interface varies depending on the interface speed.
         if platformName == "Sand":
            speed = ethPhyIntfStatusDir.intfStatus[ key ].speed
            maxQueueLen = lanzHwStatus.maxQueueLenPerSpeed[ speed ]
            if( ( intfStatusModel.highThreshold > maxQueueLen ) or \
               ( intfStatusModel.lowThreshold > maxQueueLen ) ):
               inactiveWarning = intfStatusModel.InactiveWarningModel()
               inactiveWarning.maxQueueLength = maxQueueLen
               intfStatusModel.inactiveWarning = inactiveWarning

      showQMonLenStatus.intfStatusAll[ key ] = intfStatusModel

   return showQMonLenStatus

#-------------------------------------------------------------------------------
#show queue-monitor length global-buffer
#-------------------------------------------------------------------------------
def showQMonLenBuffer( mode ):
   return showQMonLength( mode, None, None, globalBuffer=True )

#-------------------------------------------------------------------------------
# show queue-monitor length all
#-------------------------------------------------------------------------------
def showQMonLenAll( mode ):
   return showQMonLength( mode, None, None, showAll=True )

#-------------------------------------------------------------------------------
# show queue-monitor length csv all
#-------------------------------------------------------------------------------
def showQMonLenCSVAll( mode ):
   return showQMonLength( mode, None, None, csv=True, showAll=True )

# Define queue-monitor-streaming CLI Mode
#
class LanzStreamingMode( CliMode.Lanz.LanzStreamingMode, BasicCli.ConfigModeBase ):
   name = "Queue-monitor streaming configuration"

   def __init__( self, parent, session ):
      CliMode.Lanz.LanzStreamingMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setMaxConnections( mode, args ):
      number = args[ 'MAXCONNECTIONS' ]
      lanzConfig.maxConnections = number

   def defaultMaxConnections( mode, args ):
      lanzConfig.maxConnections = 10

   def shutdown( mode, args ):
      lanzConfig.streaming = False
      # remove the default Vrf if automatically configured
      if not lanzConfig.explicitDefault:
         del lanzConfig.streamingVrf[ DEFAULT_VRF ]

   def noShutdown( mode, args ):
      if not lanzConfig.streamingVrf:
         lanzConfig.streamingVrf[ DEFAULT_VRF ] = True
      lanzConfig.streaming = True

   def setVrf( mode, args ):
      vrfName = args.get( 'VRF' )
      if vrfName == DEFAULT_VRF:
         lanzConfig.explicitDefault = True
      if not vrfExists( vrfName ):
         mode.addErrorAndStop( "VRF %s not configured." % vrfName )
      if vrfName in lanzConfig.streamingVrf:
         return
      if not toggleMultiVrfStreamingEnabled():
         lanzConfig.streamingVrf.clear()
      lanzConfig.streamingVrf[ vrfName ] = True
      # delete default VRF if set by system
      if not lanzConfig.explicitDefault:
         del lanzConfig.streamingVrf[ DEFAULT_VRF ]

   def setDefaultVrf( mode, args ):
      if vrf := args.get( 'VRF' ):
         deleteVrf = True
         if vrf == DEFAULT_VRF:
            lanzConfig.explicitDefault = False
         # Don't delete the default VRF if it's the only one left
         if DEFAULT_VRF in lanzConfig.streamingVrf and \
            len( lanzConfig.streamingVrf ) == 1:
            deleteVrf = False
            mode.addWarning( "No other VRFs available, not removing" )

         if deleteVrf:
            del lanzConfig.streamingVrf[ vrf ]

         if len( lanzConfig.streamingVrf ) < 1:
            lanzConfig.streamingVrf[ DEFAULT_VRF ] = True
      else:
         # If no vrf specified, delete all but default
         vrfs = ( vrf for vrf in lanzConfig.streamingVrf if vrf != DEFAULT_VRF )
         # Initialize default VRF here
         lanzConfig.streamingVrf[ DEFAULT_VRF ] = True
         lanzConfig.explicitDefault = False

         for vrf in vrfs:
            del lanzConfig.streamingVrf[ vrf ]

def aclNameIs( mode, aclType, aclName ):
   AclCliLib.setServiceAcl( mode, serviceName, IPPROTO_TCP,
                            aclConfig, aclCpConfig, aclName, aclType,
                            port=[ AclLib.getServByName( IPPROTO_TCP,
                                                         serviceName ) ] )

def aclNameDel( mode, aclType ):
   AclCliLib.noServiceAcl( mode, serviceName, aclConfig, aclCpConfig, None, aclType )

def ipAclNameIs( mode, args ):
   aclName = args[ 'ACL' ]
   aclNameIs( mode, 'ip', aclName )

def ipv6AclNameIs( mode, args ):
   aclName = args[ 'ACL' ]
   aclNameIs( mode, 'ipv6', aclName )

def ipAclNameDel( mode, args ):
   aclNameDel( mode, 'ip' )

def ipv6AclNameDel( mode, args ):
   aclNameDel( mode, 'ipv6' )

#-------------------------------------------------------------------------------
# [no] queue-monitor streaming
#-------------------------------------------------------------------------------
def gotoStreamingMode( mode, args ):
   childMode = mode.childMode( LanzStreamingMode )
   mode.session_.gotoChildMode( childMode )

def defaultConfig( mode, args ):
   childMode = mode.childMode( LanzStreamingMode )
   lanzConfig.streaming = False
   lanzConfig.maxConnections = 10
   aclNameDel( childMode, 'ip' )
   aclNameDel( childMode, 'ipv6' )
   childMode.setDefaultVrf( args )

#-------------------------------------------------------------------------------
# show queue-monitor streaming ip access-list [<acl-name>] [summary]
# show queue-monitor streaming ipv6 access-list [<acl-name>] [summary]
#-------------------------------------------------------------------------------
def showQMonStreamingAcl( mode, aclName, aclType ):
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 aclName,
                                 supressVrf=True,
                                 serviceName=serviceName )

#-------------------------------------------------------------------------------
# clear queue-monitor streaming counters ip access-list
# clear queue-monitor streaming counters ipv6 access-list
#-------------------------------------------------------------------------------
def clearQMonStreamingAclCounters( mode, aclType ):
   AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

def clearQMonStreamingIpAclCounters( mode ):
   clearQMonStreamingAclCounters( mode, 'ip' )

def clearQMonStreamingIpv6AclCounters( mode ):
   clearQMonStreamingAclCounters( mode, 'ipv6' )

#-------------------------------------------------------------------------------
# show queue-monitor streaming clients
#-------------------------------------------------------------------------------
def showQMonStreamingClients( mode, args ):
   model = QueueMonitorStreamingClients( _vrf=args.get( 'VRF' ) )

   model.lanzEnabled = lanzEnabled( lanzConfig, lanzHwStatus )
   model.streamingEnabled = lanzConfig.streaming
   model.clientConnections = streamingStatus.countTotalClientConnections()
   # show queue-monitor streaming clients
   allHostnames = ListOfHostnames()
   # allHostnames in the unfiltered case is a list of all hostnames for every vrf
   if 'VRF' in args:
      vrfName = args.get( 'VRF' )
      # Check for vrf or vrf's existing clients
      if not vrfExists( vrfName ):
         mode.addErrorAndStop( "VRF %s not configured." % vrfName )
      if vrfName not in lanzConfig.streamingVrf:
         mode.addErrorAndStop( "VRF %s not configured for streaming." % vrfName )
      if vrfName not in streamingStatus.vrfClientHostname:
         return model
      clientsByVrf = streamingStatus.vrfClientHostname[ vrfName ]
      allHostnames.hostnames.extend( clientsByVrf.clientHostname.values() )
      model.clientHostnames[ vrfName ] = allHostnames
      model.clientConnections = len( clientsByVrf.clientHostname.values() )
   else:
      for vrfName, clientsByVrf in streamingStatus.vrfClientHostname.items():
         hostnames = list( clientsByVrf.clientHostname.values() )
         model.clientHostnames[ vrfName ] = ListOfHostnames( hostnames=hostnames )
   return model

# Have the Cli Agent mount all needed state from sysdb
def Plugin( entityManager ):
   global lanzConfig
   global lanzLatencyStatus
   global em
   global ethPhyIntfStatusDir
   global lanzHwStatus
   global mirroringConfig
   global mirroringHwCapability
   global lagConfig
   global streamingStatus
   global updateIntervalStatus
   global aclConfig
   global aclCpConfig
   global aclStatus
   global aclCheckpoint
   global hwEpochStatus

   em = entityManager
   lanzConfig = ConfigMount.mount( entityManager, 'lanz/config', 'Lanz::Config',
                                   'w' )
   mirroringConfig = ConfigMount.mount( entityManager, "mirroring/config",
                                      "Mirroring::Config", "w" )
   lanzHwStatus = entityManager.mount( 'lanz/hardware/status',
                                       'Lanz::Hardware::Status', 'r' )
   streamingStatus = entityManager.mount( 'lanz/streaming/status',
                                          'Lanz::StreamingStatus', 'r' )
   updateIntervalStatus = entityManager.mount( 'lanz/updateinterval/status',
                                               'Lanz::UpdateIntervalStatus', 'r' )
   mirroringHwCapability = LazyMount.mount( entityManager, "mirroring/hwcapability",
                                            "Mirroring::HwCapability", "r" )
   lagConfig = LazyMount.mount( entityManager, "lag/config",
                                "Lag::Config", "r" )
   LazyMount.mount( entityManager,
                      'interface/status/eth/phy/slice',
                      'Tac::Dir', 'ri' )
   ethPhyIntfStatusDir = LazyMount.mount( entityManager,
                                          'interface/status/eth/phy/all',
                                          'Interface::AllEthPhyIntfStatusDir', 'r' )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   hwEpochStatus = LazyMount.mount( entityManager, "hwEpoch/status",
                                    "HwEpoch::Status", "r" )
   lanzLatencyStatus = LazyMount.mount( entityManager, 'hardware/lanz/latency',
                                        "Lanz::LatencyStatus", 'r' )

#################################################################################
#
# PLEASE KEEP 'show tech-support' registration AT BOTTOM of this file, and INSERT
# new commands ABOVE here.
#
# Register 'show queue-monitor length' with 'show tech-support'.
#
#################################################################################

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-08-01 12:11:29',
   cmds=[ 'show queue-monitor length' ],
   cmdsGuard=lambda: lanzEnabled( lanzConfig, lanzHwStatus ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2022-01-26 09:03:48',
   cmds=[ 'show queue-monitor length tx-latency' ],
   cmdsGuard=lambda: lanzTxLatencyEnabled(
      lanzConfig, lanzHwStatus, lanzLatencyStatus ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2022-01-26 09:03:48',
   cmds=[ 'show queue-monitor length status' ],
   cmdsGuard=lambda: lanzSupported( lanzHwStatus ) )
