# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliParser
from CliPlugin.IntfCli import (
   IntfConfigMode,
   IntfDependentBase,
   Intf,
)
import CliPlugin.LdpCli as LdpCli
from CliPlugin.MplsCli import mplsIntfModeKw
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import Tracing

th = Tracing.Handle( "LdpIntfConfigMode" )
t0 = th.trace0

ldpConfigColl = None

class MplsLdpIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( 'Ethernet' ) or
               mode.intf.name.startswith( 'Po' ) or
               mode.intf.hardware() == 'loopback' or
               mode.intf.hardware() == 'vlan' or
               mode.intf.hardware() == 'test' or
               mode.intf.hardware() == 'switch' or
               mode.intf.isSubIntf() )

# -----------------------------------------------------------------------------------
# [no|default] mpls ldp igp sync
# -----------------------------------------------------------------------------------
class MplsLdpIgpSyncCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp igp sync'
   noOrDefaultSyntax = 'mpls ldp igp sync ...'
   data = {
      'mpls': mplsIntfModeKw,
      'ldp': LdpCli.ldpKw,
      'igp': 'IGP configuration',
      'sync': 'IGP sync configuration',
   }

   @staticmethod
   def _setIntfLdpIgpSync( mode, flag ):
      vrfName = DEFAULT_VRF # support default vrf for now
      ldpConfig = ldpConfigColl.config.get( vrfName ) # Ldp::Config
      intfConfigColl = ldpConfig.intfConfigColl # Ldp::IntfConfigColl
      intfConfig = intfConfigColl.get( mode.intf.name, None )
      if flag == 'useGlobal':
         if intfConfig is not None:
            # delete interface-specific entry
            del intfConfigColl[ mode.intf.name ]
         return
      if intfConfig is None:
         intfConfig = intfConfigColl.newMember( mode.intf.name )
      elif intfConfig.igpSync == flag:
         return
      intfConfig.igpSync = flag

   @staticmethod
   def handler( mode, args ):
      MplsLdpIgpSyncCmd._setIntfLdpIgpSync( mode, "on" )

   @staticmethod
   def noHandler( mode, args ):
      MplsLdpIgpSyncCmd._setIntfLdpIgpSync( mode, "off" )

   @staticmethod
   def defaultHandler( mode, args ):
      MplsLdpIgpSyncCmd._setIntfLdpIgpSync( mode, "useGlobal" )

MplsLdpIntfModelet.addCommandClass( MplsLdpIgpSyncCmd )

# -----------------------------------------------------------------------------------
# [no|default] mpls ldp interface
# -----------------------------------------------------------------------------------
class MplsLdpInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp interface'
   noOrDefaultSyntax = 'mpls ldp interface ...'
   data = {
      'mpls': mplsIntfModeKw,
      'ldp': LdpCli.ldpKw,
      'interface': 'Enable LDP on this interface',
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = DEFAULT_VRF # support default vrf for now
      ldpConfig = ldpConfigColl.config.get( vrfName ) # Ldp::Config
      ldpConfig.ldpEnabledIntf.add( mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = DEFAULT_VRF # support default vrf for now
      ldpConfig = ldpConfigColl.config.get( vrfName ) # Ldp::Config
      ldpConfig.ldpEnabledIntf.remove( mode.intf.name )

MplsLdpIntfModelet.addCommandClass( MplsLdpInterfaceCmd )

# -------------------------------------------------------------------------------
# Add our modelet to the interface
# -------------------------------------------------------------------------------
IntfConfigMode.addModelet( MplsLdpIntfModelet )

# -------------------------------------------------------------------------------
# Register a dependent class for all interfaces to cleanup on Intf deletion
# -------------------------------------------------------------------------------
class MplsLdpInterfaceRemover( IntfDependentBase ):
   def setDefault( self ):
      vrfName = DEFAULT_VRF
      ldpConfig = ldpConfigColl.config.get( vrfName )
      if not ldpConfig:
         t0( "No Ldp::Config entry for VRF", vrfName )
         return
      intfName = self.intf_.name
      t0( "Intf deletion. Cleanup", intfName )
      ldpConfig.ldpEnabledIntf.remove( intfName )

def Plugin( entityManager ):
   global ldpConfigColl

   ldpConfigColl = ConfigMount.mount( entityManager, "mpls/ldp/ldpConfigColl",
                                      "Ldp::LdpConfigColl", "w" )
   Intf.registerDependentClass( MplsLdpInterfaceRemover )
