#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
import CliParser
from CliMode.LdpMode import (
      MldpMode,
      )
from CliPlugin.AclCli import (
      accessGroupKwMatcher,
      inKwMatcher,
      ipAclNameMatcher,
      ipKwForServiceAclMatcher,
)
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.LdpCli as LdpCli
import CliPlugin.MplsCli as MplsCli
import CliPlugin.RcfCliLib as RcfCliLib
import CliPlugin.RouteMapCli as RouteMapCli
import Intf
from IntfRangePlugin.EthIntf import EthPhyAutoIntfType
from IntfRangePlugin.LagIntf import LagAutoIntfType
from IntfRangePlugin.LoopbackIntf import LoopbackAutoIntfType
from IntfRangePlugin.VlanIntf import VlanAutoIntfType
import LazyMount
import ReversibleSecretCli
import Tac
from TypeFuture import TacLazyType

# pkgdeps: library Arnet
# pkgdeps: library LdpLibSysdbTypes
# pkgdeps: library MplsSysdbTypes

FwdEqvClass = TacLazyType( 'Mpls::FwdEqvClass' )
IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
LdpProtoParam = TacLazyType( 'Ldp::LdpProtoParam' )
LdpP2mpFecGenericOpaqueId = TacLazyType( 'Ldp::LdpP2mpFecGenericOpaqueId' )

matcherHello = CliMatcher.KeywordMatcher( 'hello',
      helpdesc='Ldp link hello parameters' )
matcherHoldTime = CliMatcher.KeywordMatcher( 'hold-time',
      helpdesc='Max time allowed between hello msgs' )
matcherInfinite = CliMatcher.KeywordMatcher( 'infinite',
      helpdesc='Infinite hold time' )
matcherTargetedHello = CliMatcher.KeywordMatcher( 'targeted-hello',
      helpdesc='Ldp targeted hello parameters' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Network interface' )
matcherFec = CliMatcher.KeywordMatcher( 'fec',
      helpdesc='LDP FEC configuration' )
matcherFilter = CliMatcher.KeywordMatcher( 'filter',
      helpdesc='Filter for FECs' )
matcherDiscovery = CliMatcher.KeywordMatcher( 'discovery',
      helpdesc='Discovered LDP adjacencies' )
matcherLdpNeighbor = CliMatcher.KeywordMatcher( 'neighbor',
      helpdesc='LDP Neighbor configuration' )
matcherActive = CliMatcher.KeywordMatcher( 'active',
      helpdesc='Activate password index' )
matcherAuth = CliMatcher.KeywordMatcher( 'authentication',
      helpdesc='LDP authentication configuration' )
matcherIndex = CliMatcher.KeywordMatcher( 'index',
      helpdesc='LDP password index' )
matcherPassword = CliMatcher.KeywordMatcher( 'password',
      helpdesc='LDP authentication password' )
matcherAuthIndexInteger = CliMatcher.IntegerMatcher( 1, 2**32 - 1,
      helpdesc='Password index' )

mplsHwCapability = None

intfTypes = ( EthPhyAutoIntfType, LagAutoIntfType,
              LoopbackAutoIntfType, VlanAutoIntfType )

def entropyLabelSupportedGuard( mode, token ):
   if mplsHwCapability.mplsEntropyLabelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def clearLdpPassword( mode ):
   mode.enableLdpPassword( None, True )

def clearAuthenticationIndex( mode ):
   mode.authIndexPasswordIs( enable=False )
   mode.authIndexActiveIs( enable=False )

def clearNeighbor( mode ):
   neighborTargetedAllDefault( mode )
   mode.neighAuthIndexActiveIs( enable=False )

def clearMldp( mode, no=None ):
   mLdpMode = mode.childMode( MldpConfigMode )
   mLdpMode.shutdownMldpIs( no )
   mLdpMode.lspStaticP2mpFecBindingCollDelAll()

def setLinkReadyTimeout( mode, timeout ):
   mode.ldpLinkReadyTimeout( timeout )

def clearLinkReadyTimeout( mode ):
   mode.ldpLinkReadyTimeout( 0 )

def defaultLinkReadyTimeout( mode ):
   mode.ldpLinkReadyTimeout( -1 )

def setLinkReadyDelay( mode, delay ):
   mode.ldpLinkReadyDelay( delay )

def clearLinkReadyDelay( mode ):
   mode.ldpLinkReadyDelay( 0 )

def defaultLinkReadyDelay( mode ):
   mode.ldpLinkReadyDelay( -1 )

def shutdownLdp( mode, no=None ):
   mode.shutdownIs( no )

def entropyLabel( mode, args=None ):
   mode.entropyLabelIs( args )

def protoParam( mode, param=0 ):
   mode.protoParamIs( param )

def protoParamDefault( mode ):
   mode.protoParamIs( 0 )

def routerIdDefault( mode, force ):
   mode.routerIdIs( True, '0.0.0.0', '',
                    force if mode.session.isInteractive() else True )

def routerIdIpAddr( mode, ipAddr, force ):
   mode.routerIdIs( None, ipAddr, '',
                    force if mode.session.isInteractive() else True )

def routerIdIntfId( mode, intf, force ):
   mode.routerIdIs( None, '0.0.0.0', intf,
                    force if mode.session.isInteractive() else True )

def transportAddrDefault( mode, args=None ):
   mode.transportAddrIntfIdIs( True, '' )

def advDsplModeIs( mode, advDsplMode ):
   mode.advDsplModeIs( advDsplMode )

def neighborTargetedIp( mode, ipAddr ):
   mode.targetedIs( ipAddr )

def neighborTargetedDefault( mode, ipAddr ):
   mode.targetedDel( ipAddr )

def neighborTargetedAllDefault( mode ):
   mode.targetedDelAll()

def pseudowireAgentDel( mode ):
   mode.pseudowireAgentDel()

def fecFilterPrefixListName( mode, listName ):
   mode.fecFilterPrefixListNameIs( listName )

def fecFilterDefault( mode ):
   mode.fecFilterPrefixListNameDel()

# Link-hello
def helloHoldTime( mode, seconds ):
   mode.helloHoldTimeIs( seconds )

def helloHoldTimeInfinite( mode ):
   mode.helloHoldTimeIs( LdpProtoParam.infiniteHelloHoldTime )

def clearHelloHoldTime( mode ):
   mode.helloHoldTimeIs( LdpProtoParam.defaultBasicHelloHoldTime )

def helloInterval( mode, seconds ):
   mode.helloIntervalIs( seconds )

def clearHelloInterval( mode ):
   mode.helloIntervalIs( LdpProtoParam.defaultBasicHelloInterval )

# Targeted-hello
def targetedHelloHoldTime( mode, seconds ):
   mode.targetedHelloHoldTimeIs( seconds )

def targetedHelloHoldTimeInfinite( mode ):
   mode.targetedHelloHoldTimeIs( LdpProtoParam.infiniteHelloHoldTime )

def clearTargetedHelloHoldTime( mode ):
   mode.targetedHelloHoldTimeIs( LdpProtoParam.defaultTargetHelloHoldTime )

def targetedHelloInterval( mode, seconds ):
   mode.targetedHelloIntervalIs( seconds )

def clearTargetedHelloInterval( mode ):
   mode.targetedHelloIntervalIs( LdpProtoParam.defaultTargetHelloInterval )

def aclNameIs( mode, aclName ):
   mode.aclNameIs( aclName )

def aclNameDel( mode ):
   mode.aclNameDel()

def ldpIntfsDisabled( mode, no=None ):
   mode.ldpIntfsDisabledIs( no )

def clearSrcProtoBgpV4Lu( mode ):
   mode.allowLuTunnelRedistIs( False )

# --- TEST CLI ---
def ignoreHwUnprogrammedDefault( mode ):
   mode.ignoreHwUnprogrammedIs( False )

class LdpClearConfig:
   '''
   The purpose of this class and the map it contains it to allow easier code
   introspection from the tests, map keys are ignored here but used by the tests
   to determine wether all possible "no" completions have an associated cleaning
   handler
   '''
   def __init__( self, clearConfigHandlersMap ):
      # dictionary containing cli configuration to be cleaned
      self.clearConfigHandlersMap = clearConfigHandlersMap

   def __call__( self, mode, *args ):
      '''
      This allows this class to act as a kind of "functor" to be able to treat
      leaves cleanup handler functions the same way as the ones containing maps
      '''
      for cleanupList in self.clearConfigHandlersMap.values():
         if cleanupList is not None:
            clearConfigHandler, args = cleanupList[ 0 ], cleanupList[ 1 : ]
            clearConfigHandler( mode, *args )

clearIgpSync = LdpClearConfig( {
                 'delay': [ defaultLinkReadyDelay ],
                 'holddown': [ defaultLinkReadyTimeout ],
               } )

clearIgp = LdpClearConfig( { 'sync': [ clearIgpSync ] } )

clearDiscoveryHello = LdpClearConfig( {
                        'hold-time': [ clearHelloHoldTime ],
                        'interval': [ clearHelloInterval ],
                      } )

clearDiscoveryTargetedHello = LdpClearConfig( {
                                'hold-time': [ clearTargetedHelloHoldTime ],
                                'interval': [ clearTargetedHelloInterval ],
                              } )

clearDiscovery = LdpClearConfig( {
                   'hello': [ clearDiscoveryHello ],
                   'targeted-hello': [ clearDiscoveryTargetedHello ],
                 } )

clearPseudowireAgent = LdpClearConfig( { 'pseudowires': [ pseudowireAgentDel ] } )

clearIp = LdpClearConfig( { 'access-group': [ aclNameDel ] } )

clearGrRole = LdpClearConfig(
   { 'helper': [ lambda mode : mode.noLdpGrHelperMode() ],
     'speaker': [ lambda mode : mode.noLdpGrSpeakerMode() ] } )
clearGr = LdpClearConfig( { 'role': [ clearGrRole ] } )

clearSrcProtoBgpV4 = LdpClearConfig(
      { 'labeled-unicast': [ clearSrcProtoBgpV4Lu ] } )
clearSrcProtoBgp = LdpClearConfig( { 'ipv4': [ clearSrcProtoBgpV4 ] } )
clearSrcProto = LdpClearConfig( { 'bgp': [ clearSrcProtoBgp ] } )
clearTunnel = LdpClearConfig( { 'source-protocol': [ clearSrcProto ] } )
clearAuthentication = LdpClearConfig( { 'index': [ clearAuthenticationIndex ] } )

clearMplsLdp = LdpClearConfig( {
                 'shutdown': None, # executed first below
                 'router-id': [ routerIdDefault, True ],
                 'transport-address': [ transportAddrDefault ],
                 'password': [ clearLdpPassword ],
                 'authentication': [ clearAuthentication ],
                 'igp': [ clearIgp ],
                 'fec': [ fecFilterDefault ],
                 'discovery': [ clearDiscovery ],
                 'neighbor': [ clearNeighbor ],
                 'pseudowires': [ clearPseudowireAgent ],
                 'proto-param': [ protoParamDefault ],
                 'ip': [ clearIp ],
                 'mldp': [ clearMldp ],
                 'graceful-restart': [ clearGr ],
                 'interface': [ ldpIntfsDisabled, 'default' ],
                 'label': [ LdpCli.LdpConfigMode.ldpLabelLocalTermination,
                             'implicit-null' ],
                 'entropy-label': [ entropyLabel, CliCommand.NAME_DEFAULT ],
                 'tunnel': [ clearTunnel ],
                 # --- TEST CLI ---
                 'ignore-hw-unprogrammed': [ ignoreHwUnprogrammedDefault ],
               } )

# --------------------------------------------------------------------------------
# [ no | default ] mpls ldp
# --------------------------------------------------------------------------------
class MplsLdpCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': MplsCli.mplsNodeForConfig,
      'ldp': LdpCli.ldpKw,
   }

   @staticmethod
   def handler( mode, args ):
      mplsLdpMode = mode.childMode( LdpCli.LdpConfigMode )
      mode.session_.gotoChildMode( mplsLdpMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsLdpMode = mode.childMode( LdpCli.LdpConfigMode )
      # shutdown should be done first to take down the agent
      shutdownLdp( mplsLdpMode )
      clearMplsLdp( mplsLdpMode )

BasicCli.GlobalConfigMode.addCommandClass( MplsLdpCmd )

# --------------------------------------------------------------------------------
# [ no | default ] discovery hello hold-time ( HOLD_TIME | infinite )
# --------------------------------------------------------------------------------
class DiscoveryHelloHoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'discovery hello hold-time ( HOLD_TIME | infinite )'
   noOrDefaultSyntax = 'discovery hello hold-time ...'
   data = {
      'discovery': matcherDiscovery,
      'hello': matcherHello,
      'hold-time': matcherHoldTime,
      'HOLD_TIME': CliMatcher.IntegerMatcher( 1, 65534,
         helpdesc='Link hello hold time in seconds' ),
      'infinite': matcherInfinite,
   }

   @staticmethod
   def handler( mode, args ):
      if 'infinite' in args:
         helloHoldTimeInfinite( mode )
      else:
         helloHoldTime( mode, args[ 'HOLD_TIME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return clearHelloHoldTime( mode )

LdpCli.LdpConfigMode.addCommandClass( DiscoveryHelloHoldTimeCmd )

# --------------------------------------------------------------------------------
# [ no | default ] discovery hello interval INTERVAL
# --------------------------------------------------------------------------------
class DiscoveryHelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'discovery hello interval INTERVAL'
   noOrDefaultSyntax = 'discovery hello interval ...'
   data = {
      'discovery': matcherDiscovery,
      'hello': matcherHello,
      'interval': 'Time between hello transmitions',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Link hello interval in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      return helloInterval( mode, args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return clearHelloInterval( mode )

LdpCli.LdpConfigMode.addCommandClass( DiscoveryHelloIntervalCmd )

# --------------------------------------------------------------------------------
# [ no | default ] discovery targeted-hello hold-time ( HOLD_TIME | infinite )
# --------------------------------------------------------------------------------
class DiscoveryTargetedHelloHoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'discovery targeted-hello hold-time ( HOLD_TIME | infinite )'
   noOrDefaultSyntax = 'discovery targeted-hello hold-time ...'
   data = {
      'discovery': matcherDiscovery,
      'targeted-hello': 'Ldp targeted hello parameters',
      'hold-time': matcherHoldTime,
      'HOLD_TIME': CliMatcher.IntegerMatcher( 1, 65534,
         helpdesc='Targeted hold time in seconds' ),
      'infinite': matcherInfinite,
   }

   @staticmethod
   def handler( mode, args ):
      if 'infinite' in args:
         targetedHelloHoldTimeInfinite( mode )
      else:
         targetedHelloHoldTime( mode, args[ 'HOLD_TIME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return clearTargetedHelloHoldTime( mode )

LdpCli.LdpConfigMode.addCommandClass( DiscoveryTargetedHelloHoldTimeCmd )

# --------------------------------------------------------------------------------
# [ no | default ] discovery targeted-hello interval INTERVAL
# --------------------------------------------------------------------------------
class DiscoveryTargetedHelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'discovery targeted-hello interval INTERVAL'
   noOrDefaultSyntax = 'discovery targeted-hello interval ...'
   data = {
      'discovery': matcherDiscovery,
      'targeted-hello': matcherTargetedHello,
      'interval': 'Time between hello transmitions',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Link hello interval in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      return targetedHelloInterval( mode, args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return clearTargetedHelloInterval( mode )

LdpCli.LdpConfigMode.addCommandClass( DiscoveryTargetedHelloIntervalCmd )

# --------------------------------------------------------------------------------
# [ no | default ] igp sync delay [ DELAY ]
# --------------------------------------------------------------------------------
class IgpSyncDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'igp sync delay [ DELAY ]'
   noOrDefaultSyntax = 'igp sync delay ...'
   data = {
      'igp': 'IGP configuration',
      'sync': LdpCli.syncForShowKw,
      'delay': 'Time to wait before notifying IGP of link readiness',
      'DELAY': CliMatcher.IntegerMatcher( 1, 60, helpdesc='Time in seconds.' ),
   }

   @staticmethod
   def handler( mode, args ):
      return setLinkReadyDelay( mode, args.get( 'DELAY' ) )

   @staticmethod
   def noHandler( mode, args ):
      return clearLinkReadyDelay( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      return defaultLinkReadyDelay( mode )

LdpCli.LdpConfigMode.addCommandClass( IgpSyncDelayCmd )

# --------------------------------------------------------------------------------
# [ no | default ] igp sync holddown [ until-established | TIMEOUT ]
# --------------------------------------------------------------------------------
class IgpSyncHolddownCmd( CliCommand.CliCommandClass ):
   syntax = 'igp sync holddown [ until-established | TIMEOUT ]'
   noOrDefaultSyntax = 'igp sync holddown ...'
   data = {
      'igp': 'IGP configuration',
      'sync': LdpCli.syncForShowKw,
      'holddown': ( 'Time to wait before forcing an interface ready for mpls '
                     'forwarding' ),
      'until-established': 'Wait until session is established',
      'TIMEOUT': CliMatcher.IntegerMatcher( 1, 2**32 - 1,
         helpdesc='Number of seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      return setLinkReadyTimeout( mode, args.get( 'TIMEOUT', Tac.endOfTime ) )

   @staticmethod
   def noHandler( mode, args ):
      return clearLinkReadyTimeout( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      return defaultLinkReadyTimeout( mode )

LdpCli.LdpConfigMode.addCommandClass( IgpSyncHolddownCmd )

# --------------------------------------------------------------------------------
# [ no | default ] interface disabled default
# --------------------------------------------------------------------------------
class InterfaceDisabledDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'interface disabled default'
   noOrDefaultSyntax = syntax
   data = {
      'interface': 'LDP interface configuration',
      'disabled': 'Disable LDP on interfaces',
      'default': 'Default LDP setting for interfaces',
   }

   @staticmethod
   def handler( mode, args ):
      return ldpIntfsDisabled( mode, no=None )

   @staticmethod
   def noHandler( mode, args ):
      return ldpIntfsDisabled( mode, no=True )

   @staticmethod
   def defaultHandler( mode, args ):
      return ldpIntfsDisabled( mode, no='default' )

LdpCli.LdpConfigMode.addCommandClass( InterfaceDisabledDefaultCmd )

# --------------------------------------------------------------------------------
# [ no | default ] neighbor IPADDR targeted
# --------------------------------------------------------------------------------
class NeighborIpaddrTargetedCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor IPADDR targeted'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': matcherLdpNeighbor,
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
      'targeted': 'Targeted Peer',
   }

   @staticmethod
   def handler( mode, args ):
      return neighborTargetedIp( mode, args[ 'IPADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return neighborTargetedDefault( mode, args[ 'IPADDR' ] )

LdpCli.LdpConfigMode.addCommandClass( NeighborIpaddrTargetedCmd )

# --------------------------------------------------------------------------------
# [ no | default ] proto-param PROTO_PARAM
# --------------------------------------------------------------------------------
class ProtoParamCmd( CliCommand.CliCommandClass ):
   syntax = 'proto-param PROTO_PARAM'
   noOrDefaultSyntax = 'proto-param ...'
   data = {
      'proto-param': 'Protocol Parameter Setting',
      'PROTO_PARAM': CliMatcher.IntegerMatcher( 0, 2**32 - 1,
         helpdesc='Proto Parameter' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      return protoParam( mode, args[ 'PROTO_PARAM' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return protoParamDefault( mode )

LdpCli.LdpConfigMode.addCommandClass( ProtoParamCmd )

# --------------------------------------------------------------------------------
# [ no | default ] shutdown
# --------------------------------------------------------------------------------
class LdpShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown LDP',
   }

   @staticmethod
   def handler( mode, args ):
      if BasicCliUtil.confirm( mode,
                               "You are attempting to shutdown LDP on all "
                               "interfaces. Are you sure you want to shutdown? "
                               "[Y/n]" ):
         shutdownLdp( mode, no=None )

   @staticmethod
   def noHandler( mode, args ):
      shutdownLdp( mode, no=True )

   @staticmethod
   def defaultHandler( mode, args ):
      shutdownLdp( mode, no='default' )

LdpCli.LdpConfigMode.addCommandClass( LdpShutdownCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transport-address interface INTFS
# --------------------------------------------------------------------------------
class TransportAddrIntfId( CliCommand.CliCommandClass ):
   syntax = 'transport-address interface INTFS'
   noOrDefaultSyntax = 'transport-address ...'
   data = {
      'transport-address': 'Set LDP Transport Addr',
      'interface': matcherInterface,
      'INTFS': Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=intfTypes ),
   }

   @staticmethod
   def handler( mode, args ):
      # There's a quirk that we take a range of interfaces, but use only the first.
      mode.transportAddrIntfIdIs( None, next( iter( args[ 'INTFS' ] ) ) )

   noOrDefaultHandler = transportAddrDefault

LdpCli.LdpConfigMode.addCommandClass( TransportAddrIntfId )

# --------------------------------------------------------------------------------
# [ no | default ] fec filter prefix-list LISTNAME
# --------------------------------------------------------------------------------
class FecFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'fec filter prefix-list LISTNAME'
   noOrDefaultSyntax = 'fec filter ...'
   data = {
      'fec': matcherFec,
      'filter': matcherFilter,
      'prefix-list': 'Prefix list',
      'LISTNAME': RouteMapCli.prefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return fecFilterPrefixListName( mode, args[ 'LISTNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return fecFilterDefault( mode )

LdpCli.LdpConfigMode.addCommandClass( FecFilterCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ip access-group ACLNAME [ in ]
# --------------------------------------------------------------------------------
class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACLNAME [ in ]'
   noOrDefaultSyntax = 'ip access-group ...'
   data = {
      'ip': ipKwForServiceAclMatcher,
      'access-group': accessGroupKwMatcher,
      'ACLNAME': ipAclNameMatcher,
      'in': inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return aclNameIs( mode, args[ 'ACLNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return aclNameDel( mode )

LdpCli.LdpConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] router-id ( IPADDR | ( interface INTFS ) ) [ force ]
# --------------------------------------------------------------------------------
class RouterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ( IPADDR | ( interface INTFS ) ) [ force ]'
   noOrDefaultSyntax = 'router-id [ force ] ...'
   data = {
      'router-id': 'Set LDP Router-Id',
      'IPADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='IP address' ),
      'interface': matcherInterface,
      'INTFS': Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=intfTypes ),
      'force': 'Forcibly change the LDP Router-Id',
   }

   @staticmethod
   def handler( mode, args ):
      force = 'force' in args
      if not force:
         mode.addWarning( 'Change will take effect on the next agent start. Use '
                          '"force" for immediate effect.' )
      if 'IPADDR' in args:
         routerIdIpAddr( mode, args[ 'IPADDR' ], force )
      elif 'interface' in args:
         # There's a quirk that we take a range of intfs, but use only the first.
         routerIdIntfId( mode, next( iter( args[ 'INTFS' ] ) ), force )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return routerIdDefault( mode, 'force' in args )

LdpCli.LdpConfigMode.addCommandClass( RouterIdCmd )

# --------------------------------------------------------------------------------
# [ no | default ] password PASSWORD
# --------------------------------------------------------------------------------
class PasswordCommand( CliCommand.CliCommandClass ):
   syntax = 'password PASSWORD'
   noOrDefaultSyntax = 'password ...'
   data = {
      'password': matcherPassword,
      'PASSWORD': ReversibleSecretCli.defaultReversiblePwdCliExpr,
   }

   @staticmethod
   def handler( mode, args ):
      mode.enableLdpPassword( args[ 'PASSWORD' ], False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearLdpPassword( mode )

LdpCli.LdpConfigMode.addCommandClass( PasswordCommand )

# --------------------------------------------------------------------------------
# [ no | default ] authentication index INDEX password PASSWORD
# --------------------------------------------------------------------------------
class AuthIndexPasswordCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication index INDEX password PASSWORD'
   noOrDefaultSyntax = 'authentication index INDEX password ...'
   data = {
      'authentication': matcherAuth,
      'index': matcherIndex,
      'INDEX': matcherAuthIndexInteger,
      'password': matcherPassword,
      'PASSWORD': ReversibleSecretCli.defaultReversiblePwdCliExpr,
   }

   @staticmethod
   def handler( mode, args ):
      index = args[ 'INDEX' ]
      password = args[ 'PASSWORD' ]
      mode.authIndexPasswordIs( index=index, password=password )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      index = args[ 'INDEX' ]
      mode.authIndexPasswordIs( index=index, enable=False )

LdpCli.LdpConfigMode.addCommandClass( AuthIndexPasswordCmd )

# --------------------------------------------------------------------------------
# [ no | default ] authentication index INDEX active
# --------------------------------------------------------------------------------
class AuthIndexActiveCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication index INDEX active'
   noOrDefaultSyntax = 'authentication index [INDEX] active'
   data = {
      'authentication': matcherAuth,
      'index': matcherIndex,
      'INDEX': matcherAuthIndexInteger,
      'active': matcherActive,
   }

   @staticmethod
   def handler( mode, args ):
      index = args[ 'INDEX' ]
      mode.authIndexActiveIs( index=index )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.authIndexActiveIs( enable=False )

LdpCli.LdpConfigMode.addCommandClass( AuthIndexActiveCmd )

# --------------------------------------------------------------------------------
# [ no | default ] neighbor IPADDR authentication index INDEX active
# --------------------------------------------------------------------------------
class NeighborAuthIndexActiveCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor IPADDR authentication index INDEX active'
   noOrDefaultSyntax = 'neighbor IPADDR authentication index [INDEX] active'
   data = {
      'neighbor': matcherLdpNeighbor,
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
      'authentication': matcherAuth,
      'index': matcherIndex,
      'INDEX': matcherAuthIndexInteger,
      'active': matcherActive,
   }

   @staticmethod
   def handler( mode, args ):
      ipAddr = args[ 'IPADDR' ]
      index = args[ 'INDEX' ]
      mode.neighAuthIndexActiveIs( ipAddr=ipAddr, index=index )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipAddr = args[ 'IPADDR' ]
      mode.neighAuthIndexActiveIs( ipAddr=ipAddr, enable=False )

LdpCli.LdpConfigMode.addCommandClass( NeighborAuthIndexActiveCmd )

# label local-termination { implicit-null | explicit-null }
class LabelLocalTermination( CliCommand.CliCommandClass ):
   syntax = 'label local-termination ( implicit-null | explicit-null )'
   noOrDefaultSyntax = 'label local-termination ...'
   data = {
      'label': 'Label to be advertised',
      'local-termination': 'Local termination label to be advertised',
      'implicit-null': 'Advertise implicit null',
      'explicit-null': 'Advertise explicit null'
   }
   handler = LdpCli.LdpConfigMode.ldpLabelLocalTermination
   noOrDefaultHandler = handler

LdpCli.LdpConfigMode.addCommandClass( LabelLocalTermination )

# entropy-label
class LdpEntropyLabel( CliCommand.CliCommandClass ):
   syntax = 'entropy-label'
   noOrDefaultSyntax = syntax
   data = {
      'entropy-label': CliCommand.guardedKeyword( 'entropy-label',
         helpdesc='Enable entropy label signaling capability',
         guard=entropyLabelSupportedGuard ),
   }

   handler = entropyLabel
   noOrDefaultHandler = handler

LdpCli.LdpConfigMode.addCommandClass( LdpEntropyLabel )

# ---------------------------------------------------------------------------------
# [ no | default ] neighbor hello-redundancy [ none |
#                                              ( duration ( SECONDS | infinite ) ) ]
# ---------------------------------------------------------------------------------
class NeighborHelloRedundancy( CliCommand.CliCommandClass ):
   syntax = ( 'neighbor hello-redundancy [ none | ( duration ( SECONDS | '
                                                              'infinite ) ) ]' )
   noOrDefaultSyntax = 'neighbor hello-redundancy'
   data = {
      'neighbor': matcherLdpNeighbor,
      'hello-redundancy': 'Redundant targeted adjacencies',
      'none': 'Disable hello redundancy',
      'duration': ( 'Duration of redundant adjacencies after losing all link '
                     'adjacencies' ),
      'SECONDS': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                               helpdesc='Redundant adjacency duration in seconds' ),
      'infinite': 'Retain redundant targeted adjacencies indefinitely',
   }

   @staticmethod
   def handler( mode, args ):
      mode.helloRedundancyIs( 'none' not in args )
      if 'infinite' in args:
         mode.helloRedundancyTimeoutIs( Tac.endOfTime )
      else:
         mode.helloRedundancyTimeoutIs( args.get( 'SECONDS' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.helloRedundancyIs( False )
      mode.helloRedundancyTimeoutIs( None )

LdpCli.LdpConfigMode.addCommandClass( NeighborHelloRedundancy )

# --------------------------------------------------------------------------------
# [ no | default ] neighbor end-of-lib
# --------------------------------------------------------------------------------
class NeighborEndOfLib( CliCommand.CliCommandClass ):
   syntax = 'neighbor end-of-lib'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': matcherLdpNeighbor,
      'end-of-lib': 'Signal label advertisement completion',
   }

   @staticmethod
   def handler( mode, args ):
      mode.endOfLibIs( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.endOfLibIs( False )

LdpCli.LdpConfigMode.addCommandClass( NeighborEndOfLib )

# ------------------------------------------------------------------------------
# mpls ldp configuration mode. Like ldp,
# mldp is not enabled till "no shutdown" is also configured
# under mldp.
# ------------------------------------------------------------------------------
class MldpConfigMode( MldpMode, BasicCli.ConfigModeBase ):
   name = "Mpls Ldp Configuration"

   def __init__( self, parent, session, vrfName='default' ):
      self.vrfName = vrfName
      MldpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def shutdownMldpIs( self, no ):
      config = LdpCli.getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.mLdpEnabled = bool( no )

   def lspRootOpaqueId( self, args ):
      root = args[ 'NODE' ]
      opaqueId = args[ 'OPAQUE_ID' ]
      rangeSize = args.get( 'SIZE' )
      egressVlanId = args.get( 'VLAN_ID', 0 )
      config = LdpCli.getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if rangeSize:
         if opaqueId + rangeSize > 2**32:
            self.addErrorAndStop( 'Opaque ID range is too large' )
      else:
         rangeSize = 1
      for i in range( opaqueId, opaqueId + rangeSize ):
         p2mpGenericFecId = LdpP2mpFecGenericOpaqueId( IpGenAddr( root ), i )
         config.staticP2mpFecBindingColl.newMember( p2mpGenericFecId, egressVlanId )

   def lspRootOpaqueIdDefault( self, args ):
      root = args[ 'NODE' ]
      opaqueId = args[ 'OPAQUE_ID' ]
      rangeSize = args.get( 'SIZE' )
      config = LdpCli.getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      if rangeSize:
         if opaqueId + rangeSize > 2**32:
            self.addErrorAndStop( 'Opaque ID range is too large' )
      else:
         rangeSize = 1
      for i in range( opaqueId, opaqueId + rangeSize ):
         p2mpGenericFecId = LdpP2mpFecGenericOpaqueId( IpGenAddr( root ), i )
         del config.staticP2mpFecBindingColl[ p2mpGenericFecId ]

   def lspStaticP2mpFecBindingCollDelAll( self ):
      config = LdpCli.getLdpConfigColl(
                  self.entityManager ).config.newMember( self.vrfName )
      config.staticP2mpFecBindingColl.clear()

# mpls ldp mldp, add only if feature toggle is enabled
# --------------------------------------------------------------------------------
# [ no | default ] mldp
# --------------------------------------------------------------------------------
class MldpCmd( CliCommand.CliCommandClass ):
   syntax = 'mldp'
   noOrDefaultSyntax = syntax
   data = {
      'mldp': 'MLDP configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MldpConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearMldp( mode )

LdpCli.LdpConfigMode.addCommandClass( MldpCmd )

# --------------------------------------------------------------------------------
# [ no | default ] shutdown
# --------------------------------------------------------------------------------
class MldpShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown mLDP',
   }

   @staticmethod
   def handler( mode, args ):
      mode.shutdownMldpIs( no=None )

   @staticmethod
   def noHandler( mode, args ):
      mode.shutdownMldpIs( no=True )

   @staticmethod
   def defaultHandler( mode, args ):
      mode.shutdownMldpIs( no='default' )

MldpConfigMode.addCommandClass( MldpShutdownCmd )

# --------------------------------------------------------------------------------
# [ no | default ] p2mp-lsp root NODE opaque-id ( OPAQUE_ID |
#                  ( range OPAQUE_ID SIZE ) ) [ vlan VLAN_ID ]
# --------------------------------------------------------------------------------
class LspRootOpaqueId( CliCommand.CliCommandClass ):
   syntax = ( 'p2mp-lsp root NODE opaque-id ( OPAQUE_ID | ( range OPAQUE_ID SIZE ) )'
              ' [ vlan VLAN_ID ]' )
   noOrDefaultSyntax = ( 'p2mp-lsp root NODE opaque-id ( OPAQUE_ID |'
                         '( range OPAQUE_ID SIZE ) ) ...' )
   data = {
      'p2mp-lsp': CliCommand.hiddenKeyword( 'p2mp-lsp' ),
      'root': 'Root node address',
      'NODE': IpAddrMatcher.IpAddrMatcher( helpdesc='P2mp root node address' ),
      'opaque-id': 'Opaque ID',
      'OPAQUE_ID': CliMatcher.IntegerMatcher( 1, 2**32 - 1,
         helpdesc='Opaque-id' ),
      'range': "Configure a range of opaque IDs",
      'SIZE': CliMatcher.IntegerMatcher( 1, 2**32 - 1,
         helpdesc='Size of the range' ),
      'vlan': 'Configure an outgoing VLAN',
      # allow VLAN IDs of 14 bits because VLAN might come from the extended range
      'VLAN_ID': CliMatcher.IntegerMatcher( 1, 2**14 - 1,
                                            helpdesc='outgoing VLAN ID' )
   }

   handler = MldpConfigMode.lspRootOpaqueId
   noOrDefaultHandler = MldpConfigMode.lspRootOpaqueIdDefault

MldpConfigMode.addCommandClass( LspRootOpaqueId )

matcherGracefulRestart = CliMatcher.KeywordMatcher( 'graceful-restart',
   helpdesc='LDP graceful restart' )
matcherRole = CliMatcher.KeywordMatcher( 'role',
   helpdesc='LDP graceful restart role' )
matcherTimer = CliMatcher.KeywordMatcher( 'timer',
   helpdesc='LDP graceful restart timer' )
grTimeoutMatcher = CliMatcher.IntegerMatcher( 1, 3600,
   helpdesc='Number of seconds' )

# Gr Helper timers
# --------------------------------------------------------------------------------
# [ no | default ] timer neighbor-liveness TIMEOUT
# --------------------------------------------------------------------------------
class TimerNeighborLivenessCmd( CliCommand.CliCommandClass ):
   syntax = 'timer neighbor-liveness TIMEOUT'
   noOrDefaultSyntax = 'timer neighbor-liveness ...'
   data = {
      'timer': matcherTimer,
      'neighbor-liveness': ( 'Maximum time to preserve the session '
                              'forwarding-state before the session is '
                              'reestablished' ),
      'TIMEOUT': grTimeoutMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return mode.neighborLivenessIs( args[ 'TIMEOUT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noNeighborLiveness()

LdpCli.LdpGrHelperConfigMode.addCommandClass( TimerNeighborLivenessCmd )

# --------------------------------------------------------------------------------
# [ no | default ] timer recovery [ TIMEOUT ]
# --------------------------------------------------------------------------------
class TimerRecoveryCmd( CliCommand.CliCommandClass ):
   syntax = 'timer recovery TIMEOUT'
   noOrDefaultSyntax = 'timer recovery ...'
   data = {
      'timer': matcherTimer,
      'recovery': ( 'Maximum time to preserve the session forwarding-state '
                     'after the session has reestablished' ),
      'TIMEOUT': grTimeoutMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return mode.recoveryIs( args[ 'TIMEOUT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noRecovery()

LdpCli.LdpGrHelperConfigMode.addCommandClass( TimerRecoveryCmd )

# Gr Speaker timers
# --------------------------------------------------------------------------------
# [ no | default ] timer state-holding [ TIMEOUT ]
# --------------------------------------------------------------------------------
class TimerStateHoldingCmd( CliCommand.CliCommandClass ):
   syntax = 'timer state-holding TIMEOUT'
   noOrDefaultSyntax = 'timer state-holding ...'
   data = {
      'timer': matcherTimer,
      'state-holding': 'Time to preserve the forwarding-state after restart',
      'TIMEOUT': grTimeoutMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return mode.stateHoldingIs( args[ 'TIMEOUT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noStateHolding()

LdpCli.LdpGrSpeakerConfigMode.addCommandClass( TimerStateHoldingCmd )
# --------------------------------------------------------------------------------
# [ no | default ] timer reconnect [ TIMEOUT ]
# --------------------------------------------------------------------------------

class TimerReconnectCmd( CliCommand.CliCommandClass ):
   syntax = 'timer reconnect TIMEOUT'
   noOrDefaultSyntax = 'timer reconnect ...'
   data = {
      'timer': matcherTimer,
      'reconnect': ( 'Time the peer should preserve the forwarding-state '
                      'after restart' ),
      'TIMEOUT': grTimeoutMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return mode.reconnectIs( args[ 'TIMEOUT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noReconnect()

LdpCli.LdpGrSpeakerConfigMode.addCommandClass( TimerReconnectCmd )

# --------------------------------------------------------------------------------
# [ no | default ] graceful-restart role helper
# --------------------------------------------------------------------------------
class GracefulRestartRoleHelperCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart role helper'
   noOrDefaultSyntax = syntax
   data = {
      'graceful-restart': matcherGracefulRestart,
      'role': matcherRole,
      'helper': 'LDP graceful restart helper',
   }

   @staticmethod
   def handler( mode, args ):
      return mode.gotoLdpGrHelperMode()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noLdpGrHelperMode()

LdpCli.LdpConfigMode.addCommandClass( GracefulRestartRoleHelperCmd )

# --------------------------------------------------------------------------------
# [ no | default ] graceful-restart role speaker
# --------------------------------------------------------------------------------
class GracefulRestartRoleSpeakerCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart role speaker'
   noOrDefaultSyntax = syntax
   data = {
      'graceful-restart': matcherGracefulRestart,
      'role': matcherRole,
      'speaker': 'LDP graceful restart speaker',
   }

   @staticmethod
   def handler( mode, args ):
      return mode.gotoLdpGrSpeakerMode()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noLdpGrSpeakerMode()

LdpCli.LdpConfigMode.addCommandClass( GracefulRestartRoleSpeakerCmd )

# ---------------------------------------------------------------------------------
# [no|default] tunnel source-protocol bgp ipv4 labeled-unicast [ rcf FUNCTION ]
# in 'mpls ldp' configuration mode.
#
# Enables redistribution from BGP LU into LDP. (AID8019)
# ---------------------------------------------------------------------------------

class RedistributeLuCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel source-protocol bgp ipv4 labeled-unicast [ rcf FUNCTION ]'
   noOrDefaultSyntax = 'tunnel source-protocol bgp ipv4 labeled-unicast ...'

   data = {
      'tunnel': 'Tunnel protocols',
      'source-protocol': 'Configure the tunnel source',
      'bgp': 'BGP routes',
      'ipv4': 'IPv4 related',
      'labeled-unicast': 'Labeled-unicast sub-address family',
      'rcf': 'Routing control functions configuration to filter and modify tunnels',
      'FUNCTION': RcfCliLib.rcfFunctionMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      rcfName = args.get( 'FUNCTION' )
      if rcfName:
         if rcfName.endswith( '()' ):
            # Remove the trailing () characters
            rcfName = rcfName[ : -2 ]
      else:
         rcfName = ''
      mode.allowLuTunnelRedistIs( True, rcfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearSrcProtoBgpV4Lu( mode )

LdpCli.LdpConfigMode.addCommandClass( RedistributeLuCmd )

# -----------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -----------------------------------------------------------------------------------
def Plugin( entityManager ):
   global mplsHwCapability

   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
