# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import BasicCli
import CliCommand
import CliMatcher
import CliMode.LineSystem
import CliParser
import ConfigMount
import FileUrl
import Intf.IntfRange
import IntfRangePlugin.XcvrSlot
import Url
from CliPlugin.LineSystemPortRule import LineSystemArbitraryPortMatcher
from TypeFuture import TacLazyType

#-------------------------------------------------------------------------------
# Matchers used in commands
#-------------------------------------------------------------------------------
portRangeMatcher = Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( IntfRangePlugin.XcvrSlot.XcvrSlotType, ),
         dollarCompHelpText="Port list end",
         priority=CliParser.PRIO_HIGH )

#-------------------------------------------------------------------------------
# TAC Types used in commands
#-------------------------------------------------------------------------------

ControlPathConfig = TacLazyType( "LineSystem::ControlPathConfig" )

#-------------------------------------------------------------------------------
# Global Sysdb path aliases
#-------------------------------------------------------------------------------

lineSystemCliConfigSliceDir = None

#-------------------------------------------------------------------------------
# Helper methods: utilities to provide easy access to various Sysdb entities
#-------------------------------------------------------------------------------

def getLineSystemConfigDir( port ):
   """
   Get slicified dir for specified port

   Parameters
   ----------
   port: str, port identifier (e.g. "Port3", "Port5/1", etc.)

   Returns
   -------
   configDir: LineSystem::LineSystemCliConfigDir
   """
   configDir = lineSystemCliConfigSliceDir.get( "FixedSystem" )
   portNum = port.lower().strip( "port" )
   if "/" in portNum:
      # Only modular ports will have a "/" in them. If not present,
      # assume port belongs to a FixedSystem.
      card, _ = portNum.split( "/" )
      sliceId = "Linecard%s" % card
      configDir = lineSystemCliConfigSliceDir.get( sliceId )
   return configDir

def getLineSystemConfig( port ):
   """
   Get slicified dir for specified port

   Parameters
   ----------
   port: str, port identifier (e.g. "Port3", "Port5/1", etc.)

   Returns
   -------
   config: LineSystem::LineSystemCliConfig
   """
   configDir = getLineSystemConfigDir( port )
   # Create new config object. If it already exists in the collection, the API
   # will return the existing object.
   return configDir.newLineSystemCliConfig( port.capitalize() )

def deleteLineSystemConfig( port ):
   """
   Delete CLI config object associated with the port.

   Parameters
   ----------
   port: str, port identifier (e.g. "Port3", "Port5/1", etc.)
   """
   configDir = getLineSystemConfigDir( port )
   # Unlike normal python dictionaries, we don't get a KeyError if we attempt
   # to remove a key that isn't present.
   del configDir.lineSystemCliConfig[ port.capitalize() ]

#-------------------------------------------------------------------------------
# CLI mode classes
#-------------------------------------------------------------------------------

class LineSystemConfigMode( CliMode.LineSystem.LineSystemMode,
                            BasicCli.ConfigModeBase ):
   name = "Line system configuration"

   def __init__( self, parent, session ):
      CliMode.LineSystem.LineSystemMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class PortRangeConfigMode( CliMode.LineSystem.PortRangeMode,
                           BasicCli.ConfigModeBase ):
   name = "Line system port range configuration"

   def __init__( self, parent, session, portRange ):
      CliMode.LineSystem.PortRangeMode.__init__( self, portRange )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def _showActiveHelper( self, showAll=False ):
      """
      Displays active config of all ports in port range in the "line system"
      section.

      Implementation is very similar to BasicCliUtil.showRunningConfigWithFilter.
      Basically builds correct view of "show active" based on port config mode.
      Main difference is that the function in BasicCliUtil stops after the first
      matching child mode, and we need to include all child modes in the current
      configuration mode.
      """
      if showAll:
         urlFunc = FileUrl.localRunningConfigAll
      else:
         urlFunc = FileUrl.localRunningConfig
      url = urlFunc( *Url.urlArgsFromMode( self ) )

      output = ""
      try:
         with url.open() as f:
            # Running-config indents are three-spaces long and are magic-
            # string'd in a number of places. Well, now there's one more...
            runningConfigIndent = "   "
            if isinstance( self.param_, str ):
               # this allows CliTestMode.lineSystemPortRangeConfigMode.testParams
               # to work
               portNums = [ self.param_ ]
            else:
               portNums = [ str( i ) for i in self.param_.irPathList_ ]
            irPathExp = "^{}port ({})$".format( runningConfigIndent,
                                                "|".join( portNums ) )
            portsMatched = [ False ] * len( portNums )
            currPortMatched = False
            parentMatched = False
            for line in f:
               if currPortMatched:
                  # Found a port. Keep printing until we reach next unindented
                  # block, then break. Include "!" lines if we see them unless
                  # we've already found all the ports.
                  if( line.startswith( runningConfigIndent * 2 ) or
                      ( line.startswith( runningConfigIndent + "!" ) and
                        not all( portsMatched ) ) ):
                     output += line
                     continue

               # If we've found all the ports and reached here, we've printed all
               # lines from the previous port, so we're done.
               if all( portsMatched ):
                  break

               # Attempt to match against port section. If not parentMatched, that
               # means we haven't come across the "line system" section yet, so
               # don't bother processing the line
               currPortMatched = ( parentMatched and
                                   re.match( irPathExp, line ) is not None )
               if currPortMatched:
                  output += line
                  idx = portsMatched.count( True )
                  portsMatched[ idx ] = True
                  continue

               # If we haven't found it yet, search for the "line system" section
               if not parentMatched:
                  if line.startswith( "line system" ):
                     assert not any( portsMatched )
                     parentMatched = True
                     output += line
                     continue

            print( output, end="" )

      except OSError as e:
         self.addError( "Cannot display running-config: %s" % e )

   def showActive( self ):
      self._showActiveHelper( showAll=False )

   def showActiveAll( self, showDetail ):
      self._showActiveHelper( showAll=True )

#-------------------------------------------------------------------------------
# (config)#line system
# (config-ls)#
#-------------------------------------------------------------------------------

class LineSystemCmd( CliCommand.CliCommandClass ):
   syntax = "line system"
   noOrDefaultSyntax = syntax
   data = {
      "line": "Line system settings",
      "system": "Line system settings",
   }

   @staticmethod
   def handler( mode, args ):
      newMode = mode.childMode( LineSystemConfigMode )
      mode.session_.gotoChildMode( newMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for slices in lineSystemCliConfigSliceDir.values():
         slices.lineSystemCliConfig.clear()

BasicCli.GlobalConfigMode.addCommandClass( LineSystemCmd )

#-------------------------------------------------------------------------------
# (config-ls)# port PORTRANGE
# (config-ls-portRANGE)#
#-------------------------------------------------------------------------------

class PortRangeCmd( CliCommand.CliCommandClass ):
   syntax = "STARTUP_PORTRANGE | PORTRANGE"
   noOrDefaultSyntax = "PORTRANGE"
   data = {
      # Use hidden node to match port-range commands at startup-config time,
      # before EntityMib is available to provide port information.
      "STARTUP_PORTRANGE": CliCommand.Node( matcher=LineSystemArbitraryPortMatcher(),
                                            hidden=True ),
      "PORTRANGE": portRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      """
      Standard command handler. Enters port-range mode for specified port range.
      Objects for ports specified in range will be created if needed.
      """
      portRange = args.get( "STARTUP_PORTRANGE" ) or args[ "PORTRANGE" ]
      childMode = mode.childMode( PortRangeConfigMode, portRange=portRange )
      mode.session_.gotoChildMode( childMode )
      for port in Intf.IntfRange.intfListFromCanonical( [ portRange ] ):
         getLineSystemConfig( port )

   @staticmethod
   def noHandler( mode, args ):
      """
      Handler for "no PORTRANGE" command. Deletes the config object for ports in
      specified range. Consumers of the config should apply default settings.
      """
      portRange = args.get( "STARTUP_PORTRANGE" ) or args[ "PORTRANGE" ]
      for port in Intf.IntfRange.intfListFromCanonical( [ portRange ] ):
         deleteLineSystemConfig( port )

   @staticmethod
   def defaultHandler( mode, args ):
      """
      Handler for "default PORTRANGE" command. If an object for a specified port
      exists, set all of its attributes back to the default value.
      """
      portRange = args.get( "STARTUP_PORTRANGE" ) or args[ "PORTRANGE" ]
      for port in Intf.IntfRange.intfListFromCanonical( [ portRange ] ):
         config = getLineSystemConfig( port )
         config.reset()

LineSystemConfigMode.addCommandClass( PortRangeCmd )

#-------------------------------------------------------------------------------
# [ no | default ] ( booster | pre-amp ) disabled
#-------------------------------------------------------------------------------

boosterKw = CliMatcher.KeywordMatcher( "booster", helpdesc="Booster settings" )
preAmpKw = CliMatcher.KeywordMatcher( "pre-amp", helpdesc="Pre-amp settings" )

class AmplifierDisabledCmd( CliCommand.CliCommandClass ):
   syntax = "( booster | pre-amp ) disabled"
   noOrDefaultSyntax = syntax
   data = {
      "booster": boosterKw,
      "pre-amp": preAmpKw,
      "disabled": "Amplifier disabled",
   }

   @staticmethod
   def _handlerHelper( mode, args, enabled ):
      ampAttr = "booster" if "booster" in args else "preAmp"
      portRange = mode.param_
      for port in Intf.IntfRange.intfListFromCanonical( [ portRange ] ):
         portConfig = getLineSystemConfig( port )
         ctrlPathCfg = getattr( portConfig, ampAttr )
         newCtrlPathCfg = ControlPathConfig( enabled=enabled,
                                             txDisable=ctrlPathCfg.txDisable )
         setattr( portConfig, ampAttr, newCtrlPathCfg )

   @staticmethod
   def handler( mode, args ):
      AmplifierDisabledCmd._handlerHelper( mode, args, enabled=False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      AmplifierDisabledCmd._handlerHelper( mode, args, enabled=True )

PortRangeConfigMode.addCommandClass( AmplifierDisabledCmd )

#-------------------------------------------------------------------------------
# [ no | default ] ( booster | pre-amp ) transmitter disabled
#-------------------------------------------------------------------------------

class TxDisabledCmd( CliCommand.CliCommandClass ):
   syntax = "( booster | pre-amp ) transmitter disabled"
   noOrDefaultSyntax = syntax
   data = {
      "booster": boosterKw,
      "pre-amp": preAmpKw,
      "transmitter": "Transmitter laser settings",
      "disabled": "Transmitter disabled",
   }

   @staticmethod
   def _handlerHelper( mode, args, disabled ):
      ampAttr = "booster" if "booster" in args else "preAmp"
      portRange = mode.param_
      for port in Intf.IntfRange.intfListFromCanonical( [ portRange ] ):
         portConfig = getLineSystemConfig( port )
         ctrlPathCfg = getattr( portConfig, ampAttr )
         newCtrlPathCfg = ControlPathConfig( enabled=ctrlPathCfg.enabled,
                                             txDisable=disabled )
         setattr( portConfig, ampAttr, newCtrlPathCfg )

   @staticmethod
   def handler( mode, args ):
      TxDisabledCmd._handlerHelper( mode, args, disabled=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      TxDisabledCmd._handlerHelper( mode, args, disabled=False )

PortRangeConfigMode.addCommandClass( TxDisabledCmd )

#-------------------------------------------------------------------------------
# Plugin method to mount line-system config paths
#-------------------------------------------------------------------------------

def Plugin( em ):
   global lineSystemCliConfigSliceDir

   sliceDirPath = "hardware/line/system/cli/config/slice"
   lineSystemCliConfigSliceDir = ConfigMount.mount( em, sliceDirPath,
                                                    "Tac::Dir", "wi" )
