# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
from CliPlugin.VirtualIntfRule import _VirtualIntfMatcher
from CliPlugin.VirtualIntfRule import VirtualIntfMatcherBase
import CliMatcher
import Tac

class LineSystemArbitraryPortMatcher( VirtualIntfMatcherBase ):
   """
   Matches a port range of the format `port<PORTNUMBER>` or
   `port<SLOTNUMBER>/<PORTNUMBER>`.
   """
   def __init__( self, **kwargs ):
      portToken = "port"
      portNumberMatcher = ArbitraryPortNumberMatcher()
      portNameMatcher = _VirtualIntfMatcher( portToken, portNumberMatcher )
      VirtualIntfMatcherBase.__init__( self, portToken, portNameMatcher,
                                       portNumberMatcher, **kwargs )

class ArbitraryPortNumberMatcher:
   def __init__( self ):
      # This matcher will be used to parse line-system port-range commands at
      # startup time, i.e. before Fru has populated the entity mib. During this
      # time, we will accept any arbitrary port-number specified. The ranges passed
      # into the matchers are mostly arbitrary.
      self.noEntmibSlotNumberMatcher_ = \
         CliMatcher.IntegerMatcher( 1, 32, helpdesc="Slot number" )
      self.noEntmibPortNumberMatcher_ = \
         CliMatcher.IntegerMatcher( 1, 128, helpdesc="Number of the port to use" )
      self.fixedSystemPortRe_ = re.compile( r"(\d+)$" )
      self.modularSystemPortRe_ = re.compile( r"(\d+)/(\d+)$" )

   def match( self, mode, context, token ):
      entmibStatus = mode.entityManager.lookup( "hardware/entmib" )
      if entmibStatus.root and entmibStatus.root.initStatus == "ok":
         # If EntityMib is populated, force no match so that parser falls back to
         # matcher that takes into account available ports.
         return CliMatcher.noMatch
      # Check against fixed-system re first, just 'cause.
      fixedSystemMatch = self.fixedSystemPortRe_.match( token )
      if( fixedSystemMatch and
          ( self.noEntmibPortNumberMatcher_.match( mode, context,
                                                   fixedSystemMatch.group( 1 ) )
            is not CliMatcher.noMatch ) ):
         return CliMatcher.MatchResult( token, token )
      # Then check against modular-system re
      modularSystemMatch = self.modularSystemPortRe_.match( token )
      if( modularSystemMatch and
          ( ( self.noEntmibSlotNumberMatcher_.match( mode, context,
                                                     modularSystemMatch.group( 1 ) )
              is not CliMatcher.noMatch ) and
            ( self.noEntmibPortNumberMatcher_.match( mode, context,
                                                     modularSystemMatch.group( 2 ) )
              is not CliMatcher.noMatch ) ) ):
         return CliMatcher.MatchResult( token, token )
      # If we get here, no match
      return CliMatcher.noMatch

   def completions( self, mode, context, token ):
      # Don't provide completions if EntityMib does not provide port information
      return []
