# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module implements the CLI for Link Flap Detection.  Specifically, it provides
the following commands:

In Global Configuration mode:
-  [no] errdisable flap-setting cause link-flap max-flaps <flap-count> time <seconds>

In EXEC mode:
-  show errdisable flap-values


Profile style CLI Design for Link Flap Detection:

A mode to create and maintain "profiles" of link-flap configuration:
(config-link-flap)

- monitor link-flap policy

Profile creation under link-flap mode: (config-link-flap):
- [no] profile <profile name> max-flaps <F> time <S> [violations <V> intervals <I>]

Changing global configuration via "link-flap" mode:
(for backward compatibility to support downgrade)
- [no] profile default max-flaps <F> time <S>

Selecting set of profiles as a default setting to be used by link-flap algorithm
- default-profiles <p1> [ <p2> <p3> ... ]

Per interface configuration: (Tri-state)

Each interface can either enable link flap monitoring, disable link flap
     monitoring or inherit from global setting.

To enable link flap monitoring with configuration parameters taken from
"default-profiles" if specified, or "profile default" in that order.

- monitor link-flap

To enable link flap monitoring with configuration parameters from the
chosen profiles

- monitor link-flap [ profiles <p1> [ <P2> <p3> .. ] ]

To disable link flap monitoring:

- no monitor link-flap

To inherit from global setting

- default monitor link-flap

To create a new Damping profile or edit an existing damping profile:

- profile <profile-name> damping

To change damping profile local fault penalty:

- penalty mac fault local <penalty>

To reset damping profile local fault penalty to the default value:

- default penalty mac fault local

To reset damping profile local fault penalty to zero:

- no penalty mac fault local

To change damping profile remote fault penalty:

- penalty mac fault remote <penalty>

To reset damping profile remote fault penalty to the default value:

- default penalty mac fault remote

To reset damping profile remote fault penalty to zero:

- no penalty mac fault remote

To change penalty thresholds in a damping profile:

- penalty threshold reuse <v1> suppression <v2> maximum <v3>

To reset penalty thresholds in a damping profile to their default values:

- default penalty threshold reuse suppression maximum

To set new penalty half-life time in seconds in a damping profile:

- penalty decay half-life <time> seconds

To set penalty half-life time in minutes in a damping profile:

- penalty decay half-life <time> minutes

To reset penalty half-life time in a damping profile to its default value:

- default penalty decay half-life
"""

import BasicCli
import CliCommand
import CliMatcher
from CliMode.LinkFlapDetection import LinkFlapMode, LinkFlapDampingMode
import CliParser
from CliPlugin import EthIntfCli, IntfCli
import CliToken.Monitor
import ConfigMount
import ShowCommand
import Tac
from TypeFuture import TacLazyType

linkFlapConfig = None
linkFlapDampingConfig = None
linkFlapDampingIntfConfigDir = {}

# List of slice IDs including 'FixedSystem', 'Linecard1', ..., 'Linecard18'
sliceNames = [ 'FixedSystem' ] + [ f'Linecard{i}' for i in range( 1, 19 ) ]

DefaultParametersType = TacLazyType( 'LinkFlap::ProfileParameters' )

# CLI command implementations.

def setLinkFlapValues( mode, flaps, seconds ):
   linkFlapConfig.maxFlaps = flaps
   linkFlapConfig.interval = seconds

def noLinkFlapValues( mode ):
   setLinkFlapValues( mode, linkFlapConfig.maxFlapsDefault,
      linkFlapConfig.intervalDefault )

def getDampingIntfConfig( sliceId, intfId ):
   sliceDir = linkFlapDampingIntfConfigDir[ sliceId ]
   return sliceDir.intfConfig.newMember( intfId )

# --------------------------------------------------------------------------------
# errdisable flap-setting cause link-flap max-flaps FLAPS time SECONDS
# --------------------------------------------------------------------------------
class LinkFlapValuesCmd( CliCommand.CliCommandClass ):
   syntax = 'errdisable flap-setting cause link-flap max-flaps FLAPS time SECONDS'
   noOrDefaultSyntax = 'errdisable flap-setting cause link-flap ...'
   data = {
      'errdisable': 'Configure error disable functionality',
      'flap-setting': 'Configure the maximum number of flaps that are allowed',
      'cause': 'Specify the errdisable cause',
      'link-flap': 'Specify the flap-setting values for link flap errdisable cause',
      'max-flaps': 'Specify the maximum number of flaps',
      'FLAPS': CliMatcher.IntegerMatcher( 1, 100, helpdesc='Max-flaps' ),
      'time': 'Specify the time period that flaps are counted (in seconds)',
      'SECONDS': CliMatcher.IntegerMatcher( 1, 1800, helpdesc='Time (in seconds)' ),
   }

   @staticmethod
   def handler( mode, args ):
      setLinkFlapValues( mode, args[ 'FLAPS' ], args[ 'SECONDS' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noLinkFlapValues( mode )

BasicCli.GlobalConfigMode.addCommandClass( LinkFlapValuesCmd )

# --------------------------------------------------------------------------------
# show errdisable flap-values
# --------------------------------------------------------------------------------
class ErrdisableFlapValuesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show errdisable flap-values'
   data = {
      'errdisable': 'Show errdisable information',
      'flap-values': ( 'Show flap values that cause an error to be recognized for '
                        'a cause' ),
   }

   @staticmethod
   def handler( mode, args ):
      pformat = '%-20.20s  %-10.10s  %-10.10s'
      print( pformat % ( 'ErrDisable Reason', 'Flaps', 'Time' ) )
      print( pformat % ( '=================', '=====', '====' ) )
      print( pformat % ( 'link-flap', linkFlapConfig.maxFlaps,
                         linkFlapConfig.interval ) )

BasicCli.addShowCommandClass( ErrdisableFlapValuesCmd )

# Generate a list combining names of all existing link-flap profiles
# and damping profiles
# This function is used as a callback by profileNameMatcher
# and allProfileNameMatcher
def profileNameMatcherKeywordFn( mode ):
   return list( linkFlapConfig.profileConfig ) + \
      list( linkFlapDampingConfig.profileConfig )

# Return profile configuration object with the given name, or None
# if profile lookup by name returned no result
# This function is used as a callback by allProfileNameMatcher
def profileNameMatcherValueFn( mode, match ):
   if match in linkFlapDampingConfig.profileConfig:
      return linkFlapDampingConfig.profileConfig[ match ]
   return linkFlapConfig.profileConfig[ match ]

# This Name Matcher enables auto-completion for existing profile
# names while allowing clients to specify new names when creating profiles
profileNameMatcher = CliMatcher.DynamicKeywordMatcher(
         keywordsFn=profileNameMatcherKeywordFn,
         value=profileNameMatcherValueFn,
         emptyTokenCompletion=[ CliParser.Completion( 'WORD',
                                                      'profile name',
                                                      literal=False ) ] )

# This Name Matcher enables auto-completion for existing profile
# names while allowing clients to specify new names when creating profiles
allProfileNameMatcher = CliMatcher.DynamicNameMatcher(
   profileNameMatcherKeywordFn,
   'Damping profile name',
   helpname='WORD',
   pattern=r'[A-Za-z0-9]+',
)

# -----------------------------
# The "link-flap" mode.
# -----------------------------
class LinkFlapConfigMode( LinkFlapMode, BasicCli.ConfigModeBase ):
   name = "Link flap configuration"

   # ----------------------------------------------------------------------------
   # Constructs a LinkFlapConfigMode
   # ----------------------------------------------------------------------------
   def __init__( self, parent, session ):
      LinkFlapMode.__init__( self, "" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# monitor link-flap policy
# --------------------------------------------------------------------------------
class MonitorLinkFlapPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor link-flap policy'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': CliToken.Monitor.monitorMatcher,
      'link-flap': 'Specify the flap-setting values for link flap errdisable cause',
      'policy': 'Manage profiles for link flap detection',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( LinkFlapConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = LinkFlapConfigMode.clear

BasicCli.GlobalConfigMode.addCommandClass( MonitorLinkFlapPolicyCmd )

# Match on user-defined profile names or on pre-defiled
# "default" profile name
class ProfileNameExpression( CliCommand.CliExpression ):
   expression = '(PROFILE_NAME | default)'
   data = {
      'PROFILE_NAME': allProfileNameMatcher,
      'default': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'default',
            helpdesc='Reflection of global setting' ),
         alias='PROFILE_NAME' ),
   }

# Link Flap profile parameters expression
class LinkFlapProfilesParamsExpression( CliCommand.CliExpression ):
   expression = ( 'max-flaps FLAPS time SECONDS '
                  '[ violations VIOLATIONS intervals INTERVALS ]' )
   data = {
      'max-flaps': 'Specify the maximum number of flaps',
      'FLAPS': CliMatcher.IntegerMatcher( 1, 100, helpdesc='Max-flaps' ),
      'time': 'Specify the time period that flaps are counted (in seconds)',
      'SECONDS': CliMatcher.IntegerMatcher( 1, 1800, helpdesc='Time (in seconds)' ),
      'violations': 'Specify how many violations to be detected',
      'VIOLATIONS': CliMatcher.IntegerMatcher( 1, 1000, helpdesc='Violations' ),
      'intervals': 'Specify the intervals for monitoring violations',
      'INTERVALS': CliMatcher.IntegerMatcher( 1, 1000, helpdesc='Intervals' ),
   }

# Handler methods implementation for ProfileAllParametersCmd
class ProfileAllParametersHandler:
   @staticmethod
   def linkFlapHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      flaps = args[ 'FLAPS' ]
      seconds = args[ 'SECONDS' ]
      if profileName == "default":
         # "default" Name replicates global setting for max-flaps and time
         # violations and intervals are ignored for this profile
         setLinkFlapValues( mode, flaps, seconds )
         if 'violations' in args:
            # FIXME: this should be `mode.addError/Warning/Message`
            print( 'violations and intervals are ignored for default profile' )
      else:
         if profileName in linkFlapDampingConfig.profileConfig:
            # Link-flap profiles can't use names already taken by damping profiles.
            # Profile name spaces can't overlap because we use the same CLI commands
            # to set default profiles and assign profiles to interfaces.
            mode.addErrorAndStop(
               msg=f"Failed to create new link-flap profile {profileName}: "
                   f"damping profile {profileName} already exists",
            )
         profileConfig = linkFlapConfig.newProfileConfig( profileName )
         profileConfig.maxFlaps = flaps
         profileConfig.interval = seconds

         if 'VIOLATIONS' in args:
            profileConfig.violations = args[ 'VIOLATIONS' ]
         if 'INTERVALS' in args:
            profileConfig.trackedIntervals = args[ 'INTERVALS' ]

   @staticmethod
   def dampingHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName == "default":
         # Damping profile can't be named "default" because this name is
         # reserved for an implicit link-flap profile.
         mode.addError(
            msg=f"Failed to create new damping profile: "
                f"\"{profileName}\" name is reserved for link-flap profiles. "
                "Consider using a different profile name",
         )
         return
      if profileName in linkFlapConfig.profileConfig:
         # Damping profiles can't use names already taken by link-flap profiles.
         # Profile name spaces can't overlap because we use the same CLI commands
         # to set default profiles and assign profiles to interfaces.
         mode.addError(
            msg=f"Failed to create new damping profile: "
                f"link-flap profile {profileName} already exists. "
                "Consider using a name not overlapping with existing "
                "profiles.",
         )
         return
      childMode = mode.childMode( LinkFlapDampingConfigMode,
                                  param=profileName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def linkFlapNoHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName == "default":
         # "default" Name replicates global setting for max-flaps and time
         noLinkFlapValues( mode )
      else:
         del linkFlapConfig.defaultProfileList[ profileName ]
         for intfConfig in linkFlapConfig.intfConfig.values():
            del intfConfig.profileList[ profileName ]
         del linkFlapConfig.profileConfig[ profileName ]

   @staticmethod
   def dampingNoHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      del linkFlapDampingConfig.defaultProfileList[ profileName ]
      for sliceDir in linkFlapDampingIntfConfigDir.values():
         for intfConfig in sliceDir.intfConfig.values():
            del intfConfig.profileList[ profileName ]
      del linkFlapDampingConfig.profileConfig[ profileName ]

   @staticmethod
   def handler( mode, args ):
      if 'damping' in args:
         ProfileAllParametersHandler.dampingHandler( mode, args )
      else:
         ProfileAllParametersHandler.linkFlapHandler( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'profile' not in args:
         # We got called via our parent mode's `noOrDefaultHandler`.
         # Clear link flap config.
         linkFlapConfig.defaultProfileList.clear()
         linkFlapConfig.intfConfig.clear()
         linkFlapConfig.profileConfig.clear()
         # Clear link flap damping config.
         linkFlapDampingConfig.defaultProfileList.clear()
         for sliceDir in linkFlapDampingIntfConfigDir.values():
            sliceDir.intfConfig.clear()
         linkFlapDampingConfig.profileConfig.clear()
      elif 'damping' in args:
         ProfileAllParametersHandler.dampingNoHandler( mode, args )
      else:
         ProfileAllParametersHandler.linkFlapNoHandler( mode, args )

# --------------------------------------------------------------------------------
# [ no ] profile ( PROFILE_NAME | default ) ( damping | (max-flaps FLAPS time SECONDS
#                                   [ violations VIOLATIONS intervals INTERVALS ] ) )
# --------------------------------------------------------------------------------
# NOTE: "default" profile name can only be used when changing default link-flap
# parameters. "profile default damping" command will fail uncondiitonally.
class ProfileAllParametersCmd( CliCommand.CliCommandClass ):
   syntax = 'profile NAME_OR_DEFAULT ( LINK_FLAP_PARAMS | damping )'
   noOrDefaultSyntax = 'profile NAME_OR_DEFAULT [ damping ] ...'
   data = {
      'profile': 'Configure profiles for link-flap detection or dynamic damping',
      'NAME_OR_DEFAULT': ProfileNameExpression,
      'LINK_FLAP_PARAMS': LinkFlapProfilesParamsExpression,
      'damping': 'The profile is for link flap damping',
   }

   handler = ProfileAllParametersHandler.handler
   noOrDefaultHandler = ProfileAllParametersHandler.noOrDefaultHandler

LinkFlapConfigMode.addCommandClass( ProfileAllParametersCmd )

# --------------------------------------------------------------------------------
# [ no | default ] default-profiles [ { PROFILES } ]
# --------------------------------------------------------------------------------
class DefaultProfilesCmd( CliCommand.CliCommandClass ):
   syntax = 'default-profiles [ { PROFILES } ]'
   noOrDefaultSyntax = 'default-profiles ...'
   data = {
      'default-profiles': 'Specify a default profile list',
      'PROFILES': profileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      profileNameList = args.get( 'PROFILES' )
      if not profileNameList:
         DefaultProfilesCmd.noOrDefaultHandler( mode, args )
         return
      for profileName in linkFlapConfig.defaultProfileList:
         if profileName not in profileNameList:
            del linkFlapConfig.defaultProfileList[ profileName ]
      for profileName in linkFlapDampingConfig.defaultProfileList:
         if profileName not in profileNameList:
            del linkFlapDampingConfig.defaultProfileList[ profileName ]
      for profileName in profileNameList:
         name = profileName.profileName
         if name in linkFlapDampingConfig.profileConfig:
            linkFlapDampingConfig.defaultProfileList[ name ] = True
         elif name in linkFlapConfig.profileConfig:
            linkFlapConfig.defaultProfileList[ name ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      linkFlapConfig.defaultProfileList.clear()
      linkFlapDampingConfig.defaultProfileList.clear()

LinkFlapConfigMode.addCommandClass( DefaultProfilesCmd )

# -----------------------------
# The Interface mode.
# -----------------------------

# Setting Interface Mode:

class LinkFlapIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del linkFlapConfig.intfConfig[ self.intf_.name ]
      for sliceDir in linkFlapDampingIntfConfigDir.values():
         if self.intf_.name in sliceDir.intfConfig:
            del sliceDir.intfConfig[ self.intf_.name ]
            break

class LinkFlapIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # This is a bit hacky, but there's no better way to recognize physical Ethernet
      # and Management interfaces than to look at their name.
      return ( isinstance( mode.intf, EthIntfCli.EthPhyIntf ) and
               ( mode.intf.name.startswith( ( 'Ethernet', 'Management' ) ) ) )

IntfCli.IntfConfigMode.addModelet( LinkFlapIntfModelet )

# --------------------------------------------------------------------------------
# [ no | default ] monitor link-flap [ profiles { PROFILES } ]
# --------------------------------------------------------------------------------
class MonitorLinkFlapCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor link-flap [ profiles { PROFILES } ]'
   noOrDefaultSyntax = 'monitor link-flap ...'
   data = {
      'monitor': 'Configure monitoring',
      'link-flap': 'Specify the flap-setting values for link flap errdisable cause',
      'profiles': 'Assign profiles for link-flap detection or link-flap damping',
      'PROFILES': profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = linkFlapConfig.newIntfConfig( mode.intf.name )
      intfConfig.enabled = True
      intfDampingConfig = getDampingIntfConfig(
         mode.intf.slotId(),
         mode.intf.name,
      )
      intfDampingConfig.enabled = True

      profileNameList = set( args.get( 'PROFILES', () ) )
      if not profileNameList:
         intfConfig.profileList.clear()
         intfDampingConfig.profileList.clear()
         return

      for profileName in intfConfig.profileList:
         if profileName not in profileNameList:
            del intfConfig.profileList[ profileName ]
      for profileName in intfDampingConfig.profileList:
         if profileName not in profileNameList:
            del intfDampingConfig.profileList[ profileName ]
      for profileName in profileNameList:
         name = profileName.profileName
         if name in linkFlapConfig.profileConfig:
            intfConfig.profileList[ name ] = True
         elif name in linkFlapDampingConfig.profileConfig:
            intfDampingConfig.profileList[ name ] = True

   @staticmethod
   def noHandler( mode, args ):
      intfConfig = linkFlapConfig.newIntfConfig( mode.intf.name )
      intfConfig.enabled = False
      intfConfig.profileList.clear()
      intfDampingConfig = getDampingIntfConfig(
         mode.intf.slotId(),
         mode.intf.name,
      )
      intfDampingConfig.enabled = False
      intfDampingConfig.profileList.clear()

   @staticmethod
   def defaultHandler( mode, args ):
      del linkFlapConfig.intfConfig[ mode.intf.name ]
      sliceDir = linkFlapDampingIntfConfigDir[ mode.intf.slotId() ]
      del sliceDir.intfConfig[ mode.intf.name ]

LinkFlapIntfModelet.addCommandClass( MonitorLinkFlapCmd )

# Damping profile mode
class LinkFlapDampingConfigMode( LinkFlapDampingMode, BasicCli.ConfigModeBase ):
   name = 'Damping profile configuration'

   def __init__( self, parent, session, param ):
      self.profile = linkFlapDampingConfig.newProfileConfig( param )
      LinkFlapDampingMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getDampProfile( self ):
      return self.profile

   def setProfileAttribute( self,
                            maxThreshold=None,
                            suppressThreshold=None,
                            reuseThreshold=None,
                            trackingThreshold=None,
                            penaltyFaultLocal=None,
                            penaltyFaultRemote=None,
                            decayHalfLife=None,
                           ):
      p = self.profile.parameters

      def orDefault( value, default ):
         return default if value is None else value

      self.profile.parameters = Tac.Value(
         'LinkFlap::ProfileParameters',
         maxThreshold=orDefault( maxThreshold, p.maxThreshold ),
         suppressThreshold=orDefault( suppressThreshold, p.suppressThreshold ),
         reuseThreshold=orDefault( reuseThreshold, p.reuseThreshold ),
         trackingThreshold=orDefault( trackingThreshold, p.trackingThreshold ),
         penaltyFaultLocal=orDefault( penaltyFaultLocal, p.penaltyFaultLocal ),
         penaltyFaultRemote=orDefault( penaltyFaultRemote, p.penaltyFaultRemote ),
         decayHalfLife=orDefault( decayHalfLife, p.decayHalfLife ),
      )

# --------------------------------------------------------------------------------
# [ no | default ] penalty mac fault ( local | remote ) PENALTY
# --------------------------------------------------------------------------------
class LinkFlapDampingFaultCmd( CliCommand.CliCommandClass ):
   syntax = 'penalty mac fault ( local | remote ) PENALTY'
   noOrDefaultSyntax = 'penalty mac fault ( local | remote ) ...'
   data = {
      'penalty': 'Link flap damping penalty configuration',
      'mac': 'Configure penalty for MAC state change',
      'fault': 'Configure penalty for MAC fault change',
      'local': 'Configure penalty for local fault change',
      'remote': 'Configure penalty for remote fault change',
      'PENALTY': CliMatcher.IntegerMatcher(
         0, 5000,
         helpdesc='0: Do not apply any penalty, 1-5000: Penalty value for fault' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'local' in args:
         mode.setProfileAttribute( penaltyFaultLocal=args[ 'PENALTY' ] )
      elif 'remote' in args:
         mode.setProfileAttribute( penaltyFaultRemote=args[ 'PENALTY' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'local' in args:
         mode.setProfileAttribute(
            penaltyFaultLocal=DefaultParametersType.penaltyFaultLocalDefault,
         )
      elif 'remote' in args:
         mode.setProfileAttribute(
            penaltyFaultRemote=DefaultParametersType.penaltyFaultRemoteDefault,
         )

LinkFlapDampingConfigMode.addCommandClass( LinkFlapDampingFaultCmd )

# --------------------------------------------------------------------------------
# [ no | default] penalty threshold reuse REUSE_THRESHOLD \
#              suppression SUPPRESS_THRESHOLD maximum MAX_THRESHOLD
# --------------------------------------------------------------------------------
class LinkFlapDampingThrehsoldCmd( CliCommand.CliCommandClass ):
   syntax = (
     'penalty threshold [ reuse REUSE_THRESHOLD ] '
     '[ suppression SUPPRESS_THRESHOLD ][ maximum MAX_THRESHOLD ] '
   )
   noOrDefaultSyntax = (
     'penalty threshold [ reuse [ REUSE_THRESHOLD ] ] '
     '[ suppression [ SUPPRESS_THRESHOLD ] ][ maximum [ MAX_THRESHOLD ] ]'
   )
   data = {
      'penalty': 'Link flap damping penalty configuration',
      'threshold': 'Link flap damping threshold configuration',
      'reuse': 'Value of penalty below which suppressed link would be reused',
      'REUSE_THRESHOLD': CliMatcher.IntegerMatcher(
         0, 1000000,
         helpdesc='Penalty reuse threshold value' ),
      'suppression': 'Value of penalty above which link would be suppressed',
      'SUPPRESS_THRESHOLD': CliMatcher.IntegerMatcher(
         0, 1000000,
         helpdesc='Penalty suppression threshold value' ),
      'maximum': 'Maximum value of penalty for a link',
      'MAX_THRESHOLD': CliMatcher.IntegerMatcher(
         0, 1000000,
         helpdesc='Penalty maximum threshold value' ),
   }

   @staticmethod
   def handler( mode, args ):
      reuseThreshold = args.get( 'REUSE_THRESHOLD' )
      if reuseThreshold is not None:
         mode.setProfileAttribute( reuseThreshold=reuseThreshold )
         mode.setProfileAttribute( trackingThreshold=int( reuseThreshold / 2 ) )

      suppressThreshold = args.get( 'SUPPRESS_THRESHOLD' )
      if suppressThreshold is not None:
         mode.setProfileAttribute( suppressThreshold=suppressThreshold )

      maxThreshold = args.get( 'MAX_THRESHOLD' )
      if maxThreshold is not None:
         mode.setProfileAttribute( maxThreshold=maxThreshold )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'reuse' in args:
         mode.setProfileAttribute(
            reuseThreshold=DefaultParametersType.reuseThresholdDefault
         )
         mode.setProfileAttribute(
            trackingThreshold=DefaultParametersType.trackingThresholdDefault,
         )
      if 'suppression' in args:
         mode.setProfileAttribute(
            suppressThreshold=DefaultParametersType.suppressThresholdDefault,
         )
      if 'maximum' in args:
         mode.setProfileAttribute(
            maxThreshold=DefaultParametersType.maxThresholdDefault
         )

LinkFlapDampingConfigMode.addCommandClass( LinkFlapDampingThrehsoldCmd )

# --------------------------------------------------------------------------------
# [ no | default ] penalty decay half-life HALF_LIFE ( seconds | minutes)
# --------------------------------------------------------------------------------
class LinkFlapDampingHalfLifeCmd( CliCommand.CliCommandClass ):
   syntax = 'penalty decay half-life HALF_LIFE UNIT'
   noOrDefaultSyntax = 'penalty decay half-life ...'
   data = {
      'penalty': 'Link flap damping penalty configuration',
      'decay': 'Configure decay rate for penalty',
      'half-life': 'Configure decay half-life for penalty',
      'HALF_LIFE': CliMatcher.IntegerMatcher(
         1, 5000,
         helpdesc='Decay half-life value for penalty' ),
      'UNIT': CliMatcher.EnumMatcher( {
         'seconds': 'Half-life time in seconds',
         'minutes': 'Half-life time in minutes' } ),
   }

   @staticmethod
   def handler( mode, args ):
      decay = args[ 'HALF_LIFE' ]
      if args[ 'UNIT' ] == 'minutes':
         decay *= 60
      mode.setProfileAttribute( decayHalfLife=float( decay ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setProfileAttribute(
         decayHalfLife=DefaultParametersType.decayHalfLifeDefault
      )

LinkFlapDampingConfigMode.addCommandClass( LinkFlapDampingHalfLifeCmd )

# Plugin initialization
def Plugin( entityManager ):
   global linkFlapConfig, linkFlapDampingConfig
   linkFlapConfig = ConfigMount.mount(
      entityManager, "interface/errdisable/linkFlapConfig", "LinkFlap::Config", "w" )
   linkFlapDampingConfig = ConfigMount.mount(
      entityManager,
      "interface/archer/config/eth/linkFlapDamping/dampingConfig",
      "LinkFlap::DampingConfig", "w" )
   IntfCli.Intf.registerDependentClass( LinkFlapIntfCleaner )
   for sliceName in sliceNames:
      linkFlapDampingIntfConfigDir[ sliceName ] = ConfigMount.mount(
         entityManager,
         f'interface/archer/config/eth/linkFlapDamping/slice/{sliceName}',
         'LinkFlap::IntfConfigDir',
         'wi',
      )
