# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This module implement the CLI for displaying and clearing LLDP status.
"""

# XXX TODO
#
# -  Need to worry about atomicity in these commands.  There are some places where
#    execution could fail if the entities change during execution of a command.
#
# -  Need to take care of line-wrapping for ASCII/hex strings.
#
# -  'show lldp errors'?
#
# -  Allow an interface filter on 'show lldp counters'?

import socket, struct
import re
import Tac
import CliParser
import Arnet
import UtmpDump
import TacSigint
import IanaAddressFamilyNumbers
import IanaMau
import LazyMount
import Plugins
import LldpConstants
import LldpLib
import Intf.IntfRange
import Intf.Log
from LldpTypes import ( PowerViaMdiInfo,
                        PowerViaMdiInfo_IeeeStandard,
                        escapedString )
from IpLibConsts import DEFAULT_VRF
from EnvironmentTypes import PortStatusTacTypes as EnvPwrPortStatus
from CliPlugin.LldpModel import ( Lldp,
                                  LldpInterface,
                                  LldpInterfaceTraffic,
                                  LldpLocal,
                                  LldpMedInfo,
                                  LldpNeighbor,
                                  LldpNeighborInfo,
                                  LldpNeighbors,
                                  LldpNeighborsDetail,
                                  LldpNeighborsList,
                                  LldpTraffic,
                                  LocalInterfaceInfo,
                                  ManagementAddr,
                                  MedCapabilitiesInfo,
                                  MedNetworkPolicyInfo,
                                  NeighborInterfaceInfo,
                                  SystemCapabilities,
                                  TlvType,
                                  UnknownOrgDefTlv,
                                  UnknownTlv )
import CliPlugin.TechSupportCli
from Toggles import LldpToggleLib

btStr = "Lldp::PowerViaMdiInfo::Bt::"
envPwr = EnvPwrPortStatus.namespace + "::"

lldpCliConfig = None
remTablesCliConfig = None
lldpConfig = None
lldpStatus = None
lldpLocalSystem = None
lldpCounterCheckpoint = None
recognizedIfTypes = [ 'unknown', 'ifIndex', 'systemPortNumber' ]

### General functions.

def checkLldpEnabled( mode ):
   if not lldpCliConfig.enabled:
      mode.addError( 'LLDP is not enabled' )
      raise CliParser.AlreadyHandledError

def lldpIterator( mode, intfs, collection ):
   intfs = intfs or Arnet.sortIntf( collection )
   for intf in intfs:
      try:
         yield intf, collection[ intf ]
      except KeyError:
         mode.addWarning( 'Interface %s does not exist' % intf )

### Formatting functions.

def mauTypeToStr( mauType ):
   name = IanaMau.mauTypeNames.get( mauType, 'Unknown' )
   return name

def addrFamilyNumToStr( addrFamilyNum ):
   name = IanaAddressFamilyNumbers.addrFamilyNames.get( addrFamilyNum, 'Unknown' )
   return name.lower()

def formatAddr( addrFamilyNum, addr, short=True, canonical=False ):
   if addrFamilyNum == IanaAddressFamilyNumbers.ipV4:
      if len( addr ) == 4:
         s = socket.inet_ntop( socket.AF_INET, addr )
      else:
         if short:
            s = '(Invalid IPv4 address)'
         else:
            s = 'Invalid IPv4 address: %s' % addr.hex( ' ', 1 )
   elif addrFamilyNum == IanaAddressFamilyNumbers.ipV6:
      if len( addr ) == 16:
         s = socket.inet_ntop( socket.AF_INET6, addr )
      else:
         if short:
            s = '(Invalid IPv6 address)'
         else:
            s = 'Invalid IPv6 address: %s' % addr.hex( ' ', 1 )
   elif addrFamilyNum == IanaAddressFamilyNumbers.all802:
      s = formatMacAddress( addr, short, canonical )
   else:
      s = addr.hex( ' ', 1 )
   return s

def formatMacAddress( addr, short=True, canonical=False ):
   if len( addr ) == 6:
      if canonical:
         s = '%02x:%02x:%02x:%02x:%02x:%02x' % struct.unpack( '>BBBBBB', addr )
      else:
         s = '%04x.%04x.%04x' % struct.unpack( '>HHH', addr )
   else:
      if short:
         s = '(Invalid MAC address)'
      else:
         s = 'Invalid MAC address: %s' % addr.hex( ' ', 1 )
   return s

def chassisIdToStr( chassisIdentifier, short=True, canonical=False ):
   subtype = chassisIdentifier.chassisIdSubtype
   chassisId = chassisIdentifier.chassisId
   if subtype in [ 'cidChassisComponent',
                   'cidInterfaceAlias',
                   'cidPortComponent',
                   'cidInterfaceName',
                   'cidLocal' ]:
      if short:
         s = escapedString( chassisId )
      else:
         s = '"%s"' % escapedString( chassisId )
   elif subtype == 'cidMacAddress':
      s = formatMacAddress( chassisId, short, canonical )
   elif subtype == 'cidNetworkAddress':
      assert chassisId # Guaranteed by [802.1AB-2005] section 9.5.2.
      addrFamilyNum = chassisId[ 0 ]
      addr = chassisId[ 1: ]
      s = formatAddr( addrFamilyNum, addr, short, canonical  )
   else:
      s = chassisId.hex( ' ', 1 )
   return s

chassisIdSubtypes = {
   'cidReserved' : 'reserved',
   'cidChassisComponent' : 'chassisComponent',
   'cidInterfaceAlias' : 'interfaceAlias',
   'cidPortComponent' : 'portComponent',
   'cidMacAddress' : 'macAddress',
   'cidNetworkAddress' : 'networkAddress',
   'cidInterfaceName' : 'interfaceName',
   'cidLocal' : 'local',
   'cidUnknown' : 'unknown',
}

def chassisIdSubtypeToStr( chassisIdentifier ):
   subtype = chassisIdentifier.chassisIdSubtype
   return chassisIdSubtypes.get( subtype )

portIdSubtypes = {
   'pidReserved' : 'reserved',
   'pidInterfaceAlias' : 'interfaceAlias',
   'pidPortComponent' : 'portComponent',
   'pidMacAddress' : 'macAddress',
   'pidNetworkAddress' : 'networkAddress',
   'pidInterfaceName' : 'interfaceName',
   'pidAgentCircuitId' : 'agentCircuitId',
   'pidLocal' : 'local',
   'pidUnknown' : 'unknown',
}

def portIdToStr( portIdentifier, short=True ):
   subtype = portIdentifier.portIdSubtype
   portId = portIdentifier.portId
   if subtype in [ 'pidInterfaceAlias',
                   'pidPortComponent',
                   'pidInterfaceName',
                   'pidLocal' ]:
      if short:
         s = escapedString( portId )
      else:
         s = '"%s"' % escapedString( portId )
   elif subtype == 'pidMacAddress':
      s = formatMacAddress( portId, short )
   elif subtype == 'pidNetworkAddress':
      assert portId # Guaranteed by [802.1AB-2005] section 9.5.3.
      addrFamilyNum = portId[ 0 ]
      addr = portId[ 1: ]
      s = formatAddr( addrFamilyNum, addr, short )
   else:
      # Subtype is either 'pidAgentCircuitId' or unknown.
      s = portId.hex( ' ', 1 )
   return s

def portIdSubtypeToStr( portIdentifier ):
   subtype = portIdentifier.portIdSubtype
   return portIdSubtypes.get( subtype )

def sysCapToStr( cap ):
   caps = []
   if cap.other:
      caps.append( 'Other' )
   if cap.repeater:
      caps.append( 'Repeater' )
   if cap.bridge:
      caps.append( 'Bridge' )
   if cap.wlanAccessPoint:
      caps.append( 'WLAN Access Point' )
   if cap.router:
      caps.append( 'Router' )
   if cap.telephone:
      caps.append( 'Telephone' )
   if cap.docsisCableDevice:
      caps.append( 'DOCSIS Cable Device' )
   if cap.stationOnly:
      caps.append( 'Station Only' )
   return ", ".join( caps )

def getSysCaps( cap, sysCaps, enabled=False ):
   if cap.other:
      sysCaps.other = enabled 
   if cap.repeater:
      sysCaps.repeater = enabled
   if cap.bridge:
      sysCaps.bridge = enabled
   if cap.wlanAccessPoint:
      sysCaps.wlanAccessPoint = enabled
   if cap.router:
      sysCaps.router = enabled
   if cap.telephone:
      sysCaps.telephone = enabled
   if cap.docsisCableDevice:
      sysCaps.docsisCableDevice = enabled
   if cap.stationOnly:
      sysCaps.stationOnly = enabled
   return sysCaps

def autoNegCapsList( autoNegCapBits ):
   autoNegCaps = []
   # Note that the order of the following tests is deliberately chosen so that the
   # capabilities are listed in a logical order in the command output.
   if autoNegCapBits.bFdxPause:
      autoNegCaps.append( 'PAUSE' )
   if autoNegCapBits.bFdxAPause:
      autoNegCaps.append( 'Asymmetric PAUSE' )
   if autoNegCapBits.bFdxSPause:
      autoNegCaps.append( 'Symmetric PAUSE' )
   if autoNegCapBits.bFdxBPause:
      autoNegCaps.append( 'Asymmetric and Symmetric PAUSE' )
   if autoNegCapBits.b10baseT:
      autoNegCaps.append( '10BASE-T (half-duplex)' )
   if autoNegCapBits.b10baseTFD:
      autoNegCaps.append( '10BASE-T (full-duplex)' )
   if autoNegCapBits.b100baseT4:
      autoNegCaps.append( '100BASE-T4' )
   if autoNegCapBits.b100baseTX:
      autoNegCaps.append( '100BASE-TX (half-duplex)' )
   if autoNegCapBits.b100baseTXFD:
      autoNegCaps.append( '100BASE-TX (full-duplex)' )
   if autoNegCapBits.b100baseT2:
      autoNegCaps.append( '100BASE-T2 (half-duplex)' )
   if autoNegCapBits.b100baseT2FD:
      autoNegCaps.append( '100BASE-T2 (full-duplex)' )
   if autoNegCapBits.b1000baseX:
      autoNegCaps.append( '1000BASE-X (half-duplex)' )
   if autoNegCapBits.b1000baseXFD:
      autoNegCaps.append( '1000BASE-X (full-duplex)' )
   if autoNegCapBits.b1000baseT:
      autoNegCaps.append( '1000BASE-T (half-duplex)' )
   if autoNegCapBits.b1000baseTFD:
      autoNegCaps.append( '1000BASE-T (full-duplex)' )
   if autoNegCapBits.bOther:
      autoNegCaps.append( 'Other' )
   return autoNegCaps

def powerPairsToStr( powerPairs ):
   if powerPairs == 1:
      s = 'signal'
   elif powerPairs == 2:
      s = 'spare'
   else:
      s = 'unknown'
   return s

#For IEEE802.3 Power Via MDI

toWatts = PowerViaMdiInfo().deciWattsToWatts

def isAtLeastIeeeStdAt( std ):
   return std in [ PowerViaMdiInfo_IeeeStandard.IeeeStandard_802_3at,
                   PowerViaMdiInfo_IeeeStandard.IeeeStandard_802_3bt ]

def isAtLeastIeeeStdBt( std ):
   return std in [ PowerViaMdiInfo_IeeeStandard.IeeeStandard_802_3bt ]

_powerTypeMapping = { 'PowerType_Type1_PSE_Device' : 'type1PseDevice',
                      'PowerType_Type1_PD_Device'  : 'type1PdDevice',
                      'PowerType_Type2_PSE_Device' : 'type2PseDevice',
                      'PowerType_Type2_PD_Device'  : 'type2PdDevice' }

def powerTypeToModel( powerType ):
   return _powerTypeMapping[ powerType ]

_pdTypes = [ 'PowerType_Type1_PD_Device', 'type1PdDevice',
             'PowerType_Type2_PD_Device', 'type2PdDevice'
             'Type3SingleSignaturePd', 'type3SingleSignaturePd',
             'Type3DualSignaturePd', 'type3DualSignaturePd',
             'Type4SingleSignaturePd', 'type4SingleSignaturePd',
             'Type4DualSignaturePd', 'type4DualSignaturePd' ]
def isPd( val ):
   return val in _pdTypes

def isPse( val ):
   return not isPd( val )

_pdPowerSourceMapping = { 'PowerSource_Unknown' : 'unknown',
                          'PowerSource_Pse_OR_PrimaryPowerSource'  : 'pse',
                          'PowerSource_Reserved_OR_BackupSource' : 'reserved',
                          'PowerSource_PseAndLocal_OR_Reserved'  : 'pseAndLocal' }
_psePowerSourceMapping = { 'PowerSource_Unknown' : 'unknown',
                           'PowerSource_Pse_OR_PrimaryPowerSource'  : 'primary',
                           'PowerSource_Reserved_OR_BackupSource' : 'backup',
                           'PowerSource_PseAndLocal_OR_Reserved'  : 'reserved' }

def powerSourceToModel( powerSource, powerType ):
   if isPd( powerType ):
      return _pdPowerSourceMapping[ powerSource ]
   else:
      return _psePowerSourceMapping[ powerSource ]

_powerPriorityMapping = { 'PowerPriority_Unknown' : 'unknown',
                          'PowerPriority_Critical'  : 'critical',
                          'PowerPriority_High' : 'high',
                          'PowerPriority_Low'  : 'low' }

def powerPriorityToModel( powerPriority ):
   return _powerPriorityMapping[ powerPriority ]

_medPdClassMapping = { 'classUnknown' : 0,
                       'class0' : 1,
                       'class1' : 2,
                       'class2' : 3,
                       'class3' : 4,
                       'class4' : 5,
                       'class5' : 6,
                       'class6' : 7,
                       'class7' : 8,
                       'class8' : 9 }

def pdClassToModel( pdClass ):
   return _medPdClassMapping[ pdClass ]

_btPowerMapping = {
      'TwoPairPoweringPse' : 'twoPairPowering',
      'FourPairPoweringSingleSignaturePse' : 'fourPairPoweringSingleSignature',
      'FourPairPoweringDualSignaturePse' : 'fourPairPoweringDualSignature',
      'PoweredSingleSignaturePd' : 'poweredSingleSignature',
      'TwoPairPoweringDualSignaturePd' : 'twoPairPoweringDualSignature',
      'FourPairPoweringDualSignaturePd' : 'fourPairPoweringDualSignature',
      'AlternativeA' : 'alternativeA',
      'AlternativeB' : 'alternativeB',
      'BothAlternatives' : 'bothAlternatives',
      'DualSignaturePowerClass1' : 'powerClass1',
      'DualSignaturePowerClass2' : 'powerClass2',
      'DualSignaturePowerClass3' : 'powerClass3',
      'DualSignaturePowerClass4' : 'powerClass4',
      'DualSignaturePowerClass5' : 'powerClass5',
      ( 'SingleSignaturePd_or_TwoPairOnlyPse', True ) : 'singleSignaturePd',
      ( 'SingleSignaturePd_or_TwoPairOnlyPse', False ) : 'twoPairOnlyPse',
      'PowerClassExt1' : 'powerClassExt1',
      'PowerClassExt2' : 'powerClassExt2',
      'PowerClassExt3' : 'powerClassExt3',
      'PowerClassExt4' : 'powerClassExt4',
      'PowerClassExt5' : 'powerClassExt5',
      'PowerClassExt6' : 'powerClassExt6',
      'PowerClassExt7' : 'powerClassExt7',
      'PowerClassExt8' : 'powerClassExt8',
      'DualSignaturePd' : 'dualSignaturePd',
      'Type3Pse' : 'type3Pse',
      'Type4Pse' : 'type4Pse',
      'Type3SingleSignaturePd' : 'type3SingleSignaturePd',
      'Type3DualSignaturePd' : 'type3DualSignaturePd',
      'Type4SingleSignaturePd' : 'type4SingleSignaturePd',
      'Type4DualSignaturePd' : 'type4DualSignaturePd',
      }

def btPowerFieldToModel( value ):
   if ( not isinstance( value, str ) and
        value[ 0 ] != 'SingleSignaturePd_or_TwoPairOnlyPse' ):
      value = value[ 0 ]

   if value not in _btPowerMapping and 'Reserved' in value:
      # This is used to map all Reserved fiedls to None
      return None
   return _btPowerMapping[ value ]

_pdClassToPowerClassExt = {
      'class0' : 'DualSignaturePd',
      'class1' : 'PowerClassExt1',
      'class2' : 'PowerClassExt2',
      'class3' : 'PowerClassExt3',
      'class4' : 'PowerClassExt4',
      'class5' : 'PowerClassExt5',
      'class6' : 'PowerClassExt6',
      'class7' : 'PowerClassExt7',
      'class8' : 'PowerClassExt8',
      'classMismatch' : 'PowerClassExtReserved0',
      'classUnknown' : 'PowerClassExtReserved0',
      'classInvalid' : 'PowerClassExtReserved0',
      }

def pdClassToPowerClassExt( value ):
   return _pdClassToPowerClassExt.get( value, None )

_medCapabilitiesDeviceTypeMapping = { 
      'MedDeviceType_NotDefined' : 'notDefined',
      'MedDeviceType_EndpointClass1' : 'endpointClass1',
      'MedDeviceType_EndpointClass2' : 'endpointClass2',
      'MedDeviceType_EndpointClass3' : 'endpointClass3',
      'MedDeviceType_NetworkConnectivity' : 'networkConnectivity',
      'MedDeviceType_Reserved' : 'reserved' }

def medDeviceTypeToModel( deviceType ):
   return _medCapabilitiesDeviceTypeMapping[ deviceType ]

def interfaceIdWithoutQuotation( portIdentifier ):
   subtype = portIdentifier.portIdSubtype
   quoted = subtype in [ 'pidInterfaceAlias',
                         'pidPortComponent',
                         'pidInterfaceName',
                         'pidLocal' ]
   interfaceId = portIdToStr( portIdentifier, short=False )
   if quoted:
      interfaceId = re.sub('^"(.*)"$', r'\1', interfaceId )
   return interfaceId

def printHexDump( data, pfx='', rowLen=16 ):
   hexDump = []
   while data:
      line = pfx + ' '.join( '%02x' % i for i in iter( data[ : rowLen ] ) )
      data = data[ rowLen: ]
      hexDump.append( line )
   return hexDump

counterNames = (
   'txFramesTotal',
   'txFramesLengthExceeded',
   'txFramesNoMemErrors',
   'rxFramesDiscardedTotal',
   'rxFramesErrors',
   'rxFramesTotal',
   'rxTLVsDiscardedTotal',
   'rxTLVsUnrecognizedTotal' )

def _lldpSessionCheckpoint( mode ):
   sessionCheckpoint = mode.session.sessionData( 'LldpStatusCli.sessionCheckpoint',
                                                 None )
   if sessionCheckpoint is None:
      sessionCheckpoint = Tac.newInstance( 'Lldp::CounterCheckpoint', 'lldp' )
      mode.session.sessionDataIs( 'LldpStatusCli.sessionCheckpoint',
                                  sessionCheckpoint )
   return sessionCheckpoint

def portCounterValues( mode, portStatus ):
   intfName = portStatus.intfId

   # Get session snapshot.
   sessionCheckpoint = _lldpSessionCheckpoint( mode )
   sessionSnapshot = sessionCheckpoint.portCounterCheckpoint.get( intfName )

   # Get global snapshot.
   globalSnapshot = lldpCounterCheckpoint.portCounterCheckpoint.get( intfName )

   # Create a dummy snapshot that was logically created at the same time as the
   # PortStatus.
   zero = Tac.Value(
      "Lldp::PortCounterCheckpoint", timestamp=portStatus.creationTime )

   # Choose the newest of the three snapshots.
   snapshots = [ zero ]
   if sessionSnapshot is not None:
      snapshots.append( sessionSnapshot )
   if globalSnapshot is not None:
      snapshots.append( globalSnapshot )
   snapshots.sort( key=lambda snapshot: snapshot.timestamp )
   snapshot = snapshots[ -1 ]

   deltas = {}
   for counterName in counterNames:
      currentValue = getattr( portStatus, counterName )
      snapshotValue = getattr( snapshot, counterName )
      assert currentValue >= snapshotValue
      deltas[ counterName ] = currentValue - snapshotValue
   return deltas

### CLI command implementations.

def doShowLldp( mode, args ):
   intfs = args.get( 'INTFS' )
   checkLldpEnabled( mode )
   lldpModel = Lldp()
   lldpModel.messageTxInterval = lldpCliConfig.messageTxInterval
   lldpModel.messageTxHoldTime = lldpCliConfig.messageTxHoldTime
   lldpModel.reinitDelay =  lldpCliConfig.reinitDelay
   mgmtAddrVrf = lldpStatus.mgmtAddrVrf or DEFAULT_VRF
   lldpModel.managementAddressVrf = mgmtAddrVrf

   tlvs = []
   if lldpCliConfig.portDescTlvEnabled:
      tlvs.append( TlvType( tlvType='portDescription' ) )
   if lldpCliConfig.sysNameTlvEnabled:
      tlvs.append( TlvType( tlvType='systemName' ) )
   if lldpCliConfig.sysDescTlvEnabled:
      tlvs.append( TlvType( tlvType='systemDescription' ) )
   if lldpCliConfig.sysCapTlvEnabled:
      tlvs.append( TlvType( tlvType='systemCapabilities' ) )
   if lldpCliConfig.mgmtAddrTlvEnabled:
      tlvs.append( TlvType( tlvType='managementAddress' ) )
      if lldpCliConfig.mgmtAddrTxOption.mgmtAddrTxMode == "mgmtAddrTxModeBest":
         mgmtAddrString = "best"
      elif lldpCliConfig.mgmtAddrTxOption.mgmtAddrTxMode == "mgmtAddrTxModeAll":
         mgmtAddrString = "all"
      else:
         assert lldpCliConfig.mgmtAddrTxOption.mgmtAddrTxMode == \
            "mgmtAddrTxModeIntfName"
         mgmtAddrString = lldpCliConfig.mgmtAddrTxOption.intfName
      lldpModel.managementAddressTransmitted = mgmtAddrString
   if lldpCliConfig.portVlanIdTlvEnabled:
      tlvs.append( TlvType( tlvType='portVlan' ) )
   if lldpCliConfig.vlanNameTlvEnabled:
      tlvs.append( TlvType( tlvType='portVlanName' ) )
   if lldpCliConfig.linkAggregation8021TlvEnabled:
      tlvs.append( TlvType( tlvType='linkAggregation8021' ) )
   if lldpCliConfig.linkAggregation8023TlvEnabled:
      tlvs.append( TlvType( tlvType='linkAggregation8023' ) )
   if lldpCliConfig.maxFrameSizeTlvEnabled:
      tlvs.append( TlvType( tlvType='maxFrameSize' ) )
   if lldpCliConfig.powerViaMdiTlvEnabled:
      tlvs.append( TlvType( tlvType='powerViaMdi' ) )
   if lldpCliConfig.medNetworkPolicyTlvEnabled:
      tlvs.append( TlvType( tlvType='medCapabilities' ) )
      tlvs.append( TlvType( tlvType='medNetworkPolicy' ) )
   if LldpToggleLib.toggleLldpSerialNumberTlvEnabled() and \
      lldpCliConfig.medSerialNumberTlvEnabled:
      tlvs.append( TlvType( tlvType='serialNumber' ) )
   lldpModel.enabledTlvTypes = tlvs

   # Note that we look at Config and not CliConfig, because CliConfig will only
   # contain interfaces that have been explicitly configured by the adminstrator,
   # whereas we want to show all LLDP-capable interfaces.
   for intfName, portConfig in lldpIterator( mode, intfs, lldpConfig.portConfig ):
      lldpIntModel = LldpInterface()
      lldpIntModel.txEnabled = portConfig.adminStatus in [ 'txOnly', 'txAndRx' ]
      lldpIntModel.rxEnabled = portConfig.adminStatus in [ 'rxOnly', 'txAndRx' ]
      lldpModel.lldpInterfaces[ intfName ] = lldpIntModel
   return lldpModel

def doShowLldpCounters( mode, args ):
   intfs = args.get( 'INTFS' )
   checkLldpEnabled( mode )
   lldpIntfTrafficDict = {}

   for intfName, portStatus in lldpIterator( mode, intfs, lldpStatus.portStatus ):
      portCounters = portCounterValues( mode, portStatus )
      lldpIntfTraffic = LldpInterfaceTraffic()
      lldpIntfTraffic.txFrames =  portCounters[ 'txFramesTotal' ]
      lldpIntfTraffic.txFramesLengthExceeded = \
            portCounters[ 'txFramesLengthExceeded' ]
      lldpIntfTraffic.rxFrames = portCounters[ 'rxFramesTotal' ]
      lldpIntfTraffic.rxErrors = portCounters[ 'rxFramesErrors' ]
      lldpIntfTraffic.rxDiscards = portCounters[ 'rxFramesDiscardedTotal' ]
      lldpIntfTraffic.tlvsDiscarded = portCounters[ 'rxTLVsDiscardedTotal' ]
      lldpIntfTraffic.tlvsUnknown = portCounters[ 'rxTLVsUnrecognizedTotal' ]
      lldpIntfTrafficDict[ intfName ] = lldpIntfTraffic
   return LldpTraffic( interfaces=lldpIntfTrafficDict )

   # XXX We're not currently displaying 'rxAgeoutsTotal' anywhere -- should we?  I
   # don't think we need to.  I believe the only reason that rxAgeoutsTotal exists in
   # the MIB is to make it more efficient for an SNMP client to figure out which
   # neighbors have been deleted.

def doShowNeighbor( intfName, nbr ):
   lldpNeighbor = LldpNeighbor()
   if nbr.sysName and nbr.sysName.value:
      lldpNeighbor.neighborDevice = escapedString( nbr.sysName.value )
   else:
      lldpNeighbor.neighborDevice = chassisIdToStr( nbr.msap.chassisIdentifier )
   lldpNeighbor.neighborPort = portIdToStr( nbr.msap.portIdentifier )
   lldpNeighbor.port = intfName
   lldpNeighbor.ttl = nbr.ttl
   return lldpNeighbor

def doShowLldpNeighbors( mode, args ):
   intfs = args.get( 'INTFS' )
   checkLldpEnabled( mode )
   lldpNeighbors = LldpNeighbors()

   # XXX I'm not sure this really belongs here.  It might make more sense to have a
   # new command (show lldp statistics?) to display this.
   if lldpStatus.remTablesLastChangeTime:
      lldpNeighbors.tablesLastChangeTime = \
            lldpStatus.remTablesLastChangeTime + Tac.utcNow() - Tac.now()
   else:
      lldpNeighbors.tablesLastChangeTime = None
   lldpNeighbors.tablesInserts = lldpStatus.remTablesInserts
   lldpNeighbors.tablesDeletes = lldpStatus.remTablesDeletes
   lldpNeighbors.tablesDrops = lldpStatus.remTablesDrops
   lldpNeighbors.tablesAgeOuts = lldpStatus.remTablesAgeouts

   for intfName, portStatus in lldpIterator( mode, intfs, lldpStatus.portStatus ):
      for _, remoteSystem in sorted( portStatus.remoteSystem.items() ):
         lldpNeighbors.lldpNeighbors.append(
               doShowNeighbor( intfName, remoteSystem ) )
         TacSigint.check()
   return lldpNeighbors

def doShowNeighborDetail( nbr ):
   lldpNeighbor = LldpNeighborInfo()
   utcOffset = Tac.utcNow() - Tac.now()
   lldpNeighbor.lastContactTime = nbr.lastRefreshTime + utcOffset
   lldpNeighbor.neighborDiscoveryTime = nbr.creationTime + utcOffset
   lldpNeighbor.lastChangeTime = nbr.lastChangeTime + utcOffset
   lldpNeighbor.ttl = nbr.ttl

   # Mandatory TLVs.
   lldpNeighbor.chassisIdType = chassisIdSubtypeToStr( nbr.msap.chassisIdentifier )
   lldpNeighbor.chassisId = chassisIdToStr( nbr.msap.chassisIdentifier, short=False )
   # XXX Should we explicitly store whether or not the following TLVs were received?
   # Optional TLVs.
   if nbr.sysName:
      lldpNeighbor.systemName = escapedString( nbr.sysName.value )
   if nbr.sysDesc:
      lldpNeighbor.systemDescription = escapedString( nbr.sysDesc.value )
   systemCaps = SystemCapabilities()
   if nbr.systemCapabilities:
      getSysCaps( nbr.systemCapabilities.sysCapSupported, systemCaps )
      getSysCaps( nbr.systemCapabilities.sysCapEnabled, systemCaps, True )
      lldpNeighbor.systemCapabilities = systemCaps

   lldpNeighborIntf = NeighborInterfaceInfo()
   lldpNeighborIntf.interfaceIdType = \
         portIdSubtypeToStr( nbr.msap.portIdentifier )
   lldpNeighborIntf.interfaceId = \
         portIdToStr( nbr.msap.portIdentifier, short=False )
   lldpNeighborIntf.interfaceId_v2 = \
         interfaceIdWithoutQuotation( nbr.msap.portIdentifier )
   if nbr.portDesc:
      lldpNeighborIntf.interfaceDescription = escapedString( nbr.portDesc.value )
   if nbr.portVlanIdTlvReceived:
      lldpNeighborIntf.portVlanId = nbr.portVlanId

   if nbr.maximumFrameSizeTlvReceived:
      lldpNeighborIntf.maxFrameSize = nbr.maximumFrameSizeInfo.maxFrameSize

   if nbr.linkAggregation8021TlvReceived:
      if not nbr.linkAggregation8021Info.status.aggCapable:
         lldpNeighborIntf.linkAggregation8021Status = 'notCapable'
      elif nbr.linkAggregation8021Info.status.aggEnabled:
         lldpNeighborIntf.linkAggregation8021Status = 'capableAndEnabled'
      else:
         lldpNeighborIntf.linkAggregation8021Status = 'capableAndDisabled'
      lldpNeighborIntf.linkAggregation8021InterfaceId = \
            nbr.linkAggregation8021Info.portId

   if nbr.linkAggregation8023TlvReceived:
      if not nbr.linkAggregation8023Info.status.aggCapable:
         lldpNeighborIntf.linkAggregation8023Status = 'notCapable'
      elif nbr.linkAggregation8023Info.status.aggEnabled:
         lldpNeighborIntf.linkAggregation8023Status = 'capableAndEnabled'
      else:
         lldpNeighborIntf.linkAggregation8023Status = 'capableAndDisabled'
      lldpNeighborIntf.linkAggregation8023InterfaceId = \
            nbr.linkAggregation8023Info.portId

   for _key, manAddr in sorted( nbr.managementAddress.items() ):
      neighborManagementAddr = ManagementAddr()
      neighborManagementAddr.addressType = \
         addrFamilyNumToStr( manAddr.key.manAddrSubtype )
      neighborManagementAddr.address = \
         formatAddr( manAddr.key.manAddrSubtype, manAddr.key.manAddr )
      if manAddr.ifSubtype not in recognizedIfTypes:
         neighborManagementAddr.interfaceNumType = 'badType'
         neighborManagementAddr.interfaceNumTypeValue = \
               LldpConstants.ifSubtypeToValue( manAddr.ifSubtype )
      else:
         neighborManagementAddr.interfaceNumType = manAddr.ifSubtype
      neighborManagementAddr.interfaceNum = manAddr.ifId
      neighborManagementAddr.oidString = manAddr.oid
      lldpNeighbor.managementAddresses.append( neighborManagementAddr ) 
   # IEEE802.1 Organizationally-Specific TLVs.
   for ppvid, portAndProtocolVlan in sorted( 
         nbr.portAndProtocolVlan.items() ):
      lldpNeighborIntf.portAndProtocolVlanSupported[ ppvid ] = \
            portAndProtocolVlan.supported
      lldpNeighborIntf.portAndProtocolVlanEnabled[ ppvid ] =  \
            portAndProtocolVlan.enabled
   
   for vName in nbr.vlanName.values():
      lldpNeighborIntf.vlanNames[ vName.vlanId ] = escapedString( vName.vlanName )

   lldpNeighborIntf.protocolIdentityInfo = []
   for _idx, info in sorted( nbr.protocolIdentityInfo.items() ):
      lldpNeighborIntf.protocolIdentityInfo.append( info.protocolId.hex( ' ', 1 ) )

   if nbr.vidUsageDigestTlvReceived:
      lldpNeighborIntf.vidUsageDigest = nbr.vidUsageDigest
   if nbr.managementVidTlvReceived:
      lldpNeighborIntf.managementVid = nbr.managementVid

   lldpNeighborIntf.ztpBootVlan = nbr.ztpBootVlan

   # IEEE802.3 Organizationally-Specific TLVs.
   if nbr.macPhyConfigStatusTlvReceived:
      if not nbr.macPhyConfigStatusInfo.autoNegSupported:
         lldpNeighborIntf.autoNegCapability = 'notCapable'
      elif nbr.macPhyConfigStatusInfo.autoNegEnabled:
         lldpNeighborIntf.autoNegCapability = 'capableAndEnabled'
      else:
         lldpNeighborIntf.autoNegCapability = 'capableAndDisabled'
      lldpNeighborIntf.autoNegAdvertisedCapabilities = \
            autoNegCapsList( nbr.macPhyConfigStatusInfo.autoNegAdvertisedCap )
      lldpNeighborIntf.operMauType = mauTypeToStr( 
            nbr.macPhyConfigStatusInfo.operMauType  )
      if lldpNeighborIntf.autoNegAdvertisedCapabilities:
         lldpNeighborIntf.autoNegAdvertisedCapabilities = \
               lldpNeighborIntf.autoNegAdvertisedCapabilities[ 1: ]
   if nbr.powerViaMdiTlvReceived:
      #pylint: disable=protected-access
      lldpNeighborIntf._powerViaMdiTlvReceived = True 
      lldpNeighborIntf.portClass = nbr.powerViaMdiInfo.portClass
      lldpNeighborIntf.powerOverEthernetSupported = nbr.powerViaMdiInfo.supported
      lldpNeighborIntf.powerOverEthernetEnabled = nbr.powerViaMdiInfo.enabled
      lldpNeighborIntf.controlAbility = nbr.powerViaMdiInfo.powerPairControllable
      lldpNeighborIntf.powerPairs = powerPairsToStr(
            nbr.powerViaMdiInfo.powerPairs )
      lldpNeighborIntf.powerClass = nbr.powerViaMdiInfo.powerClass
      lldpNeighborIntf._ieeeStandard = '802.3af'
      if isAtLeastIeeeStdAt( nbr.powerViaMdiInfo.ieeeStandard ):
         lldpNeighborIntf._ieeeStandard = '802.3at'
         at = nbr.powerViaMdiInfo.at
         lldpNeighborIntf.powerType = powerTypeToModel( at.powerType )
         lldpNeighborIntf.powerSource = powerSourceToModel( at.powerSource,
                                                            at.powerType )
         lldpNeighborIntf.powerPriority = powerPriorityToModel( at.powerPriority )
         lldpNeighborIntf.pdRequestedPowerValue = toWatts(
                                                   at.pdRequestedPowerValue )
         lldpNeighborIntf.pseAllocatedPowerValue = toWatts(
                                                   at.pseAllocatedPowerValue )
      if isAtLeastIeeeStdBt( nbr.powerViaMdiInfo.ieeeStandard ):
         lldpNeighborIntf._ieeeStandard = '802.3bt'
         bt = nbr.powerViaMdiInfo.bt
         pd = isPd( nbr.powerViaMdiInfo.at.powerType )
         pse = isPse( nbr.powerViaMdiInfo.at.powerType )
         if pd or bt.pd4Pid:
            lldpNeighborIntf.pd4Pid = bt.pd4Pid
         lldpNeighborIntf.pdRequestedPowerValueModeA = toWatts(
                                                   bt.pdRequestedPowerValueModeA )
         lldpNeighborIntf.pdRequestedPowerValueModeB = toWatts(
                                                   bt.pdRequestedPowerValueModeB )
         lldpNeighborIntf.pseAllocatedPowerValueAlternativeA = toWatts(
                                          bt.pseAllocatedPowerValueAlternativeA )
         lldpNeighborIntf.pseAllocatedPowerValueAlternativeB = toWatts(
                                          bt.pseAllocatedPowerValueAlternativeB )
         if pse or bt.psePoweringStatus != 'PsePoweringStatusReserved0':
            lldpNeighborIntf.psePoweringStatus = btPowerFieldToModel(
                                                               bt.psePoweringStatus )
            lldpNeighborIntf._psePoweringStatus = Tac.enumValue(
                                 envPwr + "PsePoweringStatus", bt.psePoweringStatus )
         if pd or bt.pdPoweredStatus != 'PdPoweredStatusReserved0':
            lldpNeighborIntf.pdPoweredStatus = btPowerFieldToModel(
                                                               bt.pdPoweredStatus )
            lldpNeighborIntf._pdPoweredStatus = Tac.enumValue(
                                 btStr + "PdPoweredStatus", bt.pdPoweredStatus )
         if pse or bt.psePowerPairsExt != 'PsePowerPairsExtReserved0':
            lldpNeighborIntf.psePowerPairsExt = btPowerFieldToModel(
                                                               bt.psePowerPairsExt )
            lldpNeighborIntf._psePowerPairsExt = Tac.enumValue(
                                 envPwr + "PsePowerPairsExt", bt.psePowerPairsExt )
         lldpNeighborIntf.dualSignaturePowerClassExtModeA = btPowerFieldToModel( (
                                       bt.dualSignaturePowerClassExtModeA, pd ) )
         lldpNeighborIntf._dualSignaturePowerClassExtModeA = Tac.enumValue(
                                    envPwr + "DualSignaturePowerClassExtMode",
                                    bt.dualSignaturePowerClassExtModeA )
         lldpNeighborIntf.dualSignaturePowerClassExtModeB = btPowerFieldToModel( (
                                       bt.dualSignaturePowerClassExtModeB, pd ) )
         lldpNeighborIntf._dualSignaturePowerClassExtModeB = Tac.enumValue(
                                    envPwr + "DualSignaturePowerClassExtMode",
                                    bt.dualSignaturePowerClassExtModeB )
         lldpNeighborIntf.powerClassExt = btPowerFieldToModel( bt.powerClassExt )
         lldpNeighborIntf._powerClassExt = Tac.enumValue(
                                    btStr + "PowerClassExt", bt.powerClassExt )
         lldpNeighborIntf.powerTypeExt = btPowerFieldToModel( bt.powerTypeExt )
         lldpNeighborIntf._powerTypeExt = Tac.enumValue(
                                    envPwr + "PowerTypeExt", bt.powerTypeExt )
         if pd or bt.pdLoad:
            lldpNeighborIntf.pdLoad = bt.pdLoad
         if pse or bt.pseMaximumAvailablePowerValue != 0:
            lldpNeighborIntf.pseMaximumAvailablePower = toWatts(
                                                   bt.pseMaximumAvailablePowerValue )
         if pse or bt.pseAutoclassSupport:
            lldpNeighborIntf.pseAutoclassSupport = bt.pseAutoclassSupport
         if pse or bt.autoclassCompleted:
            lldpNeighborIntf.autoclassCompleted = bt.autoclassCompleted
         if pd or bt.autoclassRequest:
            lldpNeighborIntf.autoclassRequest = bt.autoclassRequest
         if pd or bt.powerDownRequest:
            lldpNeighborIntf.powerDownRequest = bt.powerDownRequest
         if pd or bt.powerDownTime != 0:
            lldpNeighborIntf.powerDownTime = bt.powerDownTime

   def setMedInfo( attrName, attrValue,
                   attrValueFieldSource=None, attrValueFieldNames=None ):
      '''Creates medInfo if necessary, assign attrValue medInfo.<attrName> to and
         optionally copy some fields (attrValueFieldNames) from attrValueFieldSource
         to attrValue'''
      setattr( lldpNeighborIntf.getOrCreateLldpMedInfo(), attrName, attrValue )
      if attrValueFieldSource is not None:
         for fieldName in attrValueFieldNames:
            fieldValue = getattr( attrValueFieldSource, fieldName )
            setattr( attrValue, fieldName, fieldValue )
      return attrValue

   if nbr.medCapabilitiesTlvReceived:
      capabilities = setMedInfo( 'capabilities', MedCapabilitiesInfo(),
                                 nbr.medCapabilitiesInfo.capability,
                                 [ 'capabilities', 'networkPolicy', 'location',
                                   'inventory', 'extendedPse', 'extendedPd' ] )
      capabilities.deviceType = medDeviceTypeToModel(
                                                nbr.medCapabilitiesInfo.deviceType )

   networkPoliciesRemote = sorted( nbr.networkPolicyTlv.values() )
   if networkPoliciesRemote:
      medNetworkPolicies = []
      for netpolRemote in networkPoliciesRemote:
         netpol = MedNetworkPolicyInfo()
         netpol.applicationType = netpolRemote.appValue
         netpol.unknownPolicy = netpolRemote.unknownPolicy
         netpol.tagged = netpolRemote.tagged
         netpol.vlanId = netpolRemote.vlanId
         netpol.layer2Priority = netpolRemote.cosValue
         netpol.dscpValue = netpolRemote.dscpValue
         medNetworkPolicies.append( netpol )
      setMedInfo( 'networkPolicies', medNetworkPolicies )

   lldpNeighborIntf.populateMedLocationTLVs( nbr.locationId )
   if LldpToggleLib.toggleLldpSerialNumberTlvEnabled():
      if nbr.serialNumberTlvInfo:
         lldpNeighborIntf.medInfo = lldpNeighborIntf.getOrCreateLldpMedInfo()
         lldpNeighborIntf.populateMedInventoryTLVs( nbr )
   else:
      lldpNeighborIntf.populateMedInventoryTLVs( nbr )

   # Unknown TLVs.
   unknownTlvInformation = UnknownTlv()
   for _tlvType, unknownTlv in sorted( nbr.unknownTlv.items() ):
      unknownTlvInformation.tlvType = unknownTlv.type
      unknownTlvInformation.tlvLines = printHexDump( unknownTlv.info )
      lldpNeighborIntf.unknownTlvs.append( unknownTlvInformation )
   
   for _key, unknownOrgDefTlv in sorted( nbr.unknownOrgDefTlv.items() ):
      unknownOrgDefTlvValues = getUnknownOrgDefTlvs( unknownOrgDefTlv )
      lldpNeighborIntf.unknownOrgDefinedTlvs.append( unknownOrgDefTlvValues )
   lldpNeighbor.neighborInterfaceInfo = lldpNeighborIntf

   return lldpNeighbor

def doShowLldpNeighborsDetail( mode, args ):
   intfs = args.get( 'INTFS' )
   checkLldpEnabled( mode )
   lldpNeighborsDetail = LldpNeighborsDetail()
   for intfName, portStatus in lldpIterator( mode, intfs, lldpStatus.portStatus ):
      lldpNeighborsList = LldpNeighborsList()
      remoteSystems = list( portStatus.remoteSystem.items() )
      if not remoteSystems:
         lldpNeighborsDetail.lldpNeighbors[ intfName ] = lldpNeighborsList
      for _idx, nbr in sorted( remoteSystems ):
         lldpNeighbor = doShowNeighborDetail( nbr )
         lldpNeighborsList.lldpNeighborInfo.append( lldpNeighbor )
         TacSigint.check()
      lldpNeighborsDetail.lldpNeighbors[ intfName ] = lldpNeighborsList
   return lldpNeighborsDetail

def getUnknownOrgDefTlvs( unknownOrgDefTlv ):
   unknownOrgDefTlvValues = UnknownOrgDefTlv()
   oui = unknownOrgDefTlv.key.oui
   ouiStr = '{:02X}-{:02X}-{:02X}'.format(
         ( oui >> 16 ) & 0xff, ( oui >> 8 ) & 0xff, oui & 0xff )
   subType = unknownOrgDefTlv.key.subtype
   tlvData = []
   function = LldpLib.unknownOrgDefTlvStrFn( unknownOrgDefTlv )
   if function:
      try:
         lines = function( unknownOrgDefTlv )
      except LldpLib.MalformedTlvError as e:
         unknownOrgDefTlvValues.malformedTlvError = str( e )
      else:
         for line in lines:
            tlvData.append( line )
         unknownOrgDefTlvValues.tlvLines = tlvData
         return unknownOrgDefTlvValues
   unknownOrgDefTlvValues.ouiStr = ouiStr
   unknownOrgDefTlvValues.subType = subType
   unknownOrgDefTlvValues.tlvLines = printHexDump( unknownOrgDefTlv.orgDefInfo ) 
   return unknownOrgDefTlvValues

def powerViaMdiTlvReceived( intfId ):
   port = lldpStatus.portStatus.get( intfId )
   return port.powerViaMdiTlvReceived() if port else False

def anyLldpMedReceived( intfId ):
   port = lldpStatus.portStatus.get( intfId )
   return port.anyLldpMedReceived() if port else False

def doShowLldpLocalInfo( mode, args ):
   intfs = args.get( 'INTFS' )
   checkLldpEnabled( mode )
   lldpLocalInfo = LldpLocal()
   lldpLocalInfo.chassisIdType = \
         chassisIdSubtypeToStr( lldpLocalSystem.chassisIdentifier )
   lldpLocalInfo.chassisId = chassisIdToStr( \
         lldpLocalSystem.chassisIdentifier, short=False )

   lldpLocalInfo.systemName = escapedString( lldpLocalSystem.sysName )
   lldpLocalInfo.systemDescription = escapedString( lldpLocalSystem.sysDesc )
   systemCaps = SystemCapabilities()
   getSysCaps( lldpLocalSystem.sysCapSupported, systemCaps )
   getSysCaps( lldpLocalSystem.sysCapEnabled, systemCaps, True )
   lldpLocalInfo.systemCapabilities = systemCaps

   for key in lldpLocalSystem.managementAddressPrio.values():
      manAddr = lldpLocalSystem.managementAddress[ key ]
      localPortManagementAddr = ManagementAddr()
      localPortManagementAddr.addressType = \
            addrFamilyNumToStr( manAddr.key.manAddrSubtype )
      localPortManagementAddr.address = \
            formatAddr( manAddr.key.manAddrSubtype, manAddr.key.manAddr )
      localPortManagementAddr.interfaceNumType = manAddr.ifSubtype
      localPortManagementAddr.interfaceNum = manAddr.ifId
      localPortManagementAddr.oidString = manAddr.oid
      lldpLocalInfo.managementAddresses.append( localPortManagementAddr )

   for intfName, localPort in lldpIterator( mode, intfs, lldpLocalSystem.localPort ):
      lldpLocalInterface = LocalInterfaceInfo()
      lldpLocalInterface.interfaceIdType = \
            portIdSubtypeToStr( localPort.portIdentifier )
      lldpLocalInterface.interfaceId = \
            portIdToStr( localPort.portIdentifier, short=False )
      lldpLocalInterface.interfaceId_v2 = \
            interfaceIdWithoutQuotation( localPort.portIdentifier )

      lldpLocalInterface.interfaceDescription = escapedString( localPort.portDesc )
      
      # IEEE802.1 Organizationally-Specific TLVs.
      lldpLocalInterface.portVlanId = localPort.portVlanId
      # We do not support transmitting Port And Protocol VLAN, or Protocol
      # Identity TLVs.
      linkAggrInfo = localPort.linkAggregation8021Info
      if not linkAggrInfo.status.aggCapable:
         lldpLocalInterface.linkAggregation8021Status = 'notCapable'
      elif linkAggrInfo.status.aggEnabled:
         lldpLocalInterface.linkAggregation8021Status = 'capableAndEnabled'
      else:
         lldpLocalInterface.linkAggregation8021Status = 'capableAndDisabled'
      lldpLocalInterface.linkAggregation8021InterfaceId = linkAggrInfo.portId

      # IEEE802.3 Organizationally-Specific TLVs.
      linkAggrInfo = localPort.linkAggregation8023Info
      if not linkAggrInfo.status.aggCapable:
         lldpLocalInterface.linkAggregation8023Status = 'notCapable'
      elif linkAggrInfo.status.aggEnabled:
         lldpLocalInterface.linkAggregation8023Status = 'capableAndEnabled'
      else:
         lldpLocalInterface.linkAggregation8023Status = 'capableAndDisabled'
      lldpLocalInterface.linkAggregation8023InterfaceId = linkAggrInfo.portId

      if localPort.localPortPoeInfo and localPort.localPortPoeInfo.readyToSendTlv():
         poePs = localPort.localPortPoeInfo.poePortStatus
         bt = poePs.lldpPoeNegotiationSupport == 'AtBtSupported'
         # pylint: disable=protected-access
         lldpLocalInterface._powerViaMdiTlvReceived = True
         lldpLocalInterface._ieeeStandard = '802.3bt' if bt else '802.3at'
         lldpLocalInterface.portClass = 'pClassPSE'
         lldpLocalInterface.powerOverEthernetSupported = True
         lldpLocalInterface.powerOverEthernetEnabled = ( poePs.poePortState ==
                                                         'powered' )
         lldpLocalInterface.controlAbility = poePs.powerPairControllable
         lldpLocalInterface.powerPairs = poePs.powerPairs
         if poePs.pdClass != 'invalid':
            lldpLocalInterface.powerClass = pdClassToModel( poePs.pdClass )
         lldpLocalInterface.powerType = powerTypeToModel( poePs.powerType )
         lldpLocalInterface.powerSource = powerSourceToModel( poePs.powerSource,
                                                              poePs.powerType )
         lldpLocalInterface.powerPriority = powerPriorityToModel(
                                                              poePs.powerPriority )
         rs = localPort.localPortPoeInfo.remoteSystem
         powerRequested = rs and rs.powerViaMdiInfo.at.pdRequestedPowerValue > 0.0

         if powerRequested:
            lldpLocalInterface.pdRequestedPowerValue = \
                        float( rs.powerViaMdiInfo.at.pdRequestedPowerValue ) / 10.0
         else:
            lldpLocalInterface.pdRequestedPowerValue = poePs.grantedPower
         lldpLocalInterface.pseAllocatedPowerValue = poePs.grantedPower
         if bt:
            lldpLocalInterface.pseAllocatedPowerValueAlternativeA = \
                                                      poePs.grantedPowerAlternativeA
            lldpLocalInterface.pseAllocatedPowerValueAlternativeB = \
                                                      poePs.grantedPowerAlternativeB
            if rs:
               lldpLocalInterface.pdRequestedPowerValueModeA = \
                     float( rs.powerViaMdiInfo.bt.pdRequestedPowerValueModeA ) / 10.0
               lldpLocalInterface.pdRequestedPowerValueModeB = \
                     float( rs.powerViaMdiInfo.bt.pdRequestedPowerValueModeB ) / 10.0
            else:
               lldpLocalInterface.pdRequestedPowerValueModeA = \
                              lldpLocalInterface.pseAllocatedPowerValueAlternativeA
               lldpLocalInterface.pdRequestedPowerValueModeB = \
                              lldpLocalInterface.pseAllocatedPowerValueAlternativeB
            lldpLocalInterface.psePoweringStatus = btPowerFieldToModel(
                                                            poePs.psePoweringStatus )
            lldpLocalInterface.dualSignaturePowerClassExtModeA = \
                  btPowerFieldToModel( ( poePs.dualSignaturePowerClassExtModeA,
                                         False ) )
            lldpLocalInterface._dualSignaturePowerClassExtModeA = Tac.enumValue(
                                          envPwr + "DualSignaturePowerClassExtMode",
                                          poePs.dualSignaturePowerClassExtModeA )
            lldpLocalInterface.dualSignaturePowerClassExtModeB = \
                  btPowerFieldToModel( ( poePs.dualSignaturePowerClassExtModeB,
                                         False ) )
            lldpLocalInterface._dualSignaturePowerClassExtModeB = Tac.enumValue(
                                          envPwr + "DualSignaturePowerClassExtMode",
                                          poePs.dualSignaturePowerClassExtModeB )
            classExt = pdClassToPowerClassExt( poePs.pdClass )
            lldpLocalInterface.powerClassExt = btPowerFieldToModel( classExt )
            lldpLocalInterface._powerClassExt = Tac.enumValue(
                                      btStr + "PowerClassExt", classExt )
            lldpLocalInterface.powerTypeExt = btPowerFieldToModel(
                                                               poePs.powerTypeExt )
            lldpLocalInterface._powerTypeExt = Tac.enumValue(
                                       envPwr + "PowerTypeExt", poePs.powerTypeExt )
            lldpLocalInterface.pseMaximumAvailablePower = \
                                                      poePs.pseMaximumGrantedPower
            lldpLocalInterface.pseAutoclassSupport = poePs.pseAutoclassSupport
            lldpLocalInterface.autoclassCompleted = poePs.autoclassCompleted

      lldpLocalInterface.maxFrameSize = localPort.maximumFrameSizeInfo.maxFrameSize
      
      if anyLldpMedReceived( localPort.intfId ):
         if not lldpLocalInterface.medInfo:
            lldpLocalInterface.medInfo = LldpMedInfo()
         localCap = localPort.medCapabilities.capability
         capabilities = MedCapabilitiesInfo()
         capabilities.capabilities = localCap.capabilities
         capabilities.networkPolicy = localCap.networkPolicy
         capabilities.location = localCap.location
         capabilities.extendedPse = localCap.extendedPse
         capabilities.extendedPd = localCap.extendedPd
         capabilities.inventory = localCap.inventory
         capabilities.deviceType = medDeviceTypeToModel(
                                             localPort.medCapabilities.deviceType )
         lldpLocalInterface.medInfo.capabilities = capabilities
      
      if LldpToggleLib.toggleLldpSerialNumberTlvEnabled():
         if not lldpLocalInterface.medInfo:
            lldpLocalInterface.medInfo = LldpMedInfo()
         lldpLocalInterface.medInfo.serialNumberTlvInfo = escapedString(
            lldpLocalSystem.serialNumberTlvInfo )
      localNetworkPolicyTlvs = localPort.localNetworkPolicyTlv
      if localNetworkPolicyTlvs:
         medNetworkPolicies = []
         for netpol in sorted( localNetworkPolicyTlvs.values() ):
            medNetpol = MedNetworkPolicyInfo()
            medNetpol.applicationType = netpol.appValue
            medNetpol.unknownPolicy = netpol.unknownPolicy
            medNetpol.tagged = netpol.tagged
            medNetpol.vlanId = netpol.vlanId
            medNetpol.layer2Priority = netpol.cosValue
            medNetpol.dscpValue = netpol.dscpValue
            medNetworkPolicies.append( medNetpol )
         if not lldpLocalInterface.medInfo:
            lldpLocalInterface.medInfo = LldpMedInfo()
         lldpLocalInterface.medInfo.networkPolicies = medNetworkPolicies

      # Unknown Org defined TLVs
      for _, unknownOrgDefTlv in sorted(
            localPort.unknownOrgDefTlv.items() ):
         lldpLocalUnknownOrgDefTlv = getUnknownOrgDefTlvs( unknownOrgDefTlv )
         lldpLocalInterface.unknownOrgDefinedTlvs.append( lldpLocalUnknownOrgDefTlv )

      lldpLocalInterface.ztpBootVlan = localPort.ztpBootVlan

      lldpLocalInfo.localInterfaceInfo[ intfName ] = lldpLocalInterface
      TacSigint.check()

   # IEEE802.1 Organizationally-Specific TLVs.
   # VlanName TLV
   for intfName, lldpLocalInterface in lldpLocalInfo.localInterfaceInfo.items():
      portStatus = lldpStatus.portStatus.get( intfName )
      if not lldpLocalInterface or not portStatus:
         continue
      for vName in portStatus.sentVlanName.values():
         lldpLocalInterface.vlanNames[ vName.vlanId ] = \
               escapedString( vName.vlanName )
      lldpLocalInterface.vlanNamesIncomplete = portStatus.vlanNamesIncomplete
      lldpLocalInterface.vlanNamesChanged = portStatus.vlanNamesChanged
      lldpLocalInfo.localInterfaceInfo[ intfName ] = lldpLocalInterface

   return lldpLocalInfo

def doClearLldpCounters( mode, args ):
   if 'session' in args:
      checkpoint = _lldpSessionCheckpoint( mode )
   else:
      Intf.Log.logClearCounters( "lldp" )
      checkpoint = lldpCounterCheckpoint

   for portStatus in lldpStatus.portStatus.values():
      snapshot = Tac.Value( "Lldp::PortCounterCheckpoint", timestamp=Tac.now() )
      for counterName in counterNames:
         setattr( snapshot, counterName, getattr( portStatus, counterName ) )
      checkpoint.portCounterCheckpoint[ portStatus.intfId ] = snapshot

def doClearLldpTable( mode, args ):
   info = UtmpDump.getUserInfo()
   who = info['user']
   where = info['tty']
   timestamp = Tac.now()
   remTablesCliConfig.remTablesClearRequest = \
      Tac.Value( "Lldp::ClearRequest", timestamp, who, where )

#------------------------------------------------------
# Register Lldp show commands into "show tech-support".
#------------------------------------------------------

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2010-01-01 00:08:00',
   cmds=[ 'show lldp',
          'show lldp counters',
          'show lldp local-info',
          'show lldp neighbors detail' ],
   summaryCmds=[ 'show lldp neighbors' ] )

def Plugin( entityManager ):
   global lldpCliConfig, remTablesCliConfig, lldpConfig, lldpStatus, lldpLocalSystem
   global lldpCounterCheckpoint
   lldpCliConfig = LazyMount.mount(
      entityManager, "l2discovery/lldp/cliConfig", "Lldp::CliConfig", "w" )
   remTablesCliConfig = LazyMount.mount(
      entityManager, "l2discovery/lldp/remTablesCliConfig",
      "Lldp::RemTablesCliConfig", "w" )
   lldpConfig = LazyMount.mount(
      entityManager, "l2discovery/lldp/config", "Lldp::Config", "r" )
   lldpStatus = LazyMount.mount(
      entityManager, "l2discovery/lldp/status/all", "Lldp::AllStatus", "r" )
   lldpLocalSystem = LazyMount.mount(
      entityManager, "l2discovery/lldp/localSystem", "Lldp::LocalSystem", "r" )
   lldpCounterCheckpoint = LazyMount.mount(
      entityManager, "l2discovery/lldp/counterCheckpoint",
      "Lldp::CounterCheckpoint", "w" )
   Plugins.loadPlugins( 'LldpPlugin' )
