# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import ConfigMgmtMode
from CliPlugin.ArchiveCliModel import ArchiveModel
import ArchiveLib
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
import CliToken.Clear
import ShowCommand
import Url
import Tac

########## "management archive" #######
#
# The SSD file archive management mode archive:
#
# management archive
#
# ...
# ...
#
# exit
#
# Supported Commands in management archive mode:
#
# shutdown
# no shutdown
# quotapct #   (where # is replaced by a number in the range 0 - 100)
# destination <URL>     where <URL> can be 'drive:' or 'flash:' depending on the
#                       system hardware configuration.
#

def matchDrive( fs ):
   return fs.scheme == 'drive:'

def matchExt4Flash( fs ):
   return fs.scheme == 'flash:' and fs.mountInfo()[ 2 ] == 'ext4'

destFsUrlMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode: {
      fs.scheme : 'Filesystem'
      for fs in Url.filesystems()
      if matchDrive( fs ) or matchExt4Flash( fs )
   }
)

archiveConfig = None

class ArchiveConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   """ The ArchiveConfigMode class processes management archive mode commands """

   name = "Management Archive"

   def __init__( self, parent, session ):
      """ ArchiveConfigMode __init__ sets config and status at instantiation """
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "archive" )

   def shutdown( self, args ):
      """ shutdown sets archive status to shutdown """
      archiveConfig.shutdown = True

   def noShutdown( self, args ):
      """ noShutdown sets archive status to no shutdown """
      archiveConfig.shutdown = False

   def setQuotaPct( self, args ):
      """ setQuotaPct sets the quota reserved pct for the file archive """
      archiveConfig.quotapct = args[ 'NUMBER' ]

   def defaultQuotaPct( self, args ):
      """ defaultQuotaPct set default quota reserved pct for the file archive """
      archiveConfig.quotapct = archiveConfig.defaultQuotaReservedPct

   def setDestination( self, args ):
      """Set destination of logs archive."""
      destFsUrlName = args[ 'DESTINATION' ]
      destFs = Url.getFilesystem( destFsUrlName )

      if destFs is None:
         self.addError( 'Filesystem %s no longer available' % destFsUrlName )
         return

      name = destFs.scheme.split( ':' )[ 0 ]
      mntPt = destFs.mountInfo()[ 1 ]
      archiveConfig.dest = f'{name}:{mntPt}'

   def defaultDestination( self, args ):
      """Restore archive destination to the default."""
      archiveConfig.dest = archiveConfig.defaultDest

#------------------------------------------------------------------------------------
# show archive
#------------------------------------------------------------------------------------

def currentArchive( mode ):
   """Wrapper for handling error getting the current archive."""

   try:
      archive = ArchiveLib.Archive.currentArchive()
   except AssertionError as e:
      mode.addError( 'An error occured getting filesystem information'
                     ' for the archive destination path: %s' % e )
      return None

   if archive is None:
      mode.addError( 'Unable to use the current configured archive' )
      return None
   elif archive is False:
      mode.addError( 'No archive configured' )

   return archive

#--------------------------------------------------------------------------------
# [ no | default ] management archive
#--------------------------------------------------------------------------------
class ManagementArchiveCmd( CliCommand.CliCommandClass ):
   syntax = 'management archive'
   noOrDefaultSyntax = syntax
   data = {
      'management' : ConfigMgmtMode.managementKwMatcher,
      'archive' : 'Manage logs archival and archive space',
   }

   @staticmethod
   def handler( mode, args ):
      """ gotoArchiveConfigMode gets Cli into management archive mode """
      childMode = mode.childMode( ArchiveConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = ArchiveConfigMode.clear

BasicCli.GlobalConfigMode.addCommandClass( ManagementArchiveCmd )

#--------------------------------------------------------------------------------
# show management archive
#--------------------------------------------------------------------------------
class ShowManagementArchiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management archive'
   data = {
      'management' : ConfigMgmtMode.managementShowKwMatcher,
      'archive' : 'Show logs archival status and archive space',
   }
   cliModel = ArchiveModel
   privileged = True

   @staticmethod
   def handler( mode, args ):
      """ showArchive sets archive status so render function can display """
      archive = currentArchive( mode )
      if not archive:
         return None

      archiveModel = ArchiveModel()
      archiveStatus = archive.statusSummary

      # General information
      archiveModel.destName = archiveStatus[ 'name' ]
      archiveModel.destPath = archiveStatus[ 'path' ]
      archiveModel.shutdown = archiveStatus[ 'disabled' ]
      archiveModel.exists = archiveStatus[ 'exists' ]
      archiveModel.fileArchivalEnabled = archiveStatus[ 'fileArchivalEnabled' ]
      # Destination device information:
      archiveModel.devName = archiveStatus[ 'devName' ]
      archiveModel.devPresent = archiveStatus[ 'devPresent' ]
      archiveModel.devPhysical = archiveStatus[ 'devPhysical' ]
      archiveModel.devMounted = archiveStatus[ 'devMounted' ]
      archiveModel.devForceDisabled = archiveStatus[ 'devForceDisabled' ]
      # Filesystem information:
      archiveModel.fsMntPt = archiveStatus[ 'fsMntPt' ]
      archiveModel.fsSize = archiveStatus[ 'fsSize' ]
      archiveModel.fsUsedKiB = archiveStatus[ 'fsUsedKiB' ]
      archiveModel.fsUsedPct = archiveStatus[ 'fsUsedPct' ]
      archiveModel.fsQuotaMount = archiveStatus[ 'fsQuotaMount' ]
      # Quota information:
      archiveModel.quotaOn = archiveStatus[ 'quotasOn' ] or False
      archiveModel.quotaKiB = archiveStatus[ 'quotaKiB' ] or 0
      archiveModel.quotaPct = archiveStatus[ 'quotaPct' ] or 0
      # Archive space usage information
      archiveModel.usedKiB = archiveStatus[ 'usedKiB' ]
      archiveModel.usedPct = archiveStatus[ 'usedPct' ]

      return archiveModel

BasicCli.addShowCommandClass( ShowManagementArchiveCmd )

#--------------------------------------------------------------------------------
# [ no ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Disable logs archival',
   }

   handler = ArchiveConfigMode.shutdown
   noOrDefaultHandler = ArchiveConfigMode.noShutdown

ArchiveConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# quotapct NUMBER
#--------------------------------------------------------------------------------
class QuotapctNumberCmd( CliCommand.CliCommandClass ):
   syntax = 'quotapct NUMBER'
   noOrDefaultSyntax = syntax
   data = {
      'quotapct' : 'Set logs archival quota percentage',
      'NUMBER' : CliMatcher.IntegerMatcher( 0, 100,
         helpdesc='Set quota percentage value' ),
   }

   handler = ArchiveConfigMode.setQuotaPct
   noOrDefaultHandler = ArchiveConfigMode.defaultQuotaPct

ArchiveConfigMode.addCommandClass( QuotapctNumberCmd )

#--------------------------------------------------------------------------------
# [ no | default ] destination DESTINATION
#--------------------------------------------------------------------------------
class DestinationCmd( CliCommand.CliCommandClass ):
   syntax = 'destination DESTINATION'
   noOrDefaultSyntax = 'destination ...'
   data = {
      'destination' : 'Set logs archival destination',
      'DESTINATION' : destFsUrlMatcher
   }

   handler = ArchiveConfigMode.setDestination
   noOrDefaultHandler = ArchiveConfigMode.defaultDestination

ArchiveConfigMode.addCommandClass( DestinationCmd )

#--------------------------------------------------------------------------------
# clear management archive
#--------------------------------------------------------------------------------
class ClearManagementArchiveCmd( CliCommand.CliCommandClass ):
   syntax = 'clear management archive'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'management' : ConfigMgmtMode.managementClearKwMatcher,
      'archive' : 'Manage logs archival and archive space',
   }

   @staticmethod
   def handler( mode, args ):
      """ clearArchiveConfigMode clears the management archive file archive """
      # A special case is a clear management archive request processed here
      # directly and won't end up in the reactor. It will remove the archive
      # directories as well as archived archives and then re-create the archive
      # directories. It is not persisted in the config file since it is a one time
      # operation.
      # Note: If a clear is done when the archive is disabled, the archive
      # directories will not be rebuilt. The cron job archivecheck will rebuild
      # them if archiving is re-enabled latter.

      archive = currentArchive( mode )
      if not archive:
         return

      archive.clear()

BasicCli.EnableMode.addCommandClass( ClearManagementArchiveCmd )

def Plugin( entityManager ):
   global archiveConfig

   archiveConfig = ConfigMount.mount( entityManager, "mgmt/archive/config",
                                      "Mgmt::Archive::Config", "w" )
