# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliCommand import Node
from CliMatcher import KeywordMatcher
from IntfRangePlugin.LogicalPort import LogicalPortPoolType
import BasicCli
from CliDynamicSymbol import LazyCallback
import CliParser
import CliToken, CliToken.Hardware, ShowCommand
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import LazyMount

PoolStatuses = LazyCallback( 'LogicalPortModel.PoolStatuses' )
PoolStatus = LazyCallback( 'LogicalPortModel.PoolStatus' )

logicalPortStatusSliceDir = None

logicalPortPoolGuardFilters = []

def registerLogicalPortPoolGuard( isGuardedFunction ):
   logicalPortPoolGuardFilters.append( isGuardedFunction )

def logicalPortPoolGuard( mode, token ):
   for isGuarded in logicalPortPoolGuardFilters:
      if isGuarded():
         return CliParser.guardNotThisPlatform

   return None

def getPoolStatusesAll( poolNames ):
   poolStatuses = {}
   for sliceDir in logicalPortStatusSliceDir.values():
      for subSliceDir in sliceDir.values():
         if not poolNames:
            poolStatuses.update( subSliceDir.poolStatus )
         else:
            for pool in poolNames:
               poolStatus = subSliceDir.poolStatus.get( pool )
               if poolStatus is not None:
                  poolStatuses[ pool ] = poolStatus
   return poolStatuses

def getPoolStatuses( poolNames ):
   statuses = getPoolStatusesAll( poolNames )
   poolStatuses = PoolStatuses()
   if not statuses:
      return poolStatuses

   for name, status in statuses.items():
      poolStatus = PoolStatus().toModel( status )
      if poolStatus.interfaces:
         poolStatuses.pools[ name ] = poolStatus

   return poolStatuses

# ------------------------------------------------------------------------------
# The "show hardware logical-port ( pool | POOL_RANGE ) status" command.
# ------------------------------------------------------------------------------
class ShowLogicalPortPoolCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware logical-port ( pool | POOL_RANGE ) status'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForShow,
      'logical-port' : 'Logical ports state',
      'pool' : 'Per-pool logical ports state',
      'POOL_RANGE': IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( LogicalPortPoolType, ),
         dollarCompHelpText='Pool list end' ),
      'status': Node(
         matcher=KeywordMatcher( 'status', 'Logical port pool status' ),
         guard=logicalPortPoolGuard )
   }
   cliModel = 'LogicalPortModel.PoolStatuses'

   @staticmethod
   def handler( mode, args ):
      poolRange = args.get( 'POOL_RANGE' )
      poolNames = []

      if poolRange:
         for name in IntfRange.intfListFromCanonical( [ poolRange ] ):
            poolNames.append( IntfRange.intfTypeFromName( name )[ 1 ] )

      return getPoolStatuses( poolNames )

BasicCli.addShowCommandClass( ShowLogicalPortPoolCmd )

def Plugin( em ):
   global logicalPortStatusSliceDir
   logicalPortStatusSliceDir = LazyMount.mount( em,
         "interface/archer/status/eth/phy/logicalport/slice", "Tac::Dir", "ri" )
