#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the following commands:
#-------------------------------------------------------------------------------
# Interface Mode:
#      -  [ no ] [ default ] loop-protection
#--------------------------------------------------------------------------------

import CliCommand
import CliMatcher
import CliParser
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import LazyMount
from Toggles.LoopProtectToggleLib import toggleLoopProtectL2SubIntfEnabled

bridgingHwCapabilities = None

class LoopProtectionModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( ( mode.intf.name.startswith( 'Ethernet' ) or
               mode.intf.name.startswith( 'Po' ) ) and
               ( toggleLoopProtectL2SubIntfEnabled() or
                     not mode.intf.isSubIntf() ) )

#--------------------------------------------------------------------------------
# [ no ] [ default ] loop-protection
#--------------------------------------------------------------------------------
# Defaults to True - Interfaces protected by default, but need global enable
# and vlan enable for the protection to be active

def loopProtectSupportedGuard( mode, token ):
   if bridgingHwCapabilities.loopProtectSupported:
      return None
   return CliParser.guardNotThisPlatform

class LoopProtectionCmd( CliCommand.CliCommandClass ):
   syntax = 'loop-protection'
   noOrDefaultSyntax = syntax
   data = {
      'loop-protection': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'loop-protection', helpdesc='Enable loop protection on this interface' ),
         guard=loopProtectSupportedGuard ),
   }

   handler = "LoopProtectIntfConfigModeHandler.intfLoopProtect"
   noHandler = "LoopProtectIntfConfigModeHandler.noIntfLoopProtect"
   defaultHandler = handler

LoopProtectionModelet.addCommandClass( LoopProtectionCmd )
IntfCli.IntfConfigMode.addModelet( LoopProtectionModelet )

def Plugin( entityManager ):
   global bridgingHwCapabilities

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             'bridging/hwcapabilities',
                                             'Bridging::HwCapabilities', 'r' )
