#!/usr/bin/env python3
# Copyright (c) 2009-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Loopback interface type.  In particular, it provides
# the LoopbackIntf class.
#-------------------------------------------------------------------------------

import re
import Tac
import LazyMount
import ConfigMount
import CliCommand
import CliParser
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin import IntfCli
from CliPlugin import VirtualIntfRule
from IntfRangePlugin.LoopbackIntf import LoopbackAutoIntfType
from LoopbackIntfUtils import minLoopbackIntfNum
from LoopbackIntfUtils import maxLoopbackIntfNum

loopIntfConfigDir = None
loopIntfStatusDir = None

#-------------------------------------------------------------------------------
# A subclass for loopback interfaces
#-------------------------------------------------------------------------------
class LoopbackIntf( IntfCli.VirtualIntf ):

   #----------------------------------------------------------------------------
   # Creates a new LoopbackIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      m = re.match( r'Loopback(\d+)$', name )
      self.loopbackId = int( m.group( 1 ) )
      IntfCli.VirtualIntf.__init__( self, name, mode )
      self.intfStatuses = loopIntfStatusDir.intfStatus
      self.intfStatus = None

   #----------------------------------------------------------------------------
   # The rule for matching Loopback interface names. When this pattern matches,
   # it returns an instance of the LoopbackIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = VirtualIntfRule.VirtualIntfMatcher(
      'Loopback', minLoopbackIntfNum, maxLoopbackIntfNum, value=lambda mode,
      intf: LoopbackIntf( intf, mode ) ) # pylint: disable-msg=E0602

   #----------------------------------------------------------------------------
   # Creates the Interface::LoopbackIntfConfig object for this interface if it does
   # not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      loopIntfConfigDir.intfConfig.newMember( str( self ) )

   #----------------------------------------------------------------------------
   # Determines whether the Interface::LoopbackIntfStatus object for this interface
   # exists.
   #----------------------------------------------------------------------------
   def lookupPhysical( self ):
      self.intfStatus = self.intfStatuses.get( str( self ) )
      return self.intfStatus is not None

   #----------------------------------------------------------------------------
   # Destroys the Interface::LoopbackIntfConfig object for this interface if it
   # already exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      del loopIntfConfigDir.intfConfig[ str( self ) ]

   #----------------------------------------------------------------------------
   # Returns the LoopbackIntfConfig object for this interface.
   #----------------------------------------------------------------------------
   def config( self ):
      return loopIntfConfigDir.intfConfig.get( str( self ) )

   #----------------------------------------------------------------------------
   # Returns the LoopbackIntfStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      return self.intfStatus

   #----------------------------------------------------------------------------
   # Returns the LoopbackIntfCounterDir object for this interface.
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      # we aren't supporting counters on loopback interfaces
      return None

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def showPhysical( self, mode, intfStatusModel ):
      pass

   def bandwidth( self ):
      return 0

   def hardware( self ):
      return "loopback"

   def mtu( self ):
      return 65535

   def addrStr( self ):
      return None

   #----------------------------------------------------------------------------
   # Returns an unsorted list of LoopbackIntf objects representing all the existing
   # Interface::LoopbackIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      iDir = loopIntfStatusDir.intfStatus
      for name in iDir:
         intf = LoopbackIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   def countersSupported( self ):
      return False

   def isValidHostInetAddress( self, a1 ):
      if a1.len > 32:
         return ( False, "Prefix length must be less than or equal to 32" )
      if a1.len == 32:
         # do what the industry standard seems to do
         if ( not IpAddrMatcher.validateMulticastIpAddr( a1.address ) or
               not IpAddrMatcher.splitIpAddrToInts( a1.address )[ 0 ] ):
            return ( False, f"Not a valid host address - {a1.address}" )

         return ( True, "" )
      if IpAddrMatcher.isLoopbackIpAddr( a1.address ):
         return ( False, "IP address must not be loopback" )
      return self.isUnicastInetAddress( a1 )

   def isValidHostInet6Address( self, a1 ):
      if a1.len == 128:
         if not a1.address.isGlobalUnicast:
            return ( False, "Not a valid host address - %s" %
                     ( a1.address.stringValue ) )
         return ( True, "" )

      return self.isUnicastInet6Address( a1 )

   def setDefault( self ):
      IntfCli.VirtualIntf.setDefault( self )
      cfg = self.config()
      cfg.hwFwdIdEnabled = cfg.hwFwdIdEnabledDefault

#-------------------------------------------------------------------------------
# Register the LoopbackIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( LoopbackIntf, LoopbackAutoIntfType,
                                  withIpSupport=True )

#-------------------------------------------------------------------------------
# Adds IP-specific CLI commands to the "config-if" mode for loopback ports.
#-------------------------------------------------------------------------------
class LoopbackIntfConfigModelet( CliParser.Modelet ):
   #----------------------------------------------------------------------------
   # Returns the LoopbackIntfConfig object for this interface.
   #----------------------------------------------------------------------------
   def config( self ):
      return loopIntfConfigDir.intfConfig.get( self.mode.intf.name )

   def setLoopbackHwFwdId( self, args ):
      cfg = self.config()
      if CliCommand.isDefaultCmd( args ):
         value = cfg.hwFwdIdEnabledDefault
      else:
         value = not CliCommand.isNoCmd( args )
      cfg.hwFwdIdEnabled = value

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( "Loopback" )

#--------------------------------------------------------------------------------
# [ no | default ] hardware forwarding id
#--------------------------------------------------------------------------------
class LoopbackHardwareForwardingIdCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware forwarding id'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': 'Hardware config commands',
      'forwarding': 'Hardware forwarding config commands',
      'id': 'Enable hardware forwarding',
   }

   handler = LoopbackIntfConfigModelet.setLoopbackHwFwdId
   noOrDefaultHandler = LoopbackIntfConfigModelet.setLoopbackHwFwdId

LoopbackIntfConfigModelet.addCommandClass( LoopbackHardwareForwardingIdCmd )

#-------------------------------------------------------------------------------
# Associate the LoopbackIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( LoopbackIntfConfigModelet )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global loopIntfConfigDir, loopIntfStatusDir
   loopIntfConfigDir = ConfigMount.mount( entityManager,
                                          "interface/config/loopback/intf",
                                          "Interface::LoopbackIntfConfigDir", "w" )
   loopIntfStatusDir = LazyMount.mount( entityManager,
                                        "interface/status/loopback/intf",
                                        "Interface::LoopbackIntfStatusDir", "r" )
