# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliPlugin import IntfCli
from CliPlugin.EthIntfCli import DataplaneIntfModelet
from CliPlugin.SubIntfCli import SubIntfModelet
import CliToken
import ConfigMount
import LazyMount
import Tac

config = None
status = None
hwMacsecStatus = None
hwMacsecProxyCapability = None
trapConfig = None
sharedSecretConfig = None
SubIntfId = Tac.Type( "Arnet::SubIntfId" )
EthIntfId = Tac.Type( "Arnet::EthIntfId" )

def getProfileNames( mode=None ):
   return config.profile

#--------------------------------------------------------------------------------
# [ no | default ] mac security profile PROFILE_NAME
#--------------------------------------------------------------------------------
def macsecSupportedGuard( mode, token ):
   intfName = mode.intf.name

   def macsecGuard( statusDir, intfName ):
      for hwSlice in statusDir:
         sliceStatus = statusDir[ hwSlice ].status
         if sliceStatus and intfName in sliceStatus:
            return None
      return CliParser.guardNotThisPlatform

   def macsecProxyGuard( statusDir, intfName ):
      for hwSlice in statusDir:
         sliceStatus = statusDir[ hwSlice ].proxyCapable
         if sliceStatus and intfName in sliceStatus:
            return None
      return CliParser.guardNotThisPlatform

   def isSubintfSupported( statusDir, intfName ):
      for hwSlice in statusDir:
         sliceStatus = statusDir[ hwSlice ].status
         if sliceStatus and intfName in sliceStatus:
            if sliceStatus[ intfName ].subIntfSupported:
               return True
      return False

   if SubIntfId.isSubIntfId( intfName ) and EthIntfId.isEthIntfId( intfName ):
      intfName = EthIntfId.parentIntfIdFromSubIntf( intfName )
      if isSubintfSupported( hwMacsecStatus, intfName ):
         return None
      return macsecProxyGuard( hwMacsecProxyCapability, intfName )
   elif EthIntfId.isEthIntfId( intfName ) and EthIntfId.lane( intfName ) > 1:
      if macsecGuard( hwMacsecStatus, intfName ):
         # macsec status may not exist if lane is inactive
         # only guard if primary (/1) interface is also unsupported
         primaryIntfName = '/'.join( intfName.split( '/' )[ : -1 ] + [ '1' ] )
         return macsecGuard( hwMacsecStatus, primaryIntfName )
      else:
         return None
   else:
      return macsecGuard( hwMacsecStatus, intfName )

def delMacsecIntf( intfId ):
   # Get the intfConfig object.
   if intfId in config.intfConfig:
      intfConfig = config.intfConfig[ intfId ]
   else:
      # Nothing to do.
      return

   profileName = intfConfig.profileName
   if ( profileName in config.profile ) and \
         ( intfId in config.profile[ profileName ].intf ):
      del config.profile[ profileName ].intf[ intfId ]

   # Delete the interface config.
   del config.intfConfig[ intfId ]

class MacsecIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'mac security profile PROFILE_NAME'
   noOrDefaultSyntax = 'mac security profile ...'
   data = {
      'mac' : CliToken.Mac.macMatcherForConfigIf,
      'security' : CliCommand.guardedKeyword( 'security',
         helpdesc='Media Access Control Security (802.1AE) commands',
         guard=macsecSupportedGuard ),
      'profile' : 'MAC security profile name',
      'PROFILE_NAME' : CliMatcher.DynamicNameMatcher( getProfileNames,
         'Profile name' ),
   }

   handler = "MacsecCliHandler.configMacsecIntfCmd"

   noOrDefaultHandler = "MacsecCliHandler.noMacsecIntfCmd"

SubIntfModelet.addCommandClass( MacsecIntfCmd )
DataplaneIntfModelet.addCommandClass( MacsecIntfCmd )

matcherSecurityForConfig = CliMatcher.KeywordMatcher( 'security',
                        helpdesc='Media Access Control Security (802.1AE) commands' )

# Macsec global commands.
class MacSecurityCmd( CliCommand.CliCommandClass ):
   syntax = 'mac security'
   noOrDefaultSyntax = syntax
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'security': matcherSecurityForConfig,
   }

   handler = "MacsecCliHandler.gotoMacsecMode"
   noOrDefaultHandler = "MacsecCliHandler.noMacsecMode"

BasicCli.GlobalConfigMode.addCommandClass( MacSecurityCmd )

# config command for clearing mka counters
#--------------------------------------------------------------------------------
# clear mac security mka counters
#--------------------------------------------------------------------------------
class ClearMacSecurityMkaCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear mac security mka counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'mac': CliToken.Mac.macMatcherForClear,
      'security': 'MAC security (802.1AE) information',
      'mka': 'MAC security key agreement protocol options',
      'counters': 'MAC security counter information',
   }

   handler = "MacsecCliHandler.doClearMkaCounters"

BasicCli.EnableMode.addCommandClass( ClearMacSecurityMkaCountersCmd )

#-------------------------------------------------------------------------------
# Cleanup per-interface configuration
#-------------------------------------------------------------------------------
class IntfMacsecConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing per-interface Macsec config
   when the interface is deleted."""
   def setDefault( self ):
      delMacsecIntf( self.intf_.name )

def Plugin( entityManager ):
   global config, status, hwMacsecStatus, hwMacsecProxyCapability, trapConfig
   global sharedSecretConfig

   config = ConfigMount.mount( entityManager, 'macsec/input/cli',
                               'Macsec::Config', 'w' )
   status = LazyMount.mount( entityManager, 'macsec/status',
                             'Macsec::Status', 'r' )
   hwMacsecStatus = LazyMount.mount( entityManager,
                                     'hardware/phy/status/macsec/slice',
                                     'Tac::Dir', 'ri' )
   hwMacsecProxyCapability = LazyMount.mount( entityManager,
                                'hardware/phy/status/macsecproxy/capability/slice',
                                'Tac::Dir', 'ri' )
   trapConfig = ConfigMount.mount( entityManager,
                                   'hardware/trap/config/trapConfig',
                                   'Arnet::TrapConfig', 'w' )

   sharedSecretConfig = LazyMount.mount(
      entityManager, 'mgmt/security/sh-sec-prof/config',
      'Mgmt::Security::SharedSecretProfile::Config', 'r' )

   # register interface-deletion handler
   IntfCli.Intf.registerDependentClass( IntfMacsecConfigCleaner, priority=10 )
