# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Submodel, Dict, Int, Bool, Str, List, Float, Enum
import LazyMount
from ArnetModel import MacAddress
from IntfModels import Interface
from MacsecCommon import (
   tacCipherSuiteToCli,
   ptpBypassSupportToCli,
   trafficPolicyOnNoMkaToCli,
   intfTrafficStatus,
   getMacSecPtpBypass,
   getMacSecIntfDetailPtpBypass
)
from ReversibleSecretCli import decodeKey
from TableOutput import createTable, Format
from TableOutput import terminalWidth
import Arnet
import hashlib
from functools import partial
import Tac
import Cell
from textwrap import TextWrapper
import Toggles.MacsecCommonToggleLib as macsecToggle

keySource = Enum( values=( "preSharedKey", "dot1x", "sharedSecretProfile" ),
                  help="Pre-shared CAK/CKN, 802.1x derived CAK/CKN or CAK/CKN from"
                  " a shared-secret profile",
                  optional=True )
keySourceEnumToString = { "preSharedKey": "pre-shared key",
                          "dot1x": "dot1x",
                          "sharedSecretProfile": "shared-secret profile" }
txSakDefault = Tac.Value( "Macsec::Sak" )
staticSak = 'static SAK'

oliveConfig = None
oliveStatus = None

def getStaticSakKeyInUse( cpStatus ):
   suffix = ""
   addComma = ""

   for i in range( 4 ):
      if i in cpStatus.rxSak:
         suffix += addComma + str(i)
         addComma = ","
   if suffix != "":
      suffix = " Rx AN: " + suffix
   if cpStatus.txSak:
      suffix += " Tx AN: " + str( cpStatus.txSak.an )
   return staticSak + ":" + suffix  if suffix != "" else ""

class MacsecMessageCounterDetail( Model ):
   rxInvalid = Int( help="Invalid interface or null packets" )
   rxEapolError = Int( help="EAPOL errored packets" )
   rxBasicParamSetError = Int( help="Basic parameter set errored received packets" )
   rxUnrecognizedCkn = Int( help="Unrecognized ckn in received packets" )
   rxIcvValidationError = Int( help="ICV validation error in received packets" )
   rxLivePeerListError = Int( help="Live peerlist set errored received packets" )
   rxPotentialPeerListError = Int( help="Potential peerlist set errored"
                                        " received packets" )
   rxSakUseSetError = Int( help="SAK Use Set errored received packets" )
   rxDistSakSetError = Int( help="Dist SAK Set errored received packets" )
   rxDistCakSetError = Int( help="Dist CAK Set errored received packets" )
   rxXpnSetError = Int( help="XPN Set errored received packets" )
   rxIcvIndicatorError = Int( help="ICV Indicator errored received packets" )
   rxUnrecognizedSetError = Int( help="Unrecognized parameter set errored"
                                      " received packets" )
   txInvalid = Int( help="Invalid interface in tx packets" )

   def fromTacc( self, intfCounter ):
      self.rxInvalid = intfCounter.rxMsgCounter.invalid
      self.rxEapolError = intfCounter.rxMsgCounter.eapolErr
      self.rxBasicParamSetError = intfCounter.rxMsgCounter.basicParamSetErr
      self.rxUnrecognizedCkn = intfCounter.rxMsgCounter.unrecognizedCkn
      self.rxIcvValidationError = intfCounter.rxMsgCounter.icvValidationErr
      self.rxLivePeerListError = intfCounter.rxMsgCounter.livePeerListErr
      self.rxPotentialPeerListError = intfCounter.rxMsgCounter.potentialPeerListErr
      self.rxSakUseSetError = intfCounter.rxMsgCounter.sakUseSetErr
      self.rxDistSakSetError = intfCounter.rxMsgCounter.distSakSetErr
      self.rxDistCakSetError = intfCounter.rxMsgCounter.distCakSetErr
      self.rxXpnSetError = intfCounter.rxMsgCounter.xpnSetErr
      self.rxIcvIndicatorError = intfCounter.rxMsgCounter.icvIndicatorErr
      self.rxUnrecognizedSetError = intfCounter.rxMsgCounter.unrecognizedSetErr
      self.txInvalid = intfCounter.txMsgCounter.invalid

class MacsecMessageCountersInterface( Model ):
   rxPacketsSuccess = Int( help="Total of successfully received packets" )
   rxPacketsFailure = Int( help="Total of invalid/errored received packets" )
   txPacketsSuccess = Int( help="Total of successfully transmitted packets" )
   txPacketsFailure = Int( help="Total of invalid/errored transmitted packets" )

   details = Submodel( help="MAC security counter detailed information",
                       valueType=MacsecMessageCounterDetail,
                       optional=True )

   def fromTacc( self, intfCounter, detail=False ):

      self.rxPacketsSuccess = intfCounter.rxMsgCounter.success
      self.rxPacketsFailure = intfCounter.rxMsgCounter.failure
      self.txPacketsSuccess = intfCounter.txMsgCounter.success
      self.txPacketsFailure = intfCounter.txMsgCounter.failure

      if detail:
         self.details = MacsecMessageCounterDetail()
         self.details.fromTacc( intfCounter )

   def render( self ):
      if not self.details:
         print( "%-15d %-15d %-15d %-15d" %
                ( self.rxPacketsSuccess,
                  self.rxPacketsFailure,
                  self.txPacketsSuccess,
                  self.txPacketsFailure ) )
      if self.details:
         detail = partial( print, ' ' * 3 )
         error = partial( print, ' ' * 7 )
         detail( "Tx packet success:", self.txPacketsSuccess )
         detail( "Tx packet failure:", self.txPacketsFailure )
         error( "Tx invalid:", self.details.txInvalid )
         detail( "Rx packet success:", self.rxPacketsSuccess )
         detail( "Rx packet failure:", self.rxPacketsFailure )
         error( "Rx invalid:", self.details.rxInvalid )
         error( "Rx eapol error:", self.details.rxEapolError )
         error( "Rx basic parameter set error:", self.details.rxBasicParamSetError )
         error( "Rx unrecognized CKN error:", self.details.rxUnrecognizedCkn )
         error( "Rx ICV validation error:", self.details.rxIcvValidationError )
         error( "Rx live peer list error:", self.details.rxLivePeerListError )
         error( "Rx potential peer list error:",
                self.details.rxPotentialPeerListError )
         error( "Rx SAK use set error:", self.details.rxSakUseSetError )
         error( "Rx distributed SAK set error:", self.details.rxDistSakSetError )
         error( "Rx distributed CAK set error:", self.details.rxDistCakSetError )
         error( "Rx Extended Packet Numbering set error:",
                self.details.rxXpnSetError )
         error( "Rx ICV Indicator error:", self.details.rxIcvIndicatorError )
         error( "Rx unrecognized parameter set error:",
                self.details.rxUnrecognizedSetError )

class MacsecMessageCounters( Model ):
   interfaces = Dict( help="A mapping between interfaces and"
                            " MAC security MKA counters", keyType=Interface,
                            valueType=MacsecMessageCountersInterface )

   def render( self ):
      if not self.interfaces:
         return

      printHeader = True
      if next( iter( self.interfaces.values() ) ).details:
         printHeader = False

      if printHeader:
         print( "%-15s %-15s %-15s %-15s %-15s" %
                ( 'Interface', 'Rx Success',
                  'Rx Failure', 'Tx Success', 'Tx Failure' ) )
      for key in Arnet.sortIntf( self.interfaces ):
         if printHeader:
            # below print has a comma so that a new line is not printed
            print( "%-15s" % key, end=' ' )
         else:
            print()
            print( "Interface: %s" % key )
         self.interfaces[ key ].render()

class MacsecParticipantDetail( Model ):
   keyServerAddr = MacAddress( help="Mac address of the key server" )
   keyServerPortId = Int( help="Port Id of the key server" )
   sakTransmit = Bool( help="The participant is using the distributed"
                            " SAK for transmit" )
   llpnExhaustion = Int( help="Number of times LLPN exhaustion event detected when"
                              " this participant is the principal actor" )
   keyServerMsgId = Str( help="Message identifier of the key server" )
   keyNum = Int( help="Key number of the distributed session association key" )
   livePeerList = List( help="List of all live peers for this participant",
                        valueType=str )
   potentialPeerList = List( help=" List of all potential peers for this"
                                  " participant", valueType=str )

   def fromTacc( self, actorStatus ):
      self.keyServerAddr = actorStatus.keyServer.addr
      self.keyServerPortId = actorStatus.keyServer.portNum
      self.sakTransmit = actorStatus.sakTransmit
      self.llpnExhaustion = actorStatus.llpnExhaustion
      self.keyServerMsgId = actorStatus.distSak.keyMsgId
      self.keyNum = actorStatus.distSak.keyNum
      self.livePeerList = list( actorStatus.livePeer )
      self.potentialPeerList = list( actorStatus.potentialPeer )

class MacsecParticipant( Model ):
   msgId = Str( help="Member identifier generated for the participant" )
   electedSelf = Bool( help="The participant has elected itself as the key server" )
   success = Bool( help="The participant has successfully elected a key"
                        " server and has at least one live peer" )
   defaultActor = Bool( help="The participant is spawned from a configured"
                             " fallback key" )
   principalActor = Bool( help="The participant is also a principal actor" )
   details = Submodel( valueType=MacsecParticipantDetail, optional=True,
                       help="MAC security participant detail information" )

   def fromTacc( self, actorStatus, detail=False ):
      self.msgId = actorStatus.msgId
      self.electedSelf = actorStatus.electedSelf
      self.success = actorStatus.success
      self.defaultActor = actorStatus.defaultActor
      self.principalActor = actorStatus.principal
      if detail:
         self.details = MacsecParticipantDetail()
         self.details.fromTacc( actorStatus )

   def render( self ):
      info = partial( print, ' ' * 5 )
      info( "Member ID:", self.msgId )
      keyMgmtRole = "Key Server" if self.electedSelf else "Non Key Server"
      info( "Key management role:", keyMgmtRole )
      info( "Success:", self.success )
      info( "Principal:", self.principalActor )
      keyType = "Fallback" if self.defaultActor else "Primary"
      info( "Key type:", keyType )
      if self.details:
         sci = "None"
         if self.details.keyServerAddr:
            sci = "%s::%s"
            sci %= ( self.details.keyServerAddr, self.details.keyServerPortId )
         info( "KeyServer SCI:", sci )
         info( "SAK transmit:", self.details.sakTransmit )
         info( "LLPN exhaustion:", self.details.llpnExhaustion )
         ki = "None"
         if self.details.keyServerMsgId:
            ki = f"{self.details.keyServerMsgId}:{self.details.keyNum}"
         info( "Distributed key identifier:", ki )
         info( "Live peer list:", self.details.livePeerList )
         info( "Potential peer list:", self.details.potentialPeerList )

class MacsecParticipantsInterface( Model ):
   participants = Dict( help="A mapping between CKN and MAC security participants",
                        keyType=str, valueType=MacsecParticipant )

   def render( self ):
      if not self.participants:
         return

      info = partial( print, ' ' * 3 )
      for key in sorted( self.participants ):
         info( "CKN:", key )
         self.participants[ key ].render()
         print()

class MacsecParticipants( Model ):
   interfaces = Dict( help="A mapping between interfaces and"
                           " MAC security interface participants",
                           keyType=Interface,
                           valueType=MacsecParticipantsInterface )

   def render( self ):
      if not self.interfaces:
         return

      for key in Arnet.sortIntf( self.interfaces ):
         print( "Interface:", key )
         self.interfaces[ key ].render()

class MacsecPtpBypassSupport( Model ):
   support = Enum( values=( 'notSupported', 'multicast', 'unicast', 'controlPlane',
                            'dataPlane', 'full' ), help="Supported PTP bypasses" )

class MacsecInterfaceDetailData( Model ):
   keyServerPriority = Int( help="Configured key server priority for the interface" )
   oldKeyMsgId = Str( help="A 96 bit message identifier for the old key" )
   oldKeyMsgNum = Int( help="Message number of the old key" )
   oldKeyTransmitting = Bool( help="The old key is currently used for"
                                   " encrypting data packets" )
   oldKeyReceiving = Bool( help="The old key is currently used for"
                                " decrypting data packets" )
   latestKeyMsgId = Str( help="A 96 bit message identifier for the latest key" )
   latestKeyMsgNum = Int( help="Message number of the latest key" )
   latestKeyTransmitting = Bool( help="The latest key is currently used for"
                                      " encrypting data packets" )
   latestKeyReceiving = Bool( help="The latest key is currently used for"
                                   " decrypting data packets" )
   sessionReKeyPeriod = Float( help="Period in seconds after which the session keys"
                                    " are refreshed" )
   sakRetireDelay = Int( help="Time in seconds after which the old SAK is "
                              "retired" )
   sakTransmitDelay = Int( help="Time in seconds after which the latest key "
                                 "transmit is enabled on key server" )
   mkaLifeTime = Int( help="Time in seconds after which peers are removed from "
                           "live or potential peers list, upon no response" )
   suspensionEvent = Enum( values=( 'sso', 'suspended', ), optional=True,
         help="Event due to which MKA is suspended" )
   localSsci = Str( help="Value of local SSCI" )
   ethFlowControl = Enum( values=( "Encrypted", "Bypassed", "None" ),
                          help="Flow control frame handling" )
   keyDerivationPadding = Enum( values=( "prepend", "append" ),
                                help="Padding for short length CAK/CKN" )
   unauthBypassProtocol = List( help="List of protocols allowed in unauthorized "
                                     "state", valueType=str )
   bypassProtocol = List( help="List of protocols without MAC security protection",
                          valueType=str )
   ptpBypass = List( help="List of supported PTP bypasses",
                     valueType=MacsecPtpBypassSupport )
   traffic = Enum( values=( intfTrafficStatus ), help="Type of traffic" )
   cachedSak = Bool( help="Cached SAK is used" )
   fipsPostStatus = Enum( values=( "none", "inProgress", "passed", "failed" ),
                          help="FIPS Power-on self-test" )
   profileName = Str( help="MAC security profile name" )

   def fromTacc( self, intfStatus, cpStatus, hwPostStatus, portStatus, hwIntfStatus,
                 ptpBypassesSupported, fipsRestrictions ):
      self.keyServerPriority = portStatus.keyServerPriority
      self.sessionReKeyPeriod = intfStatus.reKeyPeriod
      self.sakRetireDelay = intfStatus.oldKeyRetirementPeriod
      self.sakTransmitDelay = intfStatus.latestKeyEnablePeriod
      self.mkaLifeTime = intfStatus.mkaLifeTime
      if portStatus.ssoTriggered:
         self.suspensionEvent = "sso"
      elif portStatus.suspended:
         self.suspensionEvent = "suspended"
      self.oldKeyMsgId = cpStatus.oldSakContext.sak.keyMsgId
      self.oldKeyMsgNum = cpStatus.oldSakContext.sak.keyNum
      self.oldKeyTransmitting = cpStatus.oldSakContext.txInstalled
      self.oldKeyReceiving = cpStatus.oldSakContext.rxInstalled
      self.latestKeyMsgId = cpStatus.latestSakContext.sak.keyMsgId
      self.latestKeyMsgNum = cpStatus.latestSakContext.sak.keyNum
      self.latestKeyTransmitting = cpStatus.latestSakContext.txInstalled
      self.latestKeyReceiving = cpStatus.latestSakContext.rxInstalled
      self.localSsci = cpStatus.mySsci.ssci

      if intfStatus.ethFlowControl == Tac.Type( "Macsec::EthFlowControl" ).bypass:
         self.ethFlowControl = "Bypassed"
      elif intfStatus.ethFlowControl == Tac.Type( "Macsec::EthFlowControl" ).encrypt:
         self.ethFlowControl = "Encrypted"
      else:
         self.ethFlowControl = "None"

      if intfStatus.keyDerivationPadding == \
                        Tac.Type( "Macsec::KeyDerivationPadding" ).append:
         self.keyDerivationPadding = "append"
      else:
         self.keyDerivationPadding = "prepend"

      # If controlledPort is disabled then no traffic is flowing.
      # Else traffic flowing is either unencrypted or encrypted.
      if not cpStatus.controlledPortEnabled:
         self.traffic = 'None'
         self.cachedSak = False
      elif cpStatus.unprotectedTraffic:
         self.traffic = 'Unprotected'
         self.cachedSak = False
      elif portStatus.numSuccessActors > 0:
         self.traffic = 'Protected'
         self.cachedSak = False
      else:
         self.traffic = 'Protected'
         self.cachedSak = True

      self.unauthBypassProtocol = []
      self.bypassProtocol = []
      if intfStatus.bypassLldpUnauth:
         self.unauthBypassProtocol.append( "LLDP" )
      if intfStatus.bypassLldp:
         self.bypassProtocol.append( "LLDP" )

      # Set the list of supported PTP bypasses
      PtpBypass = Tac.Type( "Macsec::PtpBypass" )
      if intfStatus.bypassPtp == PtpBypass.ptpBypass:
         PtpBypassSupport = Tac.Type( "Macsec::PtpBypassSupport" )
         if ptpBypassesSupported:
            for bypassSupported in ptpBypassesSupported:
               macsecPtpBypassSupport = MacsecPtpBypassSupport()
               macsecPtpBypassSupport.support = bypassSupported
               self.ptpBypass.append( macsecPtpBypassSupport )

            # If the interface is configured to bypass PTP and there is some PTP
            # bypass support, append PTP to the list of bypassed protocols.
            if PtpBypassSupport.notSupported not in ptpBypassesSupported:
               self.bypassProtocol.append( "PTP" )
         else:
            macsecPtpBypassSupport = MacsecPtpBypassSupport()
            macsecPtpBypassSupport.support = PtpBypassSupport.notSupported
            self.ptpBypass.append( macsecPtpBypassSupport )

      if macsecToggle.toggleMacsecFipsPostFailureForcedEnabled() and \
            intfStatus.fipsPostFailureForced and fipsRestrictions:
         self.fipsPostStatus = "failed"
      elif hwPostStatus is None:
         self.fipsPostStatus = "none"
      elif hwPostStatus == 'complete':
         self.fipsPostStatus = "passed"
      elif hwPostStatus == 'failed':
         self.fipsPostStatus = "failed"
      else:
         self.fipsPostStatus = "inProgress"

      self.profileName = intfStatus.profileName

class MacsecInterface( Model ):
   address = MacAddress( help="MAC address of the local port" )
   portId = Int( help="Port number of the local port" )
   controlledPort = Bool( help="MAC security is operational" )
   keyMsgId = Str( help="Message identifier of the distributed key currently used" )
   keyNum = Int( help="Key number of the distributed key currently used" )
   details = Submodel( help="MAC security Interface detail information",
                       valueType=MacsecInterfaceDetailData, optional=True )

   def fromTacc( self, portStatus, cpStatus, intfStatus, hwPostStatus, hwIntfStatus,
                 hwCapabilities, fipsRestrictions, detail=False ):

      self.address = portStatus.mySci.addr
      self.portId = portStatus.mySci.portNum
      self.controlledPort = cpStatus.controlledPortEnabled
      if intfStatus.staticSakInUse:
         self.address = cpStatus.mySsci.sci.addr
         self.portId = cpStatus.mySsci.sci.portNum
         self.keyMsgId = getStaticSakKeyInUse( cpStatus )
         self.keyNum = 0
      elif cpStatus.txSak != txSakDefault:
         self.keyMsgId = cpStatus.txSak.keyMsgId
         self.keyNum = cpStatus.txSak.keyNum
      else:
         self.keyMsgId = ""
         self.keyNum = 0

      if detail:
         self.details = MacsecInterfaceDetailData()
         self.details.fromTacc( intfStatus, cpStatus, hwPostStatus, portStatus,
                                hwIntfStatus, hwCapabilities, fipsRestrictions )

   def renderKey( self, msgId, msgNum, rx, tx ):
      if not msgId:
         return "None"

      flags = ""
      if rx:
         flags += "R"
      if tx:
         flags += "T"
      return msgId + ":" + str( msgNum ) + "(%s)" % flags

   def render( self ):
      ki = "None"
      if self.keyMsgId and staticSak not in self.keyMsgId:
         ki = self.keyMsgId + ':' + str( self.keyNum )
      elif self.keyMsgId:
         ki = self.keyMsgId

      if not self.details:
         print( "%-10s::%-6s %-20s %-30s" %
               ( self.address,
                 self.portId,
                 self.controlledPort, ki ) )
      else:
         detail = partial( print, ' ' * 3 )
         detail( "Profile:", self.details.profileName )
         detail( f"SCI: {self.address}::{self.portId}" )
         detail( "SSCI:", self.details.localSsci )
         detail( "Controlled port:", self.controlledPort )
         detail( "Key server priority:", self.details.keyServerPriority )
         detail( "Key derivation padding:", self.details.keyDerivationPadding )
         detail( "Session rekey period:", self.details.sessionReKeyPeriod )
         detail( "Flow Control:", self.details.ethFlowControl )
         detail( "RxSAK retire delay:", self.details.sakRetireDelay, "seconds" )
         detail( "TxSAK transmit delay:", self.details.sakTransmitDelay, "seconds" )
         detail( "MKA lifetime:", self.details.mkaLifeTime, "seconds" )
         if self.details.suspensionEvent == 'sso':
            detail( "MKA suspended: true(SSO)" )
         elif self.details.suspensionEvent:
            detail( "MKA suspended: true" )
         trafficStatus =  intfTrafficStatus[ self.details.traffic ]
         if self.details.cachedSak:
            trafficStatus += ' using cached SAK'
         detail( "Traffic:", trafficStatus )
         if self.details.bypassProtocol:
            detail( "Bypassed protocols:", ", ".join( self.details.bypassProtocol ) )
         if self.details.unauthBypassProtocol:
            detail( "Unauthorized state bypassed protocols:",
                  ", ".join( self.details.unauthBypassProtocol ) )
         bypassesSupported = []
         for bypassSupport in self.details.ptpBypass:
            bypassesSupported.append( bypassSupport.support )
         bypassStr = getMacSecIntfDetailPtpBypass( bypassesSupported )
         if bypassStr:
            detail( "PTP bypass:", bypassStr )
         detail( "Key in use:", ki )
         key = self.renderKey( self.details.latestKeyMsgId,
                               self.details.latestKeyMsgNum,
                               self.details.latestKeyReceiving,
                               self.details.latestKeyTransmitting )
         detail( "Latest key:", key )
         key = self.renderKey( self.details.oldKeyMsgId,
                               self.details.oldKeyMsgNum,
                               self.details.oldKeyReceiving,
                               self.details.oldKeyTransmitting )
         detail( "Old key:", key )
         if self.details.fipsPostStatus != "none":
            detail( "FIPS POST:", end=' ' )
            if self.details.fipsPostStatus == "inProgress":
               print( "In Progress" )
            else:
               print( self.details.fipsPostStatus.capitalize() )


class MacsecInterfaces( Model ):
   interfaces = Dict( help="A mapping between interfaces and"
                           " MAC security interface information",
                           keyType=Interface,
                           valueType=MacsecInterface )

   def render( self ):
      if not self.interfaces:
         return

      if next( iter( self.interfaces.values() ) ).details:
         template = "\nInterface: %s"
         info = print
      else:
         print( "%-15s %-25s %-20s %-20s" %
                ( 'Interface', 'SCI', 'Controlled Port', 'Key in Use' ) )
         template = "%-15s"
         info = partial( print, end=' ' )

      for key in Arnet.sortIntf( self.interfaces ):
         info( template % key )
         self.interfaces[ key ].render()

class MacsecStatus( Model ):
   adminStateEnabled = Bool( help="Administrative state is enabled" )
   activeProfiles = Int( help="Number of profiles configured in at least"
                              "one interface" )
   delayProtection = Bool( help="Data delay protection is enabled" )
   eapolDestMac = MacAddress( help="MACsec EAPoL destination MAC address" )
   if macsecToggle.toggleMacsecConfigurableEapolEtherTypeEnabled():
      eapolEtherType = Int( help="MACsec EAPoL Ethernet type" )
   fipsMode = Bool( help="FIPS Mode is enabled" )
   securedInterfaces = Int( help="Number of interfaces with MAC security enabled" )
   licenseEnabled = Bool( help="License is enabled" )
   hwLicenseEnabled = Bool( help="Hardware license is enabled", optional=True )

   def fromTacc( self, status, adminState ):
      self.adminStateEnabled = adminState
      self.activeProfiles = len( { intfStatus.profileName for intfStatus in
                                      status.intfStatus.values() } )
      self.delayProtection = status.delayProtection
      self.eapolDestMac = status.eapolAttr.destinationMac
      if macsecToggle.toggleMacsecConfigurableEapolEtherTypeEnabled():
         self.eapolEtherType = status.eapolAttr.etherType
      self.fipsMode = status.fipsStatus.fipsRestrictions

      def isSecured( cpStatus ):
         if cpStatus.controlledPortEnabled:
            if cpStatus.txSak == txSakDefault:
               # intfStatus can be assumed to be always present if cpStatus exist,
               # so the below operation shouldn't result in keyError.
               intfStatus = status.intfStatus[ cpStatus.intfId ]
               # Additionally count static sak interfaces which have rxSak only
               # programmed as secured. Tx Sak only interfaces will get counted
               # above.
               return bool( cpStatus.rxSak and intfStatus.staticSakInUse )
            return True
         return False

      self.securedInterfaces = \
            sum( isSecured( s ) for s in status.cpStatus.values() )
      self.licenseEnabled = status.licenseEnabled
      if oliveConfig.isOliveHw:
         self.hwLicenseEnabled = oliveStatus.macsecEnabled

   def render( self ):

      def printLine( label, content ):
         print( "%-25s %s " % ( label, content ) )

      adminState = "enabled" if self.adminStateEnabled else "disabled"
      printLine( "Administrative State:", adminState )
      printLine( "Active Profiles:", self.activeProfiles )
      printLine( "Data Delay Protection:", "no" )
      printLine( "EAPoL Destination MAC:",
                 self.eapolDestMac.displayString )
      if macsecToggle.toggleMacsecConfigurableEapolEtherTypeEnabled():
         printLine( "EAPoL Ethernet Type:",
               '0x{0:0{1}X}'.format( self.eapolEtherType, 4 ) )
      printLine( "FIPS Mode:", "yes" if self.fipsMode else "no" )
      printLine( "Secured Interfaces:", self.securedInterfaces )
      licenseContent = "enabled" if self.licenseEnabled else "disabled"
      if not self.licenseEnabled and self.hwLicenseEnabled is False:
         licenseContent += " (Hardware license not enabled)"
      printLine( "License:", licenseContent )

class MacsecProfile( Model ):
   cipher = Enum( values=tuple( tacCipherSuiteToCli.values() ),
                  help="Cipher suite for a profile" )
   ckn = Str( help="Primary connectivity association key name" )
   cakSha256Hash = Str( help="Sha256 hash of primary connectivity association key" )
   secProfName = Str( help="Primary shared secret profile name" )
   fallbackCkn = Str( help="Fallback connectivity association key name" )
   fallbackCakSha256Hash = Str( help="Sha256 hash of fallback connectivity "
                                     "association key" )
   fallbackSecProfName = Str( help="Fallback shared secret profile name" )
   source = Str( help="Cli or EosSdk agent name" )
   priority = Int( help="MACsec config priority" )
   keyRetirePolicy = Enum( values=( "immediate", "delayed" ),
                            help="Key retirement policy" )
   sakRetireDelay = Int( help="Time in seconds after which the old SAK is retired" )
   sakTransmitDelay = Int( help="Time in seconds after which the latest key "
                                 "transmit is enabled on key server" )
   mkaLifeTime = Int( help="Time in seconds after which peers are removed from "
                           "live or potential peers list, upon no response" )
   keyServerPriority = Int( help="Configured key server priority for the interface" )
   keyDerivationPadding = Enum( values=( "prepend", "append" ),
                                help="Padding for short length CAK/CKN" )
   sessionReKeyPeriod = Int( help="Period in seconds after which the session keys"
                                    " are refreshed" )
   unauthBypassProtocol = List( help="List of protocols allowed in unauthorized "
                                     "state", valueType=str )
   bypassProtocol = List( help="List of protocols without MAC security protection",
                          valueType=str )
   includeSci = Bool( help="Include secure channel identifier in data packets" )
   unprotectedTrafficPolicy = Enum ( values=trafficPolicyOnNoMkaToCli.values(),
                                     help="Unprotected traffic policy" )
   primaryKeySource = keySource
   fallbackKeySource = keySource
   sakAnMax = Int( help="Maximum value of AN of any SAK generated when the "
                        "interface configured with this profile is keyserver" )
   interfaces = List( valueType=Interface,
                      help="List of interfaces this profile is configured on" )

   def fromTacc( self, profileConfig, configSrc ):
      self.cipher = tacCipherSuiteToCli.get( profileConfig.cipherSuite, "unknown" )
      self.ckn = profileConfig.key.ckn
      self.cakSha256Hash = self.sha256Cak( profileConfig.key, configSrc )
      self.secProfName = profileConfig.secretProfileName
      self.fallbackCkn = profileConfig.defaultKey.ckn
      self.fallbackCakSha256Hash = self.sha256Cak( profileConfig.defaultKey,
                                                   configSrc )
      self.fallbackSecProfName = profileConfig.defaultSecretProfileName
      self.interfaces = list( profileConfig.intf )
      self.keyRetirePolicy = "immediate" if profileConfig.keyRetire else "delayed"
      self.unprotectedTrafficPolicy = trafficPolicyOnNoMkaToCli.get(
         profileConfig.trafficPolicyOnNoMka, "unknown" )
      self.primaryKeySource = self.getPrimaryKeySource( profileConfig )
      self.fallbackKeySource = self.getFallbackKeySource( profileConfig )
      self.sakRetireDelay = profileConfig.oldKeyRetirementPeriod
      self.sakTransmitDelay = profileConfig.latestKeyEnablePeriod
      self.mkaLifeTime = profileConfig.mkaLifeTime
      self.keyServerPriority = profileConfig.keyServerPriority
      self.keyDerivationPadding = profileConfig.keyDerivationPadding
      self.sessionReKeyPeriod = profileConfig.sessionReKeyPeriod
      self.includeSci = profileConfig.includeSci

      self.bypassProtocol = []
      self.unauthBypassProtocol = []
      if profileConfig.bypassLldpUnauth:
         self.unauthBypassProtocol.append( "LLDP" )
      if profileConfig.bypassLldp:
         self.bypassProtocol.append( "LLDP" )

      if profileConfig.ethFlowControl == Tac.Type( "Macsec::EthFlowControl" ).bypass:
         self.bypassProtocol.append( "ethernet-flow-control" )

      if profileConfig.bypassPtp == Tac.Type( "Macsec::PtpBypass" ).ptpBypass:
         self.bypassProtocol.append( "PTP" )

      if profileConfig.only1BitAnMode:
         self.sakAnMax = 1
      else:
         self.sakAnMax = 3

   def getPrimaryKeySource( self, profileConfig ):
      if profileConfig.dot1xEnabled:
         return "dot1x"
      elif self.ckn:
         return "preSharedKey"
      elif self.secProfName:
         return "sharedSecretProfile"
      else:
         return None

   def getFallbackKeySource( self, profileConfig ):
      if self.fallbackCkn:
         return "preSharedKey"
      elif self.fallbackSecProfName:
         return "sharedSecretProfile"
      else:
         return None

   def sha256Cak( self, cak, configSrc ):
      if configSrc == "cli":
         cakClearText = cak.cakSecret.getClearText()
      else:
         # SDK input has cak in type 7 obfuscate text
         cakClearText = decodeKey( cak.cak )

      if not cakClearText:
         return cakClearText
      return hashlib.sha256( cakClearText.encode() ).hexdigest()

   def _printIntfs( self, intfs, title, lineLen ):
      titleLen = len( title )
      intfStr = ", ".join( Arnet.sortIntf( intfs) )
      intfStr = title + " " + intfStr
      indent = ' ' * ( titleLen + 1 )
      wrapper = TextWrapper(width=lineLen, subsequent_indent=indent)
      intfStr = wrapper.fill( intfStr )
      print( intfStr )

   def render( self ):
      info = partial( print, ' ' * 3 )
      info( "Cipher:", self.cipher )
      info( "Primary CKN:", self.ckn )
      info( "Primary CAK SHA-256 hash:", self.cakSha256Hash )
      info( "Fallback CKN:", self.fallbackCkn if self.fallbackCkn else "" )
      info( "Fallback CAK SHA-256 hash:",
                 self.fallbackCakSha256Hash if self.fallbackCakSha256Hash else "" )
      info( "Source:", self.source )
      info( "Priority:", self.priority )
      info( "SCI Inclusion:", "enabled" if self.includeSci else "disabled" )
      info( "Key retirement policy:", self.keyRetirePolicy )
      if self.primaryKeySource:
         resultStr = keySourceEnumToString[ self.primaryKeySource ]
         if self.primaryKeySource == 'sharedSecretProfile':
            resultStr += ' ' + self.secProfName
         info( "Primary key source:", resultStr )
      if self.fallbackKeySource:
         resultStr = keySourceEnumToString[ self.fallbackKeySource ]
         if self.fallbackKeySource == 'sharedSecretProfile':
            resultStr += ' ' + self.fallbackSecProfName
         info( "Fallback key source:", resultStr )
      info( "Unprotected traffic policy:", self.unprotectedTrafficPolicy )
      info( "RxSAK retire delay:", self.sakRetireDelay, "seconds" )
      info( "TxSAK transmit delay:", self.sakTransmitDelay, "seconds" )
      info( "MKA lifetime:", self.mkaLifeTime, "seconds" )
      info( "MKA key server priority:", self.keyServerPriority )
      info( "Key derivation padding:", self.keyDerivationPadding )
      info( "Session rekey period:", self.sessionReKeyPeriod )
      info( "Bypassed protocols:", ", ".join( self.bypassProtocol ) )
      info( "Unauthorized state bypassed protocols:",
            ", ".join( self.unauthBypassProtocol ) )
      info( "Max AN value of SAK:", self.sakAnMax )
      self._printIntfs( self.interfaces, "    Configured on:", terminalWidth() - 1 )

class MacsecProfiles( Model ):
   profiles = Dict( help="Mapping between profile name and profile details",
                    keyType=str, valueType=MacsecProfile )
   def render( self ):
      for k, v in self.profiles.items():
         print( "Profile:", k )
         v.render()

class MacsecAllProfiles( Model ):
   sources = Dict( help="Mapping between source and profiles",
                   keyType=str, valueType=MacsecProfiles )

   def render( self ):
      for v in self.sources.values():
         v.render()

class MacsecSak( Model ):
   sakKeyServer = Str( help="Identifier of Key Server whose sak is programmed" )
   sakNumber = Int( help="Key number of the distributed key currently in use" )
   sakFrom = Enum( values=( "current-CA", "previous-CA", "static-SA", "None" ),
                   help="SAK generated source" )
   sakGenerated = Int( help="Number of SAK generated" )
   rxSakMaxInstallTime = Float( help="Max time taken to install Rx SAK in seconds" )
   txSakMaxInstallTime = Float( help="Max time taken to install Tx SAK in seconds" )
   forcedOldRxSakDeletion = Int( help="Number of times old sak deleted without"
         " the acknowledgement from non key server" )
   forcedNewTxSakInstallation = Int( help="Number of times new tx sak installed"
                                       " without the acknowledgement from non key"
                                       " server" )
   sakGenLivePeer = Int( help="SAK generated due to live peer" )
   sakGenRekey = Int( help="SAK generated due to rekey" )
   sakGenPN = Int( help="SAK generated due to packet number exhaustion" )
   rxSakInstallDurations = Dict( help="A mapping between rxSak install time and"
                            " count", keyType=str, valueType=int )
   txSakInstallDurations = Dict( help="A mapping between txSak install time and"
                            " count", keyType=str, valueType=int )

   def fromTacc( self, sakRecord, cpStatus, intfStatus ):
      self.sakKeyServer = cpStatus.txSak.keyMsgId
      if intfStatus.staticSakInUse:
         self.sakKeyServer = getStaticSakKeyInUse( cpStatus )

      self.sakNumber = cpStatus.txSak.keyNum
      self.sakFrom = sakRecord.sakFrom if sakRecord.sakFrom else "None"
      self.sakGenerated = sakRecord.sakGenerated
      self.rxSakMaxInstallTime = sakRecord.rxSakMaxInstallTime
      self.txSakMaxInstallTime = sakRecord.txSakMaxInstallTime
      self.forcedOldRxSakDeletion = sakRecord.forcedOldRxSakDeletion
      self.forcedNewTxSakInstallation = sakRecord.forcedNewTxSakInstallation
      self.sakGenLivePeer = sakRecord.sakGenLivePeer
      self.sakGenRekey = sakRecord.sakGenRekey
      self.sakGenPN = sakRecord.sakGenPN
      buckets = { 0: '0-1', 1: '1-2', 2: '2-3', 3: '3+' }
      for i in buckets.items():
         self.rxSakInstallDurations[ i[ 1 ] ] = 0
         self.txSakInstallDurations[ i[ 1 ] ] = 0
      for bucket in sakRecord.rxSakInstallTime:
         self.rxSakInstallDurations[ buckets[ bucket ] ] =\
               sakRecord.rxSakInstallTime[ bucket ]
      for bucket in sakRecord.txSakInstallTime:
         self.txSakInstallDurations[ buckets[ bucket ] ] =\
               sakRecord.txSakInstallTime[ bucket ]

   def getTable( self ):
      headings = ( 'Direction', '0-1', '1-2', '2-3', '3+' )
      table = createTable( headings )
      columns = []
      for head in headings:
         fmtColumn = Format( justify="right", minWidth=5 )
         if head == "Direction":
            fmtColumn = Format( justify="left", minWidth=10 )
         fmtColumn.noPadLeftIs( True )
         fmtColumn.padLimitIs( True )
         columns.append( fmtColumn )

      buckets = { 0: '0-1', 1: '1-2', 2: '2-3', 3: '3+' }
      table.formatColumns( *columns )
      data = self.rxSakInstallDurations
      table.newRow( 'Rx', data[ buckets[ 0 ] ], data[ buckets[ 1 ] ],
            data[ buckets[ 2 ] ], data[ buckets[ 3 ] ] )
      data = self.txSakInstallDurations
      table.newRow( 'Tx', data[ buckets[ 0 ] ], data[ buckets[ 1 ] ],
            data[ buckets[ 2 ] ], data[ buckets[ 3 ] ] )
      return table

   def render( self ):
      if self.sakKeyServer and staticSak not in self.sakKeyServer:
         print( "Installed SAK ID: %s:%d" % ( self.sakKeyServer, self.sakNumber ) )
      else:
         print( "Installed SAK ID:", self.sakKeyServer )
      print( "Installed SAK from:", self.sakFrom )
      print( "Total SAK generated:", self.sakGenerated )
      print( "SAK generated due to new live peer:", self.sakGenLivePeer )
      print( "SAK generated due to rekey timer:", self.sakGenRekey )
      print( "SAK generated due to packet number exhaustion:", self.sakGenPN )
      print( "SAK installation time( in seconds ):" )
      print( self.getTable().output() )
      print( "Maximum Rx installation time:", self.rxSakMaxInstallTime, "seconds" )
      print( "Maximum Tx installation time:", self.txSakMaxInstallTime, "seconds" )
      print( "Forced old Rx SAK deletion count:", self.forcedOldRxSakDeletion )
      print( "Forced new Tx SAK installation count:",
             self.forcedNewTxSakInstallation )
      print()

class MacsecSaks( Model ):
   interfaces = Dict( help="A mapping between interfaces and MAC security"
                     " interface SAKs", keyType=Interface, valueType=MacsecSak )

   def render( self ):
      if not self.interfaces:
         return

      for key in Arnet.sortIntf( self.interfaces ):
         interface = self.interfaces.get( key )
         if interface:
            print( "Interface:", key )
            self.interfaces[ key ].render()

class MacsecPtpBypassInterface( Model ):
   bypassSupport = List( help="List of supported PTP bypasses",
                         valueType=MacsecPtpBypassSupport )

   def fromTacc( self, bypassesSupported ):
      if bypassesSupported:
         for bypassSupported in bypassesSupported:
            macsecPtpBypassSupport = MacsecPtpBypassSupport()
            macsecPtpBypassSupport.support = bypassSupported
            self.bypassSupport.append( macsecPtpBypassSupport )
      else:
         PtpBypassSupport = Tac.Type( "Macsec::PtpBypassSupport" )
         macsecPtpBypassSupport = MacsecPtpBypassSupport()
         macsecPtpBypassSupport.support = PtpBypassSupport.notSupported
         self.bypassSupport.append( macsecPtpBypassSupport )

   def getBypassStr( self ):
      bypassesSupported = []
      for bypassSupport in self.bypassSupport:
         bypassesSupported.append( bypassSupport.support )
      bypassStr = getMacSecPtpBypass( bypassesSupported )
      return bypassStr

class MacsecPtpBypass( Model ):
   interfaces = Dict( help="A mapping between interfaces and PTP bypass information",
                      keyType=Interface, valueType=MacsecPtpBypassInterface )

   def render( self ):
      if not self.interfaces:
         return

      headings = ( 'Interface', 'Bypass Support' )
      table = createTable( headings )

      f1 = Format( justify="left", minWidth=15 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      f2 = Format( justify="left", minWidth=25 )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )
      table.formatColumns( f1, f2 )

      for key in Arnet.sortIntf( self.interfaces ):
         table.newRow( key, self.interfaces[ key ].getBypassStr() )

      print( table.output() )

class MacsecCapableInterface( Model ):
   ciphers = List( help="List of ciphers supported",
                      valueType=str )
   perIntfPtpBypass = Bool( help="Per Interface PTP bypass is supported" )
   ptpBypasses = List( help="List of PTP bypass packets",
                      valueType=str )
   subInterfaces = Bool( help="Subinterfaces are supported" )

   def fromTacc( self, capabilities ):
      for cipher in capabilities[ "ciphers" ]:
         self.ciphers.append( tacCipherSuiteToCli[ cipher ] )
      for ptpBypass in capabilities[ "ptpBypasses" ]:
         self.ptpBypasses.append( ptpBypassSupportToCli[ ptpBypass ] )
      self.perIntfPtpBypass = capabilities[ "perIntfPtpBypass" ]
      self.subInterfaces = capabilities[ "subInterface" ]

   def render( self ):
      ciphers = sorted( self.ciphers )
      ptpBypasses = sorted( self.ptpBypasses )
      perIntfPtpBypass = "supported" if self.perIntfPtpBypass else "unsupported"
      subInterfaces = "supported" if self.subInterfaces else "unsupported"
      print( "Cipher:", ", ".join( ciphers ) )
      print( "Per interface PTP bypass:", perIntfPtpBypass )
      print( "PTP bypass packets:", ", ".join( ptpBypasses ) )
      print( "Subinterfaces:", subInterfaces )
      print()

class MacsecCapableInterfaces( Model ):
   interfaces = Dict( keyType=Interface, valueType=MacsecCapableInterface,
         help="A mapping between MACsec capable interfaces and their capabilities " )

   def render( self ):
      for intf in Arnet.sortIntf( self.interfaces ):
         print( intf )
         self.interfaces[ intf ].render()

def Plugin( entityManager ):
   global oliveConfig, oliveStatus

   oliveConfig = LazyMount.mount( entityManager,
                                  Cell.path('hardware/olive/config' ),
                                  'Hardware::Olive::Config', 'r' )
   oliveStatus = LazyMount.mount( entityManager, 'hardware/olive/status',
                                  'Olive::Status', 'r' )
