# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements Macsec proxy configuration.
# In particular, it provides:
# - the "[no|default] mac security proxy patch" command
#

import CliCommand
import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.LagIntfCli as LagIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.SubIntfCli as SubIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.VirtualIntfRule as VirtualIntfRule
import CliToken.Mac
import ConfigMount
import LazyMount
import Tac

SubIntfId = Tac.Type( "Arnet::SubIntfId" )
EthIntfId = Tac.Type( "Arnet::EthIntfId" )

hwMacsecProxyCapability = None
macsecProxyConfig = None

def macsecProxySupported( mode, token ):
   intfName = mode.intf.name

   if SubIntfId.isSubIntfId( intfName ) and EthIntfId.isEthIntfId( intfName ):
      parentIntfName = EthIntfId.parentIntfIdFromSubIntf( intfName )
   else:
      return CliParser.guardNotThisPlatform

   for hwSlice in hwMacsecProxyCapability:
      sliceStatus = hwMacsecProxyCapability[ hwSlice ]
      if sliceStatus and parentIntfName in sliceStatus.proxyCapable:
         return None

   return CliParser.guardNotThisPlatform

# pylint: disable-next=inconsistent-return-statements
def setMacsecProxyIntf( mode, args ):
   # If an proxyIntf does not exist yet, create it now.
   intf = args[ 'INTF' ]
   clientIntf = args.get( 'EXTERNAL_INTF' )
   proxyIntf = mode.intf.name
   patchIntf = intf.name
   clientIntfId = Tac.Value( 'Arnet::IntfId', clientIntf.name if clientIntf else "" )

   if proxyIntf == patchIntf:
      mode.addError( 'patch and proxy interface should be different' )
      return False

   proxyParentIntf = proxyIntf.split( '.' )[ 0 ]
   patchParentIntf = patchIntf.split( '.' )[ 0 ]
   if patchParentIntf != proxyParentIntf:
      mode.addError( 'patch and proxy interface should have the same parent '
                     'interface' )
      return False

   if clientIntf and clientIntf.name == proxyParentIntf:
      # For RedBluff a 40G macsec port may be used as the client port.
      mode.addError( 'client port must not be the macsec proxy parent port' )
      return False

   if macsecProxyConfig.proxyConfig.get( proxyIntf ) is not None:
      mode.addError( 'Mac security proxy already configured' )
      return False
   if macsecProxyConfig.patchIntf.get( proxyIntf ) is not None:
      mode.addError( 'proxy interface is already configured as a '
                     'patch interface' )
      return False
   if macsecProxyConfig.proxyConfig.get( patchIntf ) is not None:
      mode.addError( 'patch interface is already configured as a '
                     'proxy interface' )
      return False
   if macsecProxyConfig.patchIntf.get( patchIntf ) is not None:
      mode.addError( 'patch interface is already configured to a '
                     'different proxy interface' )
      return False

   macsecProxyConfig.proxyConfig.newMember( proxyIntf, patchIntf, clientIntfId )
   macsecProxyConfig.patchIntf[ patchIntf ] = True
   macsecProxyConfig.clientIntf[ clientIntfId ] = True

def noMacsecProxyIntf( mode, args ):
   # Remove the macsec proxy configuration.
   intfId = mode.intf.name
   if intfId in macsecProxyConfig.proxyConfig:
      patchIntf = macsecProxyConfig.proxyConfig[ intfId ].patchIntf
      clientIntf = macsecProxyConfig.proxyConfig[ intfId ].clientIntf
      del macsecProxyConfig.proxyConfig[ intfId ]
      del macsecProxyConfig.patchIntf[ patchIntf ]
      del macsecProxyConfig.clientIntf[ clientIntf ]

#--------------------------------------------------------------------------------
# mac security proxy patch INTF [ external EXTERNAL_INTF ]
#--------------------------------------------------------------------------------
intfMatcher = VirtualIntfRule.IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher

class MacsecProxyIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'mac security proxy patch INTF [ external EXTERNAL_INTF ]'
   noOrDefaultSyntax = 'mac security proxy ...'
   data = {
      'mac': CliToken.Mac.macMatcherForConfigIf,
# Not using tokenSecurity from MacsecCli.py to avoid dependency cycle:
# Macsec -> MacsecProxy -> Macsec
      'security': CliCommand.guardedKeyword( 'security',
         helpdesc='Media Access Control Security (802.1AE) commands',
         guard=macsecProxySupported ),
      'proxy': 'MAC security proxy port',
      'patch': 'Patch to interface for external connectivity',
      'INTF': SubIntfCli.subMatcher,
      'external': 'Front panel interface associated with the proxy',
      'EXTERNAL_INTF': intfMatcher,
   }
   handler = setMacsecProxyIntf
   noOrDefaultHandler = noMacsecProxyIntf

SubIntfCli.SubIntfModelet.addCommandClass( MacsecProxyIntfCmd )

class MacsecProxyIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      if self.intf_.name in macsecProxyConfig.proxyConfig:
         patchIntf = macsecProxyConfig.proxyConfig[ self.intf_.name ].patchIntf
         clientIntf = macsecProxyConfig.proxyConfig[ self.intf_.name ].clientIntf
         del macsecProxyConfig.proxyConfig[ self.intf_.name ]
         del macsecProxyConfig.patchIntf[ patchIntf ]
         del macsecProxyConfig.clientIntf[ clientIntf ]

def Plugin( entityManager ):
   global macsecProxyConfig, hwMacsecProxyCapability
   macsecProxyConfig = ConfigMount.mount(
      entityManager, 'macsecproxy/config', 'MacsecProxy::Config', 'w' )
   hwMacsecProxyCapability = LazyMount.mount( entityManager,
                                'hardware/phy/status/macsecproxy/capability/slice',
                                'Tac::Dir', 'ri' )

   IntfCli.Intf.registerDependentClass( MacsecProxyIntfCleaner )
