#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Submodel, Str, Model, Enum, List, Float, Dict, Bool, Int
import datetime
import TableOutput
import Tac, Arnet
import Intf.IntfRange
from IntfModels import Interface
from CliPlugin.MaintenanceCliLib import maintenanceStateToStr, originToStr
from CliPlugin.MaintenanceCliLib import maintenanceStates, adminMaintenanceStates
from CliPlugin.MaintenanceCliLib import underGoingMaintStates
from CliPlugin.MaintenanceCliLib import utcToStr, nonDynamicUnitTypes
from CliPlugin.MaintenanceModels import MaintenanceEventInfoEntry
from CliPlugin.MaintenanceModels import UnitProfile

class MaintenanceStatus( Model ):
   state = Enum( values=maintenanceStates,
                 help="Operational state of unit under Maintenance" )
   adminState = Enum( values=adminMaintenanceStates,
                      help="Admin state of unit under Maintenance" )
   stateChangeTime = Float( help="UTC time at which the operational state of "
                            "Maintenance unit changed" )
   onBootMaintenance = Bool( help="Maintenance operation due to on-boot config",
                             optional=True )
   intfsViolatingTrafficThreshold = Bool(
         help="Interfaces part of this unit are violating traffic threshold "
         "after reaching maintenance", optional=True )
   aggInBpsRate = Int( help="Aggregate input bps rate of the Unit.", 
                       optional=True )
   aggOutBpsRate = Int( help="Aggregate output bps rate of the Unit.", 
                        optional=True )

class MaintenanceProtocolInfo( Model ):
   bgpPeers = Dict( valueType=MaintenanceStatus,
                    help="Map Bgp Peers to their Maintenance status" )

class MaintenanceAll( Model ):
   units = Dict( valueType=MaintenanceStatus, help="Map Unit name to status" )
   interfaces = Dict( valueType=MaintenanceStatus, keyType=Interface,
                      help="Map Interfaces to their Maintenance status" )
   vrfs = Dict( valueType=MaintenanceProtocolInfo,
                help="Map Vrf name to its members and their Maintenance status" )

   def render( self ):
      def createTable( tableHeadings, traffic=True ):
         """
         Function used to create and format the tables which are displayed in the
         'show maintenance' command.
         The 'Unit' and 'Dynamic Interface Unit' table have six columns while the
         'Dynamic BGP Unit' table has only 4 columns.
         If traffic is True, the function will format 6 columns else will format only
         4 columns.
         """
         table = TableOutput.createTable( tableHeadings )
         if traffic == True: # pylint: disable=singleton-comparison
            fName = TableOutput.Format( justify="left", minWidth=16, maxWidth=16,
                                        wrap=True )
            fStatus = TableOutput.Format( justify="left", minWidth=18, maxWidth=18,
                                          wrap=True )
            fLastChange = TableOutput.Format( justify="center", minWidth=15, 
                                              maxWidth=15, wrap=True )
            fRate = TableOutput.Format( justify="right", minWidth=10, maxWidth=10,
                                        wrap=True )
            fRate.padLimitIs( True )
            fFlag = TableOutput.Format( justify="center", minWidth=5, maxWidth=5,
                                        wrap=True )
            table.formatColumns( fName, fStatus, fLastChange, fRate, fRate, fFlag )
         else:
            fName = TableOutput.Format( justify="left", minWidth=24, maxWidth=24,
                                        wrap=True)
            fStatus = TableOutput.Format( justify="left", minWidth=22, maxWidth=22,
                                          wrap=True )
            fLastChange = TableOutput.Format( justify="center", minWidth=25, 
                                              maxWidth=25, wrap=True )
            fFlag = TableOutput.Format( justify="left", minWidth=5, maxWidth=5,
                                        wrap=True )
            table.formatColumns( fName, fStatus, fLastChange, fFlag )
         return table

      flagsList = [ ( "o", "On-boot maintenance" ),
                    ( "v", "Violating traffic threshold" ), ]
      if len( self.units ) or len( self.interfaces ) or len( self.vrfs ):
         print( "Flags:" )
         for flag, desc in flagsList:
            print( flag + " - " + desc )
         print( "" )

      if len( self.units ):
         tableHeadings = ( "Unit Name", "Status", 
                           ( "Time since", ("last change",) ) ,
                           ( "Aggregate Rate (Mbps)", ( "In", "Out" ) ),
                           "Flags" )
         
         table = createTable( tableHeadings )

         for unitName, unitStatus in sorted( self.units.items() ):
            flags = ""
            if unitStatus.onBootMaintenance:
               flags += "o"
            if unitStatus.intfsViolatingTrafficThreshold:
               flags += "v"

            # show the aggregate rates only if state is entering/under maintenance.
            if unitStatus.state in underGoingMaintStates:
               aggInpRateInMbps = "%.2f" % ( unitStatus.aggInBpsRate / 1000.0 /
                                             1000.0)
               aggOutRateInMbps = "%.2f" % ( unitStatus.aggOutBpsRate / 1000.0 / 
                                             1000.0 ) 
            else:
               aggInpRateInMbps = "-"
               aggOutRateInMbps = "-"

            table.newRow( unitName,
                          maintenanceStateToStr[ unitStatus.state ],
                          utcToStr( unitStatus.stateChangeTime ), 
                          aggInpRateInMbps, aggOutRateInMbps,
                          flags )
         print( table.output() )

      if len( self.interfaces ):
         tableHeadings = ( "Interface Name", "Status", 
                           ( "Time since", ("last change", ) ),
                           ( "Rate (Mbps)", ( "In", "Out" ) ), 
                           "Flags" )
         table = createTable( tableHeadings )

         for intfName in Arnet.sortIntf( self.interfaces ):
            intfStatus = self.interfaces[ intfName ]
            flags = ""
            if intfStatus.onBootMaintenance:
               flags += "o"
            if intfStatus.intfsViolatingTrafficThreshold:
               flags += "v"
           
            # show the aggregate rates only if state is entering/under maintenance.
            if intfStatus.state in underGoingMaintStates:
               aggInpRateInMbps = "%.2f" % ( intfStatus.aggInBpsRate / 1000.0 / 
                                             1000.0 )
               aggOutRateInMbps = "%.2f" % ( intfStatus.aggOutBpsRate / 1000.0 / 
                                             1000.0 )
            else:
               aggInpRateInMbps = "-"
               aggOutRateInMbps = "-"

            table.newRow( intfName, maintenanceStateToStr[ intfStatus.state ],
                          utcToStr( intfStatus.stateChangeTime ), 
                          aggInpRateInMbps, aggOutRateInMbps, flags )
         print( table.output() )

      if len( self.vrfs ):
         for vrf, bgpPeerDict in self.vrfs.items():
            vrfHeading = "(vrf: %s)" % vrf
            tableHeadings = ( ( "Bgp Neighbor", "l", ( vrfHeading, ) ),
                              "Status", "Time since last change",
                              "Flags" )
            table = createTable( tableHeadings, False )
            for peerName, bgpPeerStatus in sorted(
               bgpPeerDict.bgpPeers.items() ):
               flags = ""
               if bgpPeerStatus.onBootMaintenance:
                  flags += "o"
               table.newRow( peerName,
                             maintenanceStateToStr[ bgpPeerStatus.state ],
                             utcToStr( bgpPeerStatus.stateChangeTime ), flags )
            print( table.output() )

class UnitMaintenanceDefaultProfile( Model ):
   profileName = Str( help='Default Unit Profile Name' )
   profileAttributes = Submodel( valueType=UnitProfile,
                                 help="Attributes of the default Unit profile",
                                 optional=True )
   def render( self ):
      print( 'Unit Profile: %s' % self.profileName )
      if self.profileAttributes:
         self.profileAttributes.render()

class MaintenanceUnit( Model ):
   origin = Enum( values=nonDynamicUnitTypes,
                  help='origin of this unit', optional=True )
   state = Enum( values=maintenanceStates,
                 help='Operational state of the Maintenance Unit' )
   adminState = Enum( values=adminMaintenanceStates,
                      help='Admin state of the Maintenance Unit' )
   bgpGroups = List( valueType=str, help='List of Bgp Groups' )
   interfaceGroups = List( valueType=str,
                           help='List of Interface Groups' )
   timeOfStateChange = Float( help='Time since last state change', optional=True )
   unitProfile = Str( help="Name of the Unit Maintenance Profile "
                      "applied during Maintenance" )
   onBootMaintenance = Bool( help="Maintenance operation due to on-boot config",
                             optional=True )
   onBootExitTime = Float( help="Exit time of on-boot maintenance", optional=True )
   history = List( valueType=MaintenanceEventInfoEntry,
                   help='Historical information for unit' )
   intfsViolatingTrafficThreshold = List( valueType=Interface,
                  help="List of interfaces which are violating traffic threshold",
                  optional=True )
   totalIntfTrafficViolations = Int( help="Number of interfaces which have violated "
         "traffic threshold since maintenance", optional=True )

class MaintenanceUnits( Model ):
   maintenanceUnits = Dict( valueType=MaintenanceUnit, 
                            help="Map Unit name to Unit" )
   def render( self ):
      for key in sorted( self.maintenanceUnits.keys() ):
         unit = self.maintenanceUnits[ key ]
         print( 'Unit Name: %s' % key )
         print( '   Origin: %s' % originToStr[ unit.origin ] )
         state = maintenanceStateToStr[ unit.state ]
         if unit.onBootMaintenance:
            state += ' (on-boot)'
         print( '   Status: %s' % state ) 
         print( '   Unit Profile: %s' % unit.unitProfile )
         print( '   Time Since Last State Change: %s' % utcToStr(
            unit.timeOfStateChange ) )
         if unit.onBootExitTime and unit.onBootExitTime > Tac.utcNow():
            utcDelta = int( unit.onBootExitTime - Tac.utcNow() )
            print( '   Will come out of on-boot Maintenance after %s'
                   % datetime.timedelta( seconds=utcDelta ) )
         if unit.bgpGroups:
            print( '   Bgp Groups: ' )
            for group in sorted( unit.bgpGroups ):
               print( '     %s' % group )
         if unit.interfaceGroups:
            print( '   Interface Groups: ' )
            for group in sorted( unit.interfaceGroups ):
               print( '     %s' % group )
         if unit.totalIntfTrafficViolations:
            # Print this block only if required
            print( '   Interface Traffic Threshold violations:' )
            if len( unit.intfsViolatingTrafficThreshold ):
               print( '     Current violations: %s' % len(
                     unit.intfsViolatingTrafficThreshold ) )
               intfRange = Intf.IntfRange.intfListToCanonical(
                                       unit.intfsViolatingTrafficThreshold )
               for intfs in intfRange:
                  print( '       %s' % intfs )
            print( '     Total violations, during maintenance: %d' %
                   unit.totalIntfTrafficViolations )
         if unit.history:
            print( '   History: ' )
            for entry in unit.history:
               infoStr = utcToStr( entry.timeStamp, False ) + '  ' + \
                   entry.eventInfo + ' ' + utcToStr( entry.timeStamp )
               print( "    %s" % infoStr )
         print()


class MaintenanceInterfaceBrief( Model ):
   state = Enum( values=maintenanceStates,
                 help="Maintenance operational state of this interface" )
   adminState = Enum( values=adminMaintenanceStates,
                      help="Maintenance admin state of this interface" )
   lastInBpsRate = Float( help="Input bps rate.", optional=True )
   lastOutBpsRate = Float( help="Output bps rate.", optional=True )
   maintDownTime = Float( help="Time at which the interface was shutdown for "
                          "maintenance.", optional=True )
   trafficAboveThreshold = Bool( help="Indicates if the traffic on the interface "
                                 "is above threshold", optional=True )

class MaintenanceInterfacesBrief( Model ):
   interfaces = Dict( help="Maintenance mode information for interfaces",
                      valueType=MaintenanceInterfaceBrief, keyType=Interface )
   def render( self ):
      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( self.interfaces ):
         return

      flagsList = [ ( "v", "Violating traffic threshold" ),
                    ( "s", "Shutdown for maintenance" ), ]
      print( "Flags:" )
      for flag, desc in flagsList:
         print( flag + " - " + desc )

      headings = ( "Interface", "Status",
                   ( "Rate (Mbps)", "", ( "In", "Out" ) ),
                   "Flags" )
      table = TableOutput.createTable( headings )
      fmtName = TableOutput.Format( justify="left", minWidth=22, maxWidth=22 )
      fmtStatus = TableOutput.Format( justify="left", minWidth=22, maxWidth=22,
                                      wrap=True )
      fmtRate = TableOutput.Format( justify="right", minWidth=8, maxWidth=8 )
      fmtFlags = TableOutput.Format( justify="left", minWidth=5, maxWidth=5 )
      table.formatColumns( fmtName, fmtStatus, fmtRate, fmtRate, fmtFlags )

      for intf in Arnet.sortIntf( self.interfaces ):
         model = self.interfaces[ intf ]
         state = maintenanceStateToStr.get( model.state, "-")
         inMbps = outMbps = "-"
         flagsList = []
         # Need to check explicitly for None, because the rate could be 0 as well.
         if model.lastInBpsRate != None: # pylint: disable=singleton-comparison
            inMbps = "%.1f" % ( model.lastInBpsRate / 1000.0 / 1000.0 )
         if model.lastOutBpsRate != None: # pylint: disable=singleton-comparison
            outMbps = "%.1f" % ( model.lastOutBpsRate / 1000.0 / 1000.0 )
         if model.trafficAboveThreshold:
            flagsList.append("v")
         if model.maintDownTime:
            flagsList.append("s")
         flags = "".join(flagsList)
         table.newRow( intf, state, inMbps, outMbps, flags )
      print( table.output() )

class MaintenanceStage( Model ):
   stageName = Str( help="Maintenance stage name" )
   stageDescription = Str( help="Maintenance stage description" )

class MaintenanceStages( Model ):
   maintenanceEnterStages = List( valueType=MaintenanceStage,
                                  help="List of Maintenance enter stages" )
   maintenanceExitStages = List( valueType=MaintenanceStage,
                                 help="List of Maintenance exit stages" )
   def render( self ):
      def createTable( tableHeadings ):
         table = TableOutput.createTable( tableHeadings )
         f1 = TableOutput.Format( justify="left" )
         f2 = TableOutput.Format( justify="left" )
         f3 = TableOutput.Format( justify="left", minWidth=25, maxWidth=54,
                                  wrap=True )
         table.formatColumns( f1, f2, f3 )
         return table
      
      tableHeadings = ( "No.", "Stage", "Description" )
      if self.maintenanceEnterStages:
         print( "Maintenance Enter Stage Sequence" )
         table = createTable( tableHeadings )
         index = 1
         for maintenanceStage in self.maintenanceEnterStages:
            table.newRow( str( index ), maintenanceStage.stageName, 
                          maintenanceStage.stageDescription )
            index += 1
         print( table.output() )

      if self.maintenanceExitStages:
         print( "Maintenance Exit Stage Sequence" )
         table = createTable( tableHeadings )
         index = 1
         for maintenanceStage in self.maintenanceExitStages:
            table.newRow( str( index ), maintenanceStage.stageName, 
                          maintenanceStage.stageDescription )
            index += 1
         print( table.output() )

