# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Arnet
from CliMode.Intf import IntfMode
from CliModel import Enum, Float, Str, Int, Bool, Dict, List
from CliModel import Model, Submodel
import CliExtensions
from CliPlugin.MaintenanceCliLib import maintenanceStates, adminMaintenanceStates
from CliPlugin.MaintenanceCliLib import maintenanceStateToStr, originToStr
from CliPlugin.MaintenanceCliLib import utcToStr
from CliCommon import CliModelNotDegradable

indentLevel1 = " "
indentIncrement = "  "
indentLevel2 = indentLevel1 + indentIncrement
indentLevel3 = indentLevel2 + indentIncrement

# Based on Maintenance::MonIntfState in RateMonStatus.tac
monStates = ( "monitorInit", "monitorThreshold", "monitorPassive", "monitorStop" )

def intfMonStateStr( state ):
   if state == "monitorThreshold":
      return "Threshold monitoring"
   elif state == "monitorPassive":
      return "Passive monitoring"
   else:
      return state

def _degradeToIpGenAddress( peerKey ):
   try:
      Arnet.IpGenAddr( str( peerKey ) )
   except:
      # pylint: disable-next=raise-missing-from
      raise CliModelNotDegradable( "An IPv6 link-local address cannot be"
                                   " degraded to an IpGenAddr." )
   return peerKey

def _degradePeersDict( dictRepr ):
   """
   The revision change simply changed the type of 'peerAddress' IpGenAddress => str.
   The string representation of the peer key did not change, only its type.
   Therefore, the degrade routine only attempts to check the type of the key against
   the old model's type (IpGenAddress). If it cannot create an instance of this type,
   CliModelNotDegradable is raised as the client must be using an IPv6 link-local
   peer type which cannot be represented accurately as an IpGenAddress.
   """
   dictRepr[ 'peers' ] = { _degradeToIpGenAddress( pKey ): peer
                           for pKey, peer in dictRepr[ 'peers' ].items() }
   return dictRepr

class MaintenanceInterfaceBgpPeerStatus( Model ):
   appliedRouteMap = Str(
      help="Maintenance routemap applied for the peer", optional=True )
   appliedRouteMapIn = Str(
      help="Maintenance inbound routemap applied for the peer", optional=True )
   appliedRouteMapOut = Str(
      help="Maintenance outbound routemap applied for the peer", optional=True )

   def getKey( self, data ):
      if 'addrv6' in data and data[ 'addrv6' ]:
         ipAddr = data[ 'addrv6' ]
         del data[ 'addrv6' ]
      if 'addrv4' in data and data[ 'addrv4' ]:
         ipAddr = data[ 'addrv4' ]
         del data[ 'addrv4' ]
      return ipAddr

   def processData( self, data ):
      if 'applied_routemap_in' in data:
         data[ 'appliedRouteMapIn' ] = data[ 'applied_routemap_in' ]
      if 'applied_routemap_out' in data:
         data[ 'appliedRouteMapOut' ] = data[ 'applied_routemap_out' ]
      # if both in/out rms the same, set appliedRouteMap for b/w compatbility
      if 'applied_routemap_in' in data and 'applied_routemap_out' in data and \
         data[ 'applied_routemap_in' ] == data[ 'applied_routemap_out' ]:
         data[ 'appliedRouteMap' ] = data[ 'applied_routemap_in' ]
      return data

   def renderEntry( self, addr ):
      print( indentLevel3, "Neighbor: %s" % addr )

      if self.appliedRouteMapIn or self.appliedRouteMapOut:
         print( indentLevel3 + indentIncrement,
            "Maintenance inbound routemap: %s" % self.appliedRouteMapIn )
         print( indentLevel3 + indentIncrement,
            "Maintenance outbound routemap: % s" % self.appliedRouteMapOut )

class MaintenanceInterfaceBgpStatus( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         dictRepr = _degradePeersDict( dictRepr )
      return dictRepr

   maintState = Enum( values=maintenanceStates,
                      help='Maintenance State of the Interface' )
   vrf = Str( help='VRF Name' )
   peers = Dict( valueType=MaintenanceInterfaceBgpPeerStatus,
                 help='Maintenance state of the peers on this interface' )

   def processData( self, data ):
      if data[ 'maintenance_state' ]:
         data[ 'maintState' ] = 'underMaintenance'
      else:
         data[ 'maintState' ] = 'active'
      return data

   def render( self ):
      print( indentLevel1, "Bgp:" )
      print( indentLevel2, "Maintenance State: %s" % (
                                       maintenanceStateToStr[ self.maintState ] ) )
      print( indentLevel2, "Vrf: %s" % self.vrf )
      for addr, peer in self.peers.items():
         peer.renderEntry( addr )

class MaintenanceInterfaceRateMonStatus( Model ):
   state = Enum( values=monStates,
                 help="State of traffic rate monitoring for the interface." )
   stateTime = Float( help="The time at which the state was entered." )
   startTime = Float( help="The time at which rate monitoring was started for "
                      "this interface." )
   profile = Str( help="Name of the Interface profile being used to monitor the "
                  "interface" )
   lastSampleTime = Float( help="The time at which the last sample was taken." )
   lastInBpsRate = Float( help="Input bps rate." )
   lastOutBpsRate = Float( help="Output bps rate." )
   totalSamples = Int( help="Total number of samples collected so far." )
   numSamplesAboveThresholdPreMaint = Int(
         help="Number of samples above threshold, before maintenance." )
   numSamplesBelowThresholdPreMaint = Int(
         help="Number of samples below threshold, before maintenance." )
   numSamplesAboveThresholdPostMaint = Int(
         help="Number of samples above threshold, after maintenance." )
   numSamplesBelowThresholdPostMaint = Int(
         help="Number of samples below threshold, after maintenance." )
   maintDownTime = Float( help="Time at which the interface was shutdown for "
                          "maintenance.", optional=True )

   def render( self ):
      if not self.state:
         return
      inMbpsRate = self.lastInBpsRate / 1000.0 / 1000.0
      outMbpsRate = self.lastOutBpsRate / 1000.0 / 1000.0

      print( "  Rate Monitoring:" )
      if self.maintDownTime:
         print( "    Shutdown for maintenance since %s" %
                utcToStr( self.maintDownTime ) )
      else:
         print( "    %s since %s" %
                ( intfMonStateStr( self.state ), utcToStr( self.stateTime ) ) )
      print( "    Total samples taken: %d" % self.totalSamples )
      print( "      Before Maintenance:" )
      print( "        Below threshold: %d" % self.numSamplesBelowThresholdPreMaint )
      print( "        Above threshold: %d" % self.numSamplesAboveThresholdPreMaint )
      print( "      After Maintenance:" )
      print( "        Below threshold: %d" % self.numSamplesBelowThresholdPostMaint )
      print( "        Above threshold: %d" % self.numSamplesAboveThresholdPostMaint )
      print( "    Last sample information:" )
      print( "      Sample taken %s" % utcToStr( self.lastSampleTime ) )
      print( "      In:  %.1f Mbps" % inMbpsRate )
      print( "      Out: %.1f Mbps" % outMbpsRate )

class MaintenanceInterfaceStatus( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2 and 'bgpStatus' in dictRepr:
         dictRepr[ 'bgpStatus' ] = MaintenanceInterfaceBgpStatus().degrade(
                                      dictRepr[ 'bgpStatus' ], revision )
      return dictRepr

   state = Enum( values=maintenanceStates,
                 help="Maintenance operational state of this interface",
                 optional=True )
   adminState = Enum( values=adminMaintenanceStates,
                      help="Maintenance admin state of this interface" )
   groups = List( help="Groups that the interface is part of", valueType=str )
   intfProfile = Str( help="Name of the Interface Maintenance Profile "
                      "applied during Maintenance", optional=True )
   bgpProfile = Str( help="Name of the BGP Maintenance Profile "
                     "applied during Maintenance", optional=True )
   rateMonStatus = Submodel( valueType=MaintenanceInterfaceRateMonStatus,
                             help="Rate Monitoring status for the interface",
                             optional=True )
   bgpStatus = Submodel( valueType=MaintenanceInterfaceBgpStatus,
                         help="BGP status for the interface",
                         optional=True )
   history = List( valueType=str, help="Historical information for dynamic interface"
                   " unit" )

   def render( self ):
      if len( self.groups ):
         print( "  Groups: %s" % ', '.join( self.groups ) )
      if self.intfProfile or self.bgpProfile:
         print( "  Selected profiles from Interface groups:" )
         if self.intfProfile:
            print( "    Interface Maintenance profile: %s" % self.intfProfile )
         if self.bgpProfile:
            print( "    Bgp Maintenance profile: %s" % self.bgpProfile )
      if self.bgpStatus:
         self.bgpStatus.render()
      if self.rateMonStatus:
         self.rateMonStatus.render()
      if self.history:
         print( "  History: " )
         for info in self.history:
            print( "    %s" % info )

class MaintenanceRateMonSummary( Model ):
   numIntfsEnterMaint = Int( help="Number of interfaces entering maintenance." )
   numIntfsUnderMaint = Int( help="Number of interfaces under maintenance." )
   numIntfsThresholdViolation = Int( help="Number of interfaces which violated "
                                     "the traffic threshold after going under "
                                     "maintenance." )
   numIntfsMaintDown = Int( help="Number of interfaces shutdown for maintenance." )

   def render( self ):
      print( "Rate Monitoring:" )
      print( "  Number of interfaces Entering Maintenance: %u" %
            self.numIntfsEnterMaint )
      print( "  Number of interfaces Under Maintenance: %u" %
            self.numIntfsUnderMaint )
      print( "  Number of interfaces Under Maintenance with threshold violation: "
            "%u" % self.numIntfsThresholdViolation )
      print( "  Number of interfaces shutdown for maintenance: %u" %
             self.numIntfsMaintDown )

class MaintenanceSummary( Model ):
   numUnitsConfigured = Int( help="Number of Units configured" )
   unitStatistics = Dict( help="Map different Maintenance States"
                          " to number of Units", valueType=int )
   rateMonSummary = Submodel( valueType=MaintenanceRateMonSummary,
                              optional=True,
                              help="Interface Rate Monitoring summary" )
   intfDirectlyInMaintenance = Dict( help="Map Maintenance States to"
    " number of Interfaces directly put in Maintenance", valueType=int )
   peerDirectlyInMaintenance = Dict( help="Map Maintenance States to"
    " number of Peers directly put in Maintenance", valueType=int )

   def render( self ):
      print( "Number of Units Configured: %u" % self.numUnitsConfigured )
      for statsName, statsValue in self.unitStatistics.items():
         print( "Number of Units %s: %u" % ( maintenanceStateToStr[ statsName ],
                                            statsValue ) )
      print( "Directly Put Under Maintenance:" )
      for state, value in self.intfDirectlyInMaintenance.items():
         print( "  Number of interfaces %s: %i" % (
            maintenanceStateToStr[ state ], value ) )
      for state, value in self.peerDirectlyInMaintenance.items():
         print( "  Number of bgp peers %s: %i" % (
            maintenanceStateToStr[ state ], value ) )
      self.rateMonSummary.render()

class UnitProfile( Model ):
   onBootDuration = Int( help='Duration, in seconds, for which the unit '
                         'will be under maintenance on boot. ' )
   onBootEnabled = Bool( help='Whether onboot is enabled or not' )

   def render( self ):
      print( '   On-boot:' )
      if self.onBootEnabled:
         print( '     enabled: yes' )
      else:
         print( '     enabled: no' )
      print( '     duration: %i seconds' % self.onBootDuration )
      print()

class UnitProfiles( Model ):
   unitProfiles = Dict( valueType=UnitProfile,
                        help='Map Unit Profile name to Unit Profile' )

   def render( self ):
      for key in sorted( self.unitProfiles.keys() ):
         unitProfile = self.unitProfiles[ key ]
         print( 'Unit Profile: %s' % key )
         unitProfile.render()

class InterfaceProfile( Model ):
   rateMonLoadInterval = Int( help='Duration, in seconds, for which the traffic '
                              'needs to remain below the threshold before the '
                              'interface can be declared to be under maintenance' )
   shutdown = Bool( help='Whether to shutdown interface when the traffic goes '
                    'below threshold' )
   rateMonThreshold = Float( help='Traffic threshold, in Kbps' )
   shutdownMaxDelay = Float( help='Maximum time to wait, in seconds, before '
                             'shutting down the interface, if shutdown is '
                             'configured' )

   def render( self ):
      print( '   Rate Monitoring:' )
      print( '     load-interval: %i seconds' % int( self.rateMonLoadInterval ) )
      print( '     threshold (in/out): %i Kbps' % int( self.rateMonThreshold ) )
      print( '   shutdown: ' )
      if self.shutdown:
         print( '     enabled: yes' )
      else:
         print( '     enabled: no' )
      print( '     max-delay: %i seconds' % int( self.shutdownMaxDelay ) )
      print()

class InterfaceProfiles( Model ):
   intfProfiles = Dict( valueType=InterfaceProfile,
                        help='Map Interface Profile name to Interface Profile '
                        'information' )

   def render( self ):
      for key in sorted( self.intfProfiles.keys() ):
         interfaceProfile = self.intfProfiles[ key ]
         print( 'Interface Profile: %s' % key )
         interfaceProfile.render()

class BgpProfile( Model ):
   routeMapInOut = Str( help='Initiator route map for BGP profile' )
   routeMapIn = Str( help='Inbound initiator route map for BGP profile' )
   routeMapOut = Str( help='Outbound initiator route map for BGP profile' )

   def render( self ):
      print( '   Initiator route map: ' )
      rmIn = self.routeMapIn if self.routeMapIn else "(none)"
      print( '     Inbound: %s' % rmIn )
      rmOut = self.routeMapOut if self.routeMapOut else "(none)"
      print( '     Outbound: %s' % rmOut )
      print()

class BgpProfiles( Model ):
   bgpProfiles = Dict( valueType=BgpProfile,
                       help='Map BGP Profile name to BGP Profile information' )

   def render( self ):
      for key in sorted( self.bgpProfiles.keys() ):
         bgpProfile = self.bgpProfiles[ key ]
         print( 'Bgp Profile: %s' % key )
         bgpProfile.render()

class MaintenanceProfiles( Model ):
   intfProfiles = Dict( valueType=InterfaceProfile,
                        help="Map profile name to Interface Profile information" )
   bgpProfiles = Dict( valueType=BgpProfile,
                       help="Map profile name to BGP Profile information" )
   unitProfiles = Dict( valueType=UnitProfile,
                        help="Map profile name to Unit Profile information" )

   def render( self ):
      for key in sorted( self.intfProfiles ):
         intfProfile = self.intfProfiles[ key ]
         print( 'Interface Profile: %s' % key )
         intfProfile.render()

      for key in sorted( self.bgpProfiles ):
         bgpProfile = self.bgpProfiles[ key ]
         print( 'Bgp Profile: %s' % key )
         bgpProfile.render()

      for key in sorted( self.unitProfiles ):
         unitProfile = self.unitProfiles[ key ]
         print( 'Unit Profile: %s' % key )
         unitProfile.render()

# NOTE: Ideally, the valueType for the interfaces List below should be Interface.
# But we cant make Maintenance package depend on interface, so leaving it
# as str.
class InterfaceGroup( Model ):
   origin = Enum( values=( "userConfigured", "builtin" ),
                  help="origin of this group" )
   interfaces = List( valueType=str, help="List of Interfaces in the group" )
   bgpProfile = Str( help="BGP Maintenance Profile for this group" )
   intfProfile = Str( help="Interface Maintenance Profile for this group" )
   units = List( valueType=str, help="List of Maintenance Units of which this "
                 "group is a member of" )

   def render( self ):
      print( "  Origin: %s" % originToStr[ self.origin ] )
      print( "  Interfaces:" )
      # Can't use the IntfRange API as that'd introduce a dependency between
      # Maintenance and Intf - which we want to avoid.
      # if intfGroup.interfaces:
      #    intfRange = Intf.IntfRange.intfListToCanonical( intfGroup.interfaces )
      #    for intfs in intfRange:
      #       print "    %s" % intfs
      if len( self.interfaces ):
         print( "    %s" % ( ', '.join( IntfMode.getShortname( i )
               for i in Arnet.sortIntf( self.interfaces ) ) ) )
      print( "  Profiles:" )
      if self.intfProfile:
         print( "    Interface Profile: %s" % ( self.intfProfile ) )
      if self.bgpProfile:
         print( "    Bgp Profile: %s" % ( self.bgpProfile ) )
      print( "  Units: %s" % (
            ', '.join( sorted( self.units ) ) ) )

class InterfaceGroups( Model ):
   intfGroups = Dict( valueType=InterfaceGroup,
                      help="Map group name to Interface Group information" )

   def render( self ):
      for key in sorted( self.intfGroups ):
         group = self.intfGroups[ key ]
         print( "Interface Group: %s" % key )
         group.render()

class BgpNeighbors( Model ):
   peers = List( valueType=str, help="List of peers, which could be "
                 "either peer groups, IPv4 or IPv6 peers" )

class BgpGroup( Model ):
   origin = Enum( values=( "userConfigured", "builtin" ),
                  help="origin of this group" )
   neighbors = Dict( valueType=BgpNeighbors,
                     help="Map Peer Type to list of Peers, where peer type, "
                     "the key, could be either peerGroup, peerIpv4 or peerIpv6" )
   bgpProfile = Str( help="Bgp profile associated with group", optional=True )
   vrfName = Str( help="Name of the Vrf" )
   units = List( valueType=str,
                 help="List of Maintenance units of which this group "
                 "is a member of" )

   def render( self ):
      print( "  Origin: %s" % originToStr[ self.origin ] )
      print( "  Neighbors:" )
      if self.neighbors[ 'peerGroup' ].peers:
         print( "    Peer Group: %s" % ( ', '.join(
               sorted( self.neighbors[ 'peerGroup' ].peers ) ) ) )
      if self.neighbors[ 'peerIpv4' ].peers:
         print( "    Ipv4 Peers: %s" % ( ', '.join(
               sorted( self.neighbors[ 'peerIpv4' ].peers ) ) ) )
      if self.neighbors[ 'peerIpv6' ].peers:
         print( "    Ipv6 Peers: %s" % ( ', '.join(
               sorted( self.neighbors[ 'peerIpv6' ].peers ) ) ) )
      if self.bgpProfile:
         print( "  Bgp Profile: %s" % self.bgpProfile )
      if self.vrfName:
         print( "  Vrf: %s" % self.vrfName )
      print( "  Units: %s" % ( ', '.join( sorted( self.units ) ) ) )

class BgpGroups( Model ):
   bgpGroups = Dict( valueType=BgpGroup,
                     help="Map group name to BGP Group information" )

   def render( self ):
      for key in sorted( self.bgpGroups ):
         group = self.bgpGroups[ key ]
         print( "BGP Group: %s" % key )
         group.render()

class MaintenanceGroups( Model ):
   intfGroups = Dict( valueType=InterfaceGroup,
                      help="Map group name to Interface group information" )
   bgpGroups = Dict( valueType=BgpGroup,
                     help="Map group name to BGP Group information" )

   def render( self ):
      for key in sorted( self.intfGroups ):
         intfGroup = self.intfGroups[ key ]
         print( "Interface Group: %s" % key )
         intfGroup.render()

      for key in sorted( self.bgpGroups ):
         bgpGroup = self.bgpGroups[ key ]
         print( "Bgp Group: %s" % key )
         bgpGroup.render()

class MaintenanceEventInfoEntry( Model ):
   eventInfo = Str( help="Information about a Maintenance related event" )
   timeStamp = Float( help="Time at which the event occured" )

class MaintenanceEventInfo( Model ):
   maintEnterEvents = List( valueType=MaintenanceEventInfoEntry,
                            help='Historical information for maintenance '
                            'enter of unit' )
   maintExitEvents = List( valueType=MaintenanceEventInfoEntry,
                           help='Historical information for maintenance '
                           'exit of unit' )

class MaintenanceDebugInfo( Model ):
   maintenanceEventInfo = Dict( valueType=MaintenanceEventInfo,
                                help="Map Unit name to historical "
                                "information" )

   def render( self ):
      def printLogEntries( maintHistory, maintHeading ):
         beginTime = 0
         for entry in maintHistory:
            if beginTime == 0:
               beginTime = entry.timeStamp
               print( maintHeading % ( utcToStr( entry.timeStamp ),
                                      utcToStr( entry.timeStamp, relative=False ) ) )
            print( "%13.6f   %s" % ( entry.timeStamp - beginTime, entry.eventInfo ) )

      for key in sorted( self.maintenanceEventInfo.keys() ):
         maintenanceDebugInfo = self.maintenanceEventInfo[ key ]
         print( key )
         print( "  History:" )
         if maintenanceDebugInfo.maintEnterEvents:
            maintHeading = "    Maintenance Enter Stage Progression started %s @ %s"
            printLogEntries( maintenanceDebugInfo.maintEnterEvents, maintHeading )
         if maintenanceDebugInfo.maintExitEvents:
            maintHeading = "    Maintenance Exit Stage Progression started %s @ %s"
            printLogEntries( maintenanceDebugInfo.maintExitEvents, maintHeading )

intfProfilesHook = CliExtensions.CliHook()
bgpProfilesHook = CliExtensions.CliHook()
intfGroupsHook = CliExtensions.CliHook()
bgpGroupsHook = CliExtensions.CliHook()
showMaintenanceInterfaceHook = CliExtensions.CliHook()
showMaintenanceSummaryHook = CliExtensions.CliHook()
profilesCleanupHook = CliExtensions.CliHook()
bgpMaintenanceDebugInfoHook = CliExtensions.CliHook()
bgpMaintenanceGroupCleanupHook = CliExtensions.CliHook()
intfMaintenanceGroupCleanupHook = CliExtensions.CliHook()
