#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-from-import

from Ark import getPlatform
import CliCommand
import CliMatcher
import CliParser
import CliParserCommon
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.EthIntfModel as EthIntfModel
import CliPlugin.FruCli as FruCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.MacAddr as MacAddr
import CliPlugin.VirtualIntfRule as VirtualIntfRule
from CliPlugin.ManagementActiveIntfModel import (
   MgmtActiveIntfRedundancyStatus,
   MgmtActiveIntfRedundancyMonitorNeighbor,
   MgmtActiveIntfRedundancyMemberIntfStatus
)
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.FruCli import SlotMatcher
from CliPlugin.FruCli import SlotDescription
from CliPlugin.PhysicalIntfRule import PhysicalIntfMatcher
import ConfigMount
import Ethernet
import LazyMount
import Tac
import Tracing
from Arnet import IpGenAddr
from TypeFuture import TacLazyType

RedundancyMonitorNeighbor = TacLazyType(
   'Interface::RedundancyMonitorNeighbor' )
RedundancyMonitorNeighborInterval = TacLazyType(
   'Interface::RedundancyMonitorNeighborInterval' )
RedundancyMonitorNeighborTimeoutMult = TacLazyType(
   'Interface::RedundancyMonitorNeighborTimeoutMult' )
RedundancyFallbackDelay = TacLazyType( 'Interface::RedundancyFallbackDelay' )
RedundancyIntfSet = TacLazyType( 'Interface::RedundancyIntfSet' )
IntfOperStatus = TacLazyType( 'Interface::IntfOperStatus' )

__defaultTraceHandle__ = Tracing.Handle( 'ManagementActiveIntfCli' )
t0 = Tracing.trace0

mgmtActiveIntfConfigDir = None
mgmtActiveIntfStatusDir = None
ethPhyConfigDir = None
allIntfStatusDir = None

mgmtActiveIntfName = "Management0"

def isVeos():
   return getPlatform() == 'veos'

def _checkIfAllowed():
   return EthIntfCli.isModular() or isVeos()

def managementActiveGuard( mode, token ):
   if not _checkIfAllowed():
      return CliParser.guardNotThisPlatform
   return None

# pylint: disable-msg=W0223
# Method 'foo' is abstract in class 'Bar' but is not overridden
class ManagementActiveIntf( EthIntfCli.EthIntf ):

   #----------------------------------------------------------------------------
   # Creates a new EthIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      EthIntfCli.EthIntf.__init__( self, name, mode )
      self.mgmtActiveIntfConfigDir = mgmtActiveIntfConfigDir
      self.mgmtActiveIntfStatusDir = mgmtActiveIntfStatusDir

   helpdesc = 'Management interface on the active supervisor'

   matcher = VirtualIntfRule.VirtualIntfMatcher(
      'Management', 0, 0, helpdesc=helpdesc,
      value=lambda mode, intf: ManagementActiveIntf( intf, mode ),
      guard=managementActiveGuard )

   def config( self ):
      return self.mgmtActiveIntfConfigDir.intfConfig

   def status( self ):
      return self.mgmtActiveIntfStatusDir.intfStatus

   def getRealIntf( self ):
      status = self.status()
      if status and status.realIntfId:
         intf  = EthIntfCli.EthPhyIntf( status.realIntfId, self.mode_ )
         realIntf = intf if intf.status() else None
         return realIntf
      return None

   def getIntfCounterDir( self ):
      assert False, "getIntfCounterDir not implemented for ManagementActive"

   #----------------------------------------------------------------------------
   # Returns a list containing the MgmtActiveIntf if it exists.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      if mgmtActiveIntfStatusDir.intfStatus:
         return [ ManagementActiveIntf( mgmtActiveIntfName, mode ) ]
      return []

   #----------------------------------------------------------------------------
   # Creates the Interface::MgmtActiveIntfConfig object for this interface if it
   # does not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      self.mgmtActiveIntfConfigDir.intfConfig = ( mgmtActiveIntfName, )

   #----------------------------------------------------------------------------
   # Destroys the Interface::MgmtActiveIntfConfig object for this interface if
   # it already exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      self.mgmtActiveIntfConfigDir.intfConfig = None

   def countersSupported( self ):
      return False

   def switchportEligible( self ):
      return False

   def routingSupported( self ):
      return True

   def vrfSupported( self ):
      return True

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def hardware( self ):
      return "ethernet"

   def addrStr( self ):
      return "address is %s (bia %s)" % ( self.addr(), self.burnedInAddr() )

   def burnedInAddr( self ):
      return self.addr()

   def bandwidth( self ):
      realIntf = self.getRealIntf()
      if realIntf:
         return realIntf.bandwidth()
      return 0

   def getIntfStatusModel( self ):
      return EthIntfModel.EthPhyInterfaceStatus( name=self.status().intfId )

   def showLinkTypeSpecific( self, intfStatusModel ):
      realIntf = self.getRealIntf()
      if realIntf:
         ( duplex, autoNegotiate,
           autoNegotiateActive )= realIntf.showLinkTypeSpecific( intfStatusModel )
      else:
         duplex = "duplexUnknown"
         autoNegotiate = "unknown"
         autoNegotiateActive = False
         intfStatusModel.lanes = 0

      intfStatusModel.duplex = duplex
      intfStatusModel.autoNegotiate = autoNegotiate
      # pylint: disable-next=protected-access
      intfStatusModel._autoNegotiateActive = autoNegotiateActive

   def showLoopbackMode( self, intfStatusModel ):
      realIntf = self.getRealIntf()
      if realIntf:
         loopbackMode = realIntf.showLoopbackMode( intfStatusModel )
      else:
         loopbackMode = "loopbackNone"

      intfStatusModel.loopbackMode = loopbackMode

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesStatus()"
   #----------------------------------------------------------------------------
   def autonegActive( self ):
      realIntf = self.getRealIntf()
      if realIntf:
         return realIntf.autonegActive()
      return False

   def getIntfDuplexStr( self ):
      realIntf = self.getRealIntf()
      if realIntf:
         return realIntf.getIntfDuplexStr()
      return "duplexUnknown"

   def xcvrTypeStr( self ):
      realIntf = self.getRealIntf()
      if realIntf:
         return realIntf.xcvrTypeStr()
      return "Unknown"

   #----------------------------------------------------------------------------
   # Sets the MAC address of the interface to a specified value or the default
   # value.
   #----------------------------------------------------------------------------
   def setMacAddr( self, addr, routerMacConfigured=None, physical=False ):
      if addr is None:
         self.config().addr = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
      elif Ethernet.isUnicast( addr ):
         self.config().addr = addr
      else:
         self.mode_.addError( 'Cannot set the interface MAC address to a '
                              'multicast address' )

   #----------------------------------------------------------------------------
   # The "no interface Management 0" command, in "config" mode.
   #----------------------------------------------------------------------------
   def noInterface( self ):
      self.destroy()

   #----------------------------------------------------------------------------
   # This function is called by the "default interface management0" command.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      if isVeos():
         self.setMacAddr( None )
      if EthIntfCli.isModular():
         self.setBackupLinkToStandby( enable=False )
         self.setRedundancyMonitorLinkState( enable=False )
         self.setRedundancyMonitorNeighbor()
         self.setRedundancyFallbackDelay()
         self.clearRedundancyIntf()
      EthIntfCli.EthIntf.setDefault( self )

   #---------------------------------------------------------------------------
   # This function is called by the "backup-link standby [fallback-delay <t>]"
   #---------------------------------------------------------------------------
   def setBackupLinkToStandby( self, enable=True, fallbackDelay=None ):
      config = self.config()
      config.backupLinkStandby = enable

      if fallbackDelay is None:
         config.backupLinkFallbackDelay = (
               config.defaultFallbackDelay )
      elif fallbackDelay == "infinity":
         config.backupLinkFallbackDelay = Tac.endOfTime
      else:
         config.backupLinkFallbackDelay = fallbackDelay

   def isLegacyBackupLinkEnabled( self ):
      return self.config().backupLinkStandby

   def isRedundancyLinkStateMonitorEnabled( self ):
      return self.config().redundancyMonitorLinkState

   def isRedundancyNeighborMonitorEnabled( self ):
      return bool( self.config().redundancyMonitorNeighbor )

   def isRedundancyFallbackDelayEnabled( self ):
      return self.config().redundancyFallbackDelay != Tac.endOfTime

   def isRedundancyInterfaceEnabled( self ):
      return self.config().redundancyIntfSet

   def setRedundancyMonitorLinkState( self, enable=True ):
      config = self.config()
      config.redundancyMonitorLinkState = enable

   def setRedundancyMonitorNeighbor( self, addr=None, interval=None, mult=None ):
      config = self.config()
      neighbor = RedundancyMonitorNeighbor( IpGenAddr( addr ) )
      if interval:
         neighbor.intervalMs = interval
      if mult:
         neighbor.timeoutMult = mult
      config.redundancyMonitorNeighbor = neighbor

   def setRedundancyFallbackDelay( self, delay=None ):
      config = self.config()
      if delay is None or delay == 'infinity':
         config.redundancyFallbackDelay = Tac.endOfTime
      else:
         config.redundancyFallbackDelay = delay

   def addRedundancyIntf( self, slotId, primaryIntf, backupntfs ):
      config = self.config()
      intfSet = RedundancyIntfSet( slotId )
      intfSet.intf[ 0 ] = primaryIntf
      for index, intf in enumerate( backupntfs ):
         intfSet.intf[ index + 1 ] = intf
      config.redundancyIntfSet.addMember( intfSet )

   def removeRedundancyIntf( self, slotId ):
      config = self.config()
      del config.redundancyIntfSet[ slotId ]

   def clearRedundancyIntf( self ):
      config = self.config()
      config.redundancyIntfSet.clear()

IntfCli.Intf.addPhysicalIntfType( ManagementActiveIntf, None,
                                  withIpSupport=True,
                                  withRoutingProtoSupport=False )

def _updateMgmtRangeIntfForModular():
   if _checkIfAllowed():
      EthIntfCli.MgmtAutoIntfType.helpDesc = [
         'Slot number (0 for Active Management)',
         'Port number' ]
      EthIntfCli.MgmtAutoIntfType.registerManagementIntfClass(
         ManagementActiveIntf,
         lambda name, mode: name[ -1 ] == '0',
         [ mgmtActiveIntfName ] )

FruCli.registerModularSystemCallback( _updateMgmtRangeIntfForModular )
FruCli.registerFixedSystemCallback( _updateMgmtRangeIntfForModular )

class ManagementActiveIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, ManagementActiveIntf ) and
               isVeos() )

#--------------------------------------------------------------------------------
# [ no | default ] mac-address MAC_ADDR
#
# Note that "mac-address 0.0.0" will reset the MAC address to the default.
# Remarkably, this is consistent with the industry-standard.
#--------------------------------------------------------------------------------
class MacAddressMacaddrCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-address MAC_ADDR'
   noOrDefaultSyntax = 'mac-address ...'
   data = {
      'mac-address' : 'Set interface MAC address',
      'MAC_ADDR' : MacAddr.macAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.setMacAddr( args[ 'MAC_ADDR' ] )
      t0( 'Set ' + mode.intf.name + ' MAC address to ' + args[ 'MAC_ADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setMacAddr( None )
      t0( 'Set ' + mode.intf.name + ' MAC address to default' )

ManagementActiveIntfModelet.addCommandClass( MacAddressMacaddrCmd )

IntfCli.IntfConfigMode.addModelet( ManagementActiveIntfModelet )

class ManagementActiveIntfHaModlet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, ManagementActiveIntf ) and
             EthIntfCli.isModular())

#--------------------------------------------------------------------------------
# [ no | default ] backup-link standby [ fallback-delay ( infinity | DELAY ) ]
#--------------------------------------------------------------------------------
class BackupLinkStandbyCmd( CliCommand.CliCommandClass ):
   syntax = 'backup-link standby [ fallback-delay ( infinity | DELAY ) ]'
   noOrDefaultSyntax = 'backup-link standby ...'
   data = {
      'backup-link' : 'Backup link for the interface',
      'standby' : 'Backup link using standby',
      'fallback-delay' : 'Fallback delay to the active supervisor',
      'infinity' : "Don't ever fallback",
      'DELAY' : CliMatcher.IntegerMatcher( 0, 3600,
         helpdesc='Fallback delay in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      if ( mode.intf.isRedundancyLinkStateMonitorEnabled() or
           mode.intf.isRedundancyNeighborMonitorEnabled() or
           mode.intf.isRedundancyFallbackDelayEnabled() or
           mode.intf.isRedundancyInterfaceEnabled() ):
         mode.addErrorAndStop( 'The backup-link command cannot be '
                               'combined with redundancy commands' )

      fallbackDelay = args.get( 'DELAY' )
      if 'infinity' in args:
         fallbackDelay = 'infinity'
      mode.intf.setBackupLinkToStandby( enable=True, fallbackDelay=fallbackDelay )
      t0( 'Set', mode.intf.name, 'Backup link to standby with fallback delay:',
            ( '%s' % fallbackDelay ) if fallbackDelay else 'default' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setBackupLinkToStandby( enable=False )

ManagementActiveIntfHaModlet.addCommandClass( BackupLinkStandbyCmd )
IntfCli.IntfConfigMode.addModelet( ManagementActiveIntfHaModlet )

#--------------------------------------------------------------------------------
# [ no | default ] redundancy monitor link-state
#--------------------------------------------------------------------------------
class RedundancyMonitorLinkStateCmd( CliCommand.CliCommandClass ):
   syntax = 'redundancy monitor link-state'
   noOrDefaultSyntax = syntax
   data = {
      'redundancy' : 'Redundancy for the interface',
      'monitor' : 'interface monitor',
      'link-state' : 'Link state of interface'
   }

   @staticmethod
   def handler( mode, args ):
      if mode.intf.isLegacyBackupLinkEnabled():
         mode.addErrorAndStop( 'The backup-link command cannot be '
                               'combined with redundancy commands' )

      if mode.intf.isRedundancyNeighborMonitorEnabled():
         mode.addErrorAndStop( 'Multiple redundancy monitors not supported' )

      t0( 'redundancy monitor link-state' )
      mode.intf.setRedundancyMonitorLinkState( enable=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'no redundancy monitor link-state' )
      mode.intf.setRedundancyMonitorLinkState( enable=False )

ManagementActiveIntfHaModlet.addCommandClass( RedundancyMonitorLinkStateCmd )

#--------------------------------------------------------------------------------
# [ no | default ] redundancy monitor neighbor ipv6 V6ADDR
#                  [ interval INTERVAL milliseconds ]
#                  [ multiplier MULTIPLIER ]
#--------------------------------------------------------------------------------
class RedundancyMonitorNeighborCmd( CliCommand.CliCommandClass ):
   syntax = ( 'redundancy monitor neighbor ipv6 V6ADDR '
              '[ interval INTERVAL milliseconds ] [ multiplier MULTIPLIER ]' )
   noOrDefaultSyntax = 'redundancy monitor neighbor ...'
   data = {
      'redundancy' : 'Redundancy for the interface',
      'monitor' : 'interface monitor',
      'neighbor' : 'Neighbor',
      'ipv6' : 'IPv6 neighbor',
      'V6ADDR' : Ip6AddrMatcher( 'IPv6 address' ),
      'interval' : 'Interval between neighbor probes',
      'INTERVAL' : CliMatcher.IntegerMatcher(
         RedundancyMonitorNeighborInterval.min,
         RedundancyMonitorNeighborInterval.max,
         helpdesc='Interval in units of milliseconds' ),
      'milliseconds' : 'Units in milliseconds',
      'multiplier': 'Timeout multiplier',
      'MULTIPLIER': CliMatcher.IntegerMatcher(
         RedundancyMonitorNeighborTimeoutMult.min,
         RedundancyMonitorNeighborTimeoutMult.max,
         helpdesc='Number of missed neighbor replies after which it is timed out' )
   }

   @staticmethod
   def handler( mode, args ):
      if mode.intf.isLegacyBackupLinkEnabled():
         mode.addErrorAndStop( 'The backup-link command cannot be '
                               'combined with redundancy commands' )

      if mode.intf.isRedundancyLinkStateMonitorEnabled():
         mode.addErrorAndStop( 'Multiple redundancy monitors not supported' )

      if mode.intf.isRedundancyFallbackDelayEnabled():
         mode.addErrorAndStop( "The fallback-delay must be 'infinity' for "
                               "neighbor monitor")

      t0( 'redundancy monitor neighbor', args )
      mode.intf.setRedundancyMonitorNeighbor( args[ 'V6ADDR' ].stringValue,
                                              args.get( 'INTERVAL' ),
                                              args.get( 'MULTIPLIER' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'no redundancy monitor neighbor' )
      mode.intf.setRedundancyMonitorNeighbor( addr=None )

ManagementActiveIntfHaModlet.addCommandClass( RedundancyMonitorNeighborCmd )

#--------------------------------------------------------------------------------
# [ no | default ] redundancy fallback-delay ( DELAY seconds | infinity )
#--------------------------------------------------------------------------------
class RedundancyFallbackDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'redundancy fallback-delay ( infinity | ( DELAY seconds ) )'
   noOrDefaultSyntax = 'redundancy fallback-delay ...'
   data = {
      'redundancy' : 'Redundancy for the interface',
      'fallback-delay' : 'Fallback delay to the higher priority interface',
      'infinity' : "Don't ever fallback",
      'DELAY' : CliMatcher.IntegerMatcher(
         RedundancyFallbackDelay.min,
         RedundancyFallbackDelay.maxFinite,
         helpdesc='Fallback delay' ),
      'seconds' : 'Units in seconds'
   }

   @staticmethod
   def handler( mode, args ):
      if 'infinity' not in args and mode.intf.isLegacyBackupLinkEnabled():
         mode.addErrorAndStop( 'The backup-link command cannot be '
                               'combined with redundancy commands' )

      if 'infinity' not in args and mode.intf.isRedundancyNeighborMonitorEnabled():
         mode.addErrorAndStop( "The fallback-delay must be 'infinity' for "
                               "neighbor monitor")

      delay = args.get( 'DELAY', 'infinity' )
      t0( 'redundancy fallback-delay:', delay )
      mode.intf.setRedundancyFallbackDelay( delay )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'no redundancy fallback-delay' )
      mode.intf.setRedundancyFallbackDelay()

ManagementActiveIntfHaModlet.addCommandClass( RedundancyFallbackDelayCmd )

#--------------------------------------------------------------------------------
# [ no | default ] redundancy supervisor SLOT interface primary INTF
#                  backup BACKUP_INTFS
#--------------------------------------------------------------------------------
class SupeSlotMatcher( SlotMatcher ):
   def __init__( self, **kargs ):
      SlotMatcher.__init__( self, forceTags=[ 'Supervisor' ], **kargs )

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      result = SlotMatcher.match( self, mode, context, token )
      # If the hardware hasnt been discovered during startup we
      # get a noMatch. In this case, match the well-formed token
      if root is None and result == CliParserCommon.noMatch:
         t0( 'No match during startup config' )
         m = self.labelRegEx.match( token )
         if not mode.session.guardsEnabled() and m:
            t0( 'match well formed slot numbers during startup config' )
            label = m.group( 1 )
            result = CliParserCommon.MatchResult(
               SlotDescription( None, 'Supervisor', label, None ), token )
      return result

class RedundancyInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = ( 'redundancy supervisor SLOT interface '
              'primary INTF backup { BACKUP_INTFS }' )
   noOrDefaultSyntax = 'redundancy supervisor SLOT ...'

   data = {
      'redundancy' : 'Redundancy for the interface',
      'supervisor' : 'Supervisor',
      'SLOT' : SupeSlotMatcher(),
      'interface' : 'Redundant interfaces',
      'primary' : 'Primary interface',
      'INTF' : PhysicalIntfMatcher( 'Management' ),
      'backup' : 'Backup interfaces',
      'BACKUP_INTFS' : PhysicalIntfMatcher( 'Management' )
   }

   @staticmethod
   def _backupIntfInSameSlot( intf ):
      if '/2' in intf:
         return intf.replace( '/2', '/1' )
      else:
         return intf.replace( '/1', '/2' )

   @staticmethod
   def _backupIntfInPeerSlot( slotId ):
      return 'Management2/1' if slotId == 1 else 'Management1/1'

   @staticmethod
   def _validatePrimaryIntf( mode, slotId, primaryIntf ):
      validPrimaryIntfs = [ 'Management%d/1' % slotId, 'Management%d/2' % slotId ]
      if primaryIntf not in validPrimaryIntfs:
         mode.addErrorAndStop(
            'Invalid primary interface %s. Valid interfaces are %s' % (
            primaryIntf, ', '.join( validPrimaryIntfs ) ) )

   @staticmethod
   def _validateBackupIntfs( mode, slotId, primaryIntf, backupIntfs ):
      backupInSameSlot = RedundancyInterfaceCmd._backupIntfInSameSlot( primaryIntf )
      backupInPeerSlot = RedundancyInterfaceCmd._backupIntfInPeerSlot( slotId )
      validBackupIntfs = [ backupInSameSlot, backupInPeerSlot ]

      if not set( backupIntfs ).issubset( set( validBackupIntfs ) ):
         invalidBackupIntfs = sorted( set( backupIntfs ) - set( validBackupIntfs ) )
         mode.addErrorAndStop(
            'Invalid backup interface %s. Valid interfaces are %s' % (
            ', '.join( invalidBackupIntfs ), ', '.join( validBackupIntfs ) ) )

      if ( backupInSameSlot in backupIntfs and
           backupIntfs.index( backupInSameSlot ) != 0 ):
         mode.addErrorAndStop( "The %s interface must be first backup" % (
                               backupInSameSlot ) )

   @staticmethod
   def _dedup( intfs ):
      ret = []
      for intf in intfs:
         if intf not in ret:
            ret.append( intf )
      return ret

   @staticmethod
   def handler( mode, args ):
      if mode.intf.isLegacyBackupLinkEnabled():
         mode.addErrorAndStop( 'The backup-link command cannot be '
                               'combined with redundancy commands' )

      slotId = int( args[ 'SLOT' ].label )
      primaryIntf = args[ 'INTF' ]
      backupIntfs = RedundancyInterfaceCmd._dedup( args[ 'BACKUP_INTFS' ] )

      t0( 'redundancy interface supervisor',
          'SLOT:', slotId,
          'primary INTF:', primaryIntf,
          'backup INTFS:', ', '.join( backupIntfs ) )

      RedundancyInterfaceCmd._validatePrimaryIntf(
         mode, slotId, primaryIntf )
      RedundancyInterfaceCmd._validateBackupIntfs(
         mode, slotId, primaryIntf, backupIntfs )

      mode.intf.addRedundancyIntf( slotId, primaryIntf, backupIntfs )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      slotId = int( args[ 'SLOT' ].label )
      t0( 'no redundancy interface supervisor SLOT:', slotId )
      mode.intf.removeRedundancyIntf( slotId )

ManagementActiveIntfHaModlet.addCommandClass( RedundancyInterfaceCmd )

#--------------------------------------------------
# The "show redundancy backup-link" command.
#--------------------------------------------------
def getOperStatus( intf ):
   allIntfStatus = allIntfStatusDir.intfStatus.get( intf )
   if allIntfStatus:
      operStatus = allIntfStatus.operStatus
   else:
      operStatus = IntfOperStatus.intfOperUnknown
   return operStatus

def doShowRedBackupLink( mode, args ):
   intfRedStatusModel = MgmtActiveIntfRedundancyStatus()
   intfConfig = mgmtActiveIntfConfigDir.intfConfig
   intfStatus = mgmtActiveIntfStatusDir.intfStatus
   linkState = intfStatus and intfStatus.redundancyMonitorLinkState
   neighbor = intfStatus.redundancyMonitorNeighbor if intfStatus else None
   if intfConfig:
      intfRedStatusModel.name = intfConfig.intfId
      intfRedStatusModel.redundancy = linkState or bool( neighbor )
      if intfRedStatusModel.redundancy and intfStatus:
         intfRedStatusModel.redundancyLastConfigChangeTime = \
               intfStatus.redundancyLastConfigChangeTime
         intfRedStatusModel.redundancyIntf = intfStatus.realIntfId
         intfRedStatusModel.lastRedundancyLinkChangeTime = \
               intfStatus.lastRedundancyLinkChangeTime
         intfRedStatusModel.redundancyMonitorLinkState = linkState
         if neighbor:
            neighborModel = MgmtActiveIntfRedundancyMonitorNeighbor()
            neighborModel.addr = neighbor.addr
            neighborModel.interval = neighbor.intervalMs
            neighborModel.timeoutMultiplier = neighbor.timeoutMult
            intfRedStatusModel.redundancyMonitorNeighbor = neighborModel
         if intfStatus.redundancyFallbackDelay != Tac.endOfTime:
            intfRedStatusModel.redundancyFallbackDelay = \
                  int( intfStatus.redundancyFallbackDelay )
         for intf in intfStatus.redundancyIntf.values():
            if intf:
               member = MgmtActiveIntfRedundancyMemberIntfStatus()
               member.intf = intf
               operStatus = getOperStatus( intf )
               # Strip off 'intfOper' and return the rest of the enum name
               member.operStatus = ( operStatus[ 8 ].lower() + operStatus[ 9: ] )
               intfRedStatusModel.redundancyMemberIntfs.append( member )
   return intfRedStatusModel

def Plugin( entityManager ):
   global mgmtActiveIntfConfigDir
   global mgmtActiveIntfStatusDir
   global ethPhyConfigDir
   global allIntfStatusDir

   mgmtActiveIntfConfigDir = ConfigMount.mount(
      entityManager,
      "interface/config/eth/managementactive",
      "Interface::MgmtActiveIntfConfigDir", "w" )
   mgmtActiveIntfStatusDir = LazyMount.mount(
      entityManager,
      "interface/status/eth/managementactive",
      "Interface::MgmtActiveIntfStatusDir", "r" )
   ConfigMount.mount(
      entityManager,
      "interface/config/eth/phy/slice",
      "Tac::Dir", "wi" )
   ethPhyConfigDir = LazyMount.mount(
      entityManager,
      "interface/config/eth/phy/all",
      "Interface::AllEthPhyIntfConfigDir", "r" )
   allIntfStatusDir = LazyMount.mount(
      entityManager,
      "interface/status/all",
      "Interface::AllIntfStatusDir", "r" )

