#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
from CliPlugin.IntfModel import intfOperStatuses
from CliModel import Model
from CliModel import Bool
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Enum
from CliModel import Submodel
from ArnetModel import IpGenericAddress
from IntfModels import Interface

#-------------------------------------------------------------------------------
# EthIntf specific Eapi Models
#-------------------------------------------------------------------------------
class MgmtActiveIntfRedundancyMonitorNeighbor( Model ):
   addr = IpGenericAddress( help="Neighbor IP address" )
   interval = Int( help="Neighbor monitor interval in milliseconds" )
   timeoutMultiplier = Int( help="Neighbor reply timeout multiplier" )

class MgmtActiveIntfRedundancyMemberIntfStatus( Model ):
   intf = Interface( help="Name of the interface" )
   operStatus = Enum( help="Operational status", values=intfOperStatuses )   

class MgmtActiveIntfRedundancyStatus( Model ):
   name = Interface( help="Name of the interface", optional=True )
   redundancy = Bool( help='Is redundancy enabled', optional=True )
   redundancyIntf = Interface( help="Name of the interface", optional=True )
   lastRedundancyLinkChangeTime = Float( help="Time when last link changed",
                                  optional=True )
   redundancyLastConfigChangeTime = Float( 
      help="Time when last config changed", optional=True )
   redundancyMonitorLinkState = Bool( 
      help='Redundancy monitor link-state is enabled', optional=True )
   redundancyMonitorNeighbor = Submodel(
      valueType=MgmtActiveIntfRedundancyMonitorNeighbor,
      help='Neighbor monitor parameters',
      optional=True )
   redundancyFallbackDelay = Int( 
      help='Fallback delay to higher priority interface in seconds',
      optional=True )
   redundancyMemberIntfs = List( 
      valueType=MgmtActiveIntfRedundancyMemberIntfStatus,
      help='Member interface status used in redundancy' )
    
   def timeString( self, elapsed, output, units ):
      assert isinstance( elapsed, int )
      if units:
         _ , unit = units[ -1 ]
         assert ( unit == 1 ), "Last unit must have 1 as a unit value"
      if ( elapsed > 0 ) and units:
         localName, localUnits = units[0] 
         value = elapsed % localUnits if len( units ) > 1 else elapsed
         if( value > 0 ): # pylint: disable=superfluous-parens
            output = ( "%d %s"%( value, localName + ( "s" if value > 1 else "" ) )
                        + ( ( ", " + output ) if output else "" ) )
         return( self.timeString( elapsed//localUnits, 
                 output, units[1:] ) )
      else: 
         return( output )  # pylint: disable=superfluous-parens

   def renderRedundancy( self ):
      if self.name and EthIntfCli.isModular():
         print( "%s" % self.name.stringValue )
         if self.redundancy:
            print( "  Redundancy: Enabled" )
            if self.redundancyIntf:
               print( "  Redundancy active interface: %s" % (
                  self.redundancyIntf.stringValue ) )
            else:
               print( "  Redundancy active interface: n/a" )
            if self.lastRedundancyLinkChangeTime:
               timeDelta = int( Tac.now() - self.lastRedundancyLinkChangeTime )
               uptimeString = self.timeString( timeDelta, "",
                                             [ ( "second", 60 ), ( "minute", 60 ),
                                               ( "hour", 24 ), ( "day", 365 ),
                                               ( "year", 1 ) ] )
               uptimeString = uptimeString + " ago"
            else:
               uptimeString = "never"
            print( "  Last redundancy link change: %s" % uptimeString )
            if self.redundancyLastConfigChangeTime:
               timeDelta = int( Tac.now() - self.redundancyLastConfigChangeTime )
               uptimeString = self.timeString( timeDelta, "",
                                             [ ( "second", 60 ), ( "minute", 60 ),
                                               ( "hour", 24 ), ( "day", 365 ),
                                               ( "year", 1 ) ] )
               uptimeString = uptimeString + " ago"
               print( "  Last redundancy config change: %s" % uptimeString )
            if self.redundancyMonitorLinkState:
               print( "  Redundancy monitor: link-state" )
            if self.redundancyMonitorNeighbor:
               print( "  Redundancy monitor: neighbor %s,"
                      " interval %d milliseconds, multiplier %d" % (
                      self.redundancyMonitorNeighbor.addr,
                      self.redundancyMonitorNeighbor.interval,
                      self.redundancyMonitorNeighbor.timeoutMultiplier ) )
            if self.redundancyFallbackDelay is not None:
               print( "  Redundancy fallback-delay: %d" % (
                       self.redundancyFallbackDelay ) )
            else:
               print( "  Redundancy fallback-delay: infinity" )
            intfLen = len( self.redundancyMemberIntfs )
            if intfLen:
               primary = self.redundancyMemberIntfs[ 0 ]
               primaryStr = 'primary %s (%s)' % ( 
                  primary.intf.stringValue, primary.operStatus.lower() )
               backupStr = ""
               for backup in self.redundancyMemberIntfs[ 1: ]:
                  backupStr += ", backup %s (%s)" % ( 
                     backup.intf.stringValue, backup.operStatus.lower() )
               print( "  Redundancy interfaces: %s%s" % (
                      primaryStr, backupStr ) )
         else:
            print( "  Redundancy: Disabled" )
            print( "  Redundancy active interface: n/a" )
            print( "  Last redundancy link change: n/a" )

   def render( self ):
      self.renderRedundancy()


