#/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import LazyMount
import CliToken.Ip
import Arnet
from Arnet import IpGenPrefix
from CliPlugin import IntfCli
from CliPlugin import IraIp6RouteCliLib
from CliPlugin import IraIpRouteCliLib
from CliPlugin.Ip6AddrMatcher import ip6PrefixExpr
from CliPlugin.IpAddrMatcher import ipPrefixExpr
from CliPlugin.McastBoundaryCliModel import McastBoundary
from CliPlugin.McastCommonCli import mcastRoutingSupportedGuard, \
      mcastv6RoutingSupportedGuard
from McastCommonCliLib import AddressFamily, validateMulticastAddress
from ShowCommand import ShowCliCommandClass
from CliMatcher import KeywordMatcher
from CliCommand import Node

mcastBoundaryStatus = None
mcastBoundary6Status = None
mfibStatus = None
mfib6Status = None

multicastKwMatcher = KeywordMatcher( 'multicast', helpdesc='Multicast information' )
multicastDeprecatedNode = Node( multicastKwMatcher,
                                guard=mcastRoutingSupportedGuard )

boundaryKwMatcher = KeywordMatcher( 'boundary',
                                    helpdesc='Multicast boundary information' )

groupPrefixMatcher = ipPrefixExpr( 'Group range prefix',
   'Group range prefix mask',
   'Multicast boundary matching group address with prefix' )

groupV6PrefixMatcher = ip6PrefixExpr( 'Group range prefix',
   'Group range prefix mask',
   'Multicast boundary matching group address with prefix',
   fullMaskValid=False )

interfaceKwMatcher = KeywordMatcher( 'interface',
   helpdesc='Show multicast boundary for a specific interface' )
interfaceNode = Node( interfaceKwMatcher, guard=mcastRoutingSupportedGuard )
interfaceV6Node = Node( interfaceKwMatcher, guard=mcastv6RoutingSupportedGuard )

outKwMatcher = KeywordMatcher( 'out',
   helpdesc='Show boundaries with only control plane filtering' )

def mcastBoundaryInterfaceFromStatus( isQualified, intfId, out, af ):
   mcastBoundaryInterface = McastBoundary.McastBoundaryInterface()
   if af == AddressFamily.ipv4:
      swBoundaryStatus = mcastBoundaryStatus.softwareBoundaryStatus
      ifBoundaryStatus = mfibStatus.intfBoundaryStatus
   else:
      swBoundaryStatus = mcastBoundary6Status.softwareBoundaryStatus
      ifBoundaryStatus = mfib6Status.intfBoundaryStatus
   if intfId in swBoundaryStatus:
      prefixes = swBoundaryStatus[ intfId ].boundary
      prefixes = [ prefix for prefix in prefixes if isQualified( prefix ) ]
      for prefix in [ IpGenPrefix( key.stringValue ) for key in prefixes ]:
         mcastBoundaryInterface.boundary[ prefix ] = False
   if intfId in ifBoundaryStatus:
      prefixes = ifBoundaryStatus[ intfId ].boundary
      prefixes = [ prefix for prefix in prefixes if isQualified( prefix ) ]
      if out:
         for prefix in [ IpGenPrefix( key.stringValue ) for key in prefixes ]:
            del mcastBoundaryInterface.boundary[ prefix.stringValue ]
      else:
         for prefix in [ IpGenPrefix( key.stringValue ) for key in prefixes ]:
            mcastBoundaryInterface.boundary[ prefix ] = True
   return mcastBoundaryInterface

def validateGroupPrefix( mode, groupPrefix, af ):
   if af == AddressFamily.ipv4:
      if not IraIpRouteCliLib.isValidPrefix( groupPrefix ):
         mode.addError( "Masked bits of source range must be zero" )
         return False
   else:
      if not IraIp6RouteCliLib.isValidIpv6PrefixWithError( mode, groupPrefix ):
         mode.addError( "Masked bits of source range must be zero" )
         return False
   address = Arnet.IpGenAddrWithMask( str(groupPrefix) ).ipGenAddr
   err = validateMulticastAddress( address )
   if err != None: # pylint: disable=singleton-comparison
      mode.addError( err )
      return False
   return True

# pylint: disable-next=inconsistent-return-statements
def cmdShowIpMulticastBoundary( mode, args ):
   groupPrefix = args.get( 'GROUP' )
   intf = args.get( 'INTF' )
   if args.get( 'ipv6' ):
      af = AddressFamily.ipv6
      mBoundaryStatus = mcastBoundary6Status
      mcastFibStatus = mfib6Status
   else:
      af = AddressFamily.ipv4
      mBoundaryStatus = mcastBoundaryStatus
      mcastFibStatus = mfibStatus
   out = 'out' in args
   if groupPrefix:
      if af == AddressFamily.ipv6:
         groupPrefix = Arnet.Ip6AddrWithMask( groupPrefix.address,
                                              mask=groupPrefix.prefixLen() )
      if not validateGroupPrefix( mode, groupPrefix, af ):
         return
      grpPrefix = IpGenPrefix( str( groupPrefix ) )
      isQualified = lambda prefix: prefix.contains( grpPrefix.ipGenAddr ) and\
            prefix.len <= grpPrefix.len
   else:
      isQualified = lambda prefix: True
   mcastBoundary = McastBoundary()
   if intf:
      intfId = intf.name
      mcastBoundaryInterface = mcastBoundaryInterfaceFromStatus( isQualified,
            intfId, out, af )
      mcastBoundary.mcastBoundaryInterfaces[ intfId ] = mcastBoundaryInterface
   else:
      interfaces = \
          set( mBoundaryStatus.softwareBoundaryStatus.keys() )
      interfaces = interfaces.union( mcastFibStatus.intfBoundaryStatus )
      for intfId in interfaces:
         mcastBoundaryInterface = mcastBoundaryInterfaceFromStatus( isQualified,
               intfId, out, af )
         mcastBoundary.mcastBoundaryInterfaces[ intfId ] = mcastBoundaryInterface
   return mcastBoundary

class DeprecatedShowMulticastBoundaryInterface( ShowCliCommandClass ):
   syntax = 'show ip multicast boundary [ GROUP | ( interface INTF ) ] [ out ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'multicast': multicastDeprecatedNode,
      'boundary': boundaryKwMatcher,
      'GROUP': groupPrefixMatcher,
      'interface': interfaceNode,
      'INTF': IntfCli.Intf.matcher,
      'out': outKwMatcher
   }
   handler = cmdShowIpMulticastBoundary
   cliModel = McastBoundary

BasicCli.addShowCommandClass( DeprecatedShowMulticastBoundaryInterface )

class ShowMulticastBoundary4Interface( ShowCliCommandClass ):
   syntax = 'show multicast ipv4 boundary [ GROUP | ( interface INTF ) ] [ out ]'
   data = {
      'multicast': multicastKwMatcher,
      'ipv4': CliToken.Ip.ipv4MatcherForShow,
      'boundary': boundaryKwMatcher,
      'GROUP': groupPrefixMatcher,
      'interface': interfaceNode,
      'INTF': IntfCli.Intf.matcher,
      'out': outKwMatcher
   }
   handler = cmdShowIpMulticastBoundary
   cliModel = McastBoundary

BasicCli.addShowCommandClass( ShowMulticastBoundary4Interface )

class ShowMulticastBoundary6Interface( ShowCliCommandClass ):
   syntax = 'show multicast ipv6 boundary [ GROUP | ( interface INTF ) ] [ out ]'
   data = {
      'multicast': multicastKwMatcher,
      'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
      'boundary': boundaryKwMatcher,
      'GROUP': groupV6PrefixMatcher,
      'interface': interfaceV6Node,
      'INTF': IntfCli.Intf.matcher,
      'out': outKwMatcher
   }
   handler = cmdShowIpMulticastBoundary
   cliModel = McastBoundary

BasicCli.addShowCommandClass( ShowMulticastBoundary6Interface )

def Plugin( entityManager ):
   global mcastBoundaryStatus, mfibStatus, mcastBoundary6Status, mfib6Status

   mcastBoundaryStatus = LazyMount.mount( entityManager,
         'routing/mcastboundary/status', 'Routing::McastBoundary::Status', 'r' )
   mcastBoundary6Status = LazyMount.mount( entityManager,
         'routing6/mcastboundary/status', 'Routing::McastBoundary::Status', 'r' )
   mfibStatus = LazyMount.mount( entityManager,
         'routing/multicast/status', 'Routing::Multicast::Fib::Status', 'r' )
   mfib6Status = LazyMount.mount( entityManager,
         'routing6/multicast/status', 'Routing::Multicast::Fib::Status', 'r' )
