# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.McastDnsIntfCli as McastDnsIntfCli

LinkName = Tac.Type( "McastDns::LinkName" )
LocationTag = Tac.Type( "McastDns::LocationTag" )

#--------------------------------------------------------------------------------
# [ no | default ] mdns ipv4 [ link LINK ] [ default-tag TAG ]
#--------------------------------------------------------------------------------
class InterfaceMcastDnsCmd( CliCommand.CliCommandClass ):
   syntax = '''mdns ipv4
               [ ( link LINK [ default-tag TAG ] )
               | ( default-tag TAG [ link LINK ] ) ]'''
   noOrDefaultSyntax = 'mdns ipv4 ...'
   data = {
      'mdns': 'Enable McastDns on this interface',
      'ipv4': 'Use ipv4',
      'link': 'Specify link ID to be advertised to remote gateways',
      'LINK': CliMatcher.PatternMatcher( r'.+', helpname='WORD',
                  helpdesc='Link ID of this interface' ),
      'default-tag': 'Specify a default tag for DNS records received',
      'TAG': CliMatcher.PatternMatcher( r'.+', helpname='WORD',
                  helpdesc='Default tag name' ),
   }

   @staticmethod
   def handler( mode, args ):
      link = args.get( 'LINK' )
      if not link:
         link = ''
      if len( link ) > LinkName.maxLength:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "'%s' too long: must be no more than %d characters"
                           % ( link, LinkName.maxLength ) )
         return

      tag = args.get( 'TAG' )
      if tag:
         if len( tag ) > LocationTag.maxLength:
            # pylint: disable-next=consider-using-f-string
            mode.addError( "The default-tag must not be more than %d characters"
                  % ( LocationTag.maxLength ) )
            return
         if any( not c.isalpha() and not c.isdigit() and not c == '-' for c in tag ):
            mode.addError( "The default-tag must only contain letters, digits " +
                  "or hyphens" )
            return
      McastDnsIntfCli.enableMcastDns( mode, link, args.get( 'TAG' ), no=False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      McastDnsIntfCli.enableMcastDns( mode, link=args.get( 'LINK' ), no=True )

McastDnsIntfCli.McastDnsModelet.addCommandClass( InterfaceMcastDnsCmd )
