#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from ArnetModel import IpGenericAddress
from CliModel import Model, Dict, Str, Enum
import TableOutput

class SourceMembership( Model ):
   ''' includeIntf/excludeIntf = "Null0" for membership learned via Pimsm '''
   mode = Enum( help='Membership mode', optional=True,
                values=( "include", "exclude", "none" ) )

class GroupMembership( Model ):
   sources = Dict( keyType=IpGenericAddress,
                   valueType=SourceMembership,
                   optional=True,
                   help='Map Source to membership information' )

class VlanMembership( Model ):
   vrfName = Str( help='VRF name', optional=True )
   groups = Dict( keyType=IpGenericAddress, valueType=GroupMembership,
                  optional=True,
                  help='Map group to source membership information' )

class MembershipJoinStatus( Model ):
   vlans = Dict( keyType=int, valueType=VlanMembership,
                  help='Map VLAN ID to per-VLAN group membership information' )

   def initFromVlanMembershipTacModel( self, vlanMembership, vrfName ):
      vlanModel = self.vlans.get( vlanMembership.vlanId )
      if vlanModel is None:
         vlanModel = VlanMembership()
      vlanModel.vrfName = vrfName
      self.vlans[ vlanMembership.vlanId ] = vlanModel

      for groupAddr, groupMembership in vlanMembership.group.items():
         groupModel = vlanModel.groups.get( groupAddr )
         if groupModel is None:
            groupModel = GroupMembership()
         vlanModel.groups[ groupAddr ] = groupModel

         for sourceAddr, sourceMembership in groupMembership.source.items():
            sourceModel = groupModel.sources.get( sourceAddr )
            if sourceModel is None:
               sourceModel = SourceMembership()
            groupModel.sources[ sourceAddr ] = sourceModel

            if sourceMembership.includeIntf:
               sourceModel.mode = "include"
            elif sourceMembership.excludeIntf:
               sourceModel.mode = "exclude"
            else:
               sourceModel.mode = "none"

#Membership collection learned via Pimsm
class MembershipJoinStatusPimsm( MembershipJoinStatus ):
   def render( self ):
      print( 'EVPN Group Membership Learned Via PIM-SM' )
      print( 'EX: Filter mode Exclude' )
      print( 'IN: Filter mode Include\n' )

      headers = ( "Group", "Source", "Mode" )
      formatMode = TableOutput.Format( justify="left", minWidth=3 )
      formatLeft = TableOutput.Format( justify="left", minWidth=15 )
      formatMode.noPadLeftIs( True )
      formatLeft.noPadLeftIs( True )

      for vlanId, vlanMember in sorted( self.vlans.items() ):
         # pylint: disable-next=consider-using-f-string
         print( 'VRF: %s, SBD VLAN: %d' % ( vlanMember.vrfName, vlanId ) )
         table = TableOutput.createTable( headers )
         headers = ""
         table.formatColumns( formatLeft, formatLeft, formatMode )
         for groupAddr, groupMember in sorted( vlanMember.groups.items() ):
            for sourceAddr, sourceMembership in sorted(
                  groupMember.sources.items() ):
               if groupAddr in ( '0.0.0.0', '::' ):
                  table.newRow( '*', '*', '-' )
               elif sourceAddr in ( '0.0.0.0', '::' ):
                  table.newRow( groupAddr, '*', '-' )
               elif sourceMembership.mode == "include":
                  table.newRow( groupAddr, sourceAddr, "IN" )
               elif sourceMembership.mode == "exclude":
                  table.newRow( groupAddr, sourceAddr, "EX" )
               else:
                  table.newRow( groupAddr, sourceAddr, "-" )
         print( table.output() )

   def initFromTacModel( self, membership, enabledStatus, vrfName ):
      if vrfName is None:
         #no vrfName is specified, display membership for all vrfs
         for vlan, vlanMembership in membership.vlan.items():
            if vlan not in enabledStatus.sbdVlanToVrf:
               continue
            vrf = enabledStatus.sbdVlanToVrf.get( vlan )
            self.initFromVlanMembershipTacModel( vlanMembership, vrf )
      elif vrfName in enabledStatus.vrfToSBDVlan:
         #valid vrfName is specified
         vlanId = enabledStatus.vrfToSBDVlan.get( vrfName )
         if vlanId:
            vlanMembership = membership.vlan.get( vlanId )
            if vlanMembership:
               self.initFromVlanMembershipTacModel( vlanMembership, vrfName )

#Membership collection learned via Evpn
class MembershipJoinStatusEvpn( MembershipJoinStatus ):
   def render( self ):
      print( 'Group Membership Learned Via EVPN' )
      print( 'EX: Filter mode Exclude' )
      print( 'IN: Filter mode Include\n' )

      headers = ( "VLAN", "Group", "Source", "Mode" )
      formatVlan = TableOutput.Format( justify="left", minWidth=4 )
      formatMode = TableOutput.Format( justify="left", minWidth=3 )
      formatAddr = TableOutput.Format( justify="left", minWidth=15 )
      formatVlan.noPadLeftIs( True )
      formatMode.noPadLeftIs( True )
      formatAddr.noPadLeftIs( True )
      table = TableOutput.createTable( headers )
      table.formatColumns( formatVlan, formatAddr, formatAddr, formatMode )

      for vlanId, vlanMember in sorted( self.vlans.items() ):
         for groupAddr, groupMember in sorted( vlanMember.groups.items() ):
            for sourceAddr, sourceMembership in sorted(
                  groupMember.sources.items() ):
               if groupAddr in ( '0.0.0.0', '::' ):
                  table.newRow( vlanId, '*', '*', '-' )
               elif sourceAddr in ( '0.0.0.0', '::' ):
                  table.newRow( vlanId, groupAddr, '*', '-' )
               elif sourceMembership.mode == "include":
                  table.newRow( vlanId, groupAddr, sourceAddr, "IN" )
               elif sourceMembership.mode == "exclude":
                  table.newRow( vlanId, groupAddr, sourceAddr, "EX" )
               else:
                  table.newRow( vlanId, groupAddr, sourceAddr, "-" )
      print( table.output() )

   def initFromTacModel( self, membership, l3IntfStatusDir, vrfName, vlanId ):
      VlanIntfId = Tac.Type( 'Arnet::VlanIntfId' )
      if vlanId:
         vlanMembership = membership.vlan.get( vlanId )
         if vlanMembership:
            # verify vlan in the right vrf
            if vrfName:
               intfId = VlanIntfId.vlanIdToIntfId( vlanId )
               intfStatus = l3IntfStatusDir.intfStatus.get( intfId )
               if intfStatus is None or intfStatus.vrf != vrfName:
                  return
            self.initFromVlanMembershipTacModel( vlanMembership, None )
         return
      if vrfName is None:
         #no vrfName or vlan is specified, display membership for all vrfs
         for vlan, vlanMembership in membership.vlan.items():
            self.initFromVlanMembershipTacModel( vlanMembership, None )
      else:
         #valid vrfName is specified
         for vlan, vlanMembership in membership.vlan.items():
            intfId = VlanIntfId.vlanIdToIntfId( vlan )
            intfStatus = l3IntfStatusDir.intfStatus.get( intfId )
            if intfStatus and intfStatus.vrf == vrfName:
               self.initFromVlanMembershipTacModel( vlanMembership, vrfName )
