# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple
import Tac
from TypeFuture import TacLazyType

# Define used bit ranges for all 4 U32's of the drop reason.
dropReasonPartBitRange = (
   ( 1, 32 ), # part0: first bit is "no drop";
   ( 0, 32 ), # part1, all are used;
   ( 0, 3 ),  # part2: first 3 are used;
   ( 0, 0 ),  # part3: not used at all;
)

PacketDropReason = Tac.Type( "DropExport::PacketDropReason" )
IpProtoType = TacLazyType( 'Arnet::IpProtocolNumber' )
ReportKey = TacLazyType( 'DropExport::Report::Key' )
DropReasonType = namedtuple( "DropReasonType", "dropReasonCliKey dropReasonCliStr" )

# Define tuple of drop reason strings with string index corresponding to a bit in
# drop reason vector of 4 U32's.
dropReasonStr = (
   "no drop", # xxx_lapk: packet was not dropped when we tried to get drop reason.
   "internal reason",
   "MPLS label lookup miss",
   "MPLS invalid action",
   "MPLS invalid payload",
   "MPLS TTL check fail",
   "MPLS invalid control word",
   "L2GRE SIP lookup miss",
   "L2GRE VPNID lookup miss",
   "L2GRE tunnel error",
   "VXLAN SIP lookup miss",
   "VXLAN VN_ID lookup miss",
   "VXLAN tunnel error",
   "VLAN not valid",
   "ingress port not in VLAN member",
   "TPID mismatch",
   "IPV4 protocol error",
   "higig loopback",
   "higig mirror only",
   "higig unknown header",
   "higig unknown opcode",
   "LAG fail loopback",
   "src MAC = dest MAC",
   "IPV6 protocol error",
   "NIV VNTAG present",
   "NIV VNTAG not present",
   "NIV VNTAG format",
   "trill error frame",
   "BPDU",
   "bad UDP checksum",
   "tunnel decap ECN error",
   "IPv4 header error",
   "IPv6 header error",
   "parity error",
   "unicast RPF check fail",
   "field lookup stage",
   "field ingress stage",
   "tunnel object validation fail",
   "tunnel shim header error",
   "tunnel TTL check fail",
   "tunnel interface check failure",
   "tunnel error",
   "tunnel adaptation table 1 lookup miss",
   "tunnel adaptation table 2 lookup miss",
   "tunnel adaptation table 3 lookup miss",
   "tunnel adaptation table 4 lookup miss",
   "control frame",
   "higig header error",
   "NIV dot1p",
   "MPLS gal label",
   "STP not forwarding",
   "multicast index error",
   "protocol pkt",
   "unknown src MAC",
   "source route",
   "L2 source discard",
   "L2 source static move",
   "L2 dest discard",
   "source MAC zero",
   "L2 dest miss or PBT",
   "class based station move",
   "time sync pkt",
   "my station",
   "L3 dest discard",
   "multicast lookup miss",
   "port tag present error",
   "PVLAN VP egress filter",
   "port bitmap zero",
)

DropReasonCliMap = {
   PacketDropReason.unknownReason:
   DropReasonType( "unknownReason", "Unknown reason" ),
   PacketDropReason.ipForwardingLookupMiss:
   DropReasonType( "ipForwardingLookupMiss", "IP forwarding lookup miss" ),
   PacketDropReason.ipVersionError:
   DropReasonType( "ipVersionError", "IP version error" ),
   PacketDropReason.ipTtl01:
   DropReasonType( "ipTtl01", "IP TTL 0/1" ),
   PacketDropReason.ipMissingArp:
   DropReasonType( "ipMissingArp", "ARP unresolved" ),
   PacketDropReason.ipMissingRoute:
   DropReasonType( "ipMissingRoute", "IP missing route" ),
   PacketDropReason.ipv4ChecksumError:
   DropReasonType( "ipv4ChecksumError", "IPv4 checksum error" ),
   PacketDropReason.ipv6UnspecifiedDestination:
   DropReasonType( "ipv6UnspecifiedDestination", "IPv6 unspecified destination" ),
   PacketDropReason.ipv6MulticastSource:
   DropReasonType( "ipv6MulticastSource", "IPv6 multicast source" ),
   PacketDropReason.portNotVlanMember:
   DropReasonType( "portNotVlanMember", "Port not VLAN member" ),
   PacketDropReason.saEqualsDa:
   DropReasonType( "saEqualsDa", "Source address equals destination address" ),
   PacketDropReason.saMulticast:
   DropReasonType( "saMulticast", "Source address is multicast" ),
   PacketDropReason.rpf:
   DropReasonType( "rpf", "Reverse path forwarding" ),
   PacketDropReason.mtu:
   DropReasonType( "mtuExceeded", "MTU exceeded" ),
   PacketDropReason.hardware:
   DropReasonType( "hardwareError", "Hardware error" ),
   PacketDropReason.saNotFound:
   DropReasonType( "saNotFound", "Source Address not found" ),
   PacketDropReason.noForwardingAction:
   DropReasonType( "noForwardingAction", "No forwarding action" ),
   PacketDropReason.mplsDisabled:
   DropReasonType( "mplsDisabled", "MPLS disabled" ),
   PacketDropReason.ingressSourcePortFilter:
   DropReasonType( "ingressSourcePortFilter", "Ingress source port filter" ),
   PacketDropReason.mplsTtl01:
   DropReasonType( "mplsTtl01", "MPLS TTL 0/1" ),
   PacketDropReason.mplsLabelLookupMiss:
   DropReasonType( "mplsLabelLookupMiss", "MPLS label lookup miss" ),
   PacketDropReason.ipVersionErrorPostMplsDecap:
   DropReasonType( "ipVersionErrorPostMplsDecap",
                        "IP version error after MPLS decap" ),
   PacketDropReason.ipv4ChecksumErrorPostMplsDecap:
   DropReasonType( "ipv4ChecksumErrorPostMplsDecap",
                        "IPv4 checksum error after MPLS decap" ),
   PacketDropReason.ipTtl01PostMplsDecap:
   DropReasonType( "ipTtl01PostMplsDecap", "IP TTL 0/1 after MPLS decap" ),
   PacketDropReason.ipForwardingLookupMissPostMplsDecap:
   DropReasonType( "ipForwardingLookupMissPostMplsDecap",
                        "IP forwarding lookup miss after MPLS decap" ),
   PacketDropReason.ipMissingArpPostMplsDecap:
   DropReasonType( "ipMissingArpPostMplsDecap",
                        "ARP unresolved after MPLS decap" ),
   PacketDropReason.ipMissingRoutePostMplsDecap:
   DropReasonType( "ipMissingRoutePostMplsDecap",
                        "IP missing route after MPLS decap" ),
   PacketDropReason.ipv6UnspecifiedDestinationPostMplsDecap:
   DropReasonType( "ipv6UnspecifiedDestinationPostMplsDecap",
                        "IPv6 unspecified destination after MPLS decap" ),
   PacketDropReason.ipv6MulticastSourcePostMplsDecap:
   DropReasonType( "ipv6MulticastSourcePostMplsDecap",
                        "IPv6 multicast source after MPLS decap" ),
   PacketDropReason.ingressSpanningTreeFilter:
      DropReasonType( "ingressSpanningTreeFilter",
                      "Ingress Spanning Tree Filter" ),
   PacketDropReason.packetBufferIngressLimit:
      DropReasonType( "packetBufferIngressLimit",
                      "Packet buffer ingress limit" ),
   PacketDropReason.packetBufferSharedPoolLimit:
      DropReasonType( "packetBufferSharedPoolLimit",
                      "Packet buffer shared pool limit" ),
   PacketDropReason.packetBufferEgrPortSpSharedLimit:
      DropReasonType( "packetBufferEgrPortSpSharedLimit",
                      "Packet buffer egress port shared limit" ),
   PacketDropReason.packetBufferEgrQueueSharedLimit:
      DropReasonType( "packetBufferEgrQueueSharedLimit",
                      "Packet buffer egress queue shared limit" ),
   PacketDropReason.packetBufferEgrWred:
      DropReasonType( "packetBufferEgrWred",
                      "Packet buffer egress WRED" ),
   PacketDropReason.packetBufferGlobalRejects:
      DropReasonType( "packetBufferGlobalRejects", "Packet buffer global reject" ),
   PacketDropReason.packetBufferUnknown:
      DropReasonType( "packetBufferUnknown", "Packet buffer unknown" ),
   PacketDropReason.packetBufferVoqSharedReject:
      DropReasonType( "packetBufferVoqSharedReject",
                      "Packet buffer VOQ shared reject" ),
   PacketDropReason.packetBufferVoqWredReject:
      DropReasonType( "packetBufferVoqWredReject", "Packet buffer VOQ WRED reject" ),
   PacketDropReason.packetBufferVoqLatency:
      DropReasonType( "packetBufferVoqLatency", "Packet buffer VOQ latency" ),
   PacketDropReason.packetBufferVsqSharedReject:
      DropReasonType( "packetBufferVsqSharedReject",
                      "Packet buffer VSQ shared reject" ),
   PacketDropReason.packetBufferVsqWredReject:
      DropReasonType( "packetBufferVsqWredReject", "Packet buffer VSQ WRED reject" ),
   PacketDropReason.packetBufferQueueError:
      DropReasonType( "packetBufferQueueError", "Packet buffer queue error" ),
}

def getDropReasonStr( reason ):
   reasonPart = ( reason.part0, reason.part1, reason.part2, reason.part3 )
   if reasonPart[ 0 ] & 0x1:
      return dropReasonStr[ 0 ]

   reasonStr = ""
   reasonPartBitCount = 32 # drop reason is 4 U32's;
   for i, part in enumerate( reasonPart ):
      for bit in range( dropReasonPartBitRange[ i ][ 0 ],
                        dropReasonPartBitRange[ i ][ 1 ] ):
         if part & 0x1: # if the bit is set, drop reason present;
            reasonStr += dropReasonStr[ bit + ( i * reasonPartBitCount ) ] + '\n'
         part >>= 1

   return reasonStr.rstrip() # strip trailing '\n' since CLI will add it itself;

def getInterfaces( ethIntfStatusDir ):
   intfs = []
   for intfStatus in ethIntfStatusDir.intfStatus.values():
      intfs.append( intfStatus.intfId )
   return intfs

def getProtocolStr( protocol, protocolNumber ):
   if protocol:
      if protocol.startswith( 'ipProto' ):
         return protocol[ len( 'ipProto' ) : ].upper()
      else:
         return protocol
   else:
      return str( protocolNumber )
