#!/usr/bin/env python
# Copyright (c) 2008-2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import re
from CliPlugin import AclCliModel
from CliPlugin import EthIntfCli
from CliPlugin import MirroringModels
from CliPlugin.AclCli import ( showIpAccessLists, showIp6AccessLists,
      showMacAccessLists, getUnsupportedQualifierActionList )
from collections import namedtuple
import Arnet
from MirroringLib import metadataElementTokenDict
import Tac
from TypeFuture import TacLazyType
import six

AclFeature = TacLazyType( "Acl::AclFeature" )
AclKey = Tac.Type( "Mirroring::AclInfo" )
IntfApplication = Tac.Type( "Acl::IntfApplication" )
EthType = Tac.Type( "Arnet::EthType" ) # enum
GrePayloadType = TacLazyType( 'Mirroring::GrePayloadType' )
SampleRateConstants = TacLazyType( 'Mirroring::SampleRateConstants' )
TruncationSize = TacLazyType( 'Mirroring::TruncationSize' )
Constants = TacLazyType( 'Mirroring::Constants' )
GreTunnelIntfId = Tac.Type( "Arnet::GreTunnelIntfId" )

RuleCounter = namedtuple( 'RuleCounter', [ 'pkts', 'timestamp' ] )

def updateAclCounterData( counters, stale, aclList ):
   """
   Update the rules for a particular session or source ACL with counter information,
   queried from Sysdb. If the query timed out, indicate the counters may be stale.
   """

   if counters is None:
      return

   if aclList and len( aclList.aclList ) == 1:
      acl = aclList.aclList[ 0 ]
      aclRules = [ rule.sequenceNumber for rule in acl.sequence
                   if rule.action is not None ]
      if set( counters ) != set( aclRules ):
         return

      acl.countersEnabled = True
      acl.staleCounters = stale
      for rule in acl.sequence:
         if rule.action is None:
            continue
         aclCounter = AclCliModel.AclCounter()
         aclCounter.packetCount = counters[ rule.sequenceNumber ].pkts
         aclCounter.byteCount = None
         aclCounter.connCount = None
         aclCounter.checkpointPacketCount = 0
         aclCounter.checkpointConnCount = 0
         aclCounter.lastChangedTime = counters[ rule.sequenceNumber ].timestamp
         rule.counterData = aclCounter

def countersForAclKey( gv, aclKey ):
   counters = {}
   aclCounter = gv.aclCounterStatus.counter.get( aclKey )
   if aclCounter:
      for seq, ruleStatus in aclCounter.ruleBySequence.items():
         counters[ seq ] = RuleCounter( ruleStatus.pkts, ruleStatus.lastChangedTime )

   return counters

def queryAclCounters( gv, model ):
   """
   Query the mirroring ACL counters populated in Sysdb by AleMirroringSm at:
   mirroring/acl/counter/status. For security ACLs with mirroring action, the ACL
   counters are populated by AleAclSm and queried using showIpAccessLists. Only
   session and source interface ACLs are queried here.
   """

   if not gv.aclCounterConfig or not gv.aclCounterStatus:
      return

   gv.aclCounterConfig.counterUpdateRequestTime = Tac.now()
   try:
      Tac.waitFor( lambda: gv.aclCounterStatus.counterUpdateTime >=
                   gv.aclCounterConfig.counterUpdateRequestTime,
                   description='counters to be available',
                   warnAfter=None, sleep=True, maxDelay=0.1, timeout=10 )
      stale = False
   except Tac.Timeout:
      stale = True

   for aclName, srcAclEntry in model.sourceAclsDetails.items():
      aclTypes = []
      if srcAclEntry.ipAclList:
         aclTypes.append( 'ip' )
      if srcAclEntry.ipv6AclList:
         aclTypes.append( 'ipv6' )
      if srcAclEntry.macAclList:
         aclTypes.append( 'mac' )
      for aclType in aclTypes:
         aclKey = AclKey( aclName, aclType, 'in' )
         counters = countersForAclKey( gv, aclKey )
         updateAclCounterData( counters, stale, srcAclEntry.getList( aclType ) )

   if model.sessionAcl:
      aclKey = AclKey( model.sessionAcl, model.aclType, 'in' )
      counters = countersForAclKey( gv, aclKey )
      updateAclCounterData( counters, stale,
                            model.sessionAclDetails.getList( model.aclType ) )

   return

def doShowMirrorSessions( gv, mode, args ):
   # show mirror session should only look at mirror status for showing sessions
   # and not mirror config because if the mirror agent is dead then we do have the
   # session in mirroring config but not in mirroring status, so status will be
   # displayed
   # gv should provides:
   #  values:
   #    defaultGreHdrProto
   #    directionBoth
   #    directionRx
   #    directionTx
   #    headerRemovalDefaultVal
   #  mounts:
   #    aclStatusDp
   #    mirroringConfig
   #    mirroringHwCapability
   #    mirroringHwConfig
   #    mirroringHwStatus
   #    mirroringStatus

   name = args.get( 'SESSNAME' )
   detail = 'detail' in args
   accessList = 'access-list' in args
   detailStr = args.get( 'access-list', '' )
   summaryStr = ''
   if 'summary' in args:
      for sessionFilter in ( 'source', 'destination' ):
         if sessionFilter in args:
            summaryStr = sessionFilter.capitalize()
            break
      else:
         summaryStr = 'True'

   model = MirroringModels.MirrorSessions( _summary=summaryStr, _detail=detailStr )

   def setPayloadTypeForModel( sessionModel, hwConfigSession,
                               isEgressAclSession=None ):
      grePayloadType = hwConfigSession.grePayloadType
      if isEgressAclSession:
         sessionModel.grePayloadType = GrePayloadType.payloadTypeFullPacket
      elif grePayloadType == GrePayloadType.payloadTypeHwDefault:
         sessionModel.grePayloadType = gv.mirroringHwCapability.defaultGrePayloadType
      else:
         sessionModel.grePayloadType = grePayloadType

   def setGreMetadataForModel( sessionModel, hwConfigSession, isEgressAclSession ):
      txGreMetadataEnabled = ( hwConfigSession.txGreMetadataEnabled or
            gv.mirroringHwCapability.txGreMetadataGlobalSupported )
      if ( hwConfigSession.greTunnelKey.keyConfigured or 
            ( isEgressAclSession and not txGreMetadataEnabled ) ):
         sessionModel.greMetadata = [ 'disabled' ]
         return
      if ( isEgressAclSession and 
            gv.mirroringHwCapability.txGreMetadataPerSessionSupported ):
         sessionModel.egressAclMirroredTruncation = \
               Constants.egressMirroringTruncationSize
      greMetadata = hwConfigSession.greMetadata
      if not greMetadata:
         greMetadata = gv.mirroringHwCapability.defaultGreMetadata
      # Find corresponding metadata set in greMetadataSupported
      # to ensure correct element order is retained
      for metadata in gv.mirroringHwCapability.greMetadataSupported.values():
         hwConfigElements = set( greMetadata )
         defaultElements = set( metadata.metadataElement.values() )
         if hwConfigElements == defaultElements:
            sessionModel.greMetadata = [ metadataElementTokenDict[ elem ]
               for elem in metadata.metadataElement.values() ]

   def getUnsupportedQualifiers( intfName, aclName, aclType, vxlanIntf=False ):
      linecardName = None
      if EthIntfCli.isModular():
         result = re.search( r'(\d+)/', intfName )
         # Look for the linecard from the interface name. If
         # it isn't found, leave linecardName as None to check
         # all linecards for warnings on the ACL.
         if result:
            # pylint: disable-next=consider-using-f-string
            linecardName = 'Linecard%s' % result.group( 1 )
      else:
         linecardName = 'FixedSystem'
      if vxlanIntf:
         return None
      app = IntfApplication( intfName )
      quals, _ = getUnsupportedQualifierActionList(
         aclName, aclType, gv.aclStatusDp, feature=AclFeature.featureMirroring,
         linecard=linecardName, intfApplication=app )
      return quals

   def operStateAndReason( sessionStatus, sessName, intfName ):
      operState = 'active'
      operStateReason = ''

      if sessionStatus:
         targetStatus = ( gv.mirroringStatus.sessionStatus[ sessName ]
               .targetOperStatus.get( intfName ) )
         if targetStatus:
            reason = None
            ifHwStatus = gv.mirroringHwStatus.intfHwStatus.get( sessName )
            if ( ifHwStatus is not None
                 and ifHwStatus.hwConfigStatus.get( intfName ) is not None
                 and ifHwStatus.hwConfigStatus[ intfName ].status is False ):
               reason = ifHwStatus.hwConfigStatus[ intfName ].reason
            if targetStatus.state == "operStateInactive":
               operState = "inactive"
               operStateReason = targetStatus.reason
            elif reason is not None:
               operState = "inactive"
               operStateReason = reason
            elif targetStatus.state == "operStateUnknown":
               operState = "unknown"
               operStateReason = targetStatus.reason
         else:
            operState = "unknown"
            operStateReason = "unknown"
      else:
         operState = "unknown"
         operStateReason = "unknown"

      return ( operState, operStateReason )

   def createAclDetailsModel( mode, aclName, aclType ):
      def noShowCounterInfo( aclList ):
         """
         The ACL counter information queried in showIpAccessLists is valid only for
         security ACLs with mirroring action. For source interface ACLs and session
         ACLs, reset the counter data as showIpAccessLists updates them with counters
         for security ACLs; the counter data will be filled in queryAclCounters.
         """

         if aclList and len( aclList.aclList ) == 1:
            acl = aclList.aclList[ 0 ]
            acl.countersEnabled = False
            acl.staleCounters = False
            acl.noMatchCounter = None

            for rule in acl.sequence:
               rule.counterData = None

      model = MirroringModels.AclDetails()
      if not aclName:
         return None
      parameters = ( aclName, False, False, False )
      if aclType == 'ip':
         model.ipAclList = showIpAccessLists( mode, parameters, mirrorAcl=True )
         noShowCounterInfo( model.ipAclList )
      elif aclType == 'ipv6':
         model.ipv6AclList = showIp6AccessLists( mode, parameters, mirrorAcl=True )
         noShowCounterInfo( model.ipv6AclList )
      elif aclType == 'mac':
         model.macAclList = showMacAccessLists( mode, parameters, mirrorAcl=True )
         noShowCounterInfo( model.macAclList )

      return model

   def createAclListWithMirrorActionModel( mode, name, addRules ):
      egressAclSessions = gv.mirroringHwStatus.egressAclSessions
      ingressAclSessions = gv.mirroringHwStatus.ingressAclSession

      # should return None model if session has no ingress or egress ACL sessions
      # so that an empty Session (no ACLs) is not created
      if not name in egressAclSessions and \
         not name in ingressAclSessions:
         return None

      # create model and populate with ingress and/or egress ACLs
      model = MirroringModels.MirrorSessions.MirrorSession.AclsWithMirrorAction()

      if name in egressAclSessions:
         populateAclListWithMirrorActions( mode, 'out',
                                           egressAclSessions[ name ],
                                           model, addRules )
      if name in ingressAclSessions:
         populateAclListWithMirrorActions( mode, 'in',
                                           ingressAclSessions[ name ],
                                           model, addRules )

      return model

   def populateAclListWithMirrorActions( mode, direction,
                                         aclSessions, model,
                                         addRules ):
      for aclInfo in aclSessions.aclInfo:
         parameters = ( aclInfo.name, False, False, False )
         if aclInfo.aclType == 'ip':
            ipv4AclWithMirrorAction = MirroringModels.MirrorSessions.\
                  MirrorSession.AclsWithMirrorAction.Ipv4AclWithMirrorAction()
            acl = None if not addRules else \
                  showIpAccessLists( mode, parameters, mirrorAcl=True )
            ipv4AclWithMirrorAction.name = aclInfo.name
            ipv4AclWithMirrorAction.acl = acl
            ipv4AclWithMirrorAction.direction = direction
            model.ipAcls.append( ipv4AclWithMirrorAction )
         elif aclInfo.aclType == 'ipv6':
            ipv6AclWithMirrorAction = MirroringModels.MirrorSessions.\
                  MirrorSession.AclsWithMirrorAction.Ipv6AclWithMirrorAction()
            acl = None if not addRules else \
                  showIp6AccessLists( mode, parameters, mirrorAcl=True )
            ipv6AclWithMirrorAction.name = aclInfo.name
            ipv6AclWithMirrorAction.acl = acl
            ipv6AclWithMirrorAction.direction = direction
            model.ipv6Acls.append( ipv6AclWithMirrorAction )
         elif aclInfo.aclType == 'mac':
            macAclWithMirrorAction = MirroringModels.MirrorSessions.\
                  MirrorSession.AclsWithMirrorAction.MacAclWithMirrorAction()
            acl = None if not addRules else \
                  showMacAccessLists( mode, parameters, mirrorAcl=True )
            macAclWithMirrorAction.name = aclInfo.name
            macAclWithMirrorAction.acl = acl
            macAclWithMirrorAction.direction = direction
            model.macAcls.append( macAclWithMirrorAction )

   def isVlan( intfName ):
      return Tac.Type( 'Arnet::VlanIntfId' ).isVlanIntfId( intfName )

   def getVlanId( intfName ):
      assert isVlan( intfName )
      return Tac.Type( 'Arnet::VlanIntfId' ).vlanId( intfName )

   def isDefaultGreKeyApplied():
      # If the underlying hardware does not require "default GRE key"
      # processing, simply exit.
      if not gv.mirroringHwCapability.defaultGreKeySupported:
         return False

      for session in gv.mirroringHwConfig.session.values():
         if ( session.greTunnelKey is not None ) and \
            ( session.greTunnelKey.keyConfigured ):
            return True

      return False

   def handleSrcInSessionModel( mode, name, sessionModel, vxlan=False ):
      sessionConfig = gv.mirroringConfig.session.get( name )
      if ( not sessionConfig or
           sessionConfig.secureMonitor != mode.session.secureMonitor() ):
         return None
      sessionStatus = gv.mirroringStatus.sessionStatus.get( name )
      if vxlan:
         intfNames = sessionConfig.srcVxlanIntf
      else:
         intfNames = Arnet.sortIntf( sessionConfig.srcIntf )
      for intfName in intfNames:
         if vxlan:
            srcIntf = gv.mirroringConfig.session[ name ].srcVxlanIntf[ intfName ]
            intfNameStr = str(intfName)
         else:
            srcIntf = gv.mirroringConfig.session[ name ].srcIntf[ intfName ]
            intfNameStr = intfName
         if srcIntf.srcAcl:
            sessionModel.aclType = srcIntf.aclType
            sessionModel.sourceAcls[ intfNameStr ] = \
                  model.MirrorSession.Acl(
                  aclName=srcIntf.srcAcl, aclPriority=srcIntf.aclPriority,
                  aclType=srcIntf.aclType,
                  aclDetails=detail,
                  aclUnsupportedQualifiers=getUnsupportedQualifiers(
                     intfName, srcIntf.srcAcl, srcIntf.aclType, vxlan ) )

         if gv.mirroringHwConfig.session.get( name ):
            if vxlan:
               hwSrcIntf = gv.mirroringHwConfig.session[ \
                     name ].srcVxlanIntf.get( intfName )
               intfNameStr = str(intfName)
            else:
               hwSrcIntf = gv.mirroringHwConfig.session[ name \
                     ].srcIntf.get( intfName )
               intfNameStr = intfName
            if hwSrcIntf and hwSrcIntf.srcAcl:
               sessionModel.sourceAcls[ intfNameStr ] = model.MirrorSession.Acl(
                     aclName=hwSrcIntf.srcAcl,
                     aclPriority=hwSrcIntf.aclPriority,
                     aclType=hwSrcIntf.aclType,
                     aclDetails=detail,
                     aclUnsupportedQualifiers=getUnsupportedQualifiers(
                        intfName, srcIntf.srcAcl, srcIntf.aclType, vxlan ) )

         if sessionStatus:
            if not vxlan:
               srcStatus = sessionStatus.srcOperStatus.get( intfName )
            else:
               srcStatus = sessionStatus.srcVxlanOperStatus.get( intfName )
            if srcStatus is None: # pylint: disable=no-else-continue
               sessionModel.sourceUnknownInterfaces.append( intfName )
               continue
            else:
               if srcStatus.state == "operStateInactive":
                  if isVlan( intfName ) and \
                     gv.mirroringHwCapability.rxVlanSourceSupported:
                     sessionModel.sourceInactiveVlans.append(
                        model.MirrorSession.Vlan(
                        vlan=getVlanId( intfName ),
                        operState='inactive',
                        operStateReason=srcStatus.reason ) )
                     sessionModel.sourceVlans.append( getVlanId( intfName ) )
                  else:
                     sessionModel.sourceInactiveInterfaces.append(
                        model.MirrorSession.Interface(
                        name=intfName,
                        operState='inactive',
                        operStateReason=srcStatus.reason ) )
                     sessionModel.sourceInterfaces.append( intfName )
                  continue
         else:
            # session status not present
            sessionModel.sourceUnknownInterfaces.append( intfNameStr )
            continue
         reason = None
         ifHwStatus = gv.mirroringHwStatus.intfHwStatus.get( name )
         if ( ifHwStatus is not None
              and ifHwStatus.hwConfigStatus.get( intfName ) is not None
              and ifHwStatus.hwConfigStatus[ intfName ].status is False ):
            reason = ifHwStatus.hwConfigStatus[ intfName ].reason
         if reason is not None:
            if isVlan( intfName ) and \
                  gv.mirroringHwCapability.rxVlanSourceSupported:
               sessionModel.sourceInactiveVlans.append(
                  model.MirrorSession.Vlan(
                  vlan=getVlanId( intfName ),
                  operState='inactive',
                  operStateReason=srcStatus.reason ) )
            else:
               sessionModel.sourceInactiveInterfaces.append(
                  model.MirrorSession.Interface(
                  name=intfName,
                  operState='inactive',
                  operStateReason=reason ) )
               continue

         intfNameStr = str(intfName)
         if vxlan:
            intfNameStr = str(intfName.vti) + " VNI: " + str(intfName.vni) 
         if srcIntf.direction == gv.directionRx:
            if not vxlan and isVlan( intfName ) and \
                  gv.mirroringHwCapability.rxVlanSourceSupported:
               sessionModel.sourceRxVlans.append( getVlanId( intfName ) )
            else:
               sessionModel.sourceRxInterfaces.append( intfNameStr )
         elif srcIntf.direction == gv.directionTx:
            sessionModel.sourceTxInterfaces.append( intfNameStr )
         elif srcIntf.direction == gv.directionBoth:
            sessionModel.sourceBothInterfaces.append( intfNameStr )
         else:
            # pylint: disable-next=consider-using-f-string
            assert False, "Invalid direction %s" % srcIntf.direction
         if (not vxlan) and isVlan( intfName ) and \
               gv.mirroringHwCapability.rxVlanSourceSupported:
            sessionModel.sourceVlans.append( getVlanId( intfName ) )
         else:
            sessionModel.sourceInterfaces.append( intfNameStr )
      return None
   def createSessionModel( mode, name ):
      sessionConfig = gv.mirroringConfig.session.get( name )
      if ( not sessionConfig or
           sessionConfig.secureMonitor != mode.session.secureMonitor() ):
         return None
      sessionModel = MirroringModels.MirrorSessions.MirrorSession(
            _displayInSummary=True )
      sessionStatus = gv.mirroringStatus.sessionStatus.get( name )
      handleSrcInSessionModel( mode, name, sessionModel )
      handleSrcInSessionModel( mode, name, sessionModel, True )
      dstIp = srcIp = ttl = dscp = forwardingDrop = greHdrProto = vrf = key = None
      for intfName in (
            Arnet.sortIntf( gv.mirroringConfig.session[ name ].targetIntf ) ):
         if ( intfName.startswith( "MirrorTunnel" )
              and gv.mirroringHwConfig.session.get( name ) ):
            continue
         operState, operStateReason = operStateAndReason( sessionStatus,
                                                          name, intfName )

         interface = model.MirrorSession.Interface(
            name=intfName, operState=operState, operStateReason=operStateReason,
            srcIpGenAddr=srcIp, dstIpGenAddr=dstIp, ttl=ttl, dscp=dscp, key=key,
            greHdrProto=greHdrProto, forwardingDrop=forwardingDrop, vrf=vrf )
         sessionModel.targetInterfaces.append( interface )

      defaultKeyApplied = isDefaultGreKeyApplied()

      for sessName in ( name, name + '__fwdingDrop_' ):
         hwConfigSession = gv.mirroringHwConfig.session.get( sessName )
         if hwConfigSession:
            targetIntfs = hwConfigSession.targetIntf
            if not targetIntfs:
               continue
            intfName = next( iter( targetIntfs ) )
            operState, operStateReason = operStateAndReason( sessionStatus,
                                                             name, intfName )
            if intfName.startswith( "MirrorTunnel" ):
               greTunnelParams = gv.mirroringStatus.greTunnelKey.get( intfName )
               vxlanTunnelParams = gv.mirroringStatus.vxlanTunnelKey.get( intfName )
               if greTunnelParams:
                  dstIpGen = greTunnelParams.dstIpGenAddr
                  srcIpGen = greTunnelParams.srcIpGenAddr
                  ttl = greTunnelParams.ttl
                  dscp = greTunnelParams.dscp

                  if ( hwConfigSession.greTimestampingEnabled
                       and greTunnelParams.greHdrProto == gv.defaultGreHdrProto ):
                     greHdrProto = Tac.enumValue( EthType, "ethTypeUnknown" )
                  else:
                     greHdrProto = greTunnelParams.greHdrProto

                  if greTunnelParams.keyConfigured:
                     key = greTunnelParams.key
                  elif defaultKeyApplied:
                     key = gv.mirroringHwCapability.defaultGreKeyValue
                  else:
                     key = None
                  forwardingDrop = greTunnelParams.forwardingDrop
                  vrf = greTunnelParams.vrf

                  interface = model.MirrorSession.Interface(
                     name=intfName, operState=operState,
                     operStateReason=operStateReason,
                     srcIpGenAddr=srcIpGen, dstIpGenAddr=dstIpGen, ttl=ttl,
                     dscp=dscp, key=key, greHdrProto=greHdrProto,
                     forwardingDrop=forwardingDrop, vrf=vrf, tunnelType='gre' )
                  sessionModel.targetInterfaces.append( interface )
               elif vxlanTunnelParams:
                  dstIpGen = vxlanTunnelParams.remoteVtepIp
                  srcIpGen = vxlanTunnelParams.localVtepIp
                  ttl = vxlanTunnelParams.ttl
                  dscp = vxlanTunnelParams.dscp
                  vrf = vxlanTunnelParams.vrf
                  vti = vxlanTunnelParams.vti
                  vlan = vxlanTunnelParams.vlan
                  vni = vxlanTunnelParams.vni

                  interface = model.MirrorSession.Interface(
                     name=intfName, operState=operState,
                     operStateReason=operStateReason,
                     srcIpGenAddr=srcIpGen, dstIpGenAddr=dstIpGen, ttl=ttl,
                     dscp=dscp, vrf=vrf, tunnelType='vxlan', vti=vti, vlan=vlan,
                     vni=vni )
                  sessionModel.targetInterfaces.append( interface )

               # show monitor session detail
               sessionModel.grePayloadType = GrePayloadType.payloadTypeNone
               sessionModel.greMetadata = []
               isEgressAclSession = \
                     sessName in gv.mirroringHwStatus.egressAclSessions
               if detail:
                  if gv.mirroringHwCapability.supportedGrePayloadTypes:
                     setPayloadTypeForModel( sessionModel, hwConfigSession,
                                             isEgressAclSession=isEgressAclSession )
                  if gv.mirroringHwCapability.greMetadataSupported:
                     setGreMetadataForModel( sessionModel, hwConfigSession,
                                             isEgressAclSession )
                  if gv.mirroringHwCapability.greTimestampingSupported:
                     sessionModel.greTimestampingEnabled = (
                        hwConfigSession.greTimestampingEnabled )

               if gv.mirroringHwStatus.greNextHopIntfSet.get( intfName ) is None:
                  continue
               nextHopIntfSet = gv.mirroringHwStatus.greNextHopIntfSet[ intfName ]
               sortedNextHopIntfs = sorted( nextHopIntfSet.nextHopIntf )
               sortedTunnelIntfs = model.MirrorSession.TunnelIntfs()
               for key in sortedNextHopIntfs:
                  if ( len( nextHopIntfSet.nextHopIntf[ key ].intf ) == 1
                       and key in nextHopIntfSet.nextHopIntf[ key ].intf ):
                     sortedMemberIntfs = list() # pylint: disable=use-list-literal
                  else:
                     sortedMemberIntfs = sorted(
                        nextHopIntfSet.nextHopIntf[ key ].intf )
                  memberIntfs = model.MirrorSession.TunnelIntfs.MemberIntfs(
                     memberIntfs=sortedMemberIntfs )
                  sortedTunnelIntfs.tunnelIntfs[ key ] = memberIntfs
               if greTunnelParams:
                  sessionModel.greTunnels[ intfName ] = sortedTunnelIntfs
               elif vxlanTunnelParams:
                  sessionModel.vxlanTunnels[ intfName ] = sortedTunnelIntfs

      if gv.mirroringConfig.session[ name ].sessionAcl != "":
         sessionModel.aclType = gv.mirroringConfig.session[ name ].aclType
      sessionModel.sessionAcl = gv.mirroringConfig.session[ name ].sessionAcl

      if accessList:
         sessionModel.sessionAclDetails = createAclDetailsModel( mode,
               sessionModel.sessionAcl, sessionModel.aclType )

         # Populate details of source ACLs from the per-interface srcAcl table
         for intfName, acl in sessionModel.sourceAcls.items():
            if acl.aclName != sessionModel.sessionAcl:
               srcAclEntry = sessionModel.sourceAclsDetails.get( acl.aclName )
               if srcAclEntry is None:
                  srcAclEntry = createAclDetailsModel( mode, acl.aclName,
                                                       acl.aclType )
                  sessionModel.sourceAclsDetails[ acl.aclName ] = srcAclEntry
               elif not srcAclEntry.getList( acl.aclType ):
                  newEntry = createAclDetailsModel( mode, acl.aclName, acl.aclType )
                  if acl.aclType == 'ip':
                     srcAclEntry.ipAclList = newEntry.getList( acl.aclType )
                  elif acl.aclType == 'ipv6':
                     srcAclEntry.ipv6AclList = newEntry.getList( acl.aclType )
                  elif acl.aclType == 'mac':
                     srcAclEntry.macAclList = newEntry.getList( acl.aclType )
                  sessionModel.sourceAclsDetails[ acl.aclName ] = srcAclEntry

      # populate the acl list with mirror action
      if accessList:
         sessionModel.aclsWithMirrorAction = \
               createAclListWithMirrorActionModel( mode, name, True )
      elif detail:
         sessionModel.aclsWithMirrorAction = \
               createAclListWithMirrorActionModel( mode, name, False )

      if accessList and gv.mirroringHwCapability.aclCountersSupported:
         queryAclCounters( gv, sessionModel )

      sessionModel.truncate = gv.mirroringConfig.session[ name ].truncate
      truncateStatus = ''
      if gv.mirroringConfig.session[ name ].truncate:
         truncateStatus = 'unknown'
         if sessionStatus:
            if sessionStatus.truncateOperStatus.state == 'operStateActive':
               truncateStatus = 'active'
            elif sessionStatus.truncateOperStatus.state == 'operStateInactive':
               # pylint: disable-next=consider-using-f-string
               truncateStatus = 'inactive (%s)' % (
                     sessionStatus.truncateOperStatus.reason )
         if gv.mirroringHwCapability.truncationSizesSupported:
            if ( gv.mirroringConfig.session[ name ].truncationSize !=
                  TruncationSize.null ):
               sessionModel.truncationSize = (
                     gv.mirroringConfig.session[ name ].truncationSize )
            else:
               sessionModel.truncationSize = (
                     gv.mirroringHwCapability.truncationSizesSupported.defaultVal )

      sessionModel.truncateStatus = truncateStatus

      if ( gv.mirroringHwCapability.headerRemovalSupported
            and gv.mirroringConfig.session[ name ].headerRemovalSize ):
         sessionModel.headerRemovalSize = (
               gv.mirroringConfig.session[ name ].headerRemovalSize )
      else:
         sessionModel.headerRemovalSize = gv.headerRemovalDefaultVal

      sessionModel.mirrorDeviceName = None
      if sessionStatus and sessionStatus.mirrorDeviceName != '':
         sessionModel.mirrorDeviceName = sessionStatus.mirrorDeviceName

      sessionModel.hwProgrammingStatusSupported = (
         gv.mirroringHwCapability.hwProgrammingStatusSupported )
      sessionModel.programmedInHw = False
      for sessName in ( name, name + '__fwdingDrop_' ):
         programmedSession = gv.mirroringHwStatus.programmedSession.get( sessName )
         if programmedSession is not None:
            sessionModel.programmedInHw |= programmedSession

      sessionModel.rxSampleRate = None
      if ( gv.mirroringConfig.session[ name ].rxSampleRate !=
            SampleRateConstants.sampleRateUndefined ):
         sessionModel.rxSampleRate = gv.mirroringConfig.session[ name ].rxSampleRate

      sn = gv.mirroringConfig.session[ name ]
      sessionModel.mirrorRateLimitChip = sn.mirrorRateLimitChip
      sessionModel.mirrorRateLimit = sn.mirrorRateLimitInBps
      # if platform has overridden the configured rate limit, use that value.
      hwRateLimit = 0
      hwRateLimit = gv.mirroringHwStatus.rateLimitInKbps.get( name, 0 )
      if hwRateLimit > 0:
         sessionModel.mirrorRateLimit = hwRateLimit * 1000

      sessionModel.greTimestampingSupported = (
         gv.mirroringHwCapability.greTimestampingSupported )
      sessionModel.txMirrorAclSupported = (
         gv.mirroringHwCapability.txMirrorAclSupported )

      if ( gv.mirroringHwCapability.tcamProfileMapRequired and
           sn.mirrorTcamProfileFeature != '' ):
         sessionModel.tcamProfileFeature = (
            sn.mirrorTcamProfileFeature )

      sessionModel.congestionMonitor = \
         gv.mirroringConfig.session[ name ].congestionMonitor
      
      sessionModel.cpuQueue = \
         gv.mirroringConfig.session[ name ].cpuQueue

      return sessionModel

   # This info is not needed if mirror header removal is not supported
   model.multiDestSessions = []
   if gv.mirroringHwCapability.headerRemovalSupported:
      model.multiDestSessions = [ key for key, status in
                                  six.iteritems( gv.mirroringStatus.sessionStatus )
                                  if status.multiDestActive ]

   if name is not None:
      model.name = name
      sessionModel = createSessionModel( mode, name )
      if sessionModel:
         model.sessions[ name ] = sessionModel
      return model

   for sessionName in gv.mirroringConfig.session:
      sessionModel = createSessionModel( mode, sessionName )
      if not sessionModel:
         continue
      model.sessions[ sessionName ] = sessionModel
      # If a summary filter was applied, mark session if it doesn't match
      srcIntf = args.get( 'SRCINTF' )
      if srcIntf:
         if set( sessionModel.sourceInterfaces ).isdisjoint( srcIntf ):
            sessionModel.doNotDisplayInSummary()
      vlans = args.get( 'VLANS', [] )
      if vlans:
         vlans = vlans.ranges
         if set( sessionModel.sourceVlans ).isdisjoint( vlans ):
            sessionModel.doNotDisplayInSummary()
      elif 'destination' in args:
         if 'cpu' in args:
            intfs = [ 'Cpu' ]
         else:
            intfs = None
            for intfOpt in [ 'ETHINTF', 'LAGINTF', 'GREINTF', 'MIRRORTUNNELINTF' ]:
               if intfOpt in args:
                  intfs = args[ intfOpt ]
                  break

            assert intfs
         dests = [ dest.name.stringValue for dest in sessionModel.targetInterfaces ]
         if set( dests ).isdisjoint( intfs ):
            sessionModel.doNotDisplayInSummary()
   return model
