#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from Arnet import EthAddr as EthAddrStr
from CliMode.TapAgg import TapAggActionModeBase
from CliPlugin import PolicyMapCliLib, TapAggIntfCli, TapAggPmapCliLib
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.InternalRecircIntfCli import InternalRecircAutoIntfType
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.MacAddr import macAddrMatcher
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
from CliPlugin.TapAggActionContext import ActionContext
from CliPlugin.TapAggPmapCliLib import TapAggCliError
from TypeFuture import TacLazyType
import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import ConfigMount
import Intf.IntfRange
import LazyMount
import MultiRangeRule
import Tracing

t0 = Tracing.trace0

# Tac types
EthAddr = TacLazyType( 'Arnet::EthAddr' )
EthAddrTuple = TacLazyType( 'Arnet::EthAddrTuple' )
TapHeaderRemove = TacLazyType( 'Bridging::Input::TapHeaderRemove' )
UniqueId = TacLazyType( 'Ark::UniqueId' )
IdentityTuple = TacLazyType( 'Bridging::IdentityTuple' )

# Global variables
gv = CliGlobal.CliGlobal( dict(
   tapAggPmapConfig=None,
   tapAggHwStatus=None,
) )

#--------------------------------------------------------------------------------
# Helpers
#--------------------------------------------------------------------------------
def getIntfOpAndList( allowedRawIntfOp ):
   rawIntfOp, intfs = allowedRawIntfOp
   if intfs:
      try:
         return rawIntfOp, list( intfs )
      except TypeError:
         return rawIntfOp, [ intfs.name ]

   return rawIntfOp, []

def validateHeaderRemoveConfig( args ):
   headerRemove = TapHeaderRemove()
   err = ""
   size = args.get( 'SIZE', 0 )
   # The size has to be an even number.
   if size % 2 == 1:
      err = "The size must be an even number."
   else:
      preserveEth = 'ethernet' in args
      headerRemove = TapHeaderRemove( size, preserveEth )

   return headerRemove, err

#--------------------------------------------------------------------------------
# Guards
#--------------------------------------------------------------------------------
def guardPmapDot1qRemove( mode, token ):
   if ( gv.tapAggHwStatus.modeSupported
        and gv.tapAggHwStatus.pmapDot1qRemoveSupported ):
      return None
   return CliParser.guardNotThisPlatform

def guardPmapRemoveDot1qRange( mode, token ):
   if gv.tapAggHwStatus.dot1qRemoveInnerOnlySupported:
      return None
   # This platform doesn't support only removing the inner tag, the list must
   # include 1 (i.e. the outer tag)
   if isinstance( token, str ):
      outerTagPresent = '1' in token
   else: # token type is GenericRangeList, from MultiRangeRule
      outerTagPresent = 1 in token.values()
   if outerTagPresent:
      return None
   return CliParser.guardNotThisPlatform

def guardDropAction( mode, token ):
   if gv.tapAggHwStatus.dropActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardTimestampHeaderAction( mode, token ):
   if gv.tapAggHwStatus.policyTimestampActionSupported and \
         mode.timestampActionSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardTapAggHeaderRemove( mode, token ):
   if gv.tapAggHwStatus.modeSupported and gv.tapAggHwStatus.tapHeaderRemoveSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardActionSet( mode, token ):
   if gv.tapAggHwStatus.actionSetActionSupported:
      return None
   return CliParser.guardNotThisPlatform

#--------------------------------------------------------------------------------
# Matchers and nodes
#--------------------------------------------------------------------------------
matcherSet = CliMatcher.KeywordMatcher( 'set',
      helpdesc='Set values' )
matcherDot1Q = CliMatcher.KeywordMatcher( 'dot1q',
      helpdesc='Remove dot1q tag' )
matcherIdTag = CliMatcher.KeywordMatcher( 'id-tag',
      helpdesc='Port identity' )
matcherPortId = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Tap port identity tag' )
matcherInnerPortId = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Inner tap port identity tag' )
matcherOuter = CliMatcher.KeywordMatcher( 'outer',
      helpdesc='Remove outer tag' )
nodeInner = CliCommand.guardedKeyword( 'inner',
      helpdesc='Inner port identity',
      guard=TapAggIntfCli.qinqIdentityTaggingGuard )
matcherIntfRanger = CliCommand.Node(
      matcher=Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthPhyAutoIntfType, LagAutoIntfType,
                             SwitchAutoIntfType, InternalRecircAutoIntfType ) ) )
nodeAction = CliCommand.guardedKeyword( 'action',
      helpdesc='Create action set',
      guard=guardActionSet )

#--------------------------------------------------------------------------------
# CliExpressions 
#--------------------------------------------------------------------------------
class TimestampExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      macName = name + "_mac"
      timestampName = name + "_timestamp"
      headerName = name + "_header"
      class TimestampExpr( CliCommand.CliExpression ):
         expression = "%s %s %s" % ( macName, timestampName, headerName )
         data = {
            macName : CliCommand.singleNode(
               PolicyMapCliLib.matcherActionMac, guard=guardTimestampHeaderAction ),
            timestampName : PolicyMapCliLib.matcherActionTimestamp,
            headerName : PolicyMapCliLib.matcherActionHeader,
         }
         @staticmethod
         def adapter( mode, args, argsList ):
            if timestampName in args:
               args[ name ] = True
      return TimestampExpr

class AggNexthopGroupExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      nexthopGroupName = name + "_nexthop-group"
      class AggNexthopGroupExpr( CliCommand.CliExpression ):
         expression = '%s { NHG }' % nexthopGroupName
         data = {
            nexthopGroupName : TapAggIntfCli.matcherNexthopGroup,
            'NHG' : TapAggIntfCli.matcherNexthopGroupName,
         }
         @staticmethod
         def adapter( mode, args, argsList ):
            if not args.pop( nexthopGroupName, None ):
               return
            nexthopGroups = args.pop( 'NHG', None )
            args[ name ] = nexthopGroups
      return AggNexthopGroupExpr

class IdTagExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      idTagName = name + "_id-tag"
      innerName = name + "_inner"
      keywords = ( idTagName, innerName )
      class IdTagExpr( CliCommand.CliExpression ):
         expression = '%s PORTID [ %s INNERPORTID ]' % keywords
         data = {
            idTagName : matcherIdTag,
            'PORTID' : matcherPortId,
            innerName : nodeInner,
            'INNERPORTID' : matcherInnerPortId,
         }
      return IdTagExpr

class MacAddressExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      macAddressName = name + "_mac-address"
      destName = name + "_dest"
      srcName = name + "_src"
      keywords = ( macAddressName, destName, srcName )
      class MacAddressExpr( CliCommand.CliExpression ):
         expression = '%s %s DMAC [ %s SMAC ]' % keywords
         data = {
            macAddressName : TapAggIntfCli.matcherMacAddress,
            destName : TapAggIntfCli.matcherMacAddressDest,
            'DMAC' : macAddrMatcher,
            srcName: TapAggIntfCli.matcherMacAddressSrc,
            'SMAC' : macAddrMatcher,
         }
      return MacAddressExpr

#--------------------------------------------------------------------------------
# action mode 
# command to enter the mode:
# [no] action <name>
#--------------------------------------------------------------------------------

class TapAggActionMode( TapAggActionModeBase, BasicCli.ConfigModeBase ):
   name = "Tap Aggregation Action Configuration"

   def __init__( self, parent, session, context, mapTypeStr ):
      self.actionContext = context
      param = ( type( parent ), context.mapType(), mapTypeStr,
                context.pmapName(), context.cmapName(), context.actionName() )
      TapAggActionModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def timestampActionSupported( self ):
      return False

   def removeAction( self, actionType ):
      self.actionContext.removeAction( actionType )

   def getAction( self, actionType ):
      return self.actionContext.getAction( actionType )

   def actionExists( self, actionType ):
      return self.actionContext.actionExists( actionType )

   def onExit( self ):
      t0( 'TapAggActionMode onExit' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      t0( 'TapAggActionMode abort' )
      self.actionContext.removeActionSetChanges()

   def commitContext( self ):
      t0( 'TapAggActionMode commitContext' )
      if self.actionContext is None:
         return
      context = self.actionContext
      self.actionContext = None
      context.commit()

   def setIdentityTag( self, args ):
      t0( 'TapAggActionMode setIdentityTag' )
      portId = args[ 'PORTID' ]
      innerPortId = args.get( 'INNERPORTID', 0 )
      self.actionContext.setIdTag( portId, innerPortId )

   def delIdentityTag( self, args ):
      t0( 'TapAggActionMode delIdentityTag' )
      self.removeAction( TapAggPmapCliLib.actSetIdentityTag )

   def setAggGroup( self, groups, delete=False ):
      t0( 'TapAggActionMode setAggGroup' )
      if not isinstance( groups, list ):
         groups = [ groups ]
      if delete:
         if not self.actionExists( TapAggPmapCliLib.actSetAggregationGroup ):
            return
         tapAggAct = self.getAction( TapAggPmapCliLib.actSetAggregationGroup )
         for group in groups:
            tapAggAct.aggGroup.remove( group )
         if not tapAggAct.aggGroup and not tapAggAct.aggIntf:
            self.removeAction( TapAggPmapCliLib.actSetAggregationGroup )
      else:
         tapAggAct = self.getAction( TapAggPmapCliLib.actSetAggregationGroup )
         for group in groups:
            tapAggAct.aggGroup[ group ] = True

   def delAggGroup( self ):
      t0( 'TapAggActionMode delAggGroup' )
      if not self.actionExists( TapAggPmapCliLib.actSetAggregationGroup ):
         return
      tapAggAct = self.getAction( TapAggPmapCliLib.actSetAggregationGroup )
      tapAggAct.aggGroup.clear()
      if not tapAggAct.aggIntf:
         self.removeAction( TapAggPmapCliLib.actSetAggregationGroup )

   def setAggIntf( self, allowedRawIntfOp ):
      t0( 'TapAggActionMode setAggIntf' )
      intfOp, intfList = getIntfOpAndList( allowedRawIntfOp )
      if intfOp == 'remove':
         if not self.actionExists( TapAggPmapCliLib.actSetAggregationGroup ):
            return
         tapAggAct = self.getAction( TapAggPmapCliLib.actSetAggregationGroup )
         for intf in intfList:
            tapAggAct.aggIntf.remove( intf )
         if not tapAggAct.aggGroup and not tapAggAct.aggIntf:
            self.removeAction( TapAggPmapCliLib.actSetAggregationGroup )
      else:
         tapAggAct = self.getAction( TapAggPmapCliLib.actSetAggregationGroup )
         for intf in intfList:
            tapAggAct.aggIntf[ intf ] = True

   def delAggIntf( self ):
      t0( 'TapAggActionMode delAggIntf' )
      if not self.actionExists( TapAggPmapCliLib.actSetAggregationGroup ):
         return
      tapAggAct = self.getAction( TapAggPmapCliLib.actSetAggregationGroup )
      tapAggAct.aggIntf.clear()
      if not tapAggAct.aggGroup:
         self.removeAction( TapAggPmapCliLib.actSetAggregationGroup )

   def setStripHdrBytes( self, args ):
      t0( 'TapAggActionMode setStripHdrBytes' )
      hdrInfo = str( args[ 'VLAN_RANGE' ] )
      self.actionContext.removeDot1Q( hdrInfo )

   def delStripHdrBytes( self, args ):
      t0( 'TapAggActionMode delStripHdrBytes' )
      self.removeAction( TapAggPmapCliLib.actStripHdrBytes )

   def setMacAddress( self, args ):
      t0( 'TapAggActionMode setMacAddress' )
      if EthAddrStr( args[ 'DMAC' ] ) == EthAddr():
         self.addError( TapAggCliError[ 'setMacAddress' ] )
         return
      elif 'SMAC' in args and EthAddrStr( args[ 'SMAC' ] ) == EthAddr():
         self.addError( TapAggCliError[ 'setMacAddress' ] )
         return
      destMac = args[ 'DMAC' ]
      srcMac = args.get( 'SMAC', EthAddr() )
      ethAddrTuple = EthAddrTuple( destMac, srcMac )
      self.actionContext.setMacAddress( ethAddrTuple )

   def delMacAddress( self, args ):
      t0( 'TapAggActionMode delMacAddress' )
      self.removeAction( TapAggPmapCliLib.actSetMacAddress )

   def setTimestampHeader( self ):
      # Timestamp action not supported in action-set config mode
      pass

   def delTimestampHeader( self ):
      # Timestamp action not supported in action-set config mode
      pass

#--------------------------------------------------------------------------------
# policy-map class mode
# command to enter the mode:
# [no] policy-map type tapagg <name>
#--------------------------------------------------------------------------------
class PolicyMapClassModeTapAgg( PolicyMapCliLib.PolicyMapClassMode ):
   name = 'Policy Map TapAgg Class Configuration'

   def __init__( self, parent, session, context ):
      PolicyMapCliLib.PolicyMapClassMode.__init__( self, parent, session,
                                                   context, 'tapagg' )

   def _delAction( self, actionType ):
      policyRuleAction = self.pmapClassContext.policyRuleAction
      policyRuleAction.delAction( actionType )

   def _addAction( self, actionType, action ):
      policyRuleAction = self.pmapClassContext.policyRuleAction
      policyRuleAction.addAction( actionType, action )

   def _getAction( self, actionType ):
      policyRuleAction = self.pmapClassContext.policyRuleAction
      actions = policyRuleAction.actions()
      return actions.get( actionType )

   def _namedActionSetExists( self, name ):
      currentNamedActionSet = TapAggPmapCliLib.getNamedActionSetObj(
            self.pmapClassContext,
            self.pmapName,
            self.cmapName,
            name )
      return currentNamedActionSet is not None

   def onExit( self ):
      tapAggAct = self._getAction( TapAggPmapCliLib.actActionSet )
      if tapAggAct and not tapAggAct.namedActionSet:
         self._delAction( TapAggPmapCliLib.actActionSet )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def timestampActionSupported( self ):
      return True

   def actionSetNames( self ):
      names = []
      tapAggAct = self._getAction( TapAggPmapCliLib.actActionSet )
      if tapAggAct:
         names = tapAggAct.namedActionSet
      return names

   def delIdentityTag( self, args ):
      self._delAction( TapAggPmapCliLib.actSetIdentityTag )

   def setIdentityTag( self, args ):
      portId = args[ 'PORTID' ]
      innerPortId = args.get( 'INNERPORTID', 0 )
      self.delIdentityTag( args )
      idTagTuple = IdentityTuple( portId, innerPortId )
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.idTag.newMember( self.cmapName,
                                                                     UniqueId(),
                                                                     idTagTuple )
      self._addAction( TapAggPmapCliLib.actSetIdentityTag, tapAggAct )

   def delMacAddress( self, args ):
      self._delAction( TapAggPmapCliLib.actSetMacAddress )
 
   def delTimestampHeader( self ):
      self._delAction( TapAggPmapCliLib.actSetTimestampHeader )

   def setMacAddress( self, args ):
      if EthAddrStr( args[ 'DMAC' ] ) == EthAddr():
         self.addError( TapAggCliError[ 'setMacAddress' ] )
         return
      elif 'SMAC' in args and EthAddrStr( args[ 'SMAC' ] ) == EthAddr():
         self.addError( TapAggCliError[ 'setMacAddress' ] )
         return
      destMac = args[ 'DMAC' ]
      srcMac = args.get( 'SMAC', EthAddr() )
      self.delMacAddress( args )
      macAddress = EthAddrTuple( destMac, srcMac )
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.macAddress.newMember(
            self.cmapName, UniqueId(), macAddress )
      self._addAction( TapAggPmapCliLib.actSetMacAddress, tapAggAct )

   def delAggGroupAct( self ):
      self._delAction( TapAggPmapCliLib.actSetAggregationGroup )

   def delNexthopGroupAct( self ):
      self._delAction( TapAggPmapCliLib.actSetNexthopGroup )

   def delAggGroup( self ):
      t0( "delAggGroup" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetAggregationGroup )
      if tapAggAct:
         intfList = list( tapAggAct.aggIntf )
      else:
         intfList = []
      self.delAggGroupAct()
      groupKey = "{}_{}".format( self.pmapName, self.cmapName )
      if groupKey in gv.tapAggPmapConfig.group:
         del gv.tapAggPmapConfig.group[ groupKey ]
      if intfList:
         tapAggAct = gv.tapAggPmapConfig.tapAggActions.aggGroup.newMember(
               self.cmapName, UniqueId() )
         for intf in intfList:
            tapAggAct.aggIntf[ intf ] = True
         self._addAction( TapAggPmapCliLib.actSetAggregationGroup,
                          tapAggAct )

   def delAggIntf( self ):
      t0( "delAggIntf" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetAggregationGroup )
      if tapAggAct:
         currList = list( tapAggAct.aggGroup )
      else:
         currList = []
      self.delAggGroupAct()
      if currList:
         tapAggAct = gv.tapAggPmapConfig.tapAggActions.aggGroup.newMember(
               self.cmapName, UniqueId() )
         for group in currList:
            tapAggAct.aggGroup[ group ] = True
         self._addAction( TapAggPmapCliLib.actSetAggregationGroup,
                           tapAggAct )

   def setAggGroup( self, allowedGroupOp, delete=False ):
      t0( "setAggGroup" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetAggregationGroup )
      if tapAggAct:
         currList = list( tapAggAct.aggGroup )
         intfList = list( tapAggAct.aggIntf )
      else:
         intfList = []
         currList = []
      self.delAggGroupAct()
      groupKey = "{}_{}".format( self.pmapName, self.cmapName )
      if groupKey in gv.tapAggPmapConfig.group:
         del gv.tapAggPmapConfig.group[ groupKey ]
      self.delNexthopGroupAct()
      groupList = allowedGroupOp
      newList = []
      if delete:
         for group in currList:
            if group not in groupList:
               newList.append( group )
      else:
         newList = currList + groupList
      if not newList and not intfList:
         return
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.aggGroup.newMember(
            self.cmapName, UniqueId() )
      pmapGroup = gv.tapAggPmapConfig.group.newMember( groupKey )
      for group in newList:
         tapAggAct.aggGroup[ group ] = True
         pmapGroup.groupName[ group ] = True
      for intf in intfList:
         tapAggAct.aggIntf[ intf ] = True
      self._addAction( TapAggPmapCliLib.actSetAggregationGroup,
                       tapAggAct )

   def setAggIntf( self, allowedRawIntfOp ):
      t0( "setAggIntf" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetAggregationGroup )
      if tapAggAct:
         currList = list( tapAggAct.aggIntf )
         groupList = list( tapAggAct.aggGroup )
      else:
         currList = []
         groupList = []
      self.delAggGroupAct()
      self.delNexthopGroupAct()

      intfOp, intfList = getIntfOpAndList( allowedRawIntfOp )
      newList = []
      if intfOp == 'add':
         newList = currList + intfList
      elif intfOp == 'set':
         newList = intfList
      elif intfOp == 'remove':
         for intf in currList:
            if intf not in intfList:
               newList.append( intf )
      if not newList and not groupList:
         return
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.aggGroup.newMember(
            self.cmapName, UniqueId() )
      for intf in newList:
         tapAggAct.aggIntf[ intf ] = True
      for group in groupList:
         tapAggAct.aggGroup[ group ] = True
      self._addAction( TapAggPmapCliLib.actSetAggregationGroup,
                       tapAggAct )

   def delStripHdrBytes( self, args ):
      self._delAction( TapAggPmapCliLib.actStripHdrBytes )

   def setStripHdrBytes( self, args ):
      # VLAN_RANGE type is GenericRangeList, we use its string representation '1-2'
      hdrInfo = str( args[ 'VLAN_RANGE' ] )
      hdrType = TapAggPmapCliLib.stripHdrDot1q

      self.delStripHdrBytes( args )
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.stripHdrBytes.newMember(
            self.cmapName, UniqueId() )
      stripHdr = TapAggPmapCliLib.stripHdrAction( hdrType, hdrInfo )
      tapAggAct.stripHdrBytes = stripHdr
      self._addAction( TapAggPmapCliLib.actStripHdrBytes, tapAggAct )

   def delHeaderRemove( self, args ):
      self._delAction( TapAggPmapCliLib.actSetHeaderRemove )

   def setHeaderRemove( self, args ):
      headerRemove, err = validateHeaderRemoveConfig( args )
      if err:
         self.addError( err )
         return

      self.delHeaderRemove( args )
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.headerRemove.newMember(
            self.cmapName, UniqueId(), headerRemove )
      self._addAction( TapAggPmapCliLib.actSetHeaderRemove, tapAggAct )

   def delNexthopGroup( self ):
      t0( "delNexthopGroup" )
      self.delNexthopGroupAct()

   def setNexthopGroup( self, allowedNexthopGroupOp, delete=False ):
      t0( "setNexthopGroup" )
      tapAggAct = self._getAction( TapAggPmapCliLib.actSetNexthopGroup )
      if tapAggAct:
         currList = list( tapAggAct.nexthopGroup )
      else:
         currList = []
      self.delAggGroupAct()
      groupKey = "{}_{}".format( self.pmapName, self.cmapName )
      del gv.tapAggPmapConfig.group[ groupKey ]
      self.delNexthopGroupAct()
      nhgList = allowedNexthopGroupOp
      newList = []
      if delete:
         for nexthopGroup in currList:
            if nexthopGroup not in nhgList:
               newList.append( nexthopGroup )
      else:
         newList = currList + nhgList
      if not newList:
         return
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.nexthopGroup.newMember(
            self.cmapName, UniqueId() )
      for nexthopGroup in newList:
         tapAggAct.nexthopGroup[ nexthopGroup ] = True
      self._addAction( TapAggPmapCliLib.actSetNexthopGroup,
                       tapAggAct )

   def setTimestampHeader( self ):
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.timestampHeader.newMember(
            self.cmapName, UniqueId() )
      self._addAction( TapAggPmapCliLib.actSetTimestampHeader, tapAggAct )

   def setActionDrop( self, no ):
      self._delAction( TapAggPmapCliLib.actDrop )
      if not no:
         tapAggAct = gv.tapAggPmapConfig.tapAggActions.drop.newMember(
            self.cmapName, UniqueId() )
         self._addAction( TapAggPmapCliLib.actDrop, tapAggAct )

   def delActionGroup( self, actionName ):
      tapAggAct = self._getAction( TapAggPmapCliLib.actActionSet )
      if not tapAggAct:
         return
      del tapAggAct.namedActionSet[ actionName ]
      if self._namedActionSetExists( actionName ):
         tapAggAct.updatedActionSet[ actionName ] = True

   def setActionGroup( self, actionName ):
      # Create copy of ActionSet
      originalTapAggAct = self._getAction( TapAggPmapCliLib.actActionSet )
      tapAggAct = gv.tapAggPmapConfig.tapAggActions.actionSet.newMember(
            self.cmapName, UniqueId() )
      if originalTapAggAct:
         tapAggAct.copyAction( originalTapAggAct )
         for actionSet in originalTapAggAct.updatedActionSet:
            tapAggAct.updatedActionSet[ actionSet ] = True

      # Create ActionContext and use it to create TapAggActionMode
      actionContext = ActionContext( self.pmapClassContext, actionName, tapAggAct )
      childMode = self.childMode(
             TapAggActionMode, context=actionContext, mapTypeStr=self.mapStr_ )
      self.session_.gotoChildMode( childMode )

#--------------------------------------------------------------------------------
# remove dot1q outer VLAN_RANGE
# ( no | default ) remove dot1q outer ...
#--------------------------------------------------------------------------------
class RemoveDot1QOuterCmd( CliCommand.CliCommandClass ):
   syntax = 'remove dot1q outer VLAN_RANGE'
   noOrDefaultSyntax = 'remove dot1q outer ...'
   data = {
      'remove' : CliCommand.guardedKeyword( 'remove',
         helpdesc='Remove a header',
         guard=guardPmapDot1qRemove ),
      'dot1q' : matcherDot1Q,
      'outer' : matcherOuter,
      'VLAN_RANGE' : CliCommand.Node(
         matcher=MultiRangeRule.MultiRangeMatcher(
            rangeFn=lambda: ( 1, gv.tapAggHwStatus.dot1qRemoveMaxIndex ),
            helpdesc='Specify indices of VLAN tags to be removed',
            noSingletons=False ),
         guard=guardPmapRemoveDot1qRange ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.setStripHdrBytes( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delStripHdrBytes( args )

PolicyMapClassModeTapAgg.addCommandClass( RemoveDot1QOuterCmd )
TapAggActionMode.addCommandClass( RemoveDot1QOuterCmd )

#--------------------------------------------------------------------------------
# set aggregation-group ( { group GROUP } | GROUP ) [ id-tag PORTID
#                                                     [ inner INNERPORTID ] ]
#                                                   [ mac-address dest DMAC
#                                                     [ src SMAC ] ]
#                                                   [ mac timestamp header ]
# ( no | default ) set aggregation-group [ ( { group GROUP } | GROUP )
#                                          [ id-tag ... ] [ mac-address ... ]
#                                          [ mac timestamp header ] ]
#--------------------------------------------------------------------------------
class SetAggregationGroupIdTagCmd( CliCommand.CliCommandClass ):
   syntax = """set aggregation-group ( GROUP | { group GROUP } )
               [ ID_TAG ] [ MAC_ADDRESS ] [ TIMESTAMP ]"""
   noOrDefaultSyntax = """set aggregation-group [ ( { group GROUP } | GROUP )
                          [ id-tag ... ] [ mac-address ... ] 
                          [ TIMESTAMP ] ]"""
   data = {
      'set' : matcherSet,
      'aggregation-group' : 'List of groups to aggregate flow',
      'group' : 'Set tap group for the interface',
      'GROUP' : CliMatcher.DynamicNameMatcher(
         TapAggIntfCli.getGroupList, helpdesc='Group name' ),
      'ID_TAG' : IdTagExprFactory(),
      'id-tag' : matcherIdTag,
      'MAC_ADDRESS' : MacAddressExprFactory(),
      'mac-address' : TapAggIntfCli.matcherMacAddress,
      'TIMESTAMP' : TimestampExprFactory(),
   }

   @staticmethod
   def handler( mode, args ):
      aggGroupAdd = args[ 'GROUP' ]
      if not isinstance( aggGroupAdd, list ):
         aggGroupAdd = [ aggGroupAdd ]
      mode.setAggGroup( aggGroupAdd, delete=False )
      if 'PORTID' in args:
         mode.setIdentityTag( args )
      if 'DMAC' in args:
         mode.setMacAddress( args )
      if 'TIMESTAMP' in args:
         mode.setTimestampHeader()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      aggGroup = args.get( 'GROUP' )
      if aggGroup is None:
         mode.delAggGroup()
      else:
         mode.setAggGroup( aggGroup, delete=True )
      if 'id-tag' in args:
         mode.delIdentityTag( args )
      if 'mac-address' in args:
         mode.delMacAddress( args )
      if 'TIMESTAMP' in args:
         mode.delTimestampHeader()

PolicyMapClassModeTapAgg.addCommandClass( SetAggregationGroupIdTagCmd )
TapAggActionMode.addCommandClass( SetAggregationGroupIdTagCmd )

#--------------------------------------------------------------------------------
# set id-tag PORTID [ inner INNERPORTID ] [ mac-address dest DMAC [ src SMAC ] ]
#                   [ mac timestamp header ]
# ( no | default ) set id-tag ... [ mac-address ... ]
#                                 [ mac timestamp header ]
#--------------------------------------------------------------------------------
class SetIdTagCmd( CliCommand.CliCommandClass ):
   syntax = """set ID_TAG [ MAC_ADDRESS ] [ TIMESTAMP ]"""
   noOrDefaultSyntax = """set id-tag ... [ mac-address ... ] [ TIMESTAMP ]"""
   data = {
      'set' : matcherSet,
      'ID_TAG' : IdTagExprFactory(),
      'id-tag' : matcherIdTag,
      'MAC_ADDRESS' : MacAddressExprFactory(),
      'mac-address' : TapAggIntfCli.matcherMacAddress,
      'TIMESTAMP' : TimestampExprFactory(), 
   }

   @staticmethod
   def handler( mode, args ):
      mode.setIdentityTag( args )
      if 'DMAC' in args:
         mode.setMacAddress( args )
      if 'TIMESTAMP' in args:
         mode.setTimestampHeader()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delIdentityTag( args )
      if 'mac-address' in args:
         mode.delMacAddress( args )
      if 'TIMESTAMP' in args:
         mode.delTimestampHeader()

PolicyMapClassModeTapAgg.addCommandClass( SetIdTagCmd )
TapAggActionMode.addCommandClass( SetIdTagCmd )

#--------------------------------------------------------------------------------
# set mac-address dest DMAC [ src SMAC ]
# ( no | default ) set mac-address ...
#--------------------------------------------------------------------------------
class SetMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'set MAC_ADDRESS'
   noOrDefaultSyntax = 'set mac-address ...'
   data = {
      'set' : matcherSet,
      'MAC_ADDRESS' : MacAddressExprFactory(),
      'mac-address' : TapAggIntfCli.matcherMacAddress,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setMacAddress( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delMacAddress( args )

PolicyMapClassModeTapAgg.addCommandClass( SetMacAddressCmd )
TapAggActionMode.addCommandClass( SetMacAddressCmd )

#--------------------------------------------------------------------------------
# remove header size SIZE [ preserve ethernet ]
# ( no | default ) remove header size <trailing garbage>
#--------------------------------------------------------------------------------
class SetHeaderRemoveCmd( CliCommand.CliCommandClass ):
   syntax = 'remove header size SIZE [ preserve ethernet ]'
   noOrDefaultSyntax = 'remove header size ...'
   data = {
      'remove' : CliCommand.guardedKeyword(
         'remove',
         helpdesc='Remove a header',
         guard=guardTapAggHeaderRemove ),
      'header' : 'Remove a configurable number of bytes after the Ethernet header, '
                 'including any VLAN tags',
      'size' : 'Set the number of bytes to remove',
      'SIZE' : CliMatcher.IntegerMatcher(
         2, 62, helpdesc='The size in bytes (must be even)' ),
      'preserve' : 'Preserve the original header',
      'ethernet' : 'Preserve the original Ethernet header, including VLAN tags '
                   'if any',
   }

   @staticmethod
   def handler( mode, args ):
      mode.setHeaderRemove( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delHeaderRemove( args )

PolicyMapClassModeTapAgg.addCommandClass( SetHeaderRemoveCmd )

#--------------------------------------------------------------------------------
# set interface INTFS [ id-tag PORTID [ inner INNERPORTID ] ]
#                     [ mac-address dest DMAC [ src SMAC ] ]
#                     [ mac timestamp header ]
# ( no | default ) set interface [ INTFS [ id-tag ... ] [ mac-address ... ]
#                                        [ mac timestamp header ] ]
#--------------------------------------------------------------------------------
class SetInterfaceIdTagCmd( CliCommand.CliCommandClass ):
   syntax = """set interface INTFS [ ID_TAG ] [ MAC_ADDRESS ] [ TIMESTAMP ]"""
   noOrDefaultSyntax = """set interface [ INTFS [ id-tag ... ] [ mac-address ... ]
                                                [ TIMESTAMP ] ]"""
   data = {
      'set' : matcherSet,
      'interface' : 'List of interfaces to aggregate flow',
      'INTFS' : matcherIntfRanger,
      'ID_TAG' : IdTagExprFactory(),
      'id-tag' : matcherIdTag,
      'MAC_ADDRESS' : MacAddressExprFactory(),
      'mac-address' : TapAggIntfCli.matcherMacAddress,
      'TIMESTAMP' : TimestampExprFactory(), 
   }

   @staticmethod
   def handler( mode, args ):
      allowedRawIntfOp = args[ 'INTFS' ]
      mode.setAggIntf( ( 'add', allowedRawIntfOp ) )
      if 'PORTID' in args:
         mode.setIdentityTag( args )
      if 'DMAC' in args:
         mode.setMacAddress( args )
      if 'TIMESTAMP' in args:
         mode.setTimestampHeader()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      allowedRawIntf = args.get( 'INTFS' )
      if allowedRawIntf is None:
         mode.delAggIntf()
      else:
         allowedRawIntfOpAndList = getIntfOpAndList( ( 'remove', allowedRawIntf ) )
         mode.setAggIntf( allowedRawIntfOpAndList )
      if 'id-tag' in args:
         mode.delIdentityTag( args )
      if 'mac-address' in args:
         mode.delMacAddress( args )
      if 'TIMESTAMP' in args:
         mode.delTimestampHeader()
 
PolicyMapClassModeTapAgg.addCommandClass( SetInterfaceIdTagCmd )
TapAggActionMode.addCommandClass( SetInterfaceIdTagCmd )

#--------------------------------------------------------------------------------
# set mac timestamp header
# ( no | default ) set mac timestamp header
#--------------------------------------------------------------------------------
class SetMacTimestampHeaderCmd( CliCommand.CliCommandClass ):
   syntax = "set TIMESTAMP"
   noOrDefaultSyntax = syntax 
   data = {
      'set' : matcherSet,
      'TIMESTAMP' : TimestampExprFactory(), 
   }

   @staticmethod
   def handler( mode, args ):
      mode.setTimestampHeader()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delTimestampHeader()

PolicyMapClassModeTapAgg.addCommandClass( SetMacTimestampHeaderCmd )

#--------------------------------------------------------------------------------
# set nexthop-group { GROUP } [ mac timestamp header ]
# ( no | default ) set nexthop-group { GROUP } [ mac timestamp header ]
#--------------------------------------------------------------------------------
class SetNexthopGroupCmd( CliCommand.CliCommandClass ):
   syntax = "set NEXTHOP_GROUPS [ TIMESTAMP ]"
   noOrDefaultSyntax = syntax 
   data = {
      'set' : matcherSet,
      'NEXTHOP_GROUPS' : AggNexthopGroupExprFactory(), 
      'TIMESTAMP' : TimestampExprFactory(), 
   }

   @staticmethod
   def handler( mode, args ):
      nexthopGroups = args.pop( 'NEXTHOP_GROUPS', None )
      mode.setNexthopGroup( nexthopGroups, delete=False )
      if 'TIMESTAMP' in args:
         mode.setTimestampHeader()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nexthopGroups = args.pop( 'NEXTHOP_GROUPS', None )
      if nexthopGroups is None:
         mode.delNexthopGroup()
      else:
         mode.setNexthopGroup( nexthopGroups, delete=True )
      if 'TIMESTAMP' in args:
         mode.delTimestampHeader()

PolicyMapClassModeTapAgg.addCommandClass( SetNexthopGroupCmd )

#-------------------------------------------------------------------------------
# [ no | default ] drop 
#-------------------------------------------------------------------------------
class DropCmd( CliCommand.CliCommandClass ):
   syntax = 'drop'
   noOrDefaultSyntax = syntax
   data = {
      'drop' : CliCommand.Node( matcher=PolicyMapCliLib.matcherDrop,
                                guard=guardDropAction ),
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setActionDrop( no )

   noOrDefaultHandler = handler

PolicyMapClassModeTapAgg.addCommandClass( DropCmd )

#-------------------------------------------------------------------------------
# action ACTION_NAME
# ( no | default ) action ACTION_NAME
#-------------------------------------------------------------------------------
class ActionCmd( CliCommand.CliCommandClass ):
   syntax = 'action ACTION_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'action' : nodeAction, 
      'ACTION_NAME' : CliMatcher.DynamicNameMatcher(
         PolicyMapClassModeTapAgg.actionSetNames,
         helpdesc='Action set name' )
   }

   @staticmethod
   def handler( mode, args ):
      actionName = args.get( 'ACTION_NAME', [] )
      mode.setActionGroup( actionName=actionName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      actionName = args.get( 'ACTION_NAME', [] )
      mode.delActionGroup( actionName=actionName )

PolicyMapClassModeTapAgg.addCommandClass( ActionCmd )

def Plugin( entityManager ):
   gv.tapAggHwStatus = LazyMount.mount( entityManager, 'tapagg/hwstatus',
                                        'TapAgg::HwStatus', 'r' )
   gv.tapAggPmapConfig = ConfigMount.mount( entityManager, 'tapagg/pmapconfig',
                                            'TapAgg::PmapConfig', 'w' )
