#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.TapAggIntfCli as TapAggIntfCli
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.VlanIntfCli import VlanAutoIntfType
from CliPlugin.RecircIntfCli import RecircAutoIntfType
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
from CliPlugin.InternalRecircIntfCli import InternalRecircAutoIntfType
from CliToken.Mac import macMatcherForConfig
from MultiRangeRule import MultiRangeMatcher
import Toggles.MirroringToggleLib

matcherDzGreSwitch = CliMatcher.KeywordMatcher( 'switch',
   helpdesc='Configure DzGRE switch parameters' )
matcherEncapsulation = CliMatcher.KeywordMatcher( 'encapsulation',
   helpdesc='Encapsulation protocols in packet header' )
matcherExclusive = CliMatcher.KeywordMatcher( 'exclusive',
   helpdesc='Exclusive tap aggregation mode' )
matcherFcsError = CliMatcher.KeywordMatcher( 'fcs-error',
   helpdesc='Configure the behavior regarding packets with FCS error' )
matcherFormat = CliMatcher.KeywordMatcher( 'format',
   helpdesc='Format configuration for timestamp header' )
matcherHeader = CliMatcher.KeywordMatcher( 'header',
   helpdesc='Header configuration for timestamp' )
matcherTimestampPlacement = CliMatcher.KeywordMatcher( 'placement',
   helpdesc='Placement configuration for timestamp header' )
matcherTimestampAfter = CliMatcher.KeywordMatcher( 'after',
   helpdesc='Placement of timestamp after specified headers' )
matcherMixed = CliMatcher.KeywordMatcher( 'mixed',
   helpdesc='Mixed tap-agg and switched mode' )
matcherMode = CliMatcher.KeywordMatcher( 'mode',
   helpdesc='Configure tap aggregation mode' )
matcherModule = CliMatcher.KeywordMatcher( 'module', helpdesc='Configure module' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile',
   helpdesc='Configure TCAM profile' )
matcherReplace = CliMatcher.KeywordMatcher( 'replace',
   helpdesc='Replace field configuration for timestamp' )
matcherSourceMac = CliMatcher.KeywordMatcher( 'source-mac',
   helpdesc='Replace source MAC with 48-bit truncated IEEE 1588 timestamp' )
matcherStrip = CliMatcher.KeywordMatcher( 'strip',
   helpdesc='Remove a tag in packet header' )
matcherTapagg = CliMatcher.KeywordMatcher( 'tapagg',
   helpdesc='TapAgg type' )
matcherTimestamp = CliMatcher.KeywordMatcher( 'timestamp',
   helpdesc='Configuration to set global timestamp format in Ethernet header' )
matcherType = CliMatcher.KeywordMatcher( 'type', helpdesc='Specify type' )
matcherLinecard = MultiRangeMatcher(
   # pylint: disable-next=consider-using-f-string
   value=lambda mode, grList: [ ( 'Linecard%d' % i ) for i in grList.values() ],
   rangeFn=lambda: FruCli.rangeFn( 'Linecard' ), noSingletons=False,
   helpdesc='Linecard number', tagLong='Linecard' )
nodeDzGre = CliCommand.guardedKeyword( 'dzgre',
   helpdesc='Configure DzGRE parameters',
   guard=TapAggIntfCli.dzGreDecodeSupportedGuard )
nodeEncapsulation = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'encapsulation', helpdesc='Encapsulation protocols in packet header' ),
   guard=TapAggIntfCli.brVnTagStripGuard )
nodeExclusive = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'exclusive', helpdesc='Exclusive tap aggregation mode' ),
   guard=TapAggIntfCli.tapaggGuard )
nodeFcsError = CliCommand.Node( matcher=matcherFcsError,
   guard=TapAggIntfCli.fcsErrorModeGuard )
nodeFcs = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'fcs', helpdesc='Configure FCS behavior' ),
   guard=TapAggIntfCli.fcsErrorModeGuard )
nodeRunt = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'runt', helpdesc='Configure runt packet behavior' ),
   guard=TapAggIntfCli.runtModeGuard )

nodeNexthopGroup = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
    'nexthop-group', helpdesc='Configure nexthop-group settings' ),
    guard=TapAggIntfCli.guardNexthopGroupTapAgg )
nodeHeader = CliCommand.Node( matcher=matcherHeader,
   guard=TapAggIntfCli.timestampHeaderSupportedGuard )
nodeMixed = CliCommand.Node( matcher=matcherMixed, guard=TapAggIntfCli.tapaggGuard )
nodeProfile = CliCommand.Node( matcher=matcherProfile,
   guard=TapAggIntfCli.tcamProfileGuard )
nodeReplace = CliCommand.Node( matcher=matcherReplace,
   guard=TapAggIntfCli.timestampReplaceSmacGuard )
nodeSourceMac = CliCommand.Node( matcher=matcherSourceMac,
   guard=TapAggIntfCli.timestampReplaceSmacGuard )
nodeStrip = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'strip',
   helpdesc='Remove a tag in packet header' ),
   guard=TapAggIntfCli.brVnTagStripGuard )
nodeTimestamp = CliCommand.Node( matcher=matcherTimestamp,
   guard=TapAggIntfCli.timestampHeaderSupportedGuard )
nodeTimestampPlacement = CliCommand.Node( matcher=matcherTimestampPlacement,
   guard=TapAggIntfCli.timestampHeaderPlacementSupportedGuard )

intfRangeMatcher = IntfCli.intfRangeWithSingleExpression(
   'INTFS', explicitIntfTypes=( EthPhyAutoIntfType,
                                LagAutoIntfType,
                                VlanAutoIntfType,
                                RecircAutoIntfType,
                                SwitchAutoIntfType,
                                InternalRecircAutoIntfType ) )

#--------------------------------------------------------------------------------
# [ no | default ] encapsulation dot1br strip
#--------------------------------------------------------------------------------
class EncapsulationDot1BrStripCmd( CliCommand.CliCommandClass ):
   syntax = 'encapsulation dot1br strip'
   noOrDefaultSyntax = syntax
   data = {
      'encapsulation': nodeEncapsulation,
      'dot1br': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'dot1br',
         helpdesc='802.1 BR tag' ), guard=TapAggIntfCli.brVnTagStripGuard ),
      'strip': nodeStrip,
   }
   handler = TapAggIntfCli.brTagStripConfig
   noOrDefaultHandler = TapAggIntfCli.noBrTagStripConfig

TapAggIntfCli.TapAggConfigMode.addCommandClass( EncapsulationDot1BrStripCmd )

#--------------------------------------------------------------------------------
# [ no | default ] encapsulation vn-tag strip
#--------------------------------------------------------------------------------
class EncapsulationVnTagStripCmd( CliCommand.CliCommandClass ):
   syntax = 'encapsulation vn-tag strip'
   noOrDefaultSyntax = syntax
   data = {
      'encapsulation': nodeEncapsulation,
      'vn-tag': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'vn-tag',
         helpdesc='VN-Tag' ), guard=TapAggIntfCli.brVnTagStripGuard ),
      'strip': nodeStrip,
   }
   handler = TapAggIntfCli.vnTagStripConfig
   noOrDefaultHandler = TapAggIntfCli.noVnTagStripConfig

TapAggIntfCli.TapAggConfigMode.addCommandClass( EncapsulationVnTagStripCmd )

#--------------------------------------------------------------------------------
# [ no | default ] nexthop-group encapsulation gre protocol PROTOCOL
#--------------------------------------------------------------------------------
class NexthopGroupEncapGreProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'nexthop-group encapsulation gre protocol PROTOCOL'
   noOrDefaultSyntax = 'nexthop-group encapsulation gre protocol ...'
   data = {
      'nexthop-group': nodeNexthopGroup,
      'encapsulation': 'Encapsulation settings',
      'gre': 'GRE tunnels',
      'protocol': 'GRE protocol ID for outgoing packets',
      'PROTOCOL': CliMatcher.IntegerMatcher( 0, 0xffff, helpdesc='GRE Protocol' ),
   }
   handler = TapAggIntfCli.setGreProtocol
   noOrDefaultHandler = TapAggIntfCli.setDefaultGreProtocol

TapAggIntfCli.TapAggConfigMode.addCommandClass( NexthopGroupEncapGreProtocolCmd )

#--------------------------------------------------------------------------------
# mac fcs-error ( correct | discard | pass-through )
# [ no | default ] mac fcs-error ...
#--------------------------------------------------------------------------------
class MacFcsErrorCmd( CliCommand.CliCommandClass ):
   syntax = 'mac fcs-error ( correct | discard | pass-through )'
   noOrDefaultSyntax = 'mac fcs-error ...'
   data = {
      'mac': macMatcherForConfig,
      'fcs-error': nodeFcsError,
      'correct': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'correct',
         helpdesc='Replace the FCS with a correct one (default behavior)' ),
         guard=TapAggIntfCli.fcsErrorModeGuard ),
      'discard': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'discard',
         helpdesc='Discard the packets' ), guard=TapAggIntfCli.fcsErrorModeGuard ),
      'pass-through': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'pass-through',
         helpdesc='Let the packets pass through with their original FCS' ),
         guard=TapAggIntfCli.fcsErrorModePassthroughGuard ),
   }
   handler = TapAggIntfCli.setFcsErrorMode
   noOrDefaultHandler = TapAggIntfCli.noFcsErrorMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacFcsErrorCmd )

#--------------------------------------------------------------------------------
# mac fcs append
# [ no | default ] mac fcs ...
#--------------------------------------------------------------------------------
class MacFcsCmd( CliCommand.CliCommandClass ):
   syntax = 'mac fcs append'
   noOrDefaultSyntax = 'mac fcs ...'
   data = {
      'mac': macMatcherForConfig,
      'fcs': nodeFcs,
      'append': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'append',
         helpdesc='Append a correct FCS after the original FCS' ),
         alias='ACTION',
         guard=TapAggIntfCli.fcsModeGuard ),
   }
   handler = TapAggIntfCli.setFcsMode
   noOrDefaultHandler = TapAggIntfCli.noFcsMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacFcsCmd )

# --------------------------------------------------------------------------------
# mac runt ( drop | pad | pass-through )
# [ no | default ] mac runt ...
# --------------------------------------------------------------------------------
class MacRuntCmd( CliCommand.CliCommandClass ):
   syntax = 'mac runt MODE'
   noOrDefaultSyntax = 'mac runt ...'
   data = {
      'mac': macMatcherForConfig,
      'runt': nodeRunt,
      'MODE': CliMatcher.EnumMatcher( {
         'drop': 'Drop packets that are below the runt threshold size',
         'pad': 'Pad packets that are below the runt threshold size',
         'pass-through': 'Passthrough runt packets as is',
      } )
   }
   handler = TapAggIntfCli.setRuntMode
   noOrDefaultHandler = TapAggIntfCli.noRuntMode

if Toggles.MirroringToggleLib.toggleJ2TapAggRuntFramePassthroughEnabled():
   TapAggIntfCli.TapAggConfigMode.addCommandClass( MacRuntCmd )

#--------------------------------------------------------------------------------
# mac timestamp header placement after ( source-mac | l2 )
# [ no | default ] mac timestamp header placement ...
#--------------------------------------------------------------------------------
class MacTimestampHeaderPlacementAfterCmd( CliCommand.CliCommandClass ):
   syntax = 'mac timestamp header placement after ( source-mac | l2 )'
   noOrDefaultSyntax = 'mac timestamp header placement ...'
   data = {
      'mac': macMatcherForConfig,
      'timestamp': nodeTimestamp,
      'header': nodeHeader,
      'placement': nodeTimestampPlacement,
      'after': matcherTimestampAfter,
      'source-mac': 'Placement of timestamp relative to source MAC',
      'l2': 'Placement of timestamp relative to L2 headers',
   }
   handler = TapAggIntfCli.setTimestampHeaderPlacement
   noOrDefaultHandler = TapAggIntfCli.setDefaultTimestampHeaderPlacement

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacTimestampHeaderPlacementAfterCmd )

#--------------------------------------------------------------------------------
# mac timestamp header format ( 64-bit | 48-bit )
# [ no | default ] mac timestamp header format ...
#--------------------------------------------------------------------------------
class MacTimestampHeaderFormatCmd( CliCommand.CliCommandClass ):
   syntax = 'mac timestamp header format ( 64-bit | 48-bit )'
   noOrDefaultSyntax = 'mac timestamp header format ...'
   data = {
      'mac': macMatcherForConfig,
      'timestamp': nodeTimestamp,
      'header': nodeHeader,
      'format': matcherFormat,
      '64-bit': '64-bit IEEE 1588 timestamp format',
      '48-bit': '48-bit truncated IEEE 1588 timestamp format',
   }
   handler = TapAggIntfCli.setTimestampHeaderFormat
   noOrDefaultHandler = TapAggIntfCli.setDefaultTimestampHeaderFormat

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacTimestampHeaderFormatCmd )

#--------------------------------------------------------------------------------
# mac timestamp header eth-type <eth-type>
# [ no | default ] mac timestamp header eth-type ...
#--------------------------------------------------------------------------------

class MacTimestampHeaderEthTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'mac timestamp header eth-type ETHTYPE'
   noOrDefaultSyntax = 'mac timestamp header eth-type ...'
   data = {
      'mac': macMatcherForConfig,
      'timestamp': nodeTimestamp,
      'header': nodeHeader,
      'eth-type' : CliCommand.Node(
                     matcher=CliMatcher.KeywordMatcher(
                        'eth-type',
                        helpdesc='The two-bytes Timestamp header EtherType' ),
                     guard=TapAggIntfCli.timestampCustomEthTypeGuard ),
      'ETHTYPE': CliMatcher.IntegerMatcher( 0, 0xffff, helpdesc='EtherType value' ),
   }
   handler = TapAggIntfCli.setTimestampHeaderEthType
   noOrDefaultHandler = TapAggIntfCli.setDefaultTimestampHeaderEthType

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacTimestampHeaderEthTypeCmd )

#--------------------------------------------------------------------------------
# mac timestamp replace source-mac
# [ no | default ] mac timestamp replace source-mac ...
#--------------------------------------------------------------------------------
class MacTimestampReplaceSourceMacCmd( CliCommand.CliCommandClass ):
   syntax = 'mac timestamp replace source-mac'
   noOrDefaultSyntax = 'mac timestamp replace source-mac ...'
   data = {
      'mac': macMatcherForConfig,
      'timestamp': nodeTimestamp,
      'replace': nodeReplace,
      'source-mac': nodeSourceMac,
   }
   handler = TapAggIntfCli.timestampReplaceSmacConfig
   noOrDefaultHandler = TapAggIntfCli.setDefaultTimestampReplaceSmac

TapAggIntfCli.TapAggConfigMode.addCommandClass( MacTimestampReplaceSourceMacCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mode exclusive no-errdisable INTFS
#--------------------------------------------------------------------------------
class ModeExclusiveNoErrdisableEthintfCmd( CliCommand.CliCommandClass ):
   syntax = 'mode exclusive no-errdisable INTFS'
   noOrDefaultSyntax = syntax
   data = {
      'mode': matcherMode,
      'exclusive': nodeExclusive,
      'no-errdisable': 'Do not errdisable incompatible interfaces',
      'INTFS': intfRangeMatcher,
   }
   handler = TapAggIntfCli.setTapAggNoErrdisableIntf
   noOrDefaultHandler = TapAggIntfCli.noTapAggNoErrdisableIntf

TapAggIntfCli.TapAggConfigMode.addCommandClass( ModeExclusiveNoErrdisableEthintfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] protocol lldp trap
#--------------------------------------------------------------------------------
class ProtocolLldpTrapCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol lldp trap'
   noOrDefaultSyntax = syntax
   data = {
      'protocol': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'protocol',
         helpdesc='Specify protocol of matched packets' ),
         guard=TapAggIntfCli.lldpReceiveConfigurableGuard ),
      'lldp': 'Set action on Link Layer Discovery Protocol (LLDP) packets',
      'trap': 'Enable protocol packet redirect to CPU',
   }

   @staticmethod
   def handler( mode, args ):
      TapAggIntfCli.lldpReceiveConfig( mode, no=None )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      TapAggIntfCli.lldpReceiveConfig( mode, no=True )

TapAggIntfCli.TapAggConfigMode.addCommandClass( ProtocolLldpTrapCmd )

#--------------------------------------------------------------------------------
# [ no | default ] service-policy type tapagg mac access-list match ip
#--------------------------------------------------------------------------------
class ServicePolicyTypeTapaggMacAccessListMatchIpCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy type tapagg mac access-list match ip'
   noOrDefaultSyntax = syntax
   data = {
      'service-policy': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'service-policy', helpdesc='Configure Service Policy' ),
         guard=TapAggIntfCli.macAclMatchIpConfigurableGuard ),
      'type': matcherType,
      'tapagg': matcherTapagg,
      'mac': 'MAC',
      'access-list': 'MAC access-list',
      'match': 'Specify packets to match',
      'ip': 'Match IP packets',
   }
   handler = TapAggIntfCli.setTrafficSteeringMacAclMatchIp
   noOrDefaultHandler = TapAggIntfCli.noTrafficSteeringMacAclMatchIp

TapAggIntfCli.TapAggConfigMode.addCommandClass(
   ServicePolicyTypeTapaggMacAccessListMatchIpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] truncation size ...
#--------------------------------------------------------------------------------

# Actual "truncation size" cmd is registered by platforms
# See SandDanzTapAggConfigMode.py
class TruncationSizeCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'truncation size ...'
   data = {
      'truncation': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'truncation',
         helpdesc='Global truncation settings' ),
         guard=TapAggIntfCli.truncationGlobalSizeGuard ),
      'size': 'Configure global truncation size',
   }
   noOrDefaultHandler = TapAggIntfCli.setDefaultGlobalTruncationSize

TapAggIntfCli.TapAggConfigMode.addCommandClass( TruncationSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switch identity IDENTITY
#--------------------------------------------------------------------------------
class SwitchIdentityCmd( CliCommand.CliCommandClass ):
   syntax = 'switch identity IDENTITY'
   noOrDefaultSyntax = 'switch identity ...'
   data = {
      'switch': matcherDzGreSwitch,
      'identity': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'identity',
         helpdesc='Configure switch identity'),
         guard=TapAggIntfCli.dzGreEncodeSupportedGuard ),
      'IDENTITY': CliMatcher.IntegerMatcher( 1, 4095, helpdesc='Switch identity' ),
   }
   handler = TapAggIntfCli.setDzGreSwitchId
   noOrDefaultHandler = TapAggIntfCli.setDefaultDzGreSwitchId

TapAggIntfCli.TapAggConfigMode.addCommandClass( SwitchIdentityCmd )

#--------------------------------------------------------------------------------
# mode exclusive
# [ no | default ] mode [ exclusive ]
#--------------------------------------------------------------------------------
class ModeExclusiveCmd( CliCommand.CliCommandClass ):
   syntax = 'mode exclusive'
   noOrDefaultSyntax = 'mode [ exclusive ]'
   data = {
      'mode': 'Configure tap aggregation mode',
      'exclusive': nodeExclusive,
   }
   handler = TapAggIntfCli.setMode
   noOrDefaultHandler = TapAggIntfCli.noMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( ModeExclusiveCmd )

#--------------------------------------------------------------------------------
# mode exclusive profile ( TAPAGGPROFILES | USERPROFILE )
# ( no | default ) mode [ exclusive ] profile ( TAPAGGPROFILES | USERPROFILE )
#--------------------------------------------------------------------------------
class ModeExclusiveProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'mode exclusive profile ( TAPAGGPROFILES | USERPROFILE )'
   noOrDefaultSyntax = 'mode [ exclusive ] profile ( TAPAGGPROFILES | USERPROFILE )'
   data = {
      'mode': matcherMode,
      'exclusive': nodeExclusive,
      'profile': nodeProfile,
      'TAPAGGPROFILES': CliMatcher.DynamicKeywordMatcher(
         TapAggIntfCli.allTcamProfiles ),
      'USERPROFILE': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
         helpdesc='User-Defined TCAM Profile', helpname='WORD' ),
   }
   handler = TapAggIntfCli.setMode
   noOrDefaultHandler = TapAggIntfCli.noMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( ModeExclusiveProfileCmd )

#--------------------------------------------------------------------------------
# mode mixed module CARDNUMBERS
# ( no | default ) mode mixed module CARDNUMBERS ...
#--------------------------------------------------------------------------------
class ModeMixedModuleCardnumbersCmd( CliCommand.CliCommandClass ):
   syntax = 'mode mixed module CARDNUMBERS'
   noOrDefaultSyntax = 'mode mixed module CARDNUMBERS ...'
   data = {
      'mode': matcherMode,
      'mixed': nodeMixed,
      'module': matcherModule,
      'CARDNUMBERS': matcherLinecard,
   }
   handler = TapAggIntfCli.setMode
   noOrDefaultHandler = TapAggIntfCli.noMode

TapAggIntfCli.TapAggConfigMode.addCommandClass( ModeMixedModuleCardnumbersCmd )

#--------------------------------------------------------------------------------
# mode mixed module CARDNUMBERS profile ( TAPAGGPROFILES | USERPROFILE )
#--------------------------------------------------------------------------------
class ModeMixedModuleCardnumbersProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'mode mixed module CARDNUMBERS profile ' + \
      '( TAPAGGPROFILES | USERPROFILE )'
   data = {
      'mode': matcherMode,
      'mixed': nodeMixed,
      'module': matcherModule,
      'CARDNUMBERS': matcherLinecard,
      'profile': nodeProfile,
      'TAPAGGPROFILES': CliMatcher.DynamicKeywordMatcher(
         TapAggIntfCli.allTcamProfiles ),
      'USERPROFILE': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
         helpdesc='User-Defined TCAM Profile', helpname='WORD' ),
   }
   handler = TapAggIntfCli.setMode

TapAggIntfCli.TapAggConfigMode.addCommandClass(
   ModeMixedModuleCardnumbersProfileCmd )

#--------------------------------------------------------------------------------
# identity map type dzgre
# [ no | default ] identity map type dzgre
#--------------------------------------------------------------------------------
class IdentityMapTypeDzGreCmd( CliCommand.CliCommandClass ):
   syntax = 'identity map type dzgre'
   noOrDefaultSyntax = syntax
   data = {
      'identity' : 'Configure ID tagging',
      'map' : 'Configure ID tagging map',
      'type' : matcherType,
      'dzgre' : nodeDzGre,
   }
   handler = TapAggIntfCli.gotoIdentityMapDzGreMode
   noOrDefaultHandler = TapAggIntfCli.noIdentityMapDzGreConfig

TapAggIntfCli.TapAggConfigMode.addCommandClass( IdentityMapTypeDzGreCmd )
