#!/usr/bin/env python3
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

'''CLI commands for configuring MLAG.'''

from __future__ import absolute_import, division, print_function
import CliParser, BasicCli, LazyMount
import ConfigMount, Tracing
from CliPlugin import LagCliLib
from CliPlugin.MlagWarningCli import ReloadDelayLacpConfigWorry
from CliMode.Mlag import MlagMode
import Tac
# pylint: disable-next=relative-beyond-top-level
from .MlagWarningCli import ( reloadDelayLacpConfigMsg, addMlagShutdownWarnings,
      addMlagHitfulReconfigureWarnings )
from TypeFuture import TacLazyType
import Toggles.MlagToggleLib

IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )
IpGenAddrWithMask = Tac.Type( 'Arnet::IpGenAddrWithMask' )
Af = Tac.Type( "Arnet::AddressFamily" )

mlagConfig = None
ipConfig = None
ip6Config = None
ethPhyIntfConfigDir = None
lagConfigDir = None
cliConfig = None
mlagConfigurationState = None
mlagHwStatus = None
aclTimeoutConfig = None

tacFastMacRedirectionConfig = TacLazyType( "Mlag::FastMacRedirectionConfig" )

t0 = Tracing.trace0

def fastMacRedirectConfigurableGuard( mode, token ):
   if mlagHwStatus.fastMacRedirectionConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def egressFilterInterlockConfigurableGuard( mode, token ):
   if mlagHwStatus.egressFilterInterlockConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def updateMlagConfigState( mode ):
   # Update ConfigurationState to indicate whether Mlag has been configured and 
   # how the configuration was applied. This is to avoid going into reload delay 
   # when Mlag configuration is applied for the first time during runtime.
   if mode.session_.startupConfig() and mlagConfig.configured():
      mlagConfigurationState.state = "configuredStartup"
   elif mlagConfig.configured():
      mlagConfigurationState.state = "configuredCli"
   else:
      mlagConfigurationState.state = "unknown"

ReloadDelay = Tac.Type( "Mlag::ReloadDelay" )

#-------------------------------------------------------------------------------
# config-mlag mode
#-------------------------------------------------------------------------------
class ConfigMlagMode( MlagMode, BasicCli.ConfigModeBase ):
   name = "MLAG configuration"

   def __init__( self, parent, session ):
      MlagMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMlagMode( mode, args ):
   childMode = mode.childMode( ConfigMlagMode )
   mode.session_.gotoChildMode( childMode )

def noMlagMode( mode, args ):
   defaultMlagConfig = Tac.newInstance( 'Mlag::Config', '' )
   for attr in mlagConfig.tacType.attributeQ:
      # Skipping the below attributes:
      # 1. 'intfConfig': It is a collection of Port-Channel interfaces
      # which have 'mlag <num>' in their 'interface Port-Channel<num>'
      # configuration. Since 'interface Port-Channel<num>' configuration is 
      # outside of 'mlag configuration' we do not want to purge it here.   
      # 2. Tacc related attributes.
      if attr.name in ( 'intfConfig', 'parent', 'parentAttrName', 'isNondestructing',
            'entity', 'name' ):
         continue
      if attr.writable:
         if attr.isCollection:
            getattr( mlagConfig, attr.name ).clear()
         else:
            setattr( mlagConfig, attr.name,
                     getattr( defaultMlagConfig, attr.name ) )
   updateMlagConfigState( mode )

#-------------------------------------------------------------------------------
# [no|default] shutdown
#-------------------------------------------------------------------------------
def setEnabled( mode, args ):
   mlagConfig.enabled = True
   updateMlagConfigState( mode )

def noEnabled( mode, args ):
   addMlagShutdownWarnings( mode )
   mlagConfig.enabled = False
   updateMlagConfigState( mode )

#-------------------------------------------------------------------------------
# [no] domain-id
#-------------------------------------------------------------------------------
def setDomainId( mode, args ):
   addMlagHitfulReconfigureWarnings( mode )
   mlagConfig.domainId = args.get( 'DOMAINID', '' )
   updateMlagConfigState( mode )

def isPeerAddressInSubnet( addrWithMask, peerAddress ):
   return not addrWithMask.isAddrZero and not peerAddress.isAddrZero and\
          addrWithMask.contains( peerAddress )

# This function checks for peer address and returns error if one of the
# following is true
#  - peer address is interface's primary address
#  - peer address is interface's secondary address
#  - peer address does not belong to subnet of both primary and secondary
#    addresses
def mlagGetIntfError( intfName, peerAddress ):
   isIpv6 = peerAddress.af == Af.ipv6
   inSubnet = False
   if isIpv6:
      if intfName not in ip6Config.intf:
         errMsg = "Peer address must be in local-interface's subnet"
         return errMsg
      ip6IntfConfig = ip6Config.intf[ intfName ]

      for addr in ip6IntfConfig.addr:
         intfAddrWithMask = IpGenAddrWithMask( addr.stringValue )
         if intfAddrWithMask.v6AddrWithMask.address == peerAddress.v6Addr:
            errMsg = "Peer address cannot be local interface address"
            return errMsg
         inSubnet = isPeerAddressInSubnet( intfAddrWithMask, peerAddress )
         if inSubnet:
            return ""
   else:
      if intfName not in ipConfig.ipIntfConfig:
         errMsg = "Peer address must be in local-interface's subnet"
         return errMsg
      ipIntfConfig = ipConfig.ipIntfConfig[ intfName ]
      intfAddrWithMask = IpGenAddrWithMask( str( ipIntfConfig.addrWithMask ) )

      if intfAddrWithMask.v4AddrWithMask.address == peerAddress.v4Addr:
         errMsg = "Peer address cannot be local interface address"
         return errMsg

      inSubnet = isPeerAddressInSubnet( intfAddrWithMask, peerAddress )
      if inSubnet:
         return ""
      
      for ipAddrWithMask in ipIntfConfig.secondaryWithMask:
         intfAddrWithMask = IpGenAddrWithMask( str( ipAddrWithMask ) )
         if intfAddrWithMask.v4AddrWithMask.address == peerAddress.v4Addr:
            errMsg = "Peer address cannot be local interface address"
            return errMsg
         inSubnet = isPeerAddressInSubnet( intfAddrWithMask, peerAddress )
         if inSubnet:
            return ""

   errMsg = "Peer address must be in local-interface's subnet"
   return errMsg

#-------------------------------------------------------------------------------
# check if local interface has IP and peer address is in the subnet of local intf
#-------------------------------------------------------------------------------
def validatePeerAndLocalIntf( mode, localIntfId, peerAddress ):
   # warn about configuration if none of the below is true:
   #  - No local interface is specified but peer address is valid
   #  - Local interface has valid IP address but no peer address given
   #  - Local interface has valid IP address and Peer address is not same as local
   #    interface primary/secondary address. Peer address belongs to the same subnet
   #    as local interface primary (or) secondary address
   if not localIntfId or peerAddress.isAddrZero:
      return
   ipIntfConfig = ipConfig.ipIntfConfig.get( localIntfId )
   ip6IntfConfig = ip6Config.intf.get( localIntfId )

   if not ( ipIntfConfig or ip6IntfConfig ):
      mode.addWarning( "Local interface must have an IP address configured" )
   else:
      # see if there is any warning for peerAddress and local interface
      errMsg = mlagGetIntfError( localIntfId, peerAddress )
      if errMsg:
         mode.addWarning( errMsg )

#-------------------------------------------------------------------------------
# [no] local-interface
#-------------------------------------------------------------------------------
def setLocalInterface( mode, args ):
   intf = args[ 'VLAN_INTF' ]
   if intf.name not in intf.intfConfigDir.intfConfig:
      mode.addWarning( "Interface %s not configured" % intf.name )
   else:
      validatePeerAndLocalIntf( mode, intf.name, mlagConfig.peerAddress )
   addMlagHitfulReconfigureWarnings( mode )
   mlagConfig.localIntfId = intf.name
   updateMlagConfigState( mode )

def noLocalInterface( mode, args ):
   addMlagHitfulReconfigureWarnings( mode )
   mlagConfig.localIntfId = ""
   updateMlagConfigState( mode )

#-------------------------------------------------------------------------------
# [no] peer-address
#-------------------------------------------------------------------------------
def setPeerAddress( mode, args):
   if Toggles.MlagToggleLib.toggleMlagIPv6Enabled():
      peerAddress = args[ 'PEERADDRESS' ]
   else:
      peerAddress = IpGenAddr( args[ 'PEERADDRESS' ] )

   validatePeerAndLocalIntf( mode, mlagConfig.localIntfId, peerAddress )
   addMlagHitfulReconfigureWarnings( mode )
   mlagConfig.peerAddress = peerAddress
   mlagConfig.peerAddressConfigured = not peerAddress.isAddrZero
   updateMlagConfigState( mode )

def noPeerAddress( mode, args ):
   addMlagHitfulReconfigureWarnings( mode )
   mlagConfig.peerAddress = IpGenAddr( '0.0.0.0' )
   mlagConfig.peerAddressConfigured = False
   updateMlagConfigState( mode )

#-------------------------------------------------------------------------------
# [no|default] peer-address heartbeat <IP> [vrf <VRF>]
#-------------------------------------------------------------------------------
def setHeartbeatPeerAddress( mode, args ):
   if Toggles.MlagToggleLib.toggleMlagDPDIPv6Enabled():
      peerAddress = args[ 'HEARTBEAT' ]
      
   else:
      peerAddress = IpGenAddr( args[ 'HEARTBEAT' ] )
   vrfName = args.get( 'VRF' )
   if vrfName == 'default':
      vrfName = ''
   mlagConfig.heartbeatPeerAddress = Tac.Value( "Mlag::PeerAddressAndVrf",
                                                peerAddress, vrfName or '' )
   mlagConfig.heartbeatPeerAddressConfigured = not peerAddress.isAddrZero

def noHeartbeatPeerAddress( mode, args ):
   mlagConfig.heartbeatPeerAddress = Tac.Value( "Mlag::PeerAddressAndVrf",
                                                IpGenAddr( '0.0.0.0' ), "" )
   mlagConfig.heartbeatPeerAddressConfigured = False

#-------------------------------------------------------------------------------
# dual-primary detection delay <SECONDS> [action errdisable all-interfaces]
# [no|default] dual-primary detection
#-------------------------------------------------------------------------------
def setDualPrimaryDetection( mode, args ):
   delay = args[ 'DELAY' ]
   action = 'dualPrimaryActionErrdisableAllInterfaces' if 'action' in args else None
   mlagConfig.dualPrimaryDetectionDelay = delay
   mlagConfig.dualPrimaryAction = action or 'dualPrimaryActionNone'

def noDualPrimaryDetection( mode, args ):
   mlagConfig.dualPrimaryAction = 'dualPrimaryActionNone'
   mlagConfig.dualPrimaryDetectionDelay = 0

#-------------------------------------------------------------------------------
# dual-primary recovery delay mlag <SECONDS> non-mlag <SECONDS>
# [no|default] dual-primary recovery delay
#-------------------------------------------------------------------------------
def setDualPrimaryRecoveryDelay( mode, args ):
   mlagDelay = args[ 'DELAY1' ]
   nonMlagDelay = args[ 'DELAY2' ]

   mlagConfig.dualPrimaryMlagRecoveryDelay = mlagDelay
   mlagConfig.dualPrimaryNonMlagRecoveryDelay = nonMlagDelay

def noDualPrimaryRecoveryDelay( mode, args ):
   mlagConfig.dualPrimaryMlagRecoveryDelay = 0
   mlagConfig.dualPrimaryNonMlagRecoveryDelay = 0

#-------------------------------------------------------------------------------
# [no] peer-link
#-------------------------------------------------------------------------------
def validPeerLink( intf, mode ):
   config = intf.config()
   if not config:
      mode.addError( "Interface %s not configured" % intf.name )
      return False

   # check if the proposed peer-link is not configured with
   # the available unidirectionalLinkMode's.
   if not Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( intf.name ):
      if config.unidirectionalLinkMode != 'uniLinkModeDisabled':
         if config.unidirectionalLinkMode == 'uniLinkModeSendOnly':
            linkMode = 'unidirectional send-only'
         elif config.unidirectionalLinkMode == 'uniLinkModeReceiveOnly':
            linkMode = 'unidirectional receive-only'
         else:
            linkMode = 'unidirectional send-receive'
         mode.addError( "%s is configured with %s" % ( intf.name, linkMode ) )
         return False

   # check if Port-Channel already been configured as mlag interface.
   intfConfig = mlagConfig.intfConfig
   name = intf.name
   if name in intfConfig:
      mode.addError( "%s already configured with MLAG %s" % ( name,
                                                              intfConfig[ name ] ) )
      return False

   # check if the proposed peer-link is already a member of a port-channel
   if name in lagConfigDir.phyIntf and lagConfigDir.phyIntf[ name ].lag:
      channel = lagConfigDir.phyIntf[ name ].lag.intfId
      mode.addError( "%s already configured as member of %s" % ( name, channel ) )
      return False

   if config.intfId in ethPhyIntfConfigDir.intfConfig:
      if ( config.rxFlowcontrol == 'flowControlConfigOn' or
         config.txFlowcontrol == 'flowControlConfigOn' ):
         mode.addError( '%s has flow control configured' % name )
         return False

   # MLAG: check that no member has flowControl configured
   flowPorts = []
   for member in LagCliLib.channelPorts( mode, config.intfId, lacpOnly=False,
                                         status=Tac.Type( "Lag::LagStatusFilter" ).\
                                         filterOnActiveAndInactive,
                                         useLagConfig=True ):
      ethPhyIntfConfig = ethPhyIntfConfigDir.intfConfig.get( member )
      if ethPhyIntfConfig and \
         ( ethPhyIntfConfig.rxFlowcontrol == 'flowControlConfigOn' or
           ethPhyIntfConfig.txFlowcontrol == 'flowControlConfigOn' ):
         flowPorts.append(member)
   if flowPorts:
      mode.addError( 'Flow control is configured on %s' % ', '.join( flowPorts ) )
      return False

   sic = cliConfig.switchIntfConfig.get( name )
   invalidModeMapping = { 'dot1qTunnel' : 'dot1q-tunnel',
                          'tap' : 'tap',
                          'tool' : 'tool' }
   if sic and sic.switchportMode in invalidModeMapping:
      mode.addError( '%s is configured in %s mode' % 
                     ( name, invalidModeMapping[ sic.switchportMode ] ) )
      return False

   return True

def setPeerLink( mode, args ):
   intf = args[ 'ETHINTF' ]
   if validPeerLink( intf, mode ):
      addMlagHitfulReconfigureWarnings( mode )
      mlagConfig.peerLinkIntfId = intf.name
      updateMlagConfigState( mode )

def noPeerLink( mode, args ):
   addMlagHitfulReconfigureWarnings( mode )
   mlagConfig.peerLinkIntfId = ""
   updateMlagConfigState( mode )

# This function checks the Mlag Configuration and adds warning if LACP Standby 
# is configured and the Non-Mlag reload delay < Mlag reload delay
def checkMlagReloadDelayConfig( mode, reloadDelayMlag, reloadDelayNonMlag ):
   configWorry = ReloadDelayLacpConfigWorry( mlagConfig ) 
   shouldBeWorried = configWorry.shouldBeWorried()                          
   if shouldBeWorried:
      mode.addWarning( reloadDelayLacpConfigMsg %
                       ( reloadDelayNonMlag, reloadDelayMlag ) )

#-------------------------------------------------------------------------------
# reload-delay [mlag|non-mlag] {<seconds>|infinity}
#-------------------------------------------------------------------------------
def setReloadDelay( mode, args ):
   delay = args.get( 'RELOAD_DELAY', mlagConfig.reloadDelayInfinity )
   checkMlagReloadDelayConfig( mode, delay, mlagConfig.reloadDelayNonMlag.delay )
   mlagConfig.reloadDelay = ReloadDelay( "reloadDelayConfigured", delay )

def setReloadDelayMlag( mode, args ):
   setReloadDelay( mode, args )
   mlagConfig.reloadDelayMlagConfigured = True

def setReloadDelayNonMlag( mode, args ):
   delay = args.get( 'RELOAD_DELAY', mlagConfig.reloadDelayInfinity )
   mlagConfig.reloadDelayNonMlag = ReloadDelay( "reloadDelayConfigured", delay )
   checkMlagReloadDelayConfig( mode, mlagConfig.reloadDelay.delay, delay )

def noReloadDelayMlag( mode, args ):
   mlagConfig.reloadDelay = ReloadDelay()
   checkMlagReloadDelayConfig( mode, mlagConfig.reloadDelay.delay,
                               mlagConfig.reloadDelayNonMlag.delay)
   mlagConfig.reloadDelayMlagConfigured = False

def noReloadDelayNonMlag( mode, args ):
   mlagConfig.reloadDelayNonMlag = ReloadDelay()
   checkMlagReloadDelayConfig( mode, mlagConfig.reloadDelay.delay,
                               mlagConfig.reloadDelayNonMlag.delay)

#-------------------------------------------------------------------------------
# reload-delay mode lacp standby
#-------------------------------------------------------------------------------
def setLacpStandby( mode, args ):
   mlagConfig.lacpStandby = True
   checkMlagReloadDelayConfig( mode, mlagConfig.reloadDelay.delay, 
                               mlagConfig.reloadDelayNonMlag.delay )

#-------------------------------------------------------------------------------
# [ no | default ] inactive mac-address destination peer-link [ direct | indirect ]
#-------------------------------------------------------------------------------
def setFastMacRedirect( mode, args ):
   if args[ 'TRAFFIC_DIRECT' ] == 'indirect':
      mlagConfig.fastMacRedirectionConfig = \
         tacFastMacRedirectionConfig.fastMacRedirectionConfigured
   else:
      mlagConfig.fastMacRedirectionConfig = \
         tacFastMacRedirectionConfig.fastMacRedirectionUnconfigured
   
def defaultFastMacRedirectConfig( mode, args ):
   mlagConfig.fastMacRedirectionConfig = \
      tacFastMacRedirectionConfig.defaultFastMacRedirection

#-------------------------------------------------------------------------------
# [no|default] interface mlag interlock filter egress
#-------------------------------------------------------------------------------
def setMlagIntfEgressAclInterlock( mode, args ):
   mlagConfig.mlagIntfEgressAclInterlockConfigured = True

def defaultMlagIntfEgressAclInterlock( mode, args ):
   mlagConfig.mlagIntfEgressAclInterlockConfigured = False

#-------------------------------------------------------------------------------
# clear mlag interface interlock counters
#-------------------------------------------------------------------------------
def clearMlagInterlockCounters( mode, args ):
   aclTimeoutConfig.clearCounters += 1

#-------------------------------------------------------------------------------
# Mounts
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global mlagConfig, ipConfig, ip6Config, lagConfigDir, cliConfig
   global ethPhyIntfConfigDir
   global mlagConfigurationState
   global mlagHwStatus
   global aclTimeoutConfig
   if Toggles.MlagToggleLib.toggleMlagL2SubinterfacesEnabled():
      mlagConfig = LazyMount.mount( entityManager, 'mlag/input/config/cli',
                                    'Mlag::Config', 'w' )
   else:
      mlagConfig = LazyMount.mount( entityManager, 'mlag/config',
                                    'Mlag::Config', 'w' )

   mlagConfigurationState = LazyMount.mount( entityManager,
                                             "mlag/configurationState",
                                             "Mlag::ConfigurationState", "w" )
   mlagHwStatus = LazyMount.mount( entityManager, "mlag/hardware/status",
                                   "Mlag::Hardware::Status", "r" )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ip6Config = LazyMount.mount( entityManager, "ip6/config", "Ip6::Config", "r" )
   ConfigMount.mount( entityManager, "interface/config/eth/phy/slice",
                    "Tac::Dir", "wi" )
   ethPhyIntfConfigDir = LazyMount.mount( entityManager,
                                          "interface/config/eth/phy/all",
                                          "Interface::AllEthPhyIntfConfigDir", "r" )
   lagConfigDir = LazyMount.mount( entityManager, "lag/input/config/cli",
                                   "Lag::Input::Config", "r" )
   cliConfig = LazyMount.mount( entityManager, "bridging/input/config/cli",
                                "Bridging::Input::CliConfig", "r" )
   aclTimeoutConfig = LazyMount.mount( entityManager,
                                       "mlag/aclInstall/timeoutConfig",
                                       "Mlag::AclTimeoutConfig", "w" )

