#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpmwith %{_libdir}/SysdbMountProfiles/ConfigAgent-Mld

import Tac
import Tracing, QuickTrace
import ConfigMount, SharedMem, SmashLazyMount
import Arnet
from CliPlugin import VrfCli

traceIntf = Tracing.trace4
traceGroup = Tracing.trace5
traceSource = Tracing.trace6

qv = QuickTrace.Var
qtraceIntf = QuickTrace.trace4
qtraceGroup = QuickTrace.trace5
qtraceSource = QuickTrace.trace6

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF name' )

class MldConfig:

   def __init__( self, entityManager ):

      self.em = entityManager
      self.intfConfig = ConfigMount.mount( self.em, "routing6/mld/config",
                                           "Routing::Mld::IntfConfig", "w" )

   def intf( self, intfId ):
      return self.intfConfig.configIntf.get( intfId )

   def intfIs( self, intfId ):

      intf = self.intf( intfId )
      if intf is not None:
         return intf

      traceIntf( "Adding interface", intfId, "to config" )
      qtraceIntf( "Adding interface ", qv( intfId ), " to config" )

      intf = self.intfConfig.configIntf.newMember( intfId )
      intf.staticConfig = ()
      intf.staticConfig.staticGroupConfig = ( "direct", )
      intf.querierConfig = ( intfId, )

      return intf

   def intfDel( self, intfId ):
      if intfId not in self.intfConfig.configIntf:
         return

      traceIntf( "Deleting interface", intfId, "from config" )
      qtraceIntf( "Deleting interface ", qv( intfId ), " from config" )

      del self.intfConfig.configIntf[ intfId ]

   def intfUpdate( self, intfId ):
      '''Deletes the inteface from config if all the attributes are back to the
      default value'''

      intf = self.intf( intfId )
      if intf is None:
         return

      qc = intf.querierConfig
      if ( not intf.enabled and
           qc.queryInterval == qc.queryIntervalDefault and
           qc.queryResponseInterval == qc.queryResponseIntervalDefault and
           qc.lastListenerQueryInterval == qc.lastListenerQueryIntervalDefault and
           qc.startupQueryInterval == qc.startupQueryIntervalDefault and
           qc.startupQueryCount == qc.startupQueryCountDefault and
           qc.robustness == qc.robustnessDefault and
           qc.lastListenerQueryCount == qc.lastListenerQueryCountDefault and
           not intf.staticConfig.staticGroupConfig.sourceByGroup and
           intf.staticAccessList == intf.staticAccessListDefault ) :
         self.intfDel( intfId )

   def group( self, intfId, groupAddr ):
      intf = self.intf( intfId )
      if intf is None:
         return None

      if isinstance( groupAddr, str ):
         groupAddr = Arnet.Ip6Addr( groupAddr )

      return intf.staticConfig.staticGroupConfig.sourceByGroup.get( groupAddr )

   def groupIs( self, intfId, groupAddr ):
      group = self.group( intfId, groupAddr )
      if group is not None:
         return group

      intf = self.intfIs( intfId )

      if isinstance( groupAddr, str ):
         groupAddr = Arnet.Ip6Addr( groupAddr )

      traceGroup( "Adding group", groupAddr, "to intf", intfId )
      qtraceGroup( "Adding group ", qv( groupAddr ), " to interface ", qv( intfId ) )

      return intf.staticConfig.staticGroupConfig.sourceByGroup.newMember( groupAddr )

   def groupDel( self, intfId, groupAddr ):
      intf = self.intf( intfId )
      if intf is None:
         return

      if isinstance( groupAddr, str ):
         groupAddr = Arnet.Ip6Addr( groupAddr )

      if groupAddr in intf.staticConfig.staticGroupConfig.sourceByGroup:
         traceGroup( "Deleting group", groupAddr, "from intf", intfId )
         qtraceGroup( "Deleting group ", qv( groupAddr ), " from interface ",
                      qv( intfId ) )
         del intf.staticConfig.staticGroupConfig.sourceByGroup[ groupAddr ]

   def source( self, intfId, groupAddr, sourceAddr ):
      group = self.group( intfId, groupAddr )
      if group is None:
         return None

      if isinstance( sourceAddr, str ):
         sourceAddr = Arnet.Ip6Addr( sourceAddr )

      return group.sourceAddr.get( sourceAddr )

   def sourceIs( self, intfId, groupAddr, sourceAddr ):
      group = self.groupIs( intfId, groupAddr )

      if isinstance( sourceAddr, str ):
         sourceAddr = Arnet.Ip6Addr( sourceAddr )

      traceSource( "Adding source", sourceAddr, "to group",
                   groupAddr, "on intf", intfId )
      qtraceSource( "Adding source ", qv( sourceAddr ), " to group ",
                    qv( groupAddr ), " on interface ", qv( intfId ) )

      group.sourceAddr[ sourceAddr ] = True

   def sourceDel( self, intfId, groupAddr, sourceAddr ):
      group = self.group( intfId, groupAddr )
      if group is None:
         return

      if isinstance( sourceAddr, str ):
         sourceAddr = Arnet.Ip6Addr( sourceAddr )

      if sourceAddr in group.sourceAddr:
         traceSource( "Deleting source", sourceAddr, "from group",
                      groupAddr, "on intf", intfId )
         qtraceSource( "Deleting source ", qv( sourceAddr ), " from group ",
                       qv( groupAddr ), " on interface ", qv( intfId ) )
         del group.sourceAddr[ sourceAddr ]
         if len( group.sourceAddr ) == 0:
            self.groupDel( intfId, groupAddr )

   def aclIs( self, intfId, aclName ):
      intf = self.intfIs( intfId )

      traceIntf( "Adding acl", aclName, "to intf", intfId )
      qtraceIntf( "Adding acl ", qv( aclName ), " to interface ",
                   qv( intfId ) )

      intf.staticAccessList = aclName

   def aclDel( self, intfId ):
      intf = self.intf( intfId )
      if intf is None:
         return

      aclName = intf.staticAccessList

      traceIntf( "Deleting acl", aclName, "from intf", intfId )
      qtraceIntf( "Deleting acl ", qv( aclName ), " from interface ", qv( intfId ) )

      intf.staticAccessList = intf.staticAccessListDefault
      

   def queryIntervalIs( self, intfId, queryInterval=None ):
      intf = self.intfIs( intfId )
      if queryInterval is None:
         queryInterval = intf.querierConfig.queryIntervalDefault
      intf.querierConfig.queryInterval = queryInterval

   def queryResponseIntervalIs( self, intfId, queryResponseInterval=None ):
      intf = self.intfIs( intfId )
      if queryResponseInterval is None:
         queryResponseInterval = intf.querierConfig.queryResponseIntervalDefault
      intf.querierConfig.queryResponseInterval = queryResponseInterval

   def startupQueryIntervalIs( self, intfId, queryInterval=None ):
      intf = self.intfIs( intfId )
      if queryInterval is None:
         queryInterval = intf.querierConfig.startupQueryIntervalDefault
      intf.querierConfig.startupQueryInterval = queryInterval

   def startupQueryCountIs( self, intfId, queryCount=None ):
      intf = self.intfIs( intfId )
      if queryCount is None:
         queryCount = intf.querierConfig.startupQueryCountDefault
      intf.querierConfig.startupQueryCount = queryCount

   def robustnessIs( self, intfId, count=None ):
      intf = self.intfIs( intfId )
      if count is None:
         count = intf.querierConfig.robustnessDefault
      intf.querierConfig.robustness = count

   def lastListenerQueryIntervalIs( self, intfId, queryInterval=None ):
      intf = self.intfIs( intfId )
      if queryInterval is None:
         queryInterval = intf.querierConfig.lastListenerQueryIntervalDefault
      intf.querierConfig.lastListenerQueryInterval = queryInterval

   def lastListenerQueryCountIs( self, intfId, queryCount=None ):
      intf = self.intfIs( intfId )
      if queryCount is None:
         queryCount = intf.querierConfig.lastListenerQueryCountDefault
      intf.querierConfig.lastListenerQueryCount = queryCount

class MldStatus:

   def __init__( self, entityManager ):

      self.em = entityManager
      self.smashMount = SharedMem.entityManager( sysdbEm=self.em )
      self.readerInfo = SmashLazyMount.mountInfo( 'reader' )

      self.gmpStatusByVrf = {}

      self.mldQuerierStatusByVrf = {}

      self.mldStatusByVrf = {}
      self.mldStatusByVrf[ "staticGroup" ] = {}
      self.mldStatusByVrf[ "dynamicGroup" ] = {}

      self.mldStatus = Tac.newInstance( "Routing::Mld::Smash::Status", "Cli" )
      self.mldQuerierStatus = \
         Tac.newInstance( "Routing::Mld::Smash::QuerierStatus", "Cli" )


   def getGmpStatus( self, vrfName ):

      vrfStatus = self.gmpStatusByVrf.get( vrfName )
      if vrfStatus is not None:
         return vrfStatus

      mountPath = "routing6/gmp/status/" + vrfName
      vrfStatus = self.smashMount.doMount( mountPath,
                                           "Routing::Gmp::Smash::Status",
                                           self.readerInfo )
      self.gmpStatusByVrf[ vrfName ] = vrfStatus
      return vrfStatus

   def getMldStatus( self, vrfName, groupType ):

      assert groupType in [ "staticGroup", "dynamicGroup" ]

      vrfStatus = self.mldStatusByVrf[ groupType ].get( vrfName )
      if vrfStatus is not None:
         return vrfStatus

      mountPath = self.mldStatus.mountPath( groupType, vrfName )
      vrfStatus = self.smashMount.doMount( mountPath,
                                           "Routing::Mld::Smash::Status",
                                           self.readerInfo )
      self.mldStatusByVrf[ groupType ][ vrfName ] = vrfStatus

      return vrfStatus

   def getMldQuerierStatus( self, vrfName ):

      vrfStatus = self.mldQuerierStatusByVrf.get( vrfName )
      if vrfStatus is not None:
         return vrfStatus

      mountPath = self.mldQuerierStatus.mountPath( vrfName )
      vrfStatus = self.smashMount.doMount( mountPath,
                                           "Routing::Mld::Smash::QuerierStatus",
                                           self.readerInfo )

      self.mldQuerierStatusByVrf[ vrfName ] = vrfStatus

      return vrfStatus


class MldAddrs:

   def __init__( self ):
      self.multicastRange = Arnet.Ip6Prefix( "ff00::/8" )
      self.ssmRange = Arnet.Ip6Prefix( "ff3::/32" )
      self.wcAddr = Arnet.Ip6Addr( "::" )

   def isWildcardAddr( self, addr ):
      if isinstance( addr, str ):
         addr = Arnet.Ip6Addr( addr )

      return addr == self.wcAddr

   def validSsmGroup( self, addr ):

      if isinstance( addr, str ):
         addr = Arnet.Ip6Addr( addr )

      return self.ssmRange.contains( addr ) # pylint: disable-msg=E1103

   def validMulticastAddr( self, addr ):
      if isinstance( addr, str ):
         addr = Arnet.Ip6Addr( addr )

      return self.multicastRange.contains( addr ) # pylint: disable-msg=E1103

   def validUnicastAddr( self, addr ):
      return not self.validMulticastAddr( addr )
