# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.TechSupportCli as TechSupportCli
from CliPlugin import ModuleModels
import CliMatcher
import LazyMount
import MultiRangeRule
import ShowCommand
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "ModuleCli" )
t0 = Tracing.trace0

cardStatus = None
entityMib = None
powerFuse = None

# ------------------------------------------------------
# The "show module" command, in "enable" mode
#    show module [ MODNAME | all ]
#-------------------------------------------------------

def _getNumMacAddrs( startAddr, endAddr ):
   # Return the number of MAC addresses in the given ( startAddr, endAddr ) range
   startAddrInt = int( startAddr.replace( ":", "" ), 16 )
   endAddrInt = int( endAddr.replace( ":", "" ), 16 )
   return int( endAddrInt - startAddrInt + 1 )

def _cardName( card ):
   # card is an EntityMib::Card object
   defaultTags = card.parent.parent.defaultChildTags.get( 'CardSlot', None )
   if defaultTags:
      defaultTags = defaultTags.tag
   else:
      defaultTags = []
   tag = 'Switchcard' if card.tag == 'SwitchcardCes' else card.tag
   if tag in defaultTags:
      return card.label
   else:
      return f"{tag}{card.label}"

def _getAllSlots( mode ):
   cards = []
   entityMibRoot = entityMib.root
   if entityMibRoot is not None and entityMibRoot.initStatus == "ok":
      slots = list( entityMibRoot.cardSlot.values() )
      cards = [ ( slot.card, None ) for slot in slots if slot.card ]
      secondary = []
      for ( card, _ ) in cards:
         if card.secondaryCard:
            secondary.append( ( card.secondaryCard, card ) )

      cards.extend( secondary )
   return cards

# Dictionary of functions that return a status string and uptime
# for the card, given the card entity mib and cli mode,
# keyed by card tag.
_cardStatusHook = {}
_cardUptimeHook = {}
_cardPowerOffReasonHook = {}

def registerCardStatusHook( cardTag, hook ):
   _cardStatusHook[ cardTag ] = hook

def registerCardUptimeHook( cardTag, hook ):
   _cardUptimeHook[ cardTag ] = hook

_stateToStatusEnumMap = {
      "ok" : "unknown",
      "unknown" : "unknown",
      "booting" : "poweringOn",
      "running" : "ok",
      "powerFailed" : "failed",
      "poweredOff" : "poweredOff",
      "poweringOn" : "poweringOn",
      "unknownInitStatus" : "unknown",
      "partialFailure" : "unknown",
      "epochRebootRequired" : "disabledUntilReboot",
      "epochUpgradeRequired" : "disabledUntilSystemUpgrade",
      "active" : "active",
      "standby" : "standby",
      "disabled" : "disabled",
      "upgradingFpga" : "upgradingFpga",
}

_powerOffReasonToShortReasonEnumMap = {
      "user configuration" : "userConfig",
      "ejector open" : "ejectorOpen",
      "critical system temperature" : "overheat",
      "system temperature critical" : "overheat",
      "card temp sensor group failure" : "tempSensorGroupFailure",
      "insufficient fans" : "insufficientFans",
      "incompatible fan" : "incompatibleFan",
      "power off not supported" : "powerOffNotSupported",
      "ssu" : "SSU",
      "slot unsupported" : "slotUnsupported",
      "insufficient power" : "insufficientPower",
      "card model incompatible" : "cardModelIncompatible",
}

def getCardStatus( mode, card ):
   if card.tag in _cardStatusHook:
      state = _cardStatusHook[ card.tag ]( mode, card )
      ret = _stateToStatusEnumMap.get( state )
      if not ret:
         # pylint: disable-next=consider-using-f-string
         assert False, "Unknown cardState value %s" % state
   else:
      status = cardStatus
      cardName = f"{card.tag}{card.label}"
      # Figure out the status of the card. There are several possibilities:
      hwCardStatus = status.cardStatus.get( cardName )
      if not hwCardStatus:
         initStatus = card.initStatus
         ret = _stateToStatusEnumMap.get( initStatus )
         if not ret:
            # pylint: disable-next=consider-using-f-string
            assert False, "Unknown initStatus value %s" % initStatus
      else:
         state = hwCardStatus.state
         ret = _stateToStatusEnumMap.get( state )
         if not ret:
            # pylint: disable-next=consider-using-f-string
            assert False, "Unknown cardState value %s" % state
   return ret

def getCardUptime( mode, card ):
   if card.tag in _cardUptimeHook:
      return _cardUptimeHook[ card.tag ]( mode, card )
   else:
      if getCardStatus( mode, card ) != 'ok':
         return None

      status = cardStatus
      cardName = f"{card.tag}{card.label}"
      hwCardStatus = status.cardStatus.get( cardName )

      if not hwCardStatus:
         return None
      uptime = Tac.now() - hwCardStatus.lastCardStateChangeTime
      return Tac.utcNow() - uptime

def getCardPowerOffReason( mode, card ):
   if getCardStatus( mode, card ) != 'poweredOff':
      # Fan spinners, which don't have chips and ECB pins, don't support power off
      isFanSpinner = card.tag == 'Fabric' and not card.chip
      if isFanSpinner:
         return _powerOffReasonToShortReasonEnumMap.get( "power off not supported" )
      return None

   cardName = f"{card.tag}{card.label}"
   requested = powerFuse.powerOffRequested
   powerOffRequest = requested.get( cardName )

   if not powerOffRequest:
      return None
   powerOffRequest = powerOffRequest.lower()
   if not _powerOffReasonToShortReasonEnumMap.get( powerOffRequest ):
      return "unknown"
   return _powerOffReasonToShortReasonEnumMap.get( powerOffRequest )

def showModule( mode, args ):
   slotDesc = args.get( 'MODNAME' )
   redundancyStatus = mode.session_.entityManager_.redundancyStatus()
   modules = ModuleModels.Modules()
   # "show module" cannot be ran in "standby rpr" or "disabled simplex" as in those
   # situations Sysdb does not have the required entities mounted that provide
   # the requested information.
   modules.redundancyMode = redundancyStatus.mode
   modules.redundancyProtocol = redundancyStatus.protocol

   # cards is the list of EntityMib::Card objects
   if slotDesc is None:
      cards = _getAllSlots( mode )
   else:
      if slotDesc.secondary and slotDesc.slot and slotDesc.slot.card:
         cards = [ ( slotDesc.secondary, slotDesc.slot.card ) ]
      elif not slotDesc.slot or not slotDesc.slot.card:
         cards = []
      else:
         cards = [ ( slotDesc.slot.card, None ) ]
   t0( 'cards are', cards )

   for ( card, parentCard ) in cards:
      if card:
         m = ModuleModels.Module()
         m.portCount = len( card.port )
         m.typeDescription = card.description
         m.modelName = card.modelName
         m.serialNumber = card.serialNum
         m.minMacAddress = card.minAddr
         m.maxMacAddress = card.maxAddr
         m.numMacAddresses = _getNumMacAddrs( card.minAddr, card.maxAddr )
         m.hardwareRev = card.hardwareRev
         m.softwareRev = card.softwareRev.split( '-' )[ 0 ]
         m.status = getCardStatus( mode, parentCard if parentCard else card )
         m.uptime = getCardUptime( mode, parentCard if parentCard else card )
         cardOffReason = getCardPowerOffReason( mode,
               parentCard if parentCard else card )
         if cardOffReason:
            m.powerOffReason = cardOffReason
         modules.modules[ _cardName( card ) ] = m

   return modules

moduleMatcher = CliMatcher.KeywordMatcher( 'module',
                                           helpdesc='Limit display to a module' )

moduleNode = CliCommand.Node( moduleMatcher, guard=FruCli.cardSlotGuard )

class ShowModuleCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show module [ all | MODNAME ]'
   data = { 'module': moduleNode,
            'all': 'Display all module status and information',
            'MODNAME': FruCli.SlotExpressionFactory( includeSecondary=True ),
          }
   cliModel = ModuleModels.Modules
   handler = showModule
BasicCli.addShowCommandClass( ShowModuleCmd )

# Register 'show module' with 'show tech support'
def _registerShowModuleInShowTechSupport():
   TechSupportCli.registerShowTechSupportCmd(
      '2010-06-11 00:00:00',
      cmds=[ 'show module' ],
      summaryCmds=['show module' ] )

FruCli.registerModularSystemCallback( _registerShowModuleInShowTechSupport )

# ------------------------------------------------------
# The "power module" command, in "config" mode
#
# The full syntax of this command is:
#     power enable module <module path>
#     no power enable module <module path>
#-------------------------------------------------------
# { key: ( tagLong, helpDesc ) }
cardTypes = { 'NUMS': ( '', 'Linecard and Supervisor numbers' ),
              'LINECARD': ( 'Linecard', 'Linecards' ),
              'SUP': ( 'Supervisor', 'Supervisors' ),
              'FABRIC': ( 'Fabric', 'Fabric Cards' ),
              'SWITCHCARD': ( 'Switchcard', 'Switchcards' ),
            }
def handleCardPower( mode, args, powerOn ):
   cardList = None
   for cardType, cardInfo in cardTypes.items():
      if cardType in args:
         cardList = args.get( cardType )
         cardList = FruCli.getCardsFromIdList( mode,
                                               list( cardList.values() ),
                                               cardInfo[ 0 ] )
         for card in cardList:
            if powerOn:
               card.powerOn()
            else:
               card.powerOff()
         break

def doNoPowerEnable( mode, args ):
   handleCardPower( mode, args, powerOn=False )

def doPowerEnable( mode, args ):
   handleCardPower( mode, args, powerOn=True )

class PowerModuleExpression( CliCommand.CliExpression ):
   expression = ' | '.join( cardTypes )
   data = {}
   for _cardType, _cardInfo in cardTypes.items():
      # pylint: disable=cell-var-from-loop
      _matcher =  MultiRangeRule.MultiRangeMatcher(
                     noSingletons=False,
                     helpdesc=_cardInfo[ 1 ],
                     tagLong=_cardInfo[ 0 ],
                     rangeFn=lambda x=_cardInfo[ 0 ]: FruCli.rangeFn( x ),
                  )
      _node = CliCommand.Node( _matcher,
                               guard=FruCli.cardTypeOrModularRprActiveSupeGuard )
      data[ _cardType ] = _node

moduleMatcher = CliMatcher.KeywordMatcher( 'module',
                                           helpdesc='Configure module power' )
powerModuleNode = CliCommand.Node( moduleMatcher,
                                   guard=FruCli.cardSlotGuard )

class PowerModuleCmd( CliCommand.CliCommandClass ):
   syntax = 'power enable module CARDLIST'
   noOrDefaultSyntax = 'power enable module CARDLIST'
   data = { 'power': 'Configure power supplies',
            'enable': 'Configure power',
            'module': powerModuleNode,
            'CARDLIST': PowerModuleExpression
   }
   handler = doPowerEnable
   noHandler = doNoPowerEnable
   defaultHandler = doPowerEnable

BasicCli.GlobalConfigMode.addCommandClass( PowerModuleCmd )

def Plugin( entityManager ):
   global cardStatus, entityMib
   global powerFuse
   cardStatus = LazyMount.mount( entityManager, "hardware/card/status",
                                 "Hardware::Card::Status", "r" )
   entityMib = LazyMount.mount( entityManager, "hardware/entmib",
                                "EntityMib::Status", "r" )
   powerFuse = LazyMount.mount( entityManager, "power/fuse/status/all",
                                "Power::SoftwareFuse", "r" )
