# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements MPLS configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for MPLS'''

import sys
import Arnet.MplsLib
import BasicCli
import Cell
from CliDynamicSymbol import LazyCallback
import CliParser
import CliCommand
import CliMatcher
import CliExtensions
import ConfigMount
from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import IpGenAddrMatcher
from CliPlugin import IraIpIntfCli
from CliPlugin import IraCommonCli
from CliPlugin import MplsModel
from CliPlugin import TechSupportCli
from CliPlugin.IraIpCli import vrfKwMatcher, vrfNameMatcher
from CliPlugin.IraNexthopGroupCli import nexthopGroupNameMatcher
from CliPlugin import TunnelFibCli # pylint: disable=W0611
from CliPlugin.TunnelCli import (
   TunnelTableIdentifier,
   readMountTunnelTable,
   tunnelIndexMatcher,
)
from CliParserCommon import namePattern
from IpLibConsts import DEFAULT_VRF
import LazyMount
import MplsLib
import ShowCommand
import SharedMem
import Smash
import Tac

from SrTePolicyCommonLib import srTePolicyStatusPath
from TypeFuture import TacLazyType
from CliMode.Mpls import StaticMulticastModeBase, TunnelStaticModeBase,\
      TunnelTerminationModeBase, TunnelTerminationVrfModeBase,\
      LfibStitchingPreferencesModeBase
import Toggles.MplsToggleLib
import Toggles.EvpnLibToggleLib
from Toggles.MplsToggleLib import (
   toggleStaticMplsPopOCEnabled )
from Toggles.RoutingLibToggleLib import (
  toggleFibGenMountPathEnabled,
)

MAX_LABEL_EXCEEDED = "More labels are specified than the hardware can handle"

# pkgdeps library RoutingLib

LabelRangeInfo = Tac.Type( 'Mpls::LabelRangeInfo' )
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
MplsLabelAction = Tac.Type( 'Arnet::MplsLabelAction' )
MplsStackEntryIndex = TacLazyType( 'Arnet::MplsStackEntryIndex' )
MplsVia = Tac.Type( 'Tunnel::TunnelTable::MplsVia' )
PayloadType = TacLazyType( 'Mpls::PayloadType' )
RouteMetric = Tac.Type( 'Mpls::RouteMetric' )
QosMapType = TacLazyType( "QosLib::QosMapType" )

em = None
cliChurnTestHelper = None
cliMplsRouteConfig = None
cliMplsRouteConfigReadOnly = None
configMode = BasicCli.GlobalConfigMode
decapLfib = None
decapLfibConsumer = None
decapLfibHw = None
evpnEthernetSegmentLfib = None
evpnEthernetSegmentLfibHw = None
fecModeSm = None
fecModeStatus = None
flexAlgoConfig = None
forwarding6Status = None
forwardingStatus = None
forwardingGenStatus = None
ip6Config = None
ip6Status = None
ipConfig = None
ipStatus = None
l3Config = None
l3ConfigDir = None
l3NHResolver = None
bgpPeerInfoStatusDir = None
lfibInfo = None
mplsLfibSourceInfoDir = None
lfibStatus = None
mldpOpaqueValueTable = None
gribiAfts = None
mplsHwCapability = None
mplsHwStatus = None
mplsRouteConfig = None
mplsRouteConfigDir = None
mplsRouteConfigMergeSm = None
mplsRoutingConfig = None
mplsTunnelConfig = None
labelManagerStatus = None
mplsRoutingStatus = None
mplsVrfLabelConfig = None
mplsTunnelTermVrfQosConfig = None
mplsTunnelTermVrfHwStatus = None
pseudowireLfib = None
pwRouteHelper = None
routingHardwareStatus = None
routing6VrfInfoDir = None
routingVrfInfoDir = None
srTePolicyStatus = None
srteForwardingStatus = None
mplsVskFecIdCliCollection = None
staticMcastLfib = None
staticTunnelConfig = None
staticTunnelTable = None
rsvpLerTunnelTable = None
transitLfib = None
rsvpLerTunnelTable = None
systemTunnelFib = None
dsfTunnelFib = None
gueTunnelFib = None
tunnelFib = None
unifiedForwarding6Status = None
unifiedForwardingStatus = None
unifiedForwardingGenStatus = None
protocolTunnelNameStatus = None
lfibVskOverrideConfig = None
lfibViaSetStatus = None

def mplsSupported():
   return mplsHwCapability.mplsSupported

def mplsSupportedGuard( mode, token ):
   if mplsHwCapability.mplsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def evpnMplsSupportedGuard( mode, token ):
   if mplsHwCapability.evpnMplsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsStaticVrfLabelSupported( mode, token ):
   if mplsHwCapability.mplsStaticVrfLabelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsIntfSupportedGuard( mode, token ):
   if mplsHwCapability.mplsIntfCfgSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsSkipEgressAclSupportedGuard( mode, token ):
   if mplsHwCapability.mplsSkipEgressAclSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsForwardIntoNexthopGroupSupportedGuard( mode, token ):
   if mplsHwCapability.mplsForwardIntoNexthopGroupSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsMultiLabelLookupGuard( mode, token ):
   if mplsHwCapability.mplsMultiLabelLookupSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsLookupFallbackIpGuard( mode, token ):
   if mplsHwCapability.mplsLookupFbIpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsTunnelTerminationGuard( mode, token ):
   if mplsHwCapability.mplsTunnelTerminationSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsTunnelEncapsulationGuard( mode, token ):
   if mplsHwCapability.mplsTunnelEncapTtlDscpModeSelectSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsPhpTtlGuard( mode, token ):
   if mplsHwCapability.mplsPhpTtlSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsPhpDscpUniformGuard( mode, token ):
   if token == 'pipe' or mplsHwCapability.mplsPhpDscpUniformSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsMulticastSupportedGuard( mode, token ):
   if mplsHwCapability.mplsMulticastSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsTunnelTerminationModeGuard( mode, token ):
   if mplsHwCapability.mplsTunnelTermCliModeSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsTunnelTerminationQosMapGuard( mode, token ):
   if mplsHwCapability.mplsTunnelTermVrfQosSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsMvpnSupportedGuard( mode, token ):
   if mplsHwCapability.mplsMvpnSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsEntropyLabelSupportedGuard( mode, token ):
   if mplsHwCapability.mplsEntropyLabelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def getMaxMplsLabelStack():
   return Tac.Type( 'Arnet::MplsStackEntryIndex' ).max + 1

def getPlatformMaxMplsLabelStack():
   return routingHardwareStatus.nexthopGroupMplsLabelStackSize

def ipRoutingEnabledAny( ):
   routingInfo = routingVrfInfoDir.get( DEFAULT_VRF )
   routing6Info = routing6VrfInfoDir.get( DEFAULT_VRF )
   return ( routingInfo and routingInfo.routing ) or \
          ( routing6Info and routing6Info.routing )

def showMplsConfigWarnings( mode ):
   if not mplsRoutingConfig.mplsRouting:
      mode.addWarning( "Mpls routing is not enabled" )
   if not ipRoutingEnabledAny():
      mode.addWarning( "Neither IP nor IPv6 routing is enabled" )

tunnelKwMatcherMplsShow = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc="MPLS tunnel information and counters" )

mplsNodeForConfig = CliCommand.guardedKeyword( 'mpls',
      helpdesc="Global MPLS configuration commands",
      guard=mplsSupportedGuard )
_ipForMplsMatcher = CliMatcher.KeywordMatcher( 'ip',
      helpdesc="Enable MPLS IP routing globally",
      # legacy support: "mpls routing"
      alternates=[ 'routing' ] )
staticMatcher = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Static MPLS configuration commands' )
topLabelMatcher = CliMatcher.KeywordMatcher( 'top-label',
      helpdesc="Specify the top-most MPLS labels" )
lfibMatcher = CliMatcher.KeywordMatcher( 'lfib',
      helpdesc="LFIB configuration" )
stitchingMatcher = CliMatcher.KeywordMatcher( 'stitching',
      helpdesc="Configure MPLS path stitching related parameters" )
preferencesMatcher = CliMatcher.KeywordMatcher( 'preferences',
      helpdesc="Enable and configure the preferences to influence LFIB stitching" )

# Used in Ldp and Isis.
bindingsKw = CliMatcher.KeywordMatcher( 'bindings', helpdesc='Label bindings' )

# Used in Ldp and Rsvp.
mplsMatcherForClear = CliMatcher.KeywordMatcher( 'mpls',
      helpdesc='Clear MPLS information' )

nhStr = "Address of the nexthop router"
intfValMatcher = IntfCli.Intf.matcher

# This is an arbitrary number >= the largest value we support on any HW platform
# for multi-pop. It is used to limit the number of labels we will accept for
# static MPLS routes. The reason for making this a constant value is for the
# startup config scenario, where these config commands are loaded before the MPLS
# HW config has been initialized.
maxIngressMplsTopLabels = 2

def topLabelRangeFn( mode, context ):
   # when guards are disabled, allow configured labels that are outside of
   # the static label range configuration
   if mode.session_.guardsEnabled():
      labelRange = labelRangeValue( LabelRangeInfo.rangeTypeStatic )
      labelMin = labelRange.base
      labelMax = labelRange.base + labelRange.size - 1
   else:
      labelMin = Arnet.MplsLib.labelMin
      labelMax = Arnet.MplsLib.labelMax
   return labelMin, labelMax

topLabelValMatcher = CliMatcher.DynamicIntegerMatcher(
      rangeFn=topLabelRangeFn,
      helpdesc='Value of the MPLS label' )
# This is used to register a second command for multi-label routes that only
# support pop. If swap is ever supported in the future, this can be modified.
# Decrement maxiter by one since one label is already given from topLabelValMatcher
topLabelsValNode = CliCommand.Node(
      matcher=topLabelValMatcher,
      maxMatches=maxIngressMplsTopLabels - 1,
      guard=mplsMultiLabelLookupGuard )

# for no/default mpls static top-label <xxxx>, there's no need to check the
# staticLabelRange configuration
_noTopLabelValMatcher = Arnet.MplsLib.labelValMatcher
allTopLabelMatcher = CliMatcher.KeywordMatcher( 'all',
      helpdesc='Remove all static MPLS LFIB entries' )
_nexthopMatcher = IpGenAddrMatcher.IpGenAddrMatcher(
      helpdesc="Address of the nexthop router" )
_nexthopOnIntfMatcher = IpGenAddrMatcher.IpGenAddrMatcher(
      helpdesc="Forwarding router's address on destination interface" )

swapLabelMatcher = CliMatcher.KeywordMatcher( 'swap-label',
      helpdesc="Specify the label to swap the top label with" )
outLabelValMatcher = Arnet.MplsLib.labelValMatcher
metricMatcher = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='Specify the metric for the route' )
metricValMatcher = CliMatcher.IntegerMatcher(
      RouteMetric.min, RouteMetric.max,
      helpdesc='Metric for the route' )
weightMatcher = CliCommand.Node( CliMatcher.KeywordMatcher( 'weight',
      helpdesc='Specify the weight for the via' ), hidden=True )
weightValMatcher = CliCommand.Node( CliMatcher.IntegerMatcher(
      1, 0xFFFFFFFF, helpdesc='Weight for the via' ), hidden=True )
nhIndexMatcher = CliMatcher.KeywordMatcher( 'index',
      helpdesc='Specify unique identifier for nexthop' )
nhIndexValMatcher = CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
      helpdesc='Identifier for the nexthop' )
_accessListMatcher = CliCommand.guardedKeyword( 'access-list',
      helpdesc="Egress access-list",
      guard=mplsSkipEgressAclSupportedGuard )

popMatcher = CliMatcher.KeywordMatcher( 'pop',
      helpdesc='Pop the top label' )
payloadTypeMatcher = CliMatcher.KeywordMatcher( 'payload-type',
      helpdesc='Specify the type of the underlying payload' )

_ipv4PayloadHelp = "Underlying payload type is Ipv4"
_ipv6PayloadHelp = "Underlying payload type is Ipv6"
_mplsPayloadHelp = "Underlying payload type is MPLS"
_autoPayloadHelp = "Auto detect Underlying payload type"
_autoPayloadTypeMatcher = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'auto',
                                         helpdesc=_autoPayloadHelp ),
      hidden=True )

hardwareMatcherForShow = CliMatcher.KeywordMatcher( "hardware",
      helpdesc="Show MPLS hardware command" )
aleMatcherForShow = CliMatcher.KeywordMatcher( "ale",
      helpdesc="Show MPLS hardware ale command" )
lspNameMatcher = CliMatcher.KeywordMatcher( 'name',
      helpdesc='Specify the LSP name' )
lspNameValMatcher = CliMatcher.PatternMatcher(
      pattern=r'[^\s]*',
      helpdesc='LSP name',
      helpname='WORD' )

class PayloadTypeWithBypassMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self ):
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      payloadTypeKwName = name + '_payload-type'
      accessListKwName = name + '_access-list'
      bypassKwName = name + '_bypass'
      mplsKwName = name + '_mpls'
      ipv4KwName = name + '_ipv4'
      ipv6KwName = name + '_ipv6'
      autoKwName = name + '_auto'

      class PayloadTypeWithBypassExpr( CliCommand.CliExpression ):
         expression = ( '{plt} ( ' # pylint: disable=consider-using-f-string
                        ' {mpls} | '
                        ' ( ( {ipv4} | {ipv6} | {auto} ) '
                        '   [ {al} {bypass} ] ) )'.format(
                           plt=payloadTypeKwName,
                           mpls=mplsKwName,
                           ipv4=ipv4KwName,
                           ipv6=ipv6KwName,
                           auto=autoKwName,
                           bypass=bypassKwName,
                           al=accessListKwName,
                           ) )
         data = {
            payloadTypeKwName: payloadTypeMatcher,
            mplsKwName: CliMatcher.KeywordMatcher( 'mpls',
               helpdesc=_mplsPayloadHelp ),
            ipv4KwName: CliMatcher.KeywordMatcher( 'ipv4',
               helpdesc=_ipv4PayloadHelp ),
            ipv6KwName: CliMatcher.KeywordMatcher( 'ipv6',
               helpdesc=_ipv6PayloadHelp ),
            autoKwName: _autoPayloadTypeMatcher,
            accessListKwName: _accessListMatcher,
            bypassKwName: CliMatcher.KeywordMatcher( "bypass",
               helpdesc="Bypass egress access-list" ),
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            # Verify that no other arg was added with the same name we want to
            # write the adaptor output to.
            assert name not in args

            if mplsKwName in args:
               args[ name ] = PayloadType.mpls
            else:
               bypass = bypassKwName in args
               if ipv4KwName in args:
                  args[ name ] = ( PayloadType.ipv4, bypass )
               elif ipv6KwName in args:
                  args[ name ] = ( PayloadType.ipv6, bypass )
               elif autoKwName in args:
                  args[ name ] = ( PayloadType.autoDecide, bypass )

      return PayloadTypeWithBypassExpr

# The NoBypass version is used for multi-pop routes and forwarding into NHGs,
# since we don't support bypassing egress ACLs for these
payloadTypeNoBypassMatcher = CliMatcher.EnumMatcher( {
      'ipv4': _ipv4PayloadHelp,
      'ipv6': _ipv6PayloadHelp,
      'mpls': _mplsPayloadHelp,
} )

mplsForShowNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'mpls', helpdesc='Show MPLS information' ),
      guard=mplsSupportedGuard )

nextHopMatcher = CliMatcher.KeywordMatcher( 'next-hop',
                                            helpdesc="MPLS next-hop configuration" )

#---------------------------------------------------------
# [no|default] mpls ip
#---------------------------------------------------------
class MplsIpCmd( CliCommand.CliCommandClass ):
   syntax = "mpls ip"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'ip': _ipForMplsMatcher,
   }
   handler = "MplsCliHandler.MplsIpCmd_handler"
   noOrDefaultHandler = "MplsCliHandler.MplsIpCmd_noOrDefaultHandler"

configMode.addCommandClass( MplsIpCmd )

#---------------------------------------------------------
# [no|default] mpls next-hop resolution allow default-route
#---------------------------------------------------------
class MplsNextHopResolutionAllowDefaultRouteCmd( CliCommand.CliCommandClass ):
   syntax = "mpls next-hop resolution allow default-route"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'next-hop': nextHopMatcher,
      'resolution': "MPLS next-hop resolution configuration",
      'allow': "Select when possible",
      'default-route': "Allow resolving MPLS nexthops over default route",
   }
   handler = "MplsCliHandler.MplsNextHopResolutionAllowDefaultRouteCmd_handler"
   noOrDefaultHandler = handler

configMode.addCommandClass( MplsNextHopResolutionAllowDefaultRouteCmd )

# Entropy label command tokens
tunnelNodeHelpStr = 'Tunnel configuration commands'
tunnelNode = CliMatcher.KeywordMatcher( 'tunnel', helpdesc=tunnelNodeHelpStr )

entropyLabelNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'entropy-label',
                                         helpdesc="Entropy label configuration" ),
      guard=mplsEntropyLabelSupportedGuard )

#-------------------------------------------------------------
# [no|default] mpls entropy-label pop in "config" mode
#-------------------------------------------------------------
class MplsEntropyLabelPopCmd( CliCommand.CliCommandClass ):
   syntax = "mpls entropy-label pop"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'entropy-label': entropyLabelNode,
      'pop': 'Enable popping entropy label at the PHP',
   }
   handler = "MplsCliHandler.MplsEntropyLabelPopCmd_handler"
   noOrDefaultHandler = "MplsCliHandler.MplsEntropyLabelPopCmd_noOrDefaultHandler"

#-------------------------------------------------------------
# [no|default] mpls tunnel entropy-label push in "config" mode
#-------------------------------------------------------------
class MplsTunnelEntropyLabelPushCmd( CliCommand.CliCommandClass ):
   syntax = "mpls tunnel entropy-label push"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'entropy-label': entropyLabelNode,
      'push': 'Enable pushing entropy label on ingress LER for all MPLS tunnels',
   }
   handler = "MplsCliHandler.MplsTunnelEntropyLabelPushCmd_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsTunnelEntropyLabelPushCmd_noOrDefaultHandler"

if Toggles.MplsToggleLib.toggleMplsEntropyLabelSupportEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( MplsTunnelEntropyLabelPushCmd )
   BasicCli.GlobalConfigMode.addCommandClass( MplsEntropyLabelPopCmd )

def initCliHelper():
   return Tac.newInstance( "Mpls::CliHelper", "MplsCli",
                           cliMplsRouteConfig.force() )

def initMplsRouteConfigMergeSm( entityManager ):
   global mplsRouteConfigMergeSm
   global mplsRouteConfig
   mplsRouteConfig = Tac.newInstance( 'Mpls::RouteConfig', 'routeConfig' )
   ctrl = Tac.Value( 'Arx::SmControl' )
   mplsRouteConfigMergeSm = Tac.newInstance( 'Mpls::RouteConfigMergerSm',
                                             mplsRouteConfig,
                                             mplsRouteConfigDir,
                                             cliMplsRouteConfigReadOnly,
                                             ctrl )

def initFecModeSm():
   global fecModeSm
   global fecModeStatus
   fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
   fecModeSm = Tac.newInstance( 'Ira::FecModeSm', l3Config, fecModeStatus )


_peerKwMatcher = CliMatcher.KeywordMatcher( 'peer',
                                   helpdesc="Route status depends on peer" )

_bgpMatcher = CliMatcher.KeywordMatcher( 'bgp',
                                         helpdesc="Border gateway protocol" )

_peerMatcher = IpGenAddrMatcher.IpGenAddrMatcher(
      helpdesc="Address of the peer" )

class BgpPeerExpression( CliCommand.CliExpression ):
   expression = '[ bgp peer [ BGP_PEER_ADDR ] ] '

   data = {
      'bgp': _bgpMatcher,
      'peer': _peerKwMatcher,
      'BGP_PEER_ADDR': _peerMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'isBgpPeerMonitored' ] = 'bgp' in args
      args[ 'bgpPeer' ] = args.get( 'BGP_PEER_ADDR',
                                      Tac.Value( 'Arnet::IpGenAddr', '' ) )

class BgpPeerAddrExpression( BgpPeerExpression ):
   expression = '[ bgp peer BGP_PEER_ADDR ] '

#------------------------------------------
# mpls static top-label <inlabel> [ intf ] <nhaddr> ...
# ( no | default ) mpls static top-label <inlabel> [ ... ]
#------------------------------------------
class MplsStaticRoutePopOrSwapCmd( CliCommand.CliCommandClass ):
   _viaSyntax = ( '( ( INTF ADDR_ON_INTF ) | ADDR ) [ BGP_PEER_EXP ] '
                  '( ( swap-label OUT_LABEL ) | ( pop PAYLOAD_TYPE_AND_BYPASS ) ) ' )
   if toggleStaticMplsPopOCEnabled():
      _viaSyntax = ( '[ index NH_INDEX ] ' + _viaSyntax )
      _viaSyntax += ( '[ name LSP_NAME ] [ metric METRIC ] [ weight WEIGHT ]' )
   else:
      _viaSyntax += ( '[ metric METRIC ] [ weight WEIGHT ]' )
   syntax = 'mpls static top-label IN_LABEL ' + _viaSyntax
   # pylint: disable-next=consider-using-f-string
   noOrDefaultSyntax = ( f"mpls static top-label ( all | ( NO_IN_LABEL "
                         f"[ { _viaSyntax } ] ) )" )
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'IN_LABEL': topLabelValMatcher,
      'NO_IN_LABEL': _noTopLabelValMatcher,
      'all': allTopLabelMatcher,
      'index': nhIndexMatcher,
      'NH_INDEX': nhIndexValMatcher,
      'INTF': intfValMatcher,
      'ADDR_ON_INTF': _nexthopOnIntfMatcher,
      'ADDR': _nexthopMatcher,
      'BGP_PEER_EXP': BgpPeerExpression,
      'pop': popMatcher,
      'PAYLOAD_TYPE_AND_BYPASS': PayloadTypeWithBypassMatcher(),
      'swap-label': swapLabelMatcher,
      'OUT_LABEL': outLabelValMatcher,
      'name': lspNameMatcher,
      'LSP_NAME': lspNameValMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
      'weight': weightMatcher,
      'WEIGHT': weightValMatcher,
   }
   handler = "MplsCliHandler.MplsStaticRoutePopOrSwapCmd_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsStaticRoutePopOrSwapCmd_noOrDefaultHandler"

configMode.addCommandClass( MplsStaticRoutePopOrSwapCmd )

#--------------------------------------------------------------------------
# [ no | default ] mpls lookup label count <1-2>
#--------------------------------------------------------------------------
def getLabelRange( mode, context ):
   if mplsHwCapability.mplsLookupLabelMaxCount == 1:
      # This is to avoid issues caused when startup config is loaded before the
      # hardware capability is set up, such as in the CliSaveTestLib and on a
      # real dut. This implementation won't write false data to the products because
      # mplsLookupLabelCount variable in Config is checked against product's
      # hardware capability before it is wrote to Status.
      return ( 1, sys.maxsize )
   else:
      return ( 1, mplsHwCapability.mplsLookupLabelMaxCount )

class MplsLookupLabelCountCmd( CliCommand.CliCommandClass ):
   syntax = "mpls lookup label count LABEL_COUNT"
   noOrDefaultSyntax = "mpls lookup label count ..."

   data = {
      'mpls': mplsNodeForConfig,
      'lookup': CliCommand.guardedKeyword( 'lookup',
                   helpdesc="MPLS route lookup options",
                   guard=mplsMultiLabelLookupGuard ),
      'label': "MPLS label configuration commands",
      'count': "Specify the number of labels considered for route lookup",
      'LABEL_COUNT': CliMatcher.DynamicIntegerMatcher(
         rangeFn=getLabelRange,
         helpdesc="Number of labels considered for route lookup" )
   }
   handler = "MplsCliHandler.MplsLookupLabelCountCmd_handler"
   noOrDefaultHandler = "MplsCliHandler.MplsLookupLabelCountCmd_noOrDefaultHandler"

configMode.addCommandClass( MplsLookupLabelCountCmd )

# --------------------------------------------------------------------------
# [ no ] mpls lookup fallback ip
# --------------------------------------------------------------------------

class MplsLookupFallbackIpCmd( CliCommand.CliCommandClass ):
   syntax = "mpls lookup fallback ip"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'lookup': CliCommand.guardedKeyword( 'lookup',
                   helpdesc="MPLS route lookup options",
                   guard=mplsLookupFallbackIpGuard ),
      'fallback': "Fallback route lookup for MPLS LFIB miss",
      'ip': "Enable IPv4/IPv6 route lookup",
   }
   handler = "MplsCliHandler.MplsLookupFallbackIpCmd_handler"
   noOrDefaultHandler = "MplsCliHandler.MplsLookupFallbackIpCmd_noOrDefaultHandler"

configMode.addCommandClass( MplsLookupFallbackIpCmd )

class MplsLabelsMatcher( CliCommand.CliExpressionFactory ):
   '''
   Matches a single or multiple labels.
   Will always yield a list of labels, regardless of if one or multiple is matched.
   '''
   def __init__( self, labelMatcher, labelsMatcher, multiLabelOptional=True ):
      self.multiLabelOptional = multiLabelOptional
      self.labelMatcher = labelMatcher
      self.labelsMatcher = labelsMatcher
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      labelKwName = name + '_label'
      labelsKwName = name + '_labels'
      multiLabelOptional = self.multiLabelOptional
      labelMatcher = self.labelMatcher
      labelsMatcher = self.labelsMatcher

      class PayloadTypeWithBypassExpr( CliCommand.CliExpression ):
         if multiLabelOptional:
            # pylint: disable-next=consider-using-f-string
            expression = '( %s [ { %s } ] )' % ( labelKwName, labelsKwName )
         else:
            # pylint: disable-next=consider-using-f-string
            expression = '( %s { %s } )' % ( labelKwName, labelsKwName )

         data = {
            labelKwName: labelMatcher,
            labelsKwName: labelsMatcher,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            # Verify that no other arg was added with the same name we want to
            # write the adaptor output to.
            assert name not in args

            label = args.get( labelKwName )
            if label is None:
               return

            labels = args.get( labelsKwName )
            # There's a weird secnario where a `Node` with `maxMatches=1`
            # isn't a `list`.
            if isinstance( labels, int ):
               labels = [ labels ]
            labels = labels or []
            labels.insert( 0, label )
            args[ name ] = labels

      return PayloadTypeWithBypassExpr

_multiTopLabelValMatcher = MplsLabelsMatcher(
      topLabelValMatcher, topLabelsValNode, multiLabelOptional=False )

def labelsAdapter( mode, args, argsList ):
   # There's a weird secnario where a `Node` with `maxMatches=1` isn't a `list`.
   labels = args.get( 'LABELS' )
   if labels is not None and not isinstance( labels, list ):
      args[ 'LABELS' ] = [ labels ]

#--------------------------------------------------------------------------
# [ no | default ] mpls static top-label LABEL { LABELS }
#                  ADDR [ bgp peer [ BGP_PEER_ADDR ] ]
#                  [ pop payload-type PTYPE ]
#                  [ metric METRIC ]
#--------------------------------------------------------------------------
class MplsStaticMultiLabelPop( CliCommand.CliCommandClass ):
   # BUG455910 There should be an optional interface here, effectively the same
   # as the single label mpls static config.
   syntax = ( 'mpls static top-label LABELS ADDR [ BGP_PEER_EXP ]'
              '[ pop payload-type PTYPE ] '
              '[ metric METRIC ] [ weight WEIGHT ]' )
   # In case of no, the ADDR is optional
   noOrDefaultSyntax = ( 'mpls static top-label LABELS [ ADDR ] '
                         '[ BGP_PEER_EXP ][ pop payload-type PTYPE ] '
                         '[ metric METRIC ] [ weight WEIGHT ]' )
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABELS': _multiTopLabelValMatcher,
      'ADDR': _nexthopMatcher,
      'BGP_PEER_EXP': BgpPeerExpression,
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
      'weight': weightMatcher,
      'WEIGHT': weightValMatcher,
   }
   handler = "MplsCliHandler.MplsStaticMultiLabelPop_handler"
   noOrDefaultHandler = "MplsCliHandler.MplsStaticMultiLabelPop_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticMultiLabelPop )

#--------------------------------------------------------------------------
# [ no | default ] mpls static top-label LABEL nexthop-group GROUP
#                  [ bgp peer BGP_PEER_ADDR ]
#                  [ pop [ payload-type PTYPE ] ] [ metric METRIC ]
#--------------------------------------------------------------------------

_nexthopGroupNode = CliCommand.guardedKeyword( 'nexthop-group',
      helpdesc="Forwarding into nexthop-group",
      guard=mplsForwardIntoNexthopGroupSupportedGuard )

class MplsStaticTopLabelNexthopGroup( CliCommand.CliCommandClass ):
   syntax = 'mpls static top-label LABEL '
   _viaSyntax = ( 'nexthop-group GROUP '
                  '[ BGP_PEER_ADDR_EXP ] '
                  '[ pop [ payload-type PTYPE ] ] ' )
   if toggleStaticMplsPopOCEnabled():
      _viaSyntax = '[ index NH_INDEX ] ' + _viaSyntax
      syntax = syntax + _viaSyntax + ( '[ name LSP_NAME ] [ metric METRIC ] '
                                      '[ weight WEIGHT ]' )
   else:
      syntax = syntax + _viaSyntax + '[ metric METRIC ] [ weight WEIGHT ]'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABEL': topLabelValMatcher,
      'index': nhIndexMatcher,
      'NH_INDEX': nhIndexValMatcher,
      'nexthop-group': _nexthopGroupNode,
      'GROUP': nexthopGroupNameMatcher,
      'BGP_PEER_ADDR_EXP': BgpPeerAddrExpression,
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'name': lspNameMatcher,
      'LSP_NAME': lspNameValMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
      'weight': weightMatcher,
      'WEIGHT': weightValMatcher,
   }
   handler = "MplsCliHandler.MplsStaticTopLabelNexthopGroup_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsStaticTopLabelNexthopGroup_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticTopLabelNexthopGroup )

#--------------------------------------------------------------------------
# [ no | default ] mpls static top-label LABEL { LABELS }
#                  nexthop-group GROUP [ bgp peer BGP_PEER_ADDR ]
#                  [ pop [ payload-type PTYPE ] ] [ metric METRIC ] [ weight Weight ]
#--------------------------------------------------------------------------
class MplsStaticMultiLabelNexthopGroup( CliCommand.CliCommandClass ):
   syntax = ( 'mpls static top-label LABELS nexthop-group GROUP '
              '[ BGP_PEER_ADDR_EXP ] '
              '[ pop [ payload-type PTYPE ] ] [ metric METRIC ] [ weight WEIGHT ]' )
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABELS': _multiTopLabelValMatcher,
      'nexthop-group': _nexthopGroupNode,
      'GROUP': nexthopGroupNameMatcher,
      'BGP_PEER_ADDR_EXP': BgpPeerAddrExpression,
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
      'weight': weightMatcher,
      'WEIGHT': weightValMatcher,
   }
   handler = "MplsCliHandler.MplsStaticMultiLabelNexthopGroup_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsStaticMultiLabelNexthopGroup_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticMultiLabelNexthopGroup )

#------------------------------------------
# [no] mpls static multicast top-label <inLabel>
#     next-hop <nhAddr> swap-label <outLabel>
#     next-hop <nhAddr> swap-label <outLabel>
#     ...
#------------------------------------------

LfibSysdbMplsViaSet = Tac.Type( 'Mpls::LfibSysdbMplsViaSet' )
LfibSysdbMplsVia = Tac.Type( 'Mpls::LfibSysdbMplsVia' )
multicastNode = CliCommand.guardedKeyword( 'multicast',
      helpdesc="MPLS multicast configuration commands",
      guard=mplsMulticastSupportedGuard )

class StaticMulticastMode( StaticMulticastModeBase, BasicCli.ConfigModeBase ):
   name = "Mpls static multicast route configuration"

   def __init__( self, parent, session, inLabel ):
      StaticMulticastModeBase.__init__( self, inLabel )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.inLabel = MplsLabel( inLabel )

      self.vias = set() # a local version of viaSet, for easier entry handling
      if staticMcastLfib.mplsViaSet.get( inLabel ):
         self._syncFromSysdbViaSet()
      else:
         newViaSet = LfibSysdbMplsViaSet( self.inLabel )
         staticMcastLfib.mplsViaSet.addMember( newViaSet )
      self.viasUpdated = False

   def onExit( self ):
      if self.viasUpdated:
         self._publishLocalViasToSysdb()

      BasicCli.ConfigModeBase.onExit( self )

   # convert staticMcastLfib.mplsViaSet[ inLabel ] into local viaSet
   # self.vias
   def _syncFromSysdbViaSet( self ):
      sysdbViaSet = staticMcastLfib.mplsViaSet[ self.inLabel ]
      for via in sysdbViaSet.mplsVia:
         nhAddr = via.nextHop.stringValue
         outLabel = via.outLabel.topLabel()
         self.vias.add( ( nhAddr, outLabel ) )
      assert len( sysdbViaSet.mplsVia ) == len( self.vias )

   # publish entries in self.vias to
   # staticMcastLfib.mplsViaSet[ self.inLabel ]
   def _publishLocalViasToSysdb( self ):
      newViaSet = LfibSysdbMplsViaSet( self.inLabel )
      for ( nhAddr, outLabel ) in self.vias:
         nextHop = Arnet.IpGenAddr( nhAddr )
         outLabelStack = Tac.Value( 'Arnet::MplsLabelOperation' )
         outLabelStack.appendLabel( MplsLabel( outLabel ) )
         newVia = LfibSysdbMplsVia( self.inLabel, nextHop, outLabelStack )
         newViaSet.mplsVia.add( newVia )
      staticMcastLfib.mplsViaSet.addMember( newViaSet )
      assert len( self.vias ) == \
         len( staticMcastLfib.mplsViaSet[ self.inLabel ].mplsVia )

#--------------------------------------------------------------------------
# [ no | default ] mpls static multicast top-label LABEL
#--------------------------------------------------------------------------
class MplsStaticMulticastConfigRoute( CliCommand.CliCommandClass ):
   syntax = 'mpls static multicast top-label LABEL'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'multicast': multicastNode,
      'top-label': topLabelMatcher,
      'LABEL': topLabelValMatcher,
   }
   handler = "MplsCliHandler.MplsStaticMulticastConfigRoute_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsStaticMulticastConfigRoute_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticMulticastConfigRoute )

#--------------------------------------------------------------------------
# [ no | default ] mpls fec ip sharing disabled
#--------------------------------------------------------------------------
class MplsDisableStaticFecSharing( CliCommand.CliCommandClass ):
   syntax = 'mpls fec ip sharing disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'fec': 'FEC configuration',
      'ip': 'IP configuration',
      'sharing': 'IP sharing configuration',
      'disabled': 'Disable FEC sharing',
   }
   handler = "MplsCliHandler.MplsDisableStaticFecSharing_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsDisableStaticFecSharing_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsDisableStaticFecSharing )

#--------------------------------------------------------------------------
# [ no | default ] mpls segment-routing is-is fec ip sharing disabled
#--------------------------------------------------------------------------
class MplsDisableIsisSrFecSharing( CliCommand.CliCommandClass ):
   syntax = 'mpls segment-routing is-is fec ip sharing disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'segment-routing': 'SR configuration',
      'is-is': 'IS-IS configuration',
      'fec': 'FEC configuration',
      'ip': 'IP configuration',
      'sharing': 'IP sharing configuration',
      'disabled': 'Disable FEC sharing',
   }
   handler = "MplsCliHandler.MplsDisableIsisSrFecSharing_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsDisableIsisSrFecSharing_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsDisableIsisSrFecSharing )

#--------------------------------------------------------------------------
# [ no | default ] next-hop ADDR swap-label LABEL
#--------------------------------------------------------------------------
class MplsStaticMulticastRouteConfigVia( CliCommand.CliCommandClass ):
   syntax = "next-hop ADDR swap-label LABEL"
   noOrDefaultSyntax = syntax
   data = {
      'next-hop': nextHopMatcher,
      'ADDR': IpAddrMatcher.IpAddrMatcher( "Address of the nexthop router" ),
      'swap-label': swapLabelMatcher,
      'LABEL': outLabelValMatcher,
   }
   handler = "MplsCliHandler.MplsStaticMulticastRouteConfigVia_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsStaticMulticastRouteConfigVia_noOrDefaultHandler"

StaticMulticastMode.addCommandClass( MplsStaticMulticastRouteConfigVia )

#------------------------------------------
# [no] mpls static vrf-label <label> vrf <vrf-name>
#------------------------------------------
class MplsStaticVrfLabelCmd( CliCommand.CliCommandClass ):
   syntax = "mpls static vrf-label IN_LABEL vrf VRF_NAME"
   noOrDefaultSyntax = "mpls static vrf-label NO_IN_LABEL [ vrf VRF_NAME ]"
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'vrf-label': CliCommand.guardedKeyword( 'vrf-label',
                      helpdesc="Specify the MPLS label to vrf mapping",
                      guard=mplsStaticVrfLabelSupported ),
      'IN_LABEL': topLabelValMatcher,
      'NO_IN_LABEL': _noTopLabelValMatcher,
      'vrf': vrfKwMatcher,
      'VRF_NAME': vrfNameMatcher,
   }
   handler = "MplsCliHandler.MplsStaticVrfLabelCmd_handler"
   noOrDefaultHandler = "MplsCliHandler.MplsStaticVrfLabelCmd_noOrDefaultHandler"

configMode.addCommandClass( MplsStaticVrfLabelCmd )

#--------------------------------------------------------------------------
# [ no | default ] mpls label range RTYPE BASE SIZE [ force ]'
#--------------------------------------------------------------------------
labelRangeKeywords = {
   LabelRangeInfo.rangeTypeStatic: 'Specify labels reserved for static MPLS routes',
   LabelRangeInfo.rangeTypeDynamic: 'Specify labels reserved for dynamic assignment',
   LabelRangeInfo.rangeTypeSrgb:
       'Specify labels reserved for IS-IS SR global segment identifiers (SIDs)',
   LabelRangeInfo.rangeTypeBgpSrgb:
       'Specify labels reserved for BGP SR global segment identifiers (SIDs)',
   LabelRangeInfo.rangeTypeSrlb:
       'Specify labels reserved for SR local segment identifiers (SIDs)',
}

ospfSrLabelRangeKeyword = {
   LabelRangeInfo.rangeTypeOspfSrgb:
       'Specify labels reserved for OSPF SR global segment identifiers (SIDs)'
   }

def labelRangeValue( rangeType ):
   return MplsLib.labelRange( mplsRoutingConfig, rangeType )

def labelRangeKeywordsFn():
   result = labelRangeKeywords
   if Toggles.MplsToggleLib.toggleOspfSegmentRoutingMplsEnabled():
      result = dict( result ) # copy
      result.update( ospfSrLabelRangeKeyword )
   return result

class RangeTypeExpression( CliCommand.CliExpression ):
   expression = 'RANGE_TYPE'
   data = {
      'RANGE_TYPE': CliMatcher.EnumMatcher( labelRangeKeywords ),
      'l2evpn': 'Specify labels reserved for L2 EVPN routes',
   }
   if Toggles.MplsToggleLib.toggleOspfSegmentRoutingMplsEnabled():
      expression = expression + ' | ospf-sr'
      data[ 'ospf-sr' ] = 'Specify labels reserved for OSPF SR global segment ' \
                          'identifiers (SIDs)'
   expression = expression + ' | ( l2evpn [ ethernet-segment ] )'
   data[ 'ethernet-segment' ] = 'Specify labels reserved for L2 EVPN A-D ' \
                                'per ES routes for split-horizon filtering'

   @staticmethod
   def adapter( mode, args, argsList ):
      if args.get( 'ethernet-segment' ):
         rangeType = LabelRangeInfo.rangeTypeL2evpnSharedEs
      else:
         rangeType = args.get( 'RANGE_TYPE' ) or \
                     args.get( 'ospf-sr' ) or \
                     args.get( 'l2evpn' )
      args[ 'RTYPE' ] = rangeType

checkConflictingLabelRangeHook = CliExtensions.CliHook()
handleUpdatedLabelRangeHook = CliExtensions.CliHook()
checkConflictingLabelRangeHook.addExtension(
      LazyCallback( "MplsCliHandler.checkConflictingLabelRangeForMplsStatic" ) )

class CfgMplsLabelRange( CliCommand.CliCommandClass ):
   syntax = 'mpls label range RTYPE BASE SIZE [ force ]'
   noOrDefaultSyntax = 'mpls label range RTYPE ...'
   data = {
      'mpls': mplsNodeForConfig,
      'label': 'Specify label range allocations',
      'range': 'Specify label range allocations',
      'RTYPE': RangeTypeExpression,
      'BASE': CliMatcher.IntegerMatcher( MplsLabel.unassignedMin,
                                         MplsLabel.max,
                                         helpdesc='Starting label to reserve' ),
      'SIZE': CliMatcher.IntegerMatcher( 0,
                                         MplsLabel.max - MplsLabel.unassignedMin + 1,
                                         helpdesc='Number of labels to reserve' ),
      'force': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'force', helpdesc='' ),
                  hidden=True ),
   }
   handler = "MplsCliHandler.CfgMplsLabelRange_handler"
   noOrDefaultHandler = "MplsCliHandler.CfgMplsLabelRange_noOrDefaultHandler"

configMode.addCommandClass( CfgMplsLabelRange )

#----------------------------------------------
# [no] mpls ip command in config-if mode
#----------------------------------------------
modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

mplsIntfModeKw = CliCommand.guardedKeyword( 'mpls',
      helpdesc='Interface level MPLS configuration commands',
      guard=mplsIntfSupportedGuard )

class IntfMplsIpCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ip'
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsIntfModeKw,
      'ip': 'Enable MPLS traffic on interface if MPLS enabled globally',
   }
   handler = "MplsCliHandler.IntfMplsIpCmd_handler"
   noOrDefaultHandler = handler

modelet.addCommandClass( IntfMplsIpCmd )

class IntfMplsConfigCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      # Need to set the config to default i.e. enable mpls
      intf = self.intf_.name
      del mplsRoutingConfig.mplsRoutingDisabledIntf[ intf ]

mplsNodeForShow = CliCommand.guardedKeyword( 'mpls',
      helpdesc='Show MPLS information',
      guard=mplsSupportedGuard )
_mplsRouteKwForShowMatcher = CliMatcher.KeywordMatcher( 'route',
                                                        helpdesc='Show MPLS routes' )
_mplsHwRouteKwForShowMatcher = CliMatcher.KeywordMatcher(
      'route', helpdesc='Show MPLS routes (detail)' )
matcherLfib = CliMatcher.KeywordMatcher( 'lfib', helpdesc='Show MPLS LFIB' )
labelValMatcher = Arnet.MplsLib.labelValMatcher
labelsValNode = CliCommand.Node(
      matcher=labelValMatcher,
      guard=mplsMultiLabelLookupGuard,
      maxMatches=maxIngressMplsTopLabels - 1 )

_filterViaTypeKwMatcher = CliMatcher.KeywordMatcher(
      'type', helpdesc="Filter via types" )

_tunnelTypeFilterValMatcher = CliMatcher.EnumMatcher( {
   'mpls': 'MPLS IP vias',
   'evpn': 'EVPN vias',
   'pseudowire': 'Pseudowire vias',
} )

_singleOrMultiLabelValMatcher = MplsLabelsMatcher( labelValMatcher, labelsValNode,
                                                   multiLabelOptional=True )
segmentRoutingKw = CliMatcher.KeywordMatcher(
      'segment-routing', helpdesc='Segment routing' )
_filterViaFlexAlgoKwMatcher = CliMatcher.KeywordMatcher(
      'flex-algo', helpdesc='Flexible Algorithm' )

flexAlgoName = CliMatcher.PatternMatcher( namePattern,
                                          helpname='ALGO-NAME',
                                          helpdesc='Algorithm Name' )

_filterViaSegmentTypeValMatcher = CliMatcher.EnumMatcher( {
   'prefix-segment': 'Prefix segment',
   'adjacency-segment': 'Adjacency segment',
} )

#-------------------------------------------------------------
# show mpls lfib route [type {mpls | evpn | pseudowire }]
# [<label> ...][prefix][gribi | static | rsvp | ldp | mldp |
# isis{{flex-algo <ALGO-NAME>} |
# { segment-routing [prefix-segment | adjacency-segment]}}
# | bgp | traffic-engineering segment-routing policy ] [detail]
#-------------------------------------------------------------
class ShowMplsLfibRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show mpls lfib route [ type TYPE ] [ LABELS ]'''
   syntax += '[ PREFIX ] [ ( gribi | static | rsvp | ldp | mldp | bgp | \
                  ( traffic-engineering segment-routing policy ) | \
                  ( isis ( ( flex-algo ALGO-NAME ) | \
                         ( segment-routing [ segmentType ] ) ) ) ) ] \
                  '
   syntax += '[ detail ]'
   data = {
      'mpls': mplsNodeForShow,
      'lfib': matcherLfib,
      'route': _mplsRouteKwForShowMatcher,
      'type': _filterViaTypeKwMatcher,
      'TYPE': _tunnelTypeFilterValMatcher,
      'LABELS': _singleOrMultiLabelValMatcher,
      'detail': "Display information in detail",
      'gribi': CliMatcher.KeywordMatcher( 'gribi',
         helpdesc='Filter gribi source routes' ),
      'PREFIX': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                   ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   allowAddr=False ),
      'static': CliMatcher.KeywordMatcher( 'static',
         helpdesc='Filter static source routes' ),
      'rsvp': CliMatcher.KeywordMatcher( 'rsvp',
         helpdesc='Filter rsvp source routes' ),
      'ldp': CliMatcher.KeywordMatcher( 'ldp',
         helpdesc='Filter ldp source routes' ),
      'mldp': CliMatcher.KeywordMatcher( 'mldp',
         helpdesc='Filter mldp source routes' ),
      'isis': CliMatcher.KeywordMatcher( 'isis',
         helpdesc='Filter isis source routes' ),
      'bgp': CliMatcher.KeywordMatcher( 'bgp',
         helpdesc='Filter bgp source routes' ),
      'flex-algo': _filterViaFlexAlgoKwMatcher,
      'ALGO-NAME': flexAlgoName,
      'segment-routing': segmentRoutingKw,
      'segmentType': _filterViaSegmentTypeValMatcher,
      'traffic-engineering': CliMatcher.KeywordMatcher( 'traffic-engineering',
         helpdesc='Filter SR TE policy source routes' ),
      'policy': CliMatcher.KeywordMatcher( 'policy', helpdesc='Policy based' )
      }
   cliModel = MplsModel.MplsRoutes
   handler = "MplsCliHandler.ShowMplsLfibRouteCmd_handler"

BasicCli.addShowCommandClass( ShowMplsLfibRouteCmd )

#------------------------------------------------------
# show mpls config route [<labels> ...]
#------------------------------------------------------
class ShowMplsConfigRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls config route [ LABELS ]"
   data = {
      'mpls': mplsNodeForShow,
      'config': "Show MPLS configuration",
      'route': _mplsRouteKwForShowMatcher,
      'LABELS': _singleOrMultiLabelValMatcher,
   }
   cliModel = MplsModel.MplsRoutes
   handler = "MplsCliHandler.ShowMplsConfigRouteCmd_handler"

BasicCli.addShowCommandClass( ShowMplsConfigRouteCmd )

#-----------------------------------------------
# show mpls route [type (mpls | evpn | pseudowire)] [<label> ...]
#-----------------------------------------------
class ShowMplsRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls route [ type TYPE ] [ LABELS ]"
   data = {
      'mpls': mplsNodeForShow,
      'route': _mplsHwRouteKwForShowMatcher,
      'type': _filterViaTypeKwMatcher,
      'TYPE': _tunnelTypeFilterValMatcher,
      'LABELS': _singleOrMultiLabelValMatcher,
   }
   cliModel = MplsModel.MplsHwStatus
   handler = "MplsCliHandler.ShowMplsRouteCmd_handler"

BasicCli.addShowCommandClass( ShowMplsRouteCmd )

#------------------------------------------
# show mpls route summary
#------------------------------------------
class ShowMplsRouteSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls route summary"
   data = {
      'mpls': mplsNodeForShow,
      'route': _mplsHwRouteKwForShowMatcher,
      'summary': "Show brief statistics of MPLS routes",
   }
   cliModel = MplsModel.MplsHwStatusSummary
   handler = "MplsCliHandler.showMplsHwRouteSummary"

BasicCli.addShowCommandClass( ShowMplsRouteSummaryCmd )

# Timestamps are made up to maintain historical order within show tech-support
TechSupportCli.registerShowTechSupportCmd( 
   '2013-11-22 11:35:37',
   cmds=[ 'show mpls route summary',
          'show mpls route',
          'show mpls lfib route detail' ],
   cmdsGuard=lambda: mplsHwCapability.mplsSupported,
   summaryCmds=[ 'show mpls route summary' ],
   summaryCmdsGuard=lambda: mplsHwCapability.mplsSupported )

#-------------------------------------------------------------------------
# The "[no] mpls tunnel static <name> <endpoint> [ <nexthop> <interface>
#    label-stack <label-stack> ] | [ Null0 ]" command.
#-------------------------------------------------------------------------
# Note: This token has been duplicated using the new parser above (as tunnelNode)
staticTunnelMatcherHelpStr = 'Static tunnel configuration'
staticTunnelMatcher = CliMatcher.KeywordMatcher(
   'static',
   helpdesc=staticTunnelMatcherHelpStr )
# XXX: Limit of name to 128 characters
tunnelNameValMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]{1,128}',
                                                  helpdesc='Name of static tunnel',
                                                  helpname='WORD' )

labelStackHelpStr = 'Specify the MPLS label(s) to push'
labelStackKeywordNode = CliMatcher.KeywordMatcher( 'label-stack',
                                                   helpdesc=labelStackHelpStr )
labelStackValNode = CliCommand.Node(
   matcher=CliMatcher.DynamicIntegerMatcher( topLabelRangeFn,
                                             helpdesc='Value of the MPLS label' ),
   maxMatches=getMaxMplsLabelStack() )
impNullTunnelNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                                   'imp-null-tunnel', helpdesc='' ),
                                     hidden=True )
endpointMatcher = CliMatcher.KeywordMatcher( 'endpoint',
      helpdesc='Match tunnel endpoint' )

def labelOperation( labels ):
   mplsLabel = Tac.Value( 'Arnet::MplsLabelOperation' )
   mplsLabel.stackSize = len( labels )
   mplsLabel.operation = MplsLabelAction.push
   indexes = list( range( len( labels ) ) )
   for labelStackIndex, labelIndex in zip( reversed( indexes ), indexes ):
      mplsLabel.labelStackIs( labelStackIndex, labels[ labelIndex ] )

   return mplsLabel

def validateLabelStackSize( mode, labels ):
   platformMaxLabels = getPlatformMaxMplsLabelStack()
   if mode.session.isInteractive() and len( labels ) > platformMaxLabels:
      # pylint: disable-next=consider-using-f-string
      errStr = ( 'The label stack depth exceeds the capability of the hardware.'
                 ' Maximum label stack depth supported: %d' % platformMaxLabels )
      mode.addError( errStr )
      return False
   return True

# Note: This command shares syntax with MplsDebugStaticTunnel in
# MplsDebugCli.py. If updating it here, please update the debug
# command accordingly.
# The syntax for NULL0_EXPR is only supported on this productized
# version and is not updated in the debug command.
class MplsStaticTunnel( CliCommand.CliCommandClass ):
   syntax = '''mpls tunnel static NAME
               ( ( ( ( TEP_V4_ADDR | TEP_V4_PREFIX ) NEXTHOP_V4 ) |
                 ( ( TEP_V6_ADDR | TEP_V6_PREFIX ) NEXTHOP_V6 ) )
               INTF ( ( label-stack { LABELS } ) | imp-null-tunnel ) ) |
               ( ( ( TEP_V4_ADDR | TEP_V4_PREFIX ) |
                   ( TEP_V6_ADDR | TEP_V6_PREFIX ) ) NULL0_EXPR ) 
            '''

   noOrDefaultSyntax = '''mpls tunnel static NAME ...'''

   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'static': staticTunnelMatcher,
      'NAME': tunnelNameValMatcher,
      'TEP_V4_ADDR': IpAddrMatcher.ipAddrMatcher,
      'TEP_V4_PREFIX': IpAddrMatcher.ipPrefixExpr(
         'IP address',
         "Subnet's mask value",
         'IP address with mask length',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
      ),
      'NEXTHOP_V4': IpAddrMatcher.ipAddrMatcher,
      # FIXME BUG200408: Un-hide IPv6 endpoints and nexthop in "mpls static tunnel"
      'TEP_V6_ADDR': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                      hidden=True ),
      'TEP_V6_PREFIX': CliCommand.Node( matcher=Ip6AddrMatcher.ip6PrefixMatcher,
                                        hidden=True ),
      'NEXTHOP_V6': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                     hidden=True ),
      'INTF': intfValMatcher,
      'label-stack': labelStackKeywordNode,
      'LABELS': labelStackValNode,
      'imp-null-tunnel': impNullTunnelNode,
      'NULL0_EXPR': CliCommand.Node( matcher=IraCommonCli.nullIntfKwMatcher )
   }

   adapter = labelsAdapter
   handler = "MplsCliHandler.MplsStaticTunnel_handler"
   noOrDefaultHandler = "MplsCliHandler.MplsStaticTunnel_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticTunnel )

#------------------------------------------
# [no] mpls tunnel static <name> <endpoint>
#     via <nexthop> <interface> label-stack <label-stack>
#     via <nexthop> <interface> label-stack <label-stack>
#     ...
#------------------------------------------

class TunnelStaticMode( TunnelStaticModeBase, BasicCli.ConfigModeBase ):
   name = "Mpls static tunnel configuration"

   def __init__( self, parent, session, tunName, tep ):
      TunnelStaticModeBase.__init__( self, tunName, tep )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------
# mpls tunnel static <name> <endpoint>
# [ no | default ] behavior is handled by the MplsStaticTunnel class
#--------------------------------------------------------------------------
class MplsTunnelStaticConfig( CliCommand.CliCommandClass ):
   syntax = '''mpls tunnel static NAME
               ( ENDPOINTV4ADDR | ENDPOINTV4PREFIX |
                 ENDPOINTV6ADDR | ENDPOINTV6PREFIX )'''

   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'static': staticTunnelMatcher,
      'NAME': tunnelNameValMatcher,
      'ENDPOINTV4ADDR': IpAddrMatcher.ipAddrMatcher,
      'ENDPOINTV4PREFIX': IpAddrMatcher.ipPrefixExpr(
         'IP address',
         "Subnet's mask value",
         'IP address with mask length',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
      ),
      # FIXME BUG200408: Un-hide IPv6 endpoints and nexthop in "mpls static tunnel"
      'ENDPOINTV6ADDR': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                         hidden=True ),
      'ENDPOINTV6PREFIX': CliCommand.Node( matcher=Ip6AddrMatcher.ip6PrefixMatcher,
                                           hidden=True ),
   }
   handler = "MplsCliHandler.MplsTunnelStaticConfig_handler"

BasicCli.GlobalConfigMode.addCommandClass( MplsTunnelStaticConfig )

#--------------------------------------------------------------------------
# [ no | default ] via <nexthop> <interface> label-stack <label-stack>
#--------------------------------------------------------------------------

class MplsTunnelStaticConfigVia( CliCommand.CliCommandClass ):
   syntax = '''via ( NEXTHOPV4 | NEXTHOPV6 ) INTF
               ( ( label-stack { LABELS } ) | imp-null-tunnel )'''
   noOrDefaultSyntax = syntax
   data = {
      'via': 'MPLS via configuration',
      'NEXTHOPV4': IpAddrMatcher.ipAddrMatcher,
      'NEXTHOPV6': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                    hidden=True ),
      'INTF': intfValMatcher,
      'label-stack': labelStackKeywordNode,
      'LABELS': labelStackValNode,
      'imp-null-tunnel': impNullTunnelNode,
   }
   handler = "MplsCliHandler.MplsTunnelStaticConfigVia_handler"
   noOrDefaultHandler = "MplsCliHandler.MplsTunnelStaticConfigVia_noOrDefaultHandler"

TunnelStaticMode.addCommandClass( MplsTunnelStaticConfigVia )

#-------------------------------------------------------------------------
# [no] mpls tunnel termination model ttl (pipe | uniform) dscp (pipe | uniform)
#-------------------------------------------------------------------------
_tunnelTermKwMatcher = CliMatcher.KeywordMatcher( 'termination',
      helpdesc="Tunnel termination configuration commands" )
_tunnelTerminationNodeWTunTermGuard = CliCommand.Node(
      matcher=_tunnelTermKwMatcher,
      guard=mplsTunnelTerminationGuard )
_tunnelTerminationNodeWCliModeGuard = CliCommand.Node(
      matcher=_tunnelTermKwMatcher,
      guard=mplsTunnelTerminationModeGuard )
_tunnelTermModelMatcher = CliMatcher.KeywordMatcher(
   "model", helpdesc="tunnel termination ttl and dscp model" )
_ttlMatcher = CliMatcher.KeywordMatcher( 'ttl', helpdesc="model for ttl" )
_dscpMatcher = CliMatcher.KeywordMatcher( 'dscp', helpdesc="model for dscp" )

_pipeHelpdesc = "Preserve the inner value"
_uniformHelpdesc = "Propagate value from outer header"

_ttlAndDscpTunnelModeValMatcher = CliMatcher.EnumMatcher( {
   'pipe': _pipeHelpdesc,
   'uniform': _uniformHelpdesc,
} )

_ttlTunnelModeValMatcher = CliCommand.Node(
   matcher=_ttlAndDscpTunnelModeValMatcher,
   storeSharedResult=True )

def getMplsTermDscpModelKeywords( mode, context ):
   ttlModel = context.sharedResult[ 'TTL_MODEL' ]
   options = {}
   if ttlModel == 'uniform':
      if mplsHwCapability.mplsTtlUniformDscpUniformModelSupported:
         options[ 'uniform' ] = _uniformHelpdesc
      if mplsHwCapability.mplsTtlUniformDscpPipeModelSupported:
         options[ 'pipe' ] = _pipeHelpdesc
   elif ttlModel == 'pipe':
      if mplsHwCapability.mplsTtlPipeDscpPipeModelSupported:
         options[ 'pipe' ] = _pipeHelpdesc
      if mplsHwCapability.mplsTtlPipeDscpUniformModelSupported:
         options[ 'uniform' ] = _uniformHelpdesc
   return options

_dscpTunnelModeValMatcher = CliMatcher.DynamicKeywordMatcher(
   getMplsTermDscpModelKeywords,
   alwaysMatchInStartupConfig=True,
   passContext=True )

class MplsTunnelTerminationModelCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls tunnel termination model ttl TTL_MODEL dscp DSCP_MODEL'
   noOrDefaultSyntax = 'mpls tunnel termination model ...'
   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'termination': _tunnelTerminationNodeWTunTermGuard,
      'model': _tunnelTermModelMatcher,
      'ttl': _ttlMatcher,
      'TTL_MODEL': _ttlTunnelModeValMatcher,
      'dscp': _dscpMatcher,
      'DSCP_MODEL': _dscpTunnelModeValMatcher
   }
   handler = "MplsCliHandler.MplsTunnelTerminationModelCmd_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.MplsTunnelTerminationModelCmd_noOrDefaultHandler"

configMode.addCommandClass( MplsTunnelTerminationModelCmd )

#-------------------------------------------------------------------------
# [no] mpls tunnel encapsulation model ttl (pipe <ttlValue> | uniform)
# exp (pipe <expValue> | uniform)
#-------------------------------------------------------------------------
_tunnelEncapsulationMatcher = CliCommand.guardedKeyword( 'encapsulation',
                              helpdesc="Tunnel encapsulation configuration commands",
                              guard=mplsTunnelEncapsulationGuard )
_tunnelEncapModelMatcher = CliMatcher.KeywordMatcher(
   "model", helpdesc="tunnel encapsulation TTL and EXP model" )

_ttlPipeModeValueMatcher = CliMatcher.KeywordMatcher( 'pipe',
                           helpdesc="Specify TTL value for pipe mode propagation" )
_expPipeModeValueMatcher = CliMatcher.KeywordMatcher( 'pipe',
                           helpdesc="Specify EXP value for pipe mode propagation" )
_ttlUniformModeValueMatcher = CliMatcher.KeywordMatcher( 'uniform',
                              helpdesc="Propagate TTL value from inner header" )
_expUniformModeValueMatcher = CliMatcher.KeywordMatcher( 'uniform',
                              helpdesc="Propagate EXP value from inner header" )
_expMatcher = CliMatcher.KeywordMatcher( 'exp', helpdesc="model for exp" )

class MplsTunnelEncapsulationModelCmd( CliCommand.CliCommandClass ):

   syntax = ( 'mpls tunnel encapsulation model ttl'
              '( ( TTL_MODEL_PIPE TTL_VALUE ) | TTL_MODEL_UNIFORM )'
              'exp'
              '( ( EXP_MODEL_PIPE EXP_VALUE ) | EXP_MODEL_UNIFORM )' )
   noOrDefaultSyntax = 'mpls tunnel encapsulation model ttl ...'
   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'encapsulation': _tunnelEncapsulationMatcher,
      'model': _tunnelEncapModelMatcher,
      'ttl': _ttlMatcher,
      'TTL_MODEL_PIPE': _ttlPipeModeValueMatcher,
      'TTL_MODEL_UNIFORM': _ttlUniformModeValueMatcher,
      'TTL_VALUE': CliMatcher.IntegerMatcher( 0, 255,
         helpdesc='TTL value between 0 and 255' ),
      'exp': _expMatcher,
      'EXP_MODEL_PIPE': _expPipeModeValueMatcher,
      'EXP_MODEL_UNIFORM': _expUniformModeValueMatcher,
      'EXP_VALUE': CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='EXP value between 0 and 7' )
   }
   handler = "MplsCliHandler.MplsTunnelEncapsulationModelCmd_handler"
   noOrDefaultHandler = handler

configMode.addCommandClass( MplsTunnelEncapsulationModelCmd )

#-------------------------------------------------------------------------
# [no|default] mpls tunnel termination php model
#    ttl ( pipe | uniform | ) dscp ( pipe | uniform )
#-------------------------------------------------------------------------

_phpDscpTunnelModeValMatcher = CliCommand.Node(
   matcher=_ttlAndDscpTunnelModeValMatcher,
   guard=mplsPhpDscpUniformGuard )

class MplsTunnelTerminationPhpModelCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls tunnel termination php model ttl TTL_MODEL dscp DSCP_MODEL'
   noOrDefaultSyntax = 'mpls tunnel termination php model ...'
   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'termination': _tunnelTerminationNodeWTunTermGuard,
      'php': CliCommand.guardedKeyword( 'php',
         helpdesc="Penultimate hop popping configuration",
         guard=mplsPhpTtlGuard ),
      'model': _tunnelTermModelMatcher,
      'ttl': _ttlMatcher,
      'TTL_MODEL': _ttlAndDscpTunnelModeValMatcher,
      'dscp': _dscpMatcher,
      'DSCP_MODEL': _phpDscpTunnelModeValMatcher,
   }
   handler = "MplsCliHandler.MplsTunnelTerminationPhpModelCmd_handler"
   noOrDefaultHandler = handler

configMode.addCommandClass( MplsTunnelTerminationPhpModelCmd )

class TunnelTerminationMode( TunnelTerminationModeBase, BasicCli.ConfigModeBase ):
   name = "Mpls tunnel termination configuration"

   def __init__( self, parent, session ):
      TunnelTerminationModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getChildMode( self, context ):
      childMode = self.childMode( TunnelTerminationVrfMode,
                                  context=context )
      return childMode

class TunnelTerminationVrfMode( TunnelTerminationVrfModeBase,
                                BasicCli.ConfigModeBase ):
   name = "Mpls tunnel termination vrf configuration"

   def __init__( self, parent, session, vrfName='' ):
      TunnelTerminationVrfModeBase.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TunnelTerminationModeCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls tunnel termination'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'termination': _tunnelTerminationNodeWCliModeGuard,
   }
   handler = "MplsCliHandler.gotoTunnelTerminationMode"
   noOrDefaultHandler = "MplsCliHandler.clearTunnelTerminationMode"

configMode.addCommandClass( TunnelTerminationModeCmd )

class TunnelTerminationVrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'vrf': "vrf configuration",
      'VRF_NAME': vrfNameMatcher,
   }
   handler = "MplsCliHandler.TunnelTerminationVrfModeCmd_handler"
   noOrDefaultHandler = \
      "MplsCliHandler.TunnelTerminationVrfModeCmd_noOrDefaultHandler"

TunnelTerminationMode.addCommandClass( TunnelTerminationVrfModeCmd )

def getDscpToTcMapNameRule( mode ):
   completions = [ vrfQosMap.qosMap.mapName for vrfQosMap in
                   mplsTunnelTermVrfQosConfig.vrfQosMap.values()
                   if vrfQosMap.qosMap.mapType == QosMapType.mplsDecapDscpToTc ]
   return completions

matcherDscpToTcMapName = CliMatcher.DynamicNameMatcher( getDscpToTcMapNameRule,
      helpdesc='Dscp to Tc Map Name',
      pattern=r'(?!default-map$)[A-Za-z0-9_:{}\[\]-]+' )
class TunnelTerminationVrfQosMapCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map dscp to traffic-class MAP_NAME'
   noOrDefaultSyntax = 'qos map dscp to traffic-class ...'
   data = {
      'qos': CliCommand.guardedKeyword( 'qos',
         helpdesc="Configure QoS parameters",
         guard=mplsTunnelTerminationQosMapGuard ),
      'map': 'Mapping configuration of different QoS parameters',
      'dscp': 'Specify a DSCP match',
      'to': 'Map to Traffic-Class',
      'traffic-class': 'Specify a Traffic-Class match',
      'MAP_NAME': matcherDscpToTcMapName,
   }

   handler = "MplsCliHandler.TunnelTerminationVrfMode_addQosMapToVrf"
   noOrDefaultHandler = "MplsCliHandler.TunnelTerminationVrfMode_delQosMapToVrf"

TunnelTerminationVrfMode.addCommandClass( TunnelTerminationVrfQosMapCmd )

class LfibStitchingPreferencesMode( LfibStitchingPreferencesModeBase,
                                    BasicCli.ConfigModeBase ):
   name = "LFIB stitching preferences configuration"

   def __init__( self, parent, session ):
      LfibStitchingPreferencesModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class LfibStitchingPreferencesModeCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls lfib stitching preferences'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'lfib': lfibMatcher,
      'stitching': stitchingMatcher,
      'preferences': preferencesMatcher,
   }
   handler = "MplsCliHandler.gotoLfibStitchingPreferencesMode"
   noOrDefaultHandler = "MplsCliHandler.delLfibStitchingPreferencesMode"

if Toggles.MplsToggleLib.toggleLfibStitchingConfigurablePreferencesEnabled():
   configMode.addCommandClass( LfibStitchingPreferencesModeCmd )

# ---------------------------------------------------------
# [no|default] mpls parsing speculative < ipv4 | ipv6 | ethernet | control-word >
# ---------------------------------------------------------
def mplsParsingSupported( mode, token ):
   if mplsHwCapability.mplsModifyParsingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

_mplsParsingMatcher = CliCommand.guardedKeyword( 'parsing',
      helpdesc="Configure parsing of MPLS packets",
      guard=mplsParsingSupported )

_speculativeParsingMatcher = CliMatcher.KeywordMatcher(
      'speculative', helpdesc="Configure speculative parsing of different types of "
      "MPLS encapsulated packets for all data plane features" )

_parseIpv4Helpdesc = "Enable parsing MPLS encapsulated IPv4 packets"
_parseIpv6Helpdesc = "Enable parsing MPLS encapsulated IPv6 packets"
_parseEthHelpdesc = "Enable parsing MPLS encapsulated Ethernet packets"
_parsePwCwHelpdesc = "Enable parsing MPLS pseudowire packets with a control word"

_mplsParsingTypeMatcher = CliMatcher.EnumMatcher( {
   'ipv4': _parseIpv4Helpdesc,
   'ipv6': _parseIpv6Helpdesc,
   'ethernet': _parseEthHelpdesc,
   'control-word': _parsePwCwHelpdesc,
} )

class MplsParsingCmd( CliCommand.CliCommandClass ):
   syntax = "mpls parsing speculative PARSING_TYPE"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'parsing': _mplsParsingMatcher,
      'speculative': _speculativeParsingMatcher,
      'PARSING_TYPE': _mplsParsingTypeMatcher,
   }
   handler = "MplsCliHandler.MplsParsingCmd_handler"
   noHandler = "MplsCliHandler.MplsParsingCmd_noHandler"
   defaultHandler = "MplsCliHandler.MplsParsingCmd_defaultHandler"

configMode.addCommandClass( MplsParsingCmd )

# ---------------------------------------------------------
# show mpls parsing speculative
# ---------------------------------------------------------

class ShowMplsParsingSpeculativeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls parsing speculative"
   data = {
      'mpls': mplsNodeForShow,
      'parsing': _mplsParsingMatcher,
      'speculative': _speculativeParsingMatcher,
      'PARSING_TYPE': _mplsParsingTypeMatcher,
   }

   cliModel = MplsModel.MplsParsingSpeculativeTableModel
   handler = "MplsCliHandler.ShowMplsParsingSpeculativeCmd_handler"

BasicCli.addShowCommandClass( ShowMplsParsingSpeculativeCmd )

#-------------------------------------------------------------------------
# show mpls tunnel termination qos maps [ vrf <vrfName> ]
# -------------------------------------------------------------------------
class ShowTunnelTermVrfQosMaps( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls tunnel termination qos maps [ vrf VRF_NAME ]"
   data = {
      'mpls': mplsNodeForShow,
      'tunnel': tunnelKwMatcherMplsShow,
      'termination': _tunnelTerminationNodeWCliModeGuard,
      'qos': CliCommand.guardedKeyword( 'qos', helpdesc="Show QoS status",
                                        guard=mplsTunnelTerminationQosMapGuard ),
      'maps': 'Show various QoS mappings',
      'vrf': 'Vrf mapping status',
      'VRF_NAME': vrfNameMatcher,
   }
   handler = "MplsCliHandler.showTunnelTermVrfQosMaps"
   cliModel = MplsModel.MplsVrfDscpToTcMapAllModel

BasicCli.addShowCommandClass( ShowTunnelTermVrfQosMaps )

# -------------------------------------------------------------------------
# show mpls tunnel static
# [ ( index INDEX )
# | ( endpoint ( ADDR4 | PREFIX4 | ADDR6 | PREFIX6 ) )
# | ( name NAME ) ]
#-------------------------------------------------------------------------
tunnelAfterMplsMatcherForShow = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc="MPLS tunnel information and counters" )

class ShowMplsTunnelStaticCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mpls tunnel static '
               '[ ( index INDEX ) '
               '| ( endpoint ( ADDR4 | PREFIX4 | ADDR6 | PREFIX6 ) ) '
               '| ( name NAME ) ]' )
   # FIXME BUG200408: Un-hide IPv6 endpoints and nexthop in "mpls static tunnel"
   data = {
      'mpls': mplsNodeForShow,
      'tunnel': tunnelAfterMplsMatcherForShow,
      'static': 'Static MPLS tunnel information',
      'index': 'Match tunnel index',
      'INDEX': tunnelIndexMatcher,
      'endpoint': endpointMatcher,
      'ADDR4': IpAddrMatcher.ipAddrMatcher,
      'PREFIX4': IpAddrMatcher.ipPrefixExpr(
         'IP address',
         "Subnet's mask value",
         'IP address with mask length',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
      ),
      'ADDR6': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                hidden=True ),
      'PREFIX6': CliCommand.Node( matcher=Ip6AddrMatcher.ip6PrefixMatcher,
                                  hidden=True ),
      'name': 'Match tunnel name',
      'NAME': tunnelNameValMatcher,
   }
   cliModel = MplsModel.StaticTunnelTable
   handler = "MplsCliHandler.ShowMplsTunnelStaticCmd_handler"

BasicCli.addShowCommandClass( ShowMplsTunnelStaticCmd )

#-------------------------------------------------------------------------
# The "show mpls tunnel fib" command
#-------------------------------------------------------------------------
class ShowMplsTunnelFib( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls tunnel fib'
   data = {
            'mpls': mplsForShowNode,
            'tunnel': tunnelKwMatcherMplsShow,
            'fib': 'Tunnel forwarding information'
          }
   cliModel = MplsModel.MplsTunnelFib
   hidden = True
   handler = "MplsCliHandler.ShowMplsTunnelFib_handler"

BasicCli.addShowCommandClass( ShowMplsTunnelFib )

#------------------------------------------
# show mpls label ranges
#------------------------------------------
class ShowMplsLabelRangeCommand( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls label ranges'
   data = {
      'mpls': mplsForShowNode,
      'label': 'Show static and dynamically allocated labels',
      'ranges': 'Show static and dynamically allocated label ranges',
   }
   cliModel = MplsModel.MplsLabelRanges
   handler = "MplsCliHandler.ShowMplsLabelRangeCommand_handler"

BasicCli.addShowCommandClass( ShowMplsLabelRangeCommand )

#-----------------------------------------------------------------------------------
# show mpls hardware ale adj [ ( ( LABEL [ { LABELS } ] ) | pending ) ]
#-----------------------------------------------------------------------------------
class ShowMplsFecRequestCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls hardware ale adj [ LABELS | pending ]'
   data = {
      'mpls': mplsForShowNode,
      'hardware': hardwareMatcherForShow,
      'ale': aleMatcherForShow,
      'adj': 'Show MPLS ale adjacency command',
      'LABELS': _singleOrMultiLabelValMatcher,
      'pending': ( 'Show MPLS routes whose ale adjacency are not programmed or in '
                   'sync with hardware' ),
   }
   cliModel = MplsModel.MplsFecRequestTable
   handler = "MplsCliHandler.ShowMplsFecRequestCmd_handler"

BasicCli.addShowCommandClass( ShowMplsFecRequestCmd )

#--------------------------------------------------------------------------------
# show tech-support mpls graceful-restart
#--------------------------------------------------------------------------------
class ShowTechMplsGr( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tech-support mpls graceful-restart' )
   data = {
      'tech-support': TechSupportCli.techSupportKwMatcher,
      'mpls': "Show aggregated status details for Mpls",
      'graceful-restart': "Graceful restart information for Mpls agent",
   }
   privileged = True
   handler = "MplsCliHandler.ShowTechMplsGr_handler"

BasicCli.addShowCommandClass( ShowTechMplsGr )

# register it to show tech support
TechSupportCli.registerShowTechSupportCmd(
   '2019-09-27 14:32:19',
   cmds=[ 'show mpls hardware ale adj' ],
   cmdsGuard=lambda: mplsHwCapability.mplsSupported )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global em
   global cliChurnTestHelper
   global cliMplsRouteConfig
   global cliMplsRouteConfigReadOnly
   global decapLfib
   global decapLfibConsumer
   global decapLfibHw
   global evpnEthernetSegmentLfib
   global evpnEthernetSegmentLfibHw
   global flexAlgoConfig
   global ip6Config
   global ip6Status
   global ipConfig
   global ipStatus
   global l3Config
   global l3ConfigDir
   global l3NHResolver
   global bgpPeerInfoStatusDir
   global lfibInfo
   global mplsLfibSourceInfoDir
   global mldpOpaqueValueTable
   global gribiAfts
   global mplsHwCapability
   global mplsHwStatus
   global mplsRouteConfigDir
   global mplsRoutingConfig
   global mplsTunnelConfig
   global labelManagerStatus
   global mplsRoutingStatus
   global mplsVrfLabelConfig
   global mplsTunnelTermVrfQosConfig
   global mplsTunnelTermVrfHwStatus
   global pseudowireLfib
   global pwRouteHelper
   global routingHardwareStatus
   global routing6VrfInfoDir
   global routingVrfInfoDir
   global srTePolicyStatus
   global srteForwardingStatus
   global mplsVskFecIdCliCollection
   global staticMcastLfib
   global rsvpLerTunnelTable
   global staticTunnelConfig
   global staticTunnelTable
   global rsvpLerTunnelTable
   global transitLfib
   global systemTunnelFib
   global dsfTunnelFib
   global gueTunnelFib
   global tunnelFib
   global protocolTunnelNameStatus
   global lfibVskOverrideConfig
   global lfibViaSetStatus

   def doMountsComplete():
      global forwardingStatus
      global forwarding6Status
      global forwardingGenStatus
      global unifiedForwardingStatus
      global unifiedForwarding6Status
      global unifiedForwardingGenStatus

      fsType = "Smash::Fib::ForwardingStatus"
      fs6Type = "Smash::Fib6::ForwardingStatus"
      fsGenType = "Smash::FibGen::ForwardingStatus"
      if toggleFibGenMountPathEnabled():
         unifiedForwardingGenStatus = smashEm.doMount( "forwardingGen/unifiedStatus",
                                                       fsGenType, readerInfo )
         forwardingGenStatus = smashEm.doMount( "forwardingGen/status", fsGenType,
                                                readerInfo )
      else:
         unifiedForwardingStatus = smashEm.doMount( "forwarding/unifiedStatus",
                                                    fsType, readerInfo )
         unifiedForwarding6Status = smashEm.doMount( "forwarding6/unifiedStatus",
                                                     fs6Type, readerInfo )
         forwardingStatus = smashEm.doMount( "forwarding/status", fsType,
                                             readerInfo )
         forwarding6Status = smashEm.doMount( "forwarding6/status", fs6Type,
                                              readerInfo )

      initFecModeSm()
      initMplsRouteConfigMergeSm( entityManager )

   em = entityManager

   mplsRoutingConfig = ConfigMount.mount( entityManager, "routing/mpls/config",
                                          "Mpls::Config", "w" )
   cliMplsRouteConfig = ConfigMount.mount( entityManager,
                                           "routing/mpls/route/input/cli",
                                           "Mpls::RouteConfigInput", "wi" )
   mplsVrfLabelConfig = ConfigMount.mount( entityManager,
                                           "routing/mpls/vrfLabel/input/cli",
                                           "Mpls::VrfLabelConfigInput", "wi" )
   mplsTunnelTermVrfQosConfig = ConfigMount.mount( entityManager,
                                          "routing/mpls/vrfLabel/qosMap/input/cli",
                                          "Mpls::VrfQosMapConfig", "wi" )
   mplsTunnelTermVrfHwStatus = LazyMount.mount( entityManager,
                                             "routing/hardware/mpls/qosMap/status",
                                             "Mpls::VrfQosMapHwStatus", "r" )
   mplsTunnelConfig = ConfigMount.mount( entityManager,
                                         "routing/mpls/tunnel/config",
                                         "Tunnel::MplsTunnelConfig", "w" )
   lfibInfo = LazyMount.mount( entityManager, "routing/mpls/lfibInfo",
                               "Mpls::LfibInfo", "r" )
   mplsLfibSourceInfoDir = LazyMount.mount( em,
         "routing/mpls/lfibSourceInfo",
         "Tac::Dir",
         "riS" )
   mplsHwStatus = LazyMount.mount( entityManager, "routing/hardware/mpls/status",
                                   "Mpls::Hardware::Status", "rS" )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
   labelManagerStatus = LazyMount.mount( entityManager, "mpls/labelManagerStatus",
                                         "Mpls::Api::LabelManagerStatus", "r" )
   mplsRoutingStatus = LazyMount.mount( entityManager, "routing/mpls/status",
                                        "Mpls::Status", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                    "routing/vrf/routingInfo/status",
                                    "Tac::Dir", "ri" )
   routing6VrfInfoDir = LazyMount.mount( entityManager,
                                    "routing6/vrf/routingInfo/status",
                                    "Tac::Dir", "ri" )
   ipConfig = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ip6Config = ConfigMount.mount( entityManager, "ip6/config", "Ip6::Config", "w" )
   ip6Status = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   l3ConfigDir = ConfigMount.mount( entityManager, "l3/intf/config",
                                    "L3::Intf::ConfigDir", "w" )
   rsvpLerTunnelTable = readMountTunnelTable(
            TunnelTableIdentifier.rsvpLerTunnelTable, entityManager )
   gribiAfts = LazyMount.mount( entityManager, "routing/gribi/afts",
                                 "Gribi::GribiAfts", "r" )
   staticTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.staticTunnelTable, entityManager )
   rsvpLerTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.rsvpLerTunnelTable, entityManager )

   staticTunnelConfig = ConfigMount.mount( entityManager, "tunnel/static/config",
                                          "Tunnel::Static::Config", "w" )
   staticMcastLfib = ConfigMount.mount( entityManager, 'mpls/staticMcast/lfib',
                                  'Mpls::LfibSysdbStatus', 'w' )
   IntfCli.Intf.registerDependentClass( IntfMplsConfigCleaner, 20 )
   cliChurnTestHelper = LazyMount.mount( entityManager,
                                         "routing/mpls/cli/churnHelper",
                                         "Mpls::CliChurnTestHelper", "w" )

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   keyshadowInfo = Smash.mountInfo( 'keyshadow' )
   readerInfo = Smash.mountInfo( 'reader' )
   srTePolicyStatus = smashEm.doMount( srTePolicyStatusPath(),
                                       'SrTePolicy::PolicyStatus', readerInfo )
   systemTunnelFib = smashEm.doMount( 'tunnel/tunnelFib',
                                      'Tunnel::TunnelFib::TunnelFib',
                                      readerInfo )
   dsfTunnelFib = smashEm.doMount(
      'dsf/tunnelFib', "Tunnel::TunnelFib::TunnelFib", readerInfo )
   gueTunnelFib = smashEm.doMount(
      "tunnel/gueTunnelFib", "Tunnel::TunnelFib::TunnelFib", readerInfo )
   tunnelFib = Tac.newInstance( "Tunnel::TunnelFib::UnifiedTunnelFibView",
                                systemTunnelFib, dsfTunnelFib, gueTunnelFib )
   pseudowireLfib = smashEm.doMount( 'mpls/pseudowireLfib', 'Mpls::LfibStatus',
                                     readerInfo )
   transitLfib = smashEm.doMount( 'mpls/transitLfib', 'Mpls::LfibStatus',
                                  readerInfo )
   decapLfib = smashEm.doMount( "mpls/decapLfib", "Mpls::LfibStatus", keyshadowInfo )
   decapLfibHw = smashEm.doMount( "hardware/mpls/decapLfib", "Mpls::LfibHwStatus",
                                  readerInfo )
   evpnEthernetSegmentLfib = smashEm.doMount( "mpls/evpnEthernetSegmentFilterLfib",
                                              "Mpls::LfibStatus", readerInfo )
   evpnEthernetSegmentLfibHw = smashEm.doMount(
                                    "hardware/mpls/evpnEthernetSegmentFilterLfib",
                                    "Mpls::LfibHwStatus", readerInfo )
   srteForwardingStatus = smashEm.doMount( "forwarding/srte/status",
                                       "Smash::Fib::ForwardingStatus", readerInfo )

   mplsVskFecIdCliCollection = smashEm.doMount(
      "mpls/mplsVskFecIdColl", "Smash::Mpls::MplsVskFecIdCliCollection", readerInfo )
   lfibVskOverrideConfig = smashEm.doMount( "te/cbf/vskoverride/mpls/config",
                                            "Mpls::Override::LfibVskOverrideConfig",
                                            readerInfo )
   lfibViaSetStatus = smashEm.doMount( "mpls/cbf/overrideLfib",
                                       "Mpls::LfibViaSetStatus",
                                       readerInfo )

   decapLfibConsumer = Tac.newInstance( 'Mpls::LfibStatusConsumerSm', decapLfib )
   mldpOpaqueValueTable = LazyMount.mount( entityManager,
                                           "mpls/ldp/mldpOpaqueValueTable",
                                           "Mpls::MldpOpaqueValueTable", "r" )
   flexAlgoConfig = LazyMount.mount( entityManager, 'te/flexalgo/config',
                                     'FlexAlgo::Config', 'r' )

   protocolTunnelNameStatus = LazyMount.mount(
                                 entityManager,
                                 'tunnel/tunnelNameStatus',
                                 'Tunnel::TunnelTable::ProtocolTunnelNameStatus',
                                 'r' )

   # We need to use a mount group because:
   # 1) we need to defer initMplsRouteConfigMergeSm() until the mounts are mounted,
   # 2) we need to access a config mounted entity
   #
   # We do not use the ConfigMount instance of the config mounted entity since we
   # just need read access to the entity at the specified path for the merge sm.
   # The flag needs to be "w" even though it's read-only so avoid conflicting
   # permissions with ConfigMount
   mg = entityManager.mountGroup()
   l3Config = mg.mount( "l3/config", "L3::Config", "ri" )
   mplsRouteConfigDir = mg.mount( 'routing/mpls/route/input/dynamic',
                                  'Tac::Dir', 'ri' )
   cliMplsRouteConfigReadOnly = mg.mount( 'routing/mpls/route/input/cli',
                                          'Mpls::RouteConfigInput', 'w' )
   l3NHResolverPath = Tac.Type( 'Mpls::RouteConfigMergeSm' ).l3NHResolverPath()
   l3NHResolver = LazyMount.mount( entityManager, l3NHResolverPath,
                                   'Routing::NexthopStatus', 'r' )
   bgpPeerInfoStatusDirPath = Cell.path(
      'routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' )
   bgpPeerInfoStatusDir = LazyMount.mount( entityManager,
      bgpPeerInfoStatusDirPath, 'Tac::Dir', 'ri' )
   mg.close( doMountsComplete )

   ctx = Tac.newInstance( "Mpls::PluginContext" )
   ctx.entityManager = entityManager.cEntityManager()
   pluginLoader = Tac.newInstance( "Mpls::PluginLoader" )
   pluginLoader.loadPlugins( "MplsCliPlugin", ctx, "", "" )
   pwRouteHelper = ctx.pwRouteHelper

#--------------------------------------------------------------------------
# register show tech-support extended evpn
#--------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2017-11-03 12:06:08',
   cmds=[ 'show mpls route', 'show mpls label range' ],
   extended='evpn' )

