#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericPrefix
from CliModel import Bool, Float, Int, List, Model, Str
from TableOutput import createTable, Format
from CliPlugin import TunnelModels
from HumanReadable import formatValueSi

class MplsTunnelCountersEntry( Model ):
   tunnelIndex = Int( help="Index of the tunnel entry in the source tunnel table" )
   tunnelType = TunnelModels.TunnelType
   endpoint = IpGenericPrefix( help="Endpoint of the tunnel", optional=True )
   vias = List( valueType=TunnelModels.IpVia, help="List of nexthops" )
   txPackets = Int( help="Total number of transmitted packets", optional=True )
   txBytes = Int( help="Total number of transmitted bytes", optional=True )
   # counterInUse is set to None when the counter feature is not enabled.
   # counterInUse is set to True to indicate that the counter for this tunnel is
   # currently active and in use.
   # counterInUse is set to False to indicate that the tunnel is not currently being
   # counted due to either a lack of available counter hardware resources or a
   # temporary unresolved tunnel state caused by an interface flap.
   counterInUse = Bool( help="Counter is in use for the tunnel entry",
         optional=True )
   tunnelName = Str( help="Name for the tunnel entry", optional=True )

   def renderMplsTunnelCountersTableRow( self, table ):
      txPacketsStr = '-'
      txBytesStr = '-'
      tunnelTypeStr = '-'
      tunnelEndpointStr = '-'

      txPacketsStr = self.txPackets if self.counterInUse else '-'
      txBytesStr = self.txBytes if self.counterInUse else '-'
      tunnelTypeStr = self.tunnelType
      tunnelNameStr = self.tunnelName if self.tunnelName is not None else ''

      if self.endpoint:
         tunnelEndpointStr = str( self.endpoint )

      table.newRow( tunnelTypeStr, self.tunnelIndex, tunnelEndpointStr,
                    txPacketsStr, txBytesStr, tunnelNameStr )

class MplsTunnelCounters( Model ):
   entries = List( valueType=MplsTunnelCountersEntry,
                   help="MPLS tunnel hardware counters" )
   _tableOutput = Bool( help="True if table output format was requested" )

   def render( self ):
      headings = ( "Tunnel Type", "Index", "Endpoint", "Packets", "Bytes",
                   "Tunnel Name" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings, tableWidth=79 )
      table.formatColumns( fl, fr, fl, fr, fr, fl )

      for mplsTunnelCountersEntry in self.entries:
         mplsTunnelCountersEntry.renderMplsTunnelCountersTableRow( table )

      print( table.output() )

class MplsTunnelCountersRateEntry( Model ):
   tunnelIndex = Int( help="Index of the tunnel entry in the source tunnel table" )
   tunnelType = TunnelModels.TunnelType # Enum.
   endpoint = IpGenericPrefix( help="Endpoint of the tunnel", optional=True )
   txPps = Float( help="Packets per second", optional=True )
   txBps = Float( help="Bits per second", optional=True )
   tunnelName = Str( help="Name for the tunnel entry", optional=True )

   def renderMplsTunnelCountersRateEntry( self, table ):
      tunnelTypeStr = self.tunnelType # Required.
      tunnelIndexStr = self.tunnelIndex # Required.

      tunnelEndpointStr = str( self.endpoint or '-' )
      txPpsStr = formatValueSi( self.txPps, precision=4, unit='pps' ) \
         if self.txPps is not None else '-'
      txBpsStr = formatValueSi( self.txBps, precision=4, unit='bps' ) \
         if self.txBps is not None else '-'
      tunnelNameStr = self.tunnelName or ''

      table.newRow( tunnelTypeStr, tunnelIndexStr, tunnelEndpointStr, txPpsStr,
                    txBpsStr, tunnelNameStr )

class MplsTunnelCountersRate( Model ):
   entries = List( valueType=MplsTunnelCountersRateEntry,
                   help="MPLS tunnel hardware rates" )

   def render( self ):
      headings = ( "Tunnel Type", "Index", "Endpoint", "Packet Rate", "Bit Rate",
                   "Tunnel Name" )
      fl = Format( justify='left' )
      fl.padLimitIs( True )
      fr = Format( justify='right' )
      fr.padLimitIs( True )

      table = createTable( headings, tableWidth=79 )
      table.formatColumns( fl, fr, fl, fr, fr, fl )

      for mplsTunnelCountersRateEntry in self.entries:
         mplsTunnelCountersRateEntry.renderMplsTunnelCountersRateEntry( table )

      print( table.output() )

class MplsTunnelInterfaceCountersEntry( Model ):
   txPackets = Int( help="Total number of transmitted packets" )
   txBytes = Int( help="Total number of transmitted bytes" )

