#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import

import AclCliLib
import Arnet
import Arnet.MplsLib
import BasicCli
import CliCommand
from ClientCommonLib import (
   getThreadLocalData,
   setThreadLocalData,
)
import CliMatcher
from CliMode.MplsUtilsMode import MplsUtilsMode
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliModel as AclCliModel
import CliPlugin.IntfCli as IntfCli
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliPlugin.IraIpCli as IraIpCli
import CliPlugin.IraNexthopGroupCli as IraNexthopGroupCli
import CliPlugin.MacAddr as MacAddr
import CliPlugin.MplsCli as MplsCli
import CliPlugin.NetworkToolsCli as NetworkToolsCli
import CliPlugin.VrfCli as VrfCli
import CliToken.Ip
import CliToken.Clear
import ConfigMount
from enum import Enum
import LazyMount
from MplsPingClientLib import getPingModel
from MplsTracerouteClientLib import getTracerouteModel
import ShowCommand
from SrTePolicyCommonLib import tacColor
from TypeFuture import TacLazyType
import Toggles
import Toggles.MplsUtilsToggleLib as Toggle

aclConfig = None
aclCpConfig = None
allNhgStatus = None
flexAlgoConfig = None
mplsHwCapability = None
rsvpLerCliClient = None

RsvpCliId = TacLazyType( 'Rsvp::RsvpCliId' )
RsvpLibConstants = TacLazyType( 'Rsvp::RsvpLibConstants' )
U32_MAX_VALUE = 0xFFFFFFFF
maxTtlVal = 64
TunnelSourceCli = TacLazyType( 'Rsvp::RsvpLerTunnelSource' ).tunnelSourceCli

def mplsLabelRange( mode, context ):
   return ( Arnet.MplsLib.labelMin, Arnet.MplsLib.labelMax )

def getNhgNames( mode ):
   # We could just reutrn all MPLS nexthop-groups here.
   # But since LspPingUtil will check this anyway, it should be fine to
   # return all nexthop-groups.
   return sorted( allNhgStatus.nexthopGroupNameToId.members() )

def nhgEntryIndexRange( mode, context ):
   return ( 0, IraNexthopGroupCli.nexthopGroupSizeMax( mode ) - 1 )

# ping and traceroute mpls currently don't support non-default VRF.
# This guard will prevent to use 'vrf' option in the cli until the VRF support is
# implemented.
def mplsUtilvVrfSupportedGuard( mode, token ):
   return CliParser.guardNonDefaultVrf

nodeMpls = CliCommand.Node( CliMatcher.KeywordMatcher( 'mpls',
      helpdesc='Send echo messages over a LSP' ),
      guard=MplsCli.mplsSupportedGuard )

# As long as the only purpose of the 'mpls ping' CLI is to configure ACLs, this
# token will be guarded by the hwEpoch of service ACLs. This will prevent both
# configuring ACLs and showing/clearing counters.
matcherPing = CliMatcher.KeywordMatcher( 'ping',
      helpdesc='MPLS Ping and Traceroute configuration' )

matcherAddrOrPrefix = IpAddrMatcher.IpAddrOrPrefixExprFactory(
   'Destination address', 'Destination prefix mask', 'Destination prefix',
   overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT )
matcherAddrOrPrefix6 = Ip6AddrMatcher.Ip6AddrOrPrefixExprFactory(
   overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT )
matcherBgp = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP route' )
matcherBgpLu = CliMatcher.KeywordMatcher( 'labeled-unicast',
                                          helpdesc='Labeled Unicast route' )
matcherBgpNexthop = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP next hop' )
matcherVpn = CliMatcher.KeywordMatcher( 'vpn', helpdesc='VPN route' )
matcherRaw = CliMatcher.KeywordMatcher( 'raw', helpdesc='Raw route' )
matcherLdp = CliMatcher.KeywordMatcher( 'ldp', helpdesc='LDP route' )
matcherRsvp = CliMatcher.KeywordMatcher( 'rsvp', helpdesc='RSVP route' )
matcherGeneric = CliMatcher.KeywordMatcher( 'generic',
      helpdesc='Generic route when protocol may change over LSP' )
matcherSegmentRouting = CliMatcher.KeywordMatcher( 'segment-routing',
      helpdesc='Segment route' )
matcherStatic = CliMatcher.KeywordMatcher(
                             'static', helpdesc='Static MPLS push label route' )
matcherNhg = CliMatcher.KeywordMatcher( 'nexthop-group',
      helpdesc='MPLS nexthop group' )
matcherTunnel = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc='Tunnel interface' )
matcherNhgForTunnel = CliMatcher.KeywordMatcher( 'nexthop-group',
      helpdesc='Nexthop group tunnel' )
matcherSrte = CliMatcher.KeywordMatcher( 'srte', helpdesc='SR-TE Policy' )
matcherEndpoint = CliMatcher.KeywordMatcher( 'endpoint',
      helpdesc='SR-TE Policy endpoint' )
matcherColor = CliMatcher.KeywordMatcher( 'color', helpdesc='SR-TE Policy color' )
matcherColorVal = CliMatcher.IntegerMatcher( tacColor.min, tacColor.max,
      helpdesc='Color Value' )
matcherTrafficAf = CliMatcher.KeywordMatcher( 'traffic-af',
      helpdesc='The type of traffic to be steered through the policy' )
matcherV4 = CliMatcher.KeywordMatcher( 'v4', helpdesc='IPv4 traffic' )
matcherV6 = CliMatcher.KeywordMatcher( 'v6', helpdesc='IPv6 traffic' )
matcherP2mp = CliMatcher.KeywordMatcher( 'p2mp',
      helpdesc='Point To MultiPoint Route' )
matcherMldp = CliMatcher.KeywordMatcher( 'mldp', helpdesc='Mldp Route' )
matcherLsdId = CliMatcher.KeywordMatcher( 'lsp-id',
      helpdesc='LSP ID opaque type' )
matcherMulticast = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc='Multicast opaque type' )
matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4', helpdesc='Ipv4 type' )
matcherIp = CliMatcher.KeywordMatcher( 'ip', helpdesc='IPv4 route' )
matcherIp6 = CliMatcher.KeywordMatcher( 'ipv6', helpdesc='IPv6 route' )
matcherPrefix = IpAddrMatcher.ipPrefixExpr( 'Destination address',
                                            'Destination prefix mask',
                                            'Destination prefix',
                                       overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT )
matcherPrefix6 = Ip6AddrMatcher.Ip6PrefixValidMatcher( 'IPv6 Address prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT )
matcherAlgorithm = CliMatcher.KeywordMatcher( 'algorithm',
                                              helpdesc='Flexible algorithm' )
matcherFlexAlgoName = CliMatcher.DynamicNameMatcher(
   lambda mode: ( [ fad.name for fad in flexAlgoConfig.definition.values() ]
                  if flexAlgoConfig else [] ),
   helpdesc='Flexible algorithm name' )
matcherSession = CliMatcher.KeywordMatcher( 'session', helpdesc='RSVP session' )
matcherSubTunnel = CliMatcher.KeywordMatcher( 'sub-tunnel',
      helpdesc='RSVP sub-tunnel' )
matcherSubTunnelId = CliMatcher.KeywordMatcher( 'id',
      helpdesc='Specify sub-tunnel by ID' )
matcherSubTunnelIdVal = CliMatcher.IntegerMatcher( 1,
      RsvpLibConstants.maxSubTunnelLimit, helpdesc='RSVP sub-tunnel range' )
matcherId = CliMatcher.KeywordMatcher( 'id', helpdesc='Specify session by ID' )
matcherIdVal = CliMatcher.IntegerMatcher( RsvpCliId.minValue, RsvpCliId.maxValue,
      helpdesc='RSVP session ID' )
matcherName = CliMatcher.KeywordMatcher( 'name', helpdesc='Specify session by name' )
matcherNameVal = CliMatcher.PatternMatcher( '[!-~]+', helpname='WORD',
      helpdesc='RSVP session name' )
matcherLsp = CliMatcher.KeywordMatcher( 'lsp', helpdesc='Specify LSP' )
matcherLspVal = CliMatcher.IntegerMatcher( RsvpCliId.minValue, RsvpCliId.maxValue,
      helpdesc='RSVP LSP ID' )
nodeEgressValidate = CliCommand.singleKeyword( 'egress-validate',
      helpdesc='Perform validation for this Label Switched Path at the egress' )
nodeAddress = CliCommand.singleKeyword( 'address',
      helpdesc=( 'endpoint address of the Label Switched Path that needs to be '
                 'validated' ) )
nodeLabel = CliCommand.guardedKeyword( 'label',
      helpdesc='Push an MPLS label', guard=MplsCli.mplsSupportedGuard )
matcherLabelValue = CliMatcher.DynamicIntegerMatcher( mplsLabelRange,
      helpdesc='Value of the MPLS label' )
matcherOpaqueVal = CliMatcher.IntegerMatcher( 1, U32_MAX_VALUE,
      helpdesc='lsp-id value' )
matcherNexthopGroup = CliMatcher.DynamicNameMatcher( getNhgNames,
      'Nexthop group name' )
matcherEntry = CliMatcher.KeywordMatcher( 'entry',
      helpdesc='Index value for the entry' )
matcherEntryIndex = CliMatcher.DynamicIntegerMatcher( nhgEntryIndexRange,
      helpdesc='Index value for the entry' )
matcherNexthop = CliMatcher.KeywordMatcher( 'nexthop',
      helpdesc='IP Address of nexthop' )
matcherNexthopIpAddr = IpAddrMatcher.IpAddrMatcher( 'Nexthop IP Address' )
matcherNexthopIp6Addr = Ip6AddrMatcher.Ip6AddrMatcher( 'Nexthop IPv6 Address' )
matcherDownstream = CliMatcher.KeywordMatcher( 'downstream',
      helpdesc='Downstream mapping information validation' )
matcherValidation = CliMatcher.KeywordMatcher( 'validation',
      helpdesc='Downstream mapping information validation' )
matcherEnabled = CliMatcher.KeywordMatcher( 'enabled',
      helpdesc='Enable downstream mapping information validation' )
lspPingVrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='LSP ping in VRF',
      guard=mplsUtilvVrfSupportedGuard )
lspTracerouteVrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='LSP traceroute in VRF',
      guard=mplsUtilvVrfSupportedGuard )
vrfExprFactory = VrfCli.VrfExprFactory( helpdesc='VRF name', inclDefaultVrf=True )
matcherSegmentTunnelIsis = CliMatcher.KeywordMatcher( 'isis',
      helpdesc='IS-IS Segment Routing Tunnel' )
matcherSegmentTunnelOspf = CliMatcher.KeywordMatcher( 'ospf',
      helpdesc='OSPF Segment Routing Tunnel' )
matcherNhgEntryBackupPing = CliMatcher.KeywordMatcher( 'backup',
      helpdesc='Ping backup entries/entry of NHG' )
matcherNhgEntryBackupTraceroute = CliMatcher.KeywordMatcher( 'backup',
      helpdesc='Traceroute backup entry of NHG' )

class LspOptionType( Enum ):
   ping = 'ping'
   mldpPing = 'mldpPing'
   traceroute = 'traceroute'
   mldpTraceroute = 'mldpTraceroute'
   mplsTracerouteSr = 'mplsTracerouteSr'
   ldpTraceroute = 'ldpTraceroute'

class SourceExpr( CliCommand.CliExpression ):
   expression = 'source SRC_IP'
   data = {
      'source' : CliCommand.singleKeyword( 'source',
         helpdesc='Specify source address' ),
      'SRC_IP' : IpGenAddrMatcher.IpGenAddrMatcher(
         'IPv4 or IPv6 source address',
         helpdesc4='IPv4 address used as a source',
         helpdesc6='IPv6 source address' )
   }

class TrafficClassExpr( CliCommand.CliExpression ):
   expression = 'traffic-class TRAFFIC_CLASS'
   data = {
      'traffic-class' : CliCommand.singleKeyword( 'traffic-class',
         helpdesc='Specify MPLS traffic class field' ),
      'TRAFFIC_CLASS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='MPLS traffic class field value' )
   }

class StandardExpr( CliCommand.CliExpression ):
   expression = 'standard STANDARD'
   data = {
      'standard' : CliCommand.singleKeyword( 'standard',
         helpdesc='Set the standard to comply with' ),
      'STANDARD' : CliMatcher.EnumMatcher( {
         'arista' : ( 'Comply with the Arista standard, '
                      'which allows interworking with older EOS devices' ),
         'ietf' : 'Comply with the IETF standard'
      } ),
   }

class LspOptionExprFactory( CliCommand.CliExpressionFactory ):
   '''
   The LSP protocols won't necessarily have the same options across all of them. This
   generator class lets us customize this by passing in other specific options that
   will be present in the generated CliExpression object.
   '''
   def __init__( self, optionType, multiLabel=False, egressValidate=False,
                 dsType=False ):
      self.optionType = optionType
      self.multiLabel = multiLabel
      self.egressValidate = egressValidate
      self.dsType = dsType
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      # Base option expr/data that's used by all protocols' CLI traceroute/ping cmds
      optionExpr = ( 'SOURCE | STANDARD | TRAFFIC_CLASS | TOS | '
                     'pad-reply | ( size SIZE ) ' )
      optionData = {
         'SOURCE' : SourceExpr,
         'STANDARD' : StandardExpr,
         'TRAFFIC_CLASS' : TrafficClassExpr,
         'TOS' : AclCliLib.TosExpr,
         'pad-reply' : CliCommand.singleKeyword( 'pad-reply',
            helpdesc='Indicates that the reply should copy the Pad TLV' ),
         'size' : CliCommand.singleKeyword( 'size',
            helpdesc='Specify packet size in bytes' ),
         'SIZE' : CliMatcher.IntegerMatcher( 120, 10000,
            helpdesc='Packet size in bytes' ),
      }

      # Add the specific expr/data fields based on the given option type
      # pylint: disable-next=consider-using-in
      if ( self.optionType == LspOptionType.ping or
           self.optionType == LspOptionType.mldpPing ):
         optionExpr += ( '| ( repeat REPEAT ) '
                         '| ( interval INTERVAL ) ' )
         optionData.update( { 'repeat' : NetworkToolsCli.repeatKwNode,
                              'REPEAT' : NetworkToolsCli.countRangeMatcher,
                              'interval' : CliCommand.singleKeyword( 'interval',
                                 helpdesc='Specify interval between requests' ),
                              'INTERVAL' : CliMatcher.IntegerMatcher( 0,
                                 U32_MAX_VALUE,
                                 helpdesc='interval in units of seconds' )
                              } )

      # pylint: disable-next=consider-using-in
      elif ( self.optionType == LspOptionType.traceroute or
             self.optionType == LspOptionType.mldpTraceroute ):
         optionExpr += ( '| ( max-ttl TTL ) '
                         '| ( timeout TIME ) ' )
         optionData.update( {
            'max-ttl' : CliCommand.singleKeyword( 'max-ttl',
               helpdesc='Maximum Time-To-Live' ),
            'TTL' : CliMatcher.IntegerMatcher( 1, maxTtlVal, helpdesc='TTL range' ),
            'timeout' : CliCommand.singleKeyword( 'timeout',
               helpdesc='Time for echo reply to be received' ),
            'TIME' : CliMatcher.IntegerMatcher( 0, U32_MAX_VALUE,
               helpdesc='timeout for echo reply' )

         } )
      elif self.optionType == LspOptionType.ldpTraceroute:
         optionExpr += '| multipath | ( base BASE ) | ( count COUNT )'
         optionData.update( {
            'multipath' : CliCommand.singleKeyword( 'multipath',
               helpdesc='Enable multipath traceroute' ),
            'base' : CliCommand.singleKeyword( 'base',
               helpdesc='Base IP to use in multipath traceroute' ),
            'BASE' : IpGenAddrMatcher.IpGenAddrMatcher(
               'IP base address',
               helpdesc4='IPv4 base address',
               helpdesc6='IPv6 base address' ),
            'count' : CliCommand.singleKeyword( 'count',
               helpdesc='Number of IPs to probe in multipath traceroute' ),
            'COUNT' : CliMatcher.IntegerMatcher( 1, 512,
               helpdesc='Number of IPs to probe in multipath traceroute' )
         } )
      elif self.optionType == LspOptionType.mplsTracerouteSr:
         # don't add any options right now
         pass
      else:
         raise ValueError( f'Invalid optionType: {self.optionType}' )
      # extra args for mldp ping and traceroute
      # pylint: disable-next=consider-using-in
      if ( self.optionType == LspOptionType.mldpPing or
           self.optionType == LspOptionType.mldpTraceroute ):
         optionExpr += ( '| ( jitter JITTER ) '
                         '| ( node-responder-id RESPADDR ) ' )
         optionData.update( {
            'jitter' : CliCommand.singleKeyword( 'jitter', helpdesc='Echo Jitter' ),
            'JITTER' : CliMatcher.IntegerMatcher( 1, U32_MAX_VALUE,
               helpdesc='jitter value in millisecs' ),
            'node-responder-id' : CliCommand.singleKeyword( 'node-responder-id',
               helpdesc='Node responder address' ),
            'RESPADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
               helpdesc='Node Responder Address' )
         } )
      if self.multiLabel:
         optionExpr += '| ( multi-label MULTI_LABEL ) '
         optionData.update( {
            'multi-label' : CliCommand.singleKeyword( 'multi-label',
               helpdesc=( 'Perform validation using the specified number of '
                          'top-labels during route lookups' ) ),
            # Change this to use an integer matcher if we ever support
            # more than 2 labels
            'MULTI_LABEL' : CliCommand.singleKeyword(
               str( MplsCli.maxIngressMplsTopLabels ),
               helpdesc=( 'Number of top-labels to use for validation '
                          'during route lookups' ) )
         } )

      if self.egressValidate:
         optionExpr += '| ( egress-validate [ address EGRESS_ADDRESS ] ) '
         optionData.update( { 'egress-validate' : nodeEgressValidate,
                             'address' : nodeAddress,
                             'EGRESS_ADDRESS' : IpGenAddrMatcher.IpGenAddrMatcher(
                                helpdesc='Expected egress address' ) } )

      if self.dsType:
         optionExpr += '| DS_TYPE '
         optionData.update( {
            'DS_TYPE' : CliCommand.singleNode(
               matcher=CliMatcher.EnumMatcher( {
                  'dsmap' : 'Use the Downstream Mapping TLV',
                  'ddmap' : 'Use the Downstream Detailed Mapping TLV',
               } ) ),
         } )

      class LspOptionExpr( CliCommand.CliExpression ):
         expression = '{ %s }' % optionExpr # pylint: disable=consider-using-f-string
         data = optionData

         @staticmethod
         def adapter( mode, args, argsList ):
            options = set()
            # No need to conditionally create this dict - just put all possible args
            # in here, since they're gated by the actual expression syntax
            optionValDict = {
               'base' : 'BASE',
               'count' : 'COUNT',
               'multi-label' : 'MULTI_LABEL',
               'multipath' : 'multipath',
               'pad-reply' : 'pad-reply',
               'repeat' : 'REPEAT',
               'size' : 'SIZE',
               'source' : 'SRC_IP',
               'standard' : 'STANDARD',
               'traffic-class' : 'TRAFFIC_CLASS',
               'max-ttl' : 'TTL',
               'timeout' : 'TIME',
               'jitter' : 'JITTER',
               'node-responder-id' : 'RESPADDR',
               'interval' : 'INTERVAL',
               }
            for option, optionVal in optionValDict.items():
               if option in args:
                  val = args[ optionVal ]
                  val = val[ 0 ] if isinstance( val, list ) else val
                  options.add( ( option, val ) )
            args[ name ] = options
            if 'DS_TYPE' in args:
               options.add( ( 'dstype', args[ 'DS_TYPE' ] ) )
            if 'egress-validate' in args:
               if 'address' in args:
                  egressAddress = args[ 'EGRESS_ADDRESS' ]
                  if isinstance( egressAddress, list ):
                     egressAddress = egressAddress[ 0 ]
                  options.add( ( 'egress-validate', egressAddress ) )
               else:
                  options.add( ( 'egress-validate', 'default' ) )
            if 'tos' in args:
               tosVal, dscpVal = 0, None
               if 'dscp' in args:
                  dscpVal = args.get( 'DSCP' )
                  if isinstance( dscpVal, list ):
                     dscpVal = dscpVal[ 0 ]
                  if not dscpVal:
                     dscpAclVal = args.get( 'DSCP_ACL' )
                     if isinstance( dscpAclVal, list ):
                        dscpAclVal = dscpAclVal[ 0 ]
                     dscpVal, _ = AclCliLib.dscpValueFromCli( mode, dscpAclVal )
                  tosVal = dscpVal << 2 # dscp val is first 6 bits of tos
               else:
                  tosVal = args[ 'TOS_VAL' ]
                  if isinstance( tosVal, list ):
                     tosVal = tosVal[ 0 ]
               options.add( ( 'tos', tosVal ) )
      return LspOptionExpr

lspPingOptionsExpr = LspOptionExprFactory( LspOptionType.ping )
lspTracerouteOptionsExpr = LspOptionExprFactory( LspOptionType.traceroute )

def genPing():
   while True:
      cmdBreak = getThreadLocalData( 'cmdBreak' )
      # cmdBreak is only set in getPingModel function
      # if socket listening times out and subprocess is not running
      if cmdBreak:
         break
      setThreadLocalData( 'render', False )
      getPingModel()
      setThreadLocalData( 'cache', None )
   setThreadLocalData( 'cmdBreak', False )
   return getThreadLocalData( 'pingHdr' )

def genTraceroute():
   while True:
      cmdBreak = getThreadLocalData( 'cmdBreak' )
      # cmdBreak is only set in getTracerouteModel function
      # if socket listening times out and subprocess is not running
      if cmdBreak:
         break
      setThreadLocalData( 'render', False )
      getTracerouteModel()
      setThreadLocalData( 'cache', None )
   setThreadLocalData( 'cmdBreak', False )
   return getThreadLocalData( 'tracerouteModel' )
#--------------------------------------------------------------------------------
# ping mpls [ VRF ] raw label LABELS ( ( nexthop NEXTHOP ) | ( MACADDR INTF ) )
#                                                                        [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingRawCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ping mpls [ VRF ] raw label { LABELS } '
                           '( ( nexthop NEXTHOP ) | ( MACADDR INTF ) ) [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'raw' : matcherRaw,
      'label' : nodeLabel,
      'LABELS' : Arnet.MplsLib.LabelValsWithExpNullExprFactory(),
      'nexthop' : matcherNexthop,
      'NEXTHOP' : matcherNexthopIpAddr,
      'MACADDR' : MacAddr.macAddrMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'OPTIONS' : lspPingOptionsExpr
   }
   handler = "MplsUtilCliHandler.handlerPingRawCmd"

BasicCli.EnableMode.addCommandClass( PingRawCmd )

# ------------------------------------------------------------------------------
# Mpls Utils configuration mode
# ------------------------------------------------------------------------------
class MplsUtilsConfigMode( MplsUtilsMode, BasicCli.ConfigModeBase ):
   name = 'Mpls Utils Configuration'

   def __init__( self, parent, session, vrfName='default' ):
      self.vrfName = vrfName
      self.mode = MplsUtilsConfigMode
      MplsUtilsMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
#--------------------------------------------------------------------------------
# ping mpls [ VRF ] ldp ip IP_ADDRESS [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingLdpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'ping mpls [ VRF ] ldp ip IP_ADDRESS [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'ldp' : matcherLdp,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'OPTIONS' : lspPingOptionsExpr
   }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingLdpCmd"

BasicCli.EnableMode.addShowCommandClass( PingLdpCmd )

#--------------------------------------------------------------------------------
# ping mpls p2mp mldp ROOTADDR ( ( lsp-id OPQVAL ) |
#                                ( multicast ipv4 SOURCEADDR GROUPADDR ) )
#                                [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingMplsP2mpMldpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'ping mpls p2mp mldp ROOTADDR '
              '( ( lsp-id OPQVAL ) | ( multicast ipv4 SOURCEADDR GROUPADDR ) ) '
              '[ OPTIONS ]'
            )
   data = {
           'ping' : NetworkToolsCli.pingMatcher,
           'mpls' : nodeMpls,
           'p2mp' : matcherP2mp,
           'mldp' : matcherMldp,
           'ROOTADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                             'Root Ip Address',
                             helpdesc4='IPv4 Root Address',
                             helpdesc6='IPv6 Root Address' ),
           'lsp-id' : matcherLsdId,
           'OPQVAL' : matcherOpaqueVal,
           'multicast' : matcherMulticast,
           'ipv4' : matcherIpv4,
           'SOURCEADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                             'IP address of the source',
                             helpdesc4='IPv4 Source Address',
                             helpdesc6='IPv6 Source Address' ),
           'GROUPADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                             'Group Ip Address',
                             helpdesc4='IPv4 Group Address',
                             helpdesc6='IPv6 Group Address' ),
           'OPTIONS' : LspOptionExprFactory( LspOptionType.mldpPing )
          }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingMplsP2mpMldpCmd"

BasicCli.EnableMode.addShowCommandClass( PingMplsP2mpMldpCmd )

#--------------------------------------------------------------------------------
# traceroute mpls p2mp mldp ROOTADDR ( ( lsp-id OPQVAL ) |
#                                      ( multicast ipv4 SOURCEADDR GROUPADDR ) )
#                                    [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteMplsP2mpMldpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'traceroute mpls p2mp mldp ROOTADDR '
              '( ( lsp-id OPQVAL ) | ( multicast ipv4 SOURCEADDR GROUPADDR ) ) '
              '[ OPTIONS ]'
            )
   data = {
            'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'p2mp' : matcherP2mp,
            'mldp' : matcherMldp,
            'ROOTADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                              'Root Ip Address',
                              helpdesc4='IPv4 Root Address',
                              helpdesc6='IPv6 Root Address' ),
            'lsp-id' : matcherLsdId,
            'OPQVAL' : matcherOpaqueVal,
            'multicast' : matcherMulticast,
            'ipv4' : matcherIpv4,
            'SOURCEADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                                'IP address of the source',
                                helpdesc4='IPv4 Source Address',
                                helpdesc6='IPv6 Source Address' ),
            'GROUPADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                               'Group Ip Address',
                                helpdesc4='IPv4 Source Address',
                                helpdesc6='IPv6 Source Address' ),
            'OPTIONS' : LspOptionExprFactory( LspOptionType.mldpTraceroute )
           }

   cliModel = "MplsUtilModel.MplsTraceroute"
   handler = "MplsUtilCliHandler.handlerTracerouteMplsP2mpMldpCmd"

BasicCli.EnableMode.addShowCommandClass( TracerouteMplsP2mpMldpCmd )

# --------------------------------------------------------------------------------
# ping mpls [ VRF ] generic ip IP_ADDRESS [ OPTIONS ]
# --------------------------------------------------------------------------------
class PingGenericCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'ping mpls [ VRF ] generic ip IP_ADDRESS [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'generic' : matcherGeneric,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'OPTIONS' : lspPingOptionsExpr
   }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingGenericCmd"

BasicCli.EnableMode.addCommandClass( PingGenericCmd )

#--------------------------------------------------------------------------------
# ping mpls segment-routing ( ( ip IP_ADDRESS ) | ( ipv6 IPV6_ADDRESS ) )
#    [ algorithm ALGO_NAME ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingSrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'ping mpls [ VRF ] segment-routing ( ( ( ( ip IP_ADDRESS )'
              ' | ( ipv6 IPV6_ADDRESS ) | ( isis ip IP_ADDRESS )'
              ' | ( isis ipv6 IPV6_ADDRESS ) ) [ algorithm ALGO_NAME ] )'
              ' | ( ospf ip IP_ADDRESS ) ) [ OPTIONS ]' )

   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'segment-routing' : matcherSegmentRouting,
      'isis' : matcherSegmentTunnelIsis,
      'ospf' : matcherSegmentTunnelOspf,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'ipv6' : matcherIp6,
      'IPV6_ADDRESS' : matcherPrefix6,
      'algorithm' : matcherAlgorithm,
      'ALGO_NAME' : matcherFlexAlgoName,
      'OPTIONS' : lspPingOptionsExpr
   }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingSrCmd"

BasicCli.EnableMode.addShowCommandClass( PingSrCmd )

#--------------------------------------------------------------------------------
# ping mpls static ( ip IP_ADDRESS | ipv6 IPV6_ADDRESS )
#                             [ egress-validate address EGRESS_ADDRESS ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingMplsStaticCmd( CliCommand.CliCommandClass ):
   syntax = '''ping mpls static ( ( ip IP_ADDRESS ) | ( ipv6 IPV6_ADDRESS ) )
               [ egress-validate address EGRESS_ADDRESS ] [ OPTIONS ]'''
   data = { 'ping' : NetworkToolsCli.pingMatcher,
            'mpls' : nodeMpls,
            'static' : matcherStatic,
            'ip' : matcherIp,
            'IP_ADDRESS' : matcherPrefix,
            'ipv6' : matcherIp6,
            'IPV6_ADDRESS' : matcherPrefix6,
            'egress-validate' : nodeEgressValidate,
            'address' : nodeAddress,
            'EGRESS_ADDRESS' : IpGenAddrMatcher.IpGenAddrMatcher(
                                         helpdesc='Expected egress address' ),
            'OPTIONS' : LspOptionExprFactory( LspOptionType.ping,
                                              multiLabel=True )
          }
   handler = "MplsUtilCliHandler.handlerPingMplsStaticCmd"

BasicCli.EnableMode.addCommandClass( PingMplsStaticCmd )

#--------------------------------------------------------------------------------
# ping mpls nexthop-group name [entry] [options]
# XXXXX NO VRF
#--------------------------------------------------------------------------------
class PingMplsNhgCmd( CliCommand.CliCommandClass ):
   if Toggles.MplsUtilsToggleLib.toggleMplsUtilsNhgBackupEnabled():
      syntax = 'ping mpls nexthop-group NHG [ backup ] [ entry ENTRY ] [ OPTIONS ]'
      data = { 'ping' : NetworkToolsCli.pingMatcher,
               'mpls' : nodeMpls,
               'nexthop-group' : matcherNhg,
               'NHG' : matcherNexthopGroup,
               'entry' : matcherEntry,
               'ENTRY' : matcherEntryIndex,
               'OPTIONS' : LspOptionExprFactory( LspOptionType.ping,
                                                 multiLabel=True ),
               'backup' : matcherNhgEntryBackupPing
            }
   else:
      syntax = 'ping mpls nexthop-group NHG [ entry ENTRY ] [ OPTIONS ]'
      data = { 'ping' : NetworkToolsCli.pingMatcher,
               'mpls' : nodeMpls,
               'nexthop-group' : matcherNhg,
               'NHG' : matcherNexthopGroup,
               'entry' : matcherEntry,
               'ENTRY' : matcherEntryIndex,
               'OPTIONS' : LspOptionExprFactory( LspOptionType.ping,
                                                 multiLabel=True )
            }

   handler = "MplsUtilCliHandler.handlerPingMplsNhgCmd"

BasicCli.EnableMode.addCommandClass( PingMplsNhgCmd )
#--------------------------------------------------------------------------------
# ping mpls tunnel nexthop-group ENDPOINT [entry] [options]
#--------------------------------------------------------------------------------
class PingMplsNhgTunnelCmd( CliCommand.CliCommandClass ):
   syntax = 'ping mpls tunnel nexthop-group ENDPOINT [ entry ENTRY ] [ OPTIONS ]'
   data = { 'ping': NetworkToolsCli.pingMatcher,
            'mpls' : nodeMpls,
            'tunnel' : matcherTunnel,
            'nexthop-group' : matcherNhgForTunnel,
            'ENDPOINT': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                          ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                          ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
            'entry' : matcherEntry,
            'ENTRY' : matcherEntryIndex,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.ping, multiLabel=True,
                                              egressValidate=True )
          }
   handler = "MplsUtilCliHandler.handlerPingMplsNhgTunnelCmd"

BasicCli.EnableMode.addCommandClass( PingMplsNhgTunnelCmd )

#--------------------------------------------------------------------------------
# ping mpls srte endpoint ENDPOINT color COLOR [ traffic-af ( v4 | v6 ) ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingMplsSrTeCmd( CliCommand.CliCommandClass ):
   syntax = '''ping mpls srte endpoint ENDPOINT color COLOR
               [ traffic-af ( v4 | v6 ) ] [ OPTIONS ]'''
   data = { 'ping' : NetworkToolsCli.pingMatcher,
            'mpls' : nodeMpls,
            'srte' : matcherSrte,
            'endpoint' : matcherEndpoint,
            'ENDPOINT' : IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Endpoint '
                                                            'IP/IPv6 address' ),
            'color' : matcherColor,
            'COLOR' : matcherColorVal,
            'traffic-af' : matcherTrafficAf,
            'v4' : matcherV4,
            'v6' : matcherV6,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.ping,
                                              egressValidate=True ) }
   handler = "MplsUtilCliHandler.handlerPingMplsSrTeCmd"

BasicCli.EnableMode.addCommandClass( PingMplsSrTeCmd )

def tunSpecNameList( mode ):
   return [ tunSpec.tunnelName for tunSpec in rsvpLerCliClient.tunnelSpec.values() ]

# --------------------------------------------------------------------------------
# ping mpls rsvp tunnel TUNNEL_NAME [ sub-tunnel id SUB_TUNNEL_ID ] [ OPTIONS ]
# --------------------------------------------------------------------------------
class PingRsvpTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'ping mpls rsvp tunnel TUNNEL_NAME'
              '[ sub-tunnel id SUB_TUNNEL_ID ] [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'rsvp' : matcherRsvp,
      'tunnel' : matcherTunnel,
      'TUNNEL_NAME' : CliMatcher.DynamicNameMatcher( tunSpecNameList,
                      helpdesc='Tunnel name', pattern=r'.+' ),
      'sub-tunnel' : matcherSubTunnel,
      'id' : matcherSubTunnelId,
      'SUB_TUNNEL_ID' : matcherSubTunnelIdVal,
      'OPTIONS' : lspPingOptionsExpr,
   }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingRsvpTunnelCmd"

BasicCli.EnableMode.addCommandClass( PingRsvpTunnelCmd )

#--------------------------------------------------------------------------------
# ping mpls rsvp session ( ( id SESSION_ID [ lsp LSP ] ) |
#                          ( name SESSION_NAME ) ) [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingRsvpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'ping mpls rsvp session ( ( id SESSION_ID [ lsp LSP ] ) | '
                                       '( name SESSION_NAME ) ) [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'rsvp' : matcherRsvp,
      'session' : matcherSession,
      'id' : matcherId,
      'SESSION_ID' : matcherIdVal,
      'lsp' : matcherLsp,
      'LSP' : matcherLspVal,
      'name' : matcherName,
      'SESSION_NAME' : matcherNameVal,
      'OPTIONS' : lspPingOptionsExpr,
   }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingRsvpCmd"

BasicCli.EnableMode.addShowCommandClass( PingRsvpCmd )

#--------------------------------------------------------------------------------
# ping mpls pseudowire ( ( ldp <pw-name> ) |
#                        ( vpls ldp <instance-name> group <group-name>
#                          <neighbor-address> ) )
#                      [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingPseudowireLdpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''ping mpls pseudowire
               ( ( ldp PW_NAME ) |
                 ( vpls ldp INSTANCE_NAME group GROUP_NAME
                   NEIGHBOR_ADDRESS ) ) [ OPTIONS ]'''
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'pseudowire' : 'MPLS pseudowire',
      'ldp' : 'LDP pseudowire',
      'PW_NAME' : CliMatcher.PatternMatcher( '[!-~]+', helpname='WORD',
         helpdesc='LDP pseudowire name' ),
      'vpls' : 'VPLS pseudowire',
      'INSTANCE_NAME' : CliMatcher.PatternMatcher( '[!-~]+', helpname='WORD',
         helpdesc='VPLS instance name' ),
      'group' : 'VPLS pseudowire group',
      'GROUP_NAME' : CliMatcher.PatternMatcher( '[!-~]+', helpname='WORD',
         helpdesc='VPLS pseudowire group name' ),
      'NEIGHBOR_ADDRESS' : IpAddrMatcher.IpAddrMatcher( 'Neighbor IPv4 address' ),
      'OPTIONS' : lspPingOptionsExpr,
   }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingPseudowireLdpCmd"

BasicCli.EnableMode.addCommandClass( PingPseudowireLdpCmd )

#--------------------------------------------------------------------------------
# ping mpls bgp vpn vrf ( NON_DEFAULT | DEFAULT ) ( ip PREFIX | ipv6 PREFIX6 )
# [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingVpnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''ping mpls bgp vpn VRF
               ( ip PREFIX ) | ( ipv6 PREFIX6 ) [ OPTIONS ]'''
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'bgp' : matcherBgp,
      'vpn' : matcherVpn,
      'VRF' : vrfExprFactory,
      'ip' : matcherIp,
      'PREFIX' : matcherPrefix,
      'ipv6' : matcherIp6,
      'PREFIX6' : matcherPrefix6,
      'OPTIONS' : lspPingOptionsExpr,
   }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingVpnCmd"

BasicCli.EnableMode.addShowCommandClass( PingVpnCmd )

#--------------------------------------------------------------------------------
# ping mpls bgp labeled-unicast ( ip | ipv6 ) PREFIX
# [ bgp nexthop NEXTHOP [ label-stack { LABELS } ] ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingBgpLuCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''ping mpls bgp labeled-unicast
               ( ip PREFIX [ BGP_N nexthop NEXTHOP [ label-stack { LABELS } ] ] ) |
               ( ipv6 PREFIX6 [ BGP_N nexthop NEXTHOP6 [ label-stack { LABELS } ] ] )
               [ OPTIONS ]'''
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'bgp' : matcherBgp,
      'labeled-unicast' : matcherBgpLu,
      'ip' : matcherIp,
      'PREFIX' : matcherAddrOrPrefix,
      'ipv6' : matcherIp6,
      'PREFIX6' : matcherAddrOrPrefix6,
      'BGP_N' : matcherBgpNexthop,
      'nexthop' : matcherNexthop,
      'NEXTHOP' : matcherNexthopIpAddr,
      'NEXTHOP6' : matcherNexthopIp6Addr,
      'label-stack' : MplsCli.labelStackKeywordNode,
      'LABELS' : MplsCli.labelStackValNode,
      'OPTIONS' : lspPingOptionsExpr,
   }

   cliModel = "MplsUtilModel.MplsPing"
   handler = "MplsUtilCliHandler.handlerPingBgpLuCmd"

BasicCli.EnableMode.addShowCommandClass( PingBgpLuCmd )

#--------------------------------------------------------------------------------
# traceroute mpls [ VRF ] raw label LABELS ( ( nexthop NEXTHOP ) | ( MACADDR INTF ) )
#                                                                        [ OPTIONS ]
# IPv4 only at the moment
#--------------------------------------------------------------------------------
class TracerouteRawCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traceroute mpls [ VRF ] raw label { LABELS } '
                           '( ( nexthop NEXTHOP ) | ( MACADDR INTF ) ) [ OPTIONS ]' )
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'raw' : matcherRaw,
      'label' : nodeLabel,
      'LABELS' : Arnet.MplsLib.LabelValsWithExpNullExprFactory(),
      'nexthop' : matcherNexthop,
      'NEXTHOP' : matcherNexthopIpAddr,
      'MACADDR' : MacAddr.macAddrMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'OPTIONS' : lspTracerouteOptionsExpr
   }
   handler = "MplsUtilCliHandler.handlerTracerouteRawCmd"

BasicCli.EnableMode.addCommandClass( TracerouteRawCmd )

#--------------------------------------------------------------------------------
# traceroute mpls [ VRF ] ldp ip IP_ADDRESS [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteLdpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'traceroute mpls [ VRF ] ldp ip IP_ADDRESS [ OPTIONS ]'
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspTracerouteVrfExprFactory,
      'ldp' : matcherLdp,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'OPTIONS' : LspOptionExprFactory( LspOptionType.ldpTraceroute, dsType=True )
   }

   cliModel = "MplsUtilModel.MplsTraceroute"
   handler = "MplsUtilCliHandler.handlerTracerouteLdpCmd"

BasicCli.EnableMode.addShowCommandClass( TracerouteLdpCmd )

# --------------------------------------------------------------------------------
# traceroute mpls [ VRF ] generic ip IP_ADDRESS [ OPTIONS ]
# --------------------------------------------------------------------------------
class TracerouteGenericCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'traceroute mpls [ VRF ] generic ip IP_ADDRESS [ OPTIONS ]'
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspTracerouteVrfExprFactory,
      'generic' : matcherGeneric,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute, dsType=True )
   }

   cliModel = "MplsUtilModel.MplsTraceroute"
   handler = "MplsUtilCliHandler.handlerTracerouteGenericCmd"

BasicCli.EnableMode.addShowCommandClass( TracerouteGenericCmd )

#--------------------------------------------------------------------------------
# traceroute mpls [ VRF ] segment-routing
#    ( ( ip IP_ADDRESS ) | ( ipv6 IPV6_ADDRESS ) )
#    [ algorithm ALGO_NAME ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteSrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'traceroute mpls [ VRF ] segment-routing ( ( ( ( ip IP_ADDRESS )'
              ' | ( ipv6 IPV6_ADDRESS ) | ( isis ip IP_ADDRESS )'
              ' | ( isis ipv6 IPV6_ADDRESS ) ) [ algorithm ALGO_NAME ] )'
              ' | ( ospf ip IP_ADDRESS ) ) [ OPTIONS ]' )

   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspTracerouteVrfExprFactory,
      'segment-routing' : matcherSegmentRouting,
      'isis' : matcherSegmentTunnelIsis,
      'ospf' : matcherSegmentTunnelOspf,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'ipv6' : matcherIp6,
      'IPV6_ADDRESS' : matcherPrefix6,
      'algorithm' : matcherAlgorithm,
      'ALGO_NAME' : matcherFlexAlgoName,
      'OPTIONS' : lspTracerouteOptionsExpr,
   }

   cliModel = "MplsUtilModel.MplsTraceroute"
   handler = "MplsUtilCliHandler.handlerTracerouteSrCmd"

BasicCli.EnableMode.addShowCommandClass( TracerouteSrCmd )

#--------------------------------------------------------------------------------
# traceroute mpls static ( ip | ipv6 ) prefix [ nexthop <nexthop-ip> label <label> ]
# [ options ]
#--------------------------------------------------------------------------------
class TracerouteMplsStaticCmd( CliCommand.CliCommandClass ):
   syntax = '''traceroute mpls static
               ( ( ip IP_ADDRESS [ nexthop NEXTHOP label LABEL ] ) |
                 ( ipv6 IPV6_ADDRESS [ nexthop NEXTHOP6 label LABEL ] ) )
               [ egress-validate address EGRESS_ADDRESS ] [ OPTIONS ]'''
   data = { 'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'static' : matcherStatic,
            'ip' : matcherIp,
            'IP_ADDRESS' : matcherPrefix,
            'ipv6' : matcherIp6,
            'IPV6_ADDRESS' : matcherPrefix6,
            'nexthop' : matcherNexthop,
            'NEXTHOP' : matcherNexthopIpAddr,
            'NEXTHOP6' : matcherNexthopIp6Addr,
            'label' : nodeLabel,
            'LABEL' : matcherLabelValue,
            'egress-validate' : nodeEgressValidate,
            'address' : nodeAddress,
            'EGRESS_ADDRESS' : IpGenAddrMatcher.IpGenAddrMatcher(
                                         helpdesc='Expected egress address' ),
            'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                              multiLabel=True )
          }
   handler = "MplsUtilCliHandler.handlerTracerouteMplsStaticCmd"

BasicCli.EnableMode.addCommandClass( TracerouteMplsStaticCmd )

#--------------------------------------------------------------------------------
# traceroute mpls nexthop-group name entry entry-index [options]
# XXXXX NO VRF
#--------------------------------------------------------------------------------
class TracerouteMplsNhgCmd( CliCommand.CliCommandClass ):
   if Toggles.MplsUtilsToggleLib.toggleMplsUtilsNhgBackupEnabled():
      syntax = 'traceroute mpls nexthop-group NHG [ backup ] entry ENTRY [ OPTIONS ]'
      data = { 'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
               'mpls' : nodeMpls,
               'nexthop-group' : matcherNhg,
               'NHG' : matcherNexthopGroup,
               'entry' : matcherEntry,
               'ENTRY' : matcherEntryIndex,
               'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                                multiLabel=True ),
               'backup' : matcherNhgEntryBackupTraceroute
             }
   else:
      syntax = 'traceroute mpls nexthop-group NHG entry ENTRY [ OPTIONS ]'
      data = { 'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
               'mpls' : nodeMpls,
               'nexthop-group' : matcherNhg,
               'NHG' : matcherNexthopGroup,
               'entry' : matcherEntry,
               'ENTRY' : matcherEntryIndex,
               'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                                multiLabel=True )
             }

   handler = "MplsUtilCliHandler.handlerTracerouteMplsNhgCmd"

BasicCli.EnableMode.addCommandClass( TracerouteMplsNhgCmd )

#--------------------------------------------------------------------------------
# traceroute mpls tunnel nexthop-group ENDPOINT [entry] [options]
#--------------------------------------------------------------------------------
class TracerouteMplsNhgTunnelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traceroute mpls tunnel nexthop-group ENDPOINT' 
              '[ entry ENTRY ] [ OPTIONS ]' )
   data = { 'traceroute': NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'tunnel' : matcherTunnel,
            'nexthop-group' : matcherNhgForTunnel,
            'ENDPOINT': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                           ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                           ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
            'entry' : matcherEntry,
            'ENTRY' : matcherEntryIndex,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                              multiLabel=True,
                                              egressValidate=True )
          }
   handler = "MplsUtilCliHandler.handlerTracerouteMplsNhgTunnelCmd"

BasicCli.EnableMode.addCommandClass( TracerouteMplsNhgTunnelCmd )

#--------------------------------------------------------------------------------
# traceroute mpls srte endpoint ENDPOINT color COLOR [ traffic-af ( v4 | v6 ) ]
#                                                    [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteMplsSrTeCmd( CliCommand.CliCommandClass ):
   syntax = '''traceroute mpls srte endpoint ENDPOINT color COLOR
               [ traffic-af ( v4 | v6 ) ] [ OPTIONS ]'''
   data = { 'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'srte' : matcherSrte,
            'endpoint' : matcherEndpoint,
            'ENDPOINT' : IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Endpoint '
                                                            'IP/IPv6 address' ),
            'color' : matcherColor,
            'COLOR' : matcherColorVal,
            'traffic-af' : matcherTrafficAf,
            'v4' : matcherV4,
            'v6' : matcherV6,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                              multiLabel=False,
                                              egressValidate=True ) }
   handler = "MplsUtilCliHandler.handlerTracerouteMplsSrTeCmd"

BasicCli.EnableMode.addCommandClass( TracerouteMplsSrTeCmd )

#--------------------------------------------------------------------------------
# traceroute mpls rsvp tunnel TUNNEL_NAME [ sub-tunnel id SUB_TUNNEL_ID ] [ OPTIONS ]
# --------------------------------------------------------------------------------
class TracerouteRsvpTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'traceroute mpls rsvp tunnel TUNNEL_NAME'
              '[ sub-tunnel id SUB_TUNNEL_ID ] [ OPTIONS ]' )
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'rsvp' : matcherRsvp,
      'tunnel' : matcherTunnel,
      'TUNNEL_NAME' : CliMatcher.DynamicNameMatcher( tunSpecNameList,
                      helpdesc='Tunnel name', pattern=r'.+' ),
      'sub-tunnel' : matcherSubTunnel,
      'id' : matcherSubTunnelId,
      'SUB_TUNNEL_ID' : matcherSubTunnelIdVal,
      'OPTIONS' : lspTracerouteOptionsExpr,
   }

   cliModel = "MplsUtilModel.MplsTraceroute"
   handler = "MplsUtilCliHandler.handlerTracerouteRsvpTunnelCmd"

BasicCli.EnableMode.addShowCommandClass( TracerouteRsvpTunnelCmd )

# --------------------------------------------------------------------------------
# traceroute mpls rsvp session ( ( id SESSION_ID lsp LSP ) |
#                                ( name SESSION_NAME ) ) [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteRsvpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'traceroute mpls rsvp session ( ( id SESSION_ID lsp LSP ) | '
                                             '( name SESSION_NAME ) ) [ OPTIONS ]' )
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'rsvp' : matcherRsvp,
      'session' : matcherSession,
      'id' : matcherId,
      'SESSION_ID' : matcherIdVal,
      'lsp' : matcherLsp,
      'LSP' : matcherLspVal,
      'name' : matcherName,
      'SESSION_NAME' : matcherNameVal,
      'OPTIONS' : lspTracerouteOptionsExpr,
   }

   cliModel = "MplsUtilModel.MplsTraceroute"
   handler = "MplsUtilCliHandler.handlerTracerouteRsvpCmd"

BasicCli.EnableMode.addCommandClass( TracerouteRsvpCmd )

#--------------------------------------------------------------------------------
# traceroute mpls bgp labeled-unicast ( ip | ipv6 ) PREFIX
# [ bgp nexthop NEXTHOP [ label-stack { LABELS } ] ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteBgpLuCmd( CliCommand.CliCommandClass ):
   syntax = '''traceroute mpls bgp labeled-unicast
               ( ip PREFIX [ BGP_N nexthop NEXTHOP [ label-stack { LABELS } ] ] ) |
               ( ipv6 PREFIX6 [ BGP_N nexthop NEXTHOP6 [ label-stack { LABELS } ] ] )
               [ OPTIONS ]'''
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'bgp' : matcherBgp,
      'labeled-unicast' : matcherBgpLu,
      'ip' : matcherIp,
      'PREFIX' : matcherAddrOrPrefix,
      'ipv6' : matcherIp6,
      'PREFIX6' : matcherAddrOrPrefix6,
      'BGP_N' : matcherBgpNexthop,
      'nexthop' : matcherNexthop,
      'NEXTHOP' : matcherNexthopIpAddr,
      'NEXTHOP6' : matcherNexthopIp6Addr,
      'label-stack' : MplsCli.labelStackKeywordNode,
      'LABELS' : MplsCli.labelStackValNode,
      'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute, dsType=True )
   }
   handler = "MplsUtilCliHandler.handlerTracerouteBgpLuCmd"

BasicCli.EnableMode.addCommandClass( TracerouteBgpLuCmd )

# ------------------------------------------------------------------------------
# Mpls Utils configuration commands
# ------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] ip access-group ACL_NAME [ VRF ] [ in ]
#--------------------------------------------------------------------------------
class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME [ VRF ] [ in ]'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ VRF ] [ in ]'
   data = {
      'ip' : AclCli.ipKwForServiceAclMatcher,
      'access-group' : AclCli.accessGroupKwMatcher,
      'ACL_NAME' : AclCli.ipAclNameMatcher,
      'VRF' : AclCli.vrfKwAndNameExprFactory,
      'in' : AclCli.inKwMatcher,
   }
   handler = "MplsUtilCliHandler.handlerIpAccessGroupAclnameCmd"
   noOrDefaultHandler = \
      "MplsUtilCliHandler.noOrDefaultHandlerIpAccessGroupAclnameCmd"

MplsUtilsConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 access-group ACL_NAME [ VRF ] [ in ]
#--------------------------------------------------------------------------------
class Ipv6AccessGroupAclNameCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME [ VRF ] [ in ]'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ VRF ] [ in ]'
   data = {
      'ipv6' : AclCli.ipv6KwMatcherForServiceAcl,
      'access-group' : AclCli.accessGroupKwMatcher,
      'ACL_NAME' : AclCli.ip6AclNameMatcher,
      'VRF' : AclCli.vrfKwAndNameExprFactory,
      'in' : AclCli.inKwMatcher,
   }
   handler = "MplsUtilCliHandler.handlerIpv6AccessGroupAclNameCmd"
   noOrDefaultHandler = \
      "MplsUtilCliHandler.noOrDefaultHandlerIpv6AccessGroupAclNameCmd"

MplsUtilsConfigMode.addCommandClass( Ipv6AccessGroupAclNameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mpls oam
# ( ( standard ( arista | ietf ) )
# | ( downstream validation enabled ) )
#--------------------------------------------------------------------------------
class MplsOamStandardCmd( CliCommand.CliCommandClass ):
   syntax = '''mpls oam
            ( ( standard ( arista | ietf ) )
            | ( downstream validation enabled ) )'''
   noOrDefaultSyntax = syntax
   data = { 'mpls' : MplsCli.mplsNodeForConfig,
            'oam' : 'MPLS Operations, Administration, and Management configuration',
            'standard' : ( 'Set the default standard to comply with for '
                           'ping/traceroute commands' ),
            'arista' : ( 'Comply with the Arista standard for ping/traceroute, '
                         'which allows interworking with older EOS devices' ),
            'ietf' : 'Comply with the IETF standard',
          }
   _dataDsMap = { 'downstream' : matcherDownstream,
                  'validation' : matcherValidation,
                  'enabled' : matcherEnabled,
                }
   data.update( _dataDsMap )
   handler = "MplsUtilCliHandler.handlerMplsOamStandardCmd"
   noOrDefaultHandler = "MplsUtilCliHandler.noOrDefaultHandlerMplsOamStandardCmd"

BasicCli.GlobalConfigMode.addCommandClass( MplsOamStandardCmd )

# ------------------------------------------------------------------------------
# show mpls ping [ip|ipv6] access-list [ ACL_NAME ] [ summary ] [ dynamic] [ detail ]
# NB: ACL_NAME_AND_OPTS expands out to have extra options
# ------------------------------------------------------------------------------
class ShowMplsPingAcl( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls ping [ ip | ipv6 ] access-list [ ACL_NAME_AND_OPTS ]'
   data = {
      'mpls' : MplsCli.mplsForShowNode,
      'ping' : matcherPing,
      'ip' : AclCli.ipKwForShowServiceAcl,
      'ipv6' : AclCli.ipv6KwForShowServiceAcl,
      'access-list' : AclCli.accessListKwMatcherForServiceAcl,
      'ACL_NAME_AND_OPTS' : AclCli.ipOrIpv6AclNameExpression,

   }
   cliModel = AclCliModel.AllAclList
   handler = "MplsUtilCliHandler.handlerShowMplsPingAcl"

BasicCli.addShowCommandClass( ShowMplsPingAcl )

#----------------------------------------------------------------
# clear mpls ping counters [ ip | ipv6 ] access-list
#----------------------------------------------------------------
class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear mpls ping counters [ ip | ipv6 ] access-list'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'mpls' : MplsCli.mplsMatcherForClear,
      'ping' : matcherPing,
      'counters' : AclCli.countersKwMatcher,
      'ip' : AclCli.ipKwForClearServiceAclMatcher,
      'ipv6' : AclCli.ipv6KwMatcherForClearServiceAcl,
      'access-list' : AclCli.accessListKwMatcherForServiceAcl,
   }
   handler = "MplsUtilCliHandler.handlerClearIpAclCounters"

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

#--------------------------------------------------------------------------------
# [ no | default ] mpls icmp ipv4 source-interface SRC_INTF
# MPLS ICMP configuration command for ICMP pkts
#--------------------------------------------------------------------------------
class MplsIcmpSrcInterface( CliCommand.CliCommandClass ):
   data = {
      'mpls' : MplsCli.mplsNodeForConfig,
      'icmp' : 'MPLS ICMP configuration',
      'ipv6' : 'Source IP adress-family',
      'ip' : 'Source IP adress-family',
      'source-interface' : IraIpCli.sourceInterfaceMatcherForConfig,
      'SRC_INTF' : IntfCli.Intf.matcherWithIpSupport,
   }
   if ( Toggle.toggleMplsIcmpSrcIpCliEnabled() and
      Toggle.toggleMplsIcmpSrcIpCliV4Enabled() ):
      syntax = 'mpls icmp ( ip | ipv6 ) source-interface SRC_INTF'
      noOrDefaultSyntax = 'mpls icmp ( ip | ipv6 ) source-interface ...'
   elif Toggle.toggleMplsIcmpSrcIpCliEnabled():
      syntax = 'mpls icmp ipv6 source-interface SRC_INTF'
      noOrDefaultSyntax = 'mpls icmp ipv6 source-interface ...'
   elif Toggle.toggleMplsIcmpSrcIpCliV4Enabled():
      syntax = 'mpls icmp ip source-interface SRC_INTF'
      noOrDefaultSyntax = 'mpls icmp ip source-interface ...'

   handler = "MplsUtilCliHandler.handlerMplsIcmpSrcInterface"
   noOrDefaultHandler = \
         "MplsUtilCliHandler.noOrDefaultHandlerMplsIcmpSrcInterface"

if ( Toggle.toggleMplsIcmpSrcIpCliEnabled() or
   Toggle.toggleMplsIcmpSrcIpCliV4Enabled() ):
   BasicCli.GlobalConfigMode.addCommandClass( MplsIcmpSrcInterface )

#--------------------------------------------------------------------------------
# [ no | default ] mpls icmp ( fragmentation-needed | ttl-exceeded ) tunneling
# MPLS ICMP configuration commands
#
# Commands should follow the format:
# [no|default] mpls icmp <type of ICMP msg> <type of handling>
#--------------------------------------------------------------------------------

# Enable tunneling of ICMP 'Fragmentation Needed', 'Packet Too Big' messages
# [no|default] mpls icmp fragmentation-needed tunneling
def fragNeededIcmpTunnelingGuard( mode, token ):
   if mplsHwCapability.mplsFragNeededIcmpTunnelingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class MplsIcmpTunnelingCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls icmp ( fragmentation-needed | ttl-exceeded ) tunneling'
   noOrDefaultSyntax = syntax
   data = {
      'mpls' : MplsCli.mplsNodeForConfig,
      'icmp' : 'MPLS ICMP configuration',
      'fragmentation-needed' : CliCommand.guardedKeyword( 'fragmentation-needed',
         helpdesc='Fragmentation needed, or packet too big, for LSP',
         guard=fragNeededIcmpTunnelingGuard ),
      'ttl-exceeded' : 'Time-to-live exceeded in transit',
      'tunneling' : 'Enable MPLS tunneling',
   }
   handler = "MplsUtilCliHandler.handlerMplsIcmpTunnelingCmd"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( MplsIcmpTunnelingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mpls ping
#--------------------------------------------------------------------------------
class MplsPingCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ping'
   noOrDefaultSyntax = syntax
   data = {
      'mpls' : MplsCli.mplsNodeForConfig,
      'ping' : matcherPing,
   }
   handler = "MplsUtilCliHandler.handlerMplsPingCmd"
   noOrDefaultHandler = "MplsUtilCliHandler.noOrDefaultHandlerMplsPingCmd"

BasicCli.GlobalConfigMode.addCommandClass( MplsPingCmd )

def Plugin( entityManager ):
   global aclConfig
   global aclCpConfig
   global allNhgStatus
   global mplsHwCapability
   global flexAlgoConfig
   global rsvpLerCliClient

   aclConfig = ConfigMount.mount( entityManager, 'acl/config/cli',
                                  'Acl::Input::Config', 'w' )
   aclCpConfig = ConfigMount.mount( entityManager, 'acl/cpconfig/cli',
                                  'Acl::Input::CpConfig', 'w' )
   allNhgStatus = LazyMount.mount( entityManager, 
                                   'routing/nexthopgroup/status',
                                   'Routing::NexthopGroup::Status', 'r' )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       'routing/hardware/mpls/capability',
                                       'Mpls::Hardware::Capability',
                                       'r' )
   flexAlgoConfig = LazyMount.mount( entityManager, "te/flexalgo/config",
                                     "FlexAlgo::Config", "r" )
   rsvpLerCliClient = ConfigMount.mount( entityManager, 'mpls/rsvp/lerClient/cli',
                                         'Rsvp::RsvpLerClient', 'w' )
